# Script under AGPL 3.0 or later, by Zughy
from pathlib import Path
import subprocess
import tempfile
import yaml

PATH = Path(__file__).resolve().parent
LOCALE_PATH = PATH / "locale"
TEMPLATE = LOCALE_PATH / "template.pot"
FIELDS_TO_CHECK = ('name', 'description', 'hint')

LOCALE_PATH.mkdir(exist_ok=True)

def load_yaml(path: str) -> list[str]:
    with open(path, encoding='utf-8') as f:
        file = yaml.safe_load(f)
        ret = []
        for value in file.values():
            for field in FIELDS_TO_CHECK:
                if value.get(field):
                    ret.append(value[field])
        return ret

def translate_strings(strings: list[str]):
    # uso un file temporaneo per tradurre le stringhe da Python
    with tempfile.NamedTemporaryFile("w+", delete=False, suffix=".py") as temp_f:
        temp_f.writelines(f'_("{s}")\n' for s in strings)
        temp_path = Path(temp_f.name)

    subprocess.run(["xgettext", "--no-location", f"--output={str(TEMPLATE)}", str(temp_path)], check=True)
    temp_path.unlink()

def update_pot():
    strings = []
    for file in PATH.glob("*"):
        if file.suffix in (".yml", ".yaml"):
            strings.extend(load_yaml(file))
    translate_strings(strings)

def update_po():
    for file in LOCALE_PATH.glob("*.po"):
        subprocess.run(["msgmerge", "--update", "--backup=none", str(file), f"{str(TEMPLATE)}"], check=True)


try:
    update_pot()
    update_po()
except Exception as e:
    print(f"Error: {e}")