--[[
	Lag Compensation System - Artificial Movement
	System for creating artificial player movement for testing.
]]

local function start_artificial_movement(target_player) end
local function stop_artificial_movement(target_player) end
local function cleanup_artificial_movement(name) end

local get_player_by_name = core.get_player_by_name
local core_log = core.log
local core_after = core.after

local MOVEMENT_DIRECTION_CHANGES = 6
local MOVEMENT_PUSH_INTERVAL = 0.5
local MOVEMENT_VELOCITY_STRENGTH = 20



----------------------------------------------
------------------PUBLIC API------------------
----------------------------------------------

-- Set artificial movement for a target player
function weapons_lib.lagcomp.debug_set_artificial_movement(target_player, enable)
    if not weapons_lib.lagcomp.LAG_COMP_ENABLED then
        return false
    end

    if enable then
        return start_artificial_movement(target_player)
    else
        stop_artificial_movement(target_player)
        return true
    end
end



-- Get artificial movement status for a target player
function weapons_lib.lagcomp.debug_get_artificial_movement(target_player)
    if not weapons_lib.lagcomp.LAG_COMP_ENABLED then
        return false
    end

    local p_name = target_player:get_player_name()
    local data = weapons_lib.lagcomp.artificial_movement[p_name]
    return data and data.active or false
end



----------------------------------------------
----------LOCAL FUNCTION IMPLEMENTATIONS-----
----------------------------------------------

-- Start artificial movement for a target player
function start_artificial_movement(target_player)
    local p_name = target_player:get_player_name()

    cleanup_artificial_movement(p_name)

    -- Initialize movement data
    weapons_lib.lagcomp.artificial_movement[p_name] = {
        active = true,
        direction = 1,  -- 1 for forward, -1 for backward
        push_count = 0, -- Track number of pushes for direction timing
        timer = nil
    }

    -- Start movement timer with constant pushes
    local function apply_movement_push()
        local data = weapons_lib.lagcomp.artificial_movement[p_name]
        if not data or not data.active then
            return
        end

        local player = get_player_by_name(p_name)
        if not player then
            cleanup_artificial_movement(p_name)
            return
        end

        local look_dir = player:get_look_dir()

        -- Increment push counter and change direction periodically
        data.push_count = data.push_count + 1
        if data.push_count >= MOVEMENT_DIRECTION_CHANGES then
            data.direction = -data.direction
            data.push_count = 0
        end

        local velocity_push = {
            x = look_dir.x * data.direction * MOVEMENT_VELOCITY_STRENGTH,
            y = 0,
            z = look_dir.z * data.direction * MOVEMENT_VELOCITY_STRENGTH
        }

        -- Apply controlled pushes using set_velocity to avoid stacking
        player:add_velocity(velocity_push)

        -- Schedule next push
        data.timer = core_after(MOVEMENT_PUSH_INTERVAL, apply_movement_push)
    end

    apply_movement_push()

    core_log("action", "[WeaponsLib] Started artificial movement for player " .. p_name)
    return true
end



function stop_artificial_movement(target_player)
    local p_name = target_player:get_player_name()
    cleanup_artificial_movement(p_name)
    core_log("action", "[WeaponsLib] Stopped artificial movement for player " .. p_name)
end



function cleanup_artificial_movement(name)
    local data = weapons_lib.lagcomp.artificial_movement[name]
    if not data then
        return
    end

    if data.timer then
        data.timer:cancel()
    end

    weapons_lib.lagcomp.artificial_movement[name] = nil
end
