--[[
	Lag Compensation System - Debug API
	Debug functions and utilities for testing lag compensation.
]]

local get_player_by_name = core.get_player_by_name

local PARTICLE_LIFETIME = 1.0
local PARTICLE_OFFSET_FACTOR = 0.5
local PARTICLE_SPREAD_FACTOR = 0.8



----------------------------------------------
--------------DEBUG FUNCTIONS----------------
----------------------------------------------

-- Cache debug mode for a player to avoid metadata reads during raycast
local function update_debug_mode_cache(p_name)
  local player = get_player_by_name(p_name)
  if player then
    weapons_lib.lagcomp.debug_mode_cache[p_name] = player:get_meta():get_string("lagcomp_debug") == "true"
  else
    weapons_lib.lagcomp.debug_mode_cache[p_name] = nil
  end
end



-- Get cached debug mode state
function weapons_lib.lagcomp.is_debug_enabled(p_name)
  local cached = weapons_lib.lagcomp.debug_mode_cache[p_name]
  if not cached then
    update_debug_mode_cache(p_name)
    return weapons_lib.lagcomp.debug_mode_cache[p_name] or false
  end
  return cached
end



-- Create visual debug indicator at historical position
function weapons_lib.lagcomp.create_debug_indicator(historical_pos, target_player, indicator_type)
  local player_props = target_player:get_properties()
  local collisionbox = player_props.collisionbox or weapons_lib.lagcomp.DEFAULT_COLLISIONBOX
  local eye_height = player_props.eye_height or weapons_lib.lagcomp.DEFAULT_EYE_HEIGHT

  -- historical_pos is already at eye height, calculate feet position
  local feet_pos = vector.subtract(historical_pos, {x = 0, y = eye_height, z = 0})

  local box_center = {
    x = feet_pos.x,
    y = feet_pos.y + (collisionbox[5] - collisionbox[2]) / 2,     -- Middle of hitbox height
    z = feet_pos.z
  }

  local particle_count, main_color, center_color

  if indicator_type == "raw" then
    -- Black/white particles for raw snapshots
    particle_count = 15
    main_color = function() return math.random() > PARTICLE_OFFSET_FACTOR and "white" or "black" end
    center_color = math.random() > PARTICLE_OFFSET_FACTOR and "white" or "black"
  else
    -- Red particles for interpolated positions (what the system actually uses)
    particle_count = 20
    main_color = function() return "red" end
    center_color = "red"
  end

  -- Spawn particles in a box pattern to show the hitbox
  for i = 1, particle_count do
    local width = collisionbox[4] - collisionbox[1]      -- x dimension
    local height = collisionbox[5] - collisionbox[2]     -- y dimension
    local depth = collisionbox[6] - collisionbox[3]      -- z dimension

    -- Generate random offsets that stay within the hitbox bounds
    local offset_x = (math.random() - PARTICLE_OFFSET_FACTOR) * width * PARTICLE_SPREAD_FACTOR
    local offset_y = (math.random() - PARTICLE_OFFSET_FACTOR) * height * PARTICLE_SPREAD_FACTOR
    local offset_z = (math.random() - PARTICLE_OFFSET_FACTOR) * depth * PARTICLE_SPREAD_FACTOR

    local particle_pos = vector.add(box_center, {x = offset_x, y = offset_y, z = offset_z})

    -- Create texture based on type
    local color = main_color()
    local texture = "[fill:4x4:" .. color

    core.add_particle({
      pos = particle_pos,
      velocity = {x = 0, y = 0, z = 0},
      acceleration = {x = 0, y = 0, z = 0},
      expirationtime = PARTICLE_LIFETIME,
      size = 8,
      collisiondetection = false,
      texture = texture,
      glow = 8,
    })
  end

  -- central marker particle
  local center_texture = "[fill:6x6:" .. center_color

  core.add_particle({
    pos = box_center,
    velocity = {x = 0, y = 0, z = 0},
    acceleration = {x = 0, y = 0, z = 0},
    expirationtime = PARTICLE_LIFETIME,
    size = 10,
    collisiondetection = false,
    texture = center_texture,
    glow = 12,
  })
end



----------------------------------------------
------------------PUBLIC API------------------
----------------------------------------------

-- Debug API functions
function weapons_lib.lagcomp.debug_set_simulated_ping(p_name, ping_ms)
  if not weapons_lib.lagcomp.LAG_COMP_ENABLED then
    return false, "Lag compensation is disabled"
  end

  local player = get_player_by_name(p_name)
  if not player then
    return false, "Player not found: " .. p_name
  end

  if ping_ms == 0 then
    -- Remove simulation
    if weapons_lib.lagcomp.debug_simulation[p_name] then
      weapons_lib.lagcomp.debug_simulation[p_name] = nil
    end
    return true, "Removed lag simulation for " .. p_name
  else
    -- Set up simulation
    if not weapons_lib.lagcomp.debug_simulation[p_name] then
      weapons_lib.lagcomp.debug_simulation[p_name] = {}
    end

    weapons_lib.lagcomp.debug_simulation[p_name].ping = ping_ms

    return true, string.format("Set simulated ping to %dms for %s", ping_ms, p_name)
  end
end



function weapons_lib.lagcomp.debug_get_simulated_ping(p_name)
  if weapons_lib.lagcomp.debug_simulation[p_name] then
    return weapons_lib.lagcomp.debug_simulation[p_name].ping
  end
end



-- Debug function to simulate lower server steps (larger intervals between snapshots)
function weapons_lib.lagcomp.debug_set_simulated_step(step_seconds)
  if not weapons_lib.lagcomp.LAG_COMP_ENABLED then
    return false, "Lag compensation is disabled"
  end

  if step_seconds == 0 then
    weapons_lib.lagcomp.debug_simulated_step = 0
    return true, "Disabled simulated server step - using normal snapshot frequency"
  else
    weapons_lib.lagcomp.debug_simulated_step = step_seconds
    weapons_lib.lagcomp.last_snapshot_time = 0     -- Reset timer
    return true, string.format("Set simulated server step to %.3fs (snapshots every %.3fs)", step_seconds, step_seconds)
  end
end



function weapons_lib.lagcomp.debug_get_simulated_step()
  return weapons_lib.lagcomp.debug_simulated_step
end



-- Toggle debug mode for a player and update cache
function weapons_lib.lagcomp.debug_set_mode(player, enable)
  if not weapons_lib.lagcomp.LAG_COMP_ENABLED then
    return false
  end

  local p_name = player:get_player_name()
  player:get_meta():set_string("lagcomp_debug", enable and "true" or "false")

  update_debug_mode_cache(p_name)

  return true
end



-- Get debug mode status for a player
function weapons_lib.lagcomp.debug_get_mode(player)
  local p_name = player:get_player_name()
  return weapons_lib.lagcomp.debug_mode_cache[p_name] or false
end
