--[[
	Lag Compensation System - Main API
	Main API functions and globalstep registration.
]]

-- Cache references for performance
local get_us_time = core.get_us_time
local get_connected_players = core.get_connected_players
local get_player_by_name = core.get_player_by_name
local get_player_information = core.get_player_information
local core_log = core.log
local table_insert = table.insert
local pairs = pairs
local ipairs = ipairs
local vector_distance = vector.distance

local LAG_COMP_ENABLED = weapons_lib.lagcomp.LAG_COMP_ENABLED
local MIN_PING_THRESHOLD = 0.1 -- min 100ms
local DEFAULT_PLAYER_EYE_HEIGHT = 1.475
local RECENT_INTERPOLATION_THRESHOLD = weapons_lib.lagcomp.RECENT_INTERPOLATION_THRESHOLD



-- Main function for lag-compensated hit detection
function weapons_lib.get_compensated_hit(shooter, ray_origin, ray_dir, ray_length, pierce_enabled)
  if not LAG_COMP_ENABLED then return end

  pierce_enabled = pierce_enabled or false

  local shooter_name = shooter:get_player_name()
  local debug_enabled = weapons_lib.lagcomp.is_debug_enabled(shooter_name)

  weapons_lib.print_debug(debug_enabled, shooter_name, "=== LAG COMPENSATION DEBUG ===")

  -- Get shooter's ping information (check debug simulation first)
  local avg_rtt
  local debug_sim = weapons_lib.lagcomp.debug_simulation[shooter_name]
  if debug_sim and debug_sim.ping > 0 then
    avg_rtt = debug_sim.ping / 1000     -- ms to seconds
    weapons_lib.print_debug(debug_enabled, shooter_name,
      string.format("Using SIMULATED ping: %.1fms", debug_sim.ping))
  else
    local player_info = get_player_information(shooter_name)
    if not player_info or not player_info.avg_rtt then
      weapons_lib.print_debug(debug_enabled, shooter_name, "No ping information available")
      return       -- No ping information available
    end
    avg_rtt = player_info.avg_rtt
  end
  weapons_lib.print_debug(debug_enabled, shooter_name,
    string.format("Ping %.1fms, rewinding %.3fs", avg_rtt * 1000, avg_rtt / 2))

  -- If ping is too low, lag compensation won't provide significant benefit
  if avg_rtt < MIN_PING_THRESHOLD then
    weapons_lib.print_debug(debug_enabled, shooter_name, "Ping too low, using standard raycast")
    return
  end

  -- Calculate rewind time (half of RTT)
  local current_time = get_us_time() / 1000000   -- microseconds to seconds
  local rewind_time = current_time - (avg_rtt / 2)

  weapons_lib.print_debug(debug_enabled, shooter_name, string.format("LAGCOMP: Shot at %.6fs, rewind to %.6fs (RTT: %.3fms)",
    current_time, rewind_time, avg_rtt * 1000))

  -- Show timing relationship with simulated step
  if debug_enabled then
    if weapons_lib.lagcomp.debug_simulated_step > 0 then
      weapons_lib.print_debug(debug_enabled, shooter_name,
        string.format("TIMING: Rewind is %.3fs ago, but snapshots every %.3fs",
          current_time - rewind_time, weapons_lib.lagcomp.debug_simulated_step))
    end
  end

  -- Collect historical positions of potential targets from all connected players
  local targets_with_hitboxes = {}
  local targets_found = 0

  for p_name, _ in pairs(weapons_lib.lagcomp.player_history) do
    if p_name ~= shooter_name then
      local target_player = get_player_by_name(p_name)
      if target_player then
        local historical_pos
        if debug_enabled then
          historical_pos = weapons_lib.lagcomp.get_interpolated_position_and_print_debug(p_name, rewind_time, shooter_name)
        else
          historical_pos = weapons_lib.lagcomp.get_interpolated_position(weapons_lib.lagcomp.player_history[p_name], rewind_time)
        end
        if historical_pos then
          -- Pre-pass filter by distance before intersection test
          local dist_to_ray_origin = vector_distance(ray_origin, historical_pos)
          if dist_to_ray_origin <= ray_length then
            targets_found = targets_found + 1

            -- Debug: Check if we're extrapolating vs interpolating
            if debug_enabled then
              local history = weapons_lib.lagcomp.player_history[p_name]
              if history and #history > 0 then
                local latest_snapshot_time = history[#history].time
                local oldest_snapshot_time = history[1].time

                if rewind_time > latest_snapshot_time then
                  local time_diff = rewind_time - latest_snapshot_time
                  weapons_lib.print_debug(debug_enabled, p_name,
                    string.format("Method: EXTRAPOLATION (%.3fs beyond latest snapshot)",
                      time_diff))
                elseif rewind_time < oldest_snapshot_time then
                  local time_diff = oldest_snapshot_time - rewind_time
                  weapons_lib.print_debug(debug_enabled, p_name,
                    string.format("Method: EXTRAPOLATION (%.3fs before oldest snapshot)",
                      time_diff))
                else
                  local latest_snapshot = history[#history]
                  local time_from_latest = math.abs(rewind_time - latest_snapshot.time)
                  if time_from_latest <= RECENT_INTERPOLATION_THRESHOLD then
                    weapons_lib.print_debug(debug_enabled, p_name, "Method: LINEAR INTERPOLATION (recent data)")
                  else
                    weapons_lib.print_debug(debug_enabled, p_name, "Method: HERMITE SPLINE INTERPOLATION (older data)")
                  end
                end
              end
            end

            -- Prepare target data for interpolated position
            local target_data = {
              player = target_player,
              historical_pos = historical_pos
            }

            table.insert(targets_with_hitboxes, target_data)

            if debug_enabled then
              local current_pos = target_player:get_pos()
              -- Adjust current position to eye height for fair comparison
              current_pos.y = current_pos.y + (target_player:get_properties().eye_height or DEFAULT_PLAYER_EYE_HEIGHT)
              local distance_moved = vector.distance(current_pos, historical_pos)
              weapons_lib.print_debug(debug_enabled, p_name,
                string.format("Target '%s' moved %.2f nodes in %.3fs", p_name, distance_moved, avg_rtt / 2))

              -- Show interpolation method used
              local history = weapons_lib.lagcomp.player_history[p_name]
              if history and #history > 1 then
                local velocity = target_player:get_velocity()
                weapons_lib.print_debug(debug_enabled, p_name,
                  string.format("Velocity: %.2f nodes/s (%.2f,%.2f,%.2f)",
                    vector.length(velocity), velocity.x, velocity.y, velocity.z))
              end

              -- Find the closest raw snapshot for comparison
              local closest_raw_snapshot = nil
              local closest_time_diff = math.huge

              if history then
                for _, snapshot in ipairs(history) do
                  local time_diff = math.abs(snapshot.time - rewind_time)
                  if time_diff < closest_time_diff then
                    closest_time_diff = time_diff
                    closest_raw_snapshot = snapshot
                  end
                end
              end

              -- Show raw snapshot position with black/white particles
              if closest_raw_snapshot then
                local distance_moved = vector.distance(closest_raw_snapshot.pos, historical_pos)
                weapons_lib.print_debug(debug_enabled, p_name,
                  string.format("Closest raw snapshot: (%.2f, %.2f, %.2f) [%.3fs off]",
                    closest_raw_snapshot.pos.x, closest_raw_snapshot.pos.y, closest_raw_snapshot.pos.z, closest_time_diff))
                weapons_lib.print_debug(debug_enabled, p_name,
                  string.format("RAW vs INTERP: %.3f nodes apart", distance_moved))

                -- Show the effect of simulated server step
                if weapons_lib.lagcomp.debug_simulated_step > 0 then
                  weapons_lib.print_debug(debug_enabled, p_name,
                    string.format("SIMULATED STEP: %.3fs interval affects interpolation accuracy", weapons_lib.lagcomp.debug_simulated_step))
                end

                weapons_lib.lagcomp.create_debug_indicator(closest_raw_snapshot.pos, target_player, "raw")
              end

              -- Show interpolated position with red particles
              weapons_lib.lagcomp.create_debug_indicator(historical_pos, target_player, "interpolated")
            end
          else
            weapons_lib.print_debug(debug_enabled, p_name,
              string.format("Skipping '%s' (%.2fm away, ray length %.2fm)", p_name, dist_to_ray_origin, ray_length))
          end
        else
          weapons_lib.print_debug(debug_enabled, p_name,
            string.format("Failed to get historical position for '%s'", p_name))
        end
      end
    end
  end

  -- If no valid targets, return nil
  if #targets_with_hitboxes == 0 then
    weapons_lib.print_debug(debug_enabled, shooter_name,
      string.format("No valid targets found (%d tracked players)", targets_found))
    return
  end

  weapons_lib.print_debug(debug_enabled, shooter_name,
    string.format("Testing %d targets with 3x3 grid virtual raycast", #targets_with_hitboxes))

  -- Perform 3x3 grid virtual raycast to get all hits (assisted aim like standard raycast)
  local all_hits = weapons_lib.lagcomp.perform_grid_virtual_raycast(shooter, ray_origin, ray_dir, ray_length, targets_with_hitboxes, debug_enabled, p_name)

  if #all_hits > 0 then
    weapons_lib.print_debug(debug_enabled, shooter_name,
      string.format("Found %d potential hits with lag compensation", #all_hits))
    if debug_enabled then
      for i, hit in ipairs(all_hits) do
        local target_name = hit.object:get_player_name()
        weapons_lib.print_debug(debug_enabled, shooter_name,
          string.format("%d. Target: '%s' at %.2fm", i, target_name, hit.distance))
      end
    end

    -- Convert hits to format expected by weapons_lib.apply_damage
    -- If piercing is not enabled, only return the first (closest) hit
    local formatted_hits = {}
    local hits_to_process = pierce_enabled and all_hits or {all_hits[1]}

    if pierce_enabled then
      weapons_lib.print_debug(debug_enabled, shooter_name,
        string.format("Weapon supports piercing, returning %d hits", #all_hits))
    else
      weapons_lib.print_debug(debug_enabled, shooter_name,
        "Weapon does not support piercing, returning only closest hit")
    end

    for _, hit in ipairs(hits_to_process) do
      table_insert(formatted_hits, {
        object = hit.object,
        hit_point = hit.intersection_point
      })
    end

    return formatted_hits
  else
    weapons_lib.print_debug(debug_enabled, shooter_name, "No hits detected with lag compensation")
  end
end





----------------------------------------------
-------------GLOBALSTEP-----------------------
----------------------------------------------

-- Globalstep to capture player positions at proper intervals
if LAG_COMP_ENABLED then
  core.register_globalstep(function(dtime)
    local current_time = get_us_time() / 1000000

    -- Check if we should simulate a lower server step frequency
    if weapons_lib.lagcomp.debug_simulated_step > 0 then
      if current_time - weapons_lib.lagcomp.last_snapshot_time >= weapons_lib.lagcomp.debug_simulated_step then
        weapons_lib.lagcomp.last_snapshot_time = current_time

        -- Regular snapshots for real players
        for _, player in ipairs(get_connected_players()) do
          weapons_lib.lagcomp.add_snapshot(player)
        end
      end
    else
      -- Regular snapshots for real players
      for _, player in ipairs(get_connected_players()) do
        weapons_lib.lagcomp.add_snapshot(player)
      end
    end

    -- STRESS TEST: Simulate additional players if stress test is active
    if weapons_lib.lagcomp.debug_stress_test_active then
      weapons_lib.lagcomp.debug_simulate_stress_snapshots()
    end
  end)

  -- Handle player disconnection cleanup
  core.register_on_leaveplayer(function(player)
    local p_name = player:get_player_name()

    if weapons_lib.lagcomp.player_history[p_name] then
      weapons_lib.lagcomp.player_history[p_name] = nil
    end

    weapons_lib.lagcomp.debug_mode_cache[p_name] = nil

    if weapons_lib.lagcomp.debug_simulation[p_name] then
      weapons_lib.lagcomp.debug_simulation[p_name] = nil
    end

    if weapons_lib.lagcomp.artificial_movement[p_name] then
      weapons_lib.lagcomp.artificial_movement[p_name] = nil
    end
  end)

  core_log("action", "[WeaponsLib] Lag compensation system is enabled.")
end
