--[[
	Lag Compensation System - Stress Test
	Stress testing functions for performance analysis.
]]

-- Cache references for performance
local get_us_time = core.get_us_time
local table_insert = table.insert
local table_remove = table.remove
local vector_add = vector.add
local vector_subtract = vector.subtract
local pairs = pairs

-- Constants
local STRESS_TEST_PLAYER_COUNT = 100
local STRESS_TEST_SHOOT_CHANCE = 0.5 -- 50% of players shoot each step
local STRESS_TEST_MISS_CHANCE = 0.7  -- 70% chance shots miss due to aim inaccuracy
local DEFAULT_EYE_HEIGHT = weapons_lib.lagcomp.DEFAULT_EYE_HEIGHT
local DEFAULT_COLLISIONBOX = weapons_lib.lagcomp.DEFAULT_COLLISIONBOX
local HISTORY_DURATION = weapons_lib.lagcomp.HISTORY_DURATION

local SIMULATED_DTIME = 0.1           -- Simulated dtime for movement updates
local SIMULATED_PING_MS = 0.05        -- 50ms simulated ping for stress test
local RAY_NOISE_FACTOR = 0.5          -- For centering random() output (-0.5 to 0.5)
local AIM_INACCURACY_MULTIPLIER = 0.4 -- Ray direction noise factor for miss simulation

local stress_test_simulated_players = {}
local stress_test_step_count = 0
local stress_test_timing_data = nil



function weapons_lib.lagcomp.debug_generate_stress_players()
  stress_test_simulated_players = {}
  for i = 1, STRESS_TEST_PLAYER_COUNT do
    local p_name = "stress_test_player_" .. i
    stress_test_simulated_players[p_name] = {
      pos = {x = math.random(-100, 100), y = math.random(0, 50), z = math.random(-100, 100)},
      velocity = {x = math.random(-5, 5), y = math.random(-2, 2), z = math.random(-5, 5)},
      eye_height = DEFAULT_EYE_HEIGHT
    }
  end

  -- Reset performance counter
  stress_test_step_count = 0
  stress_test_timing_data = nil
end



-- Simulate stress test snapshots (called from globalstep)
function weapons_lib.lagcomp.debug_simulate_stress_snapshots()
  local total_start_time = core.get_us_time()
  local current_time = get_us_time() / 1000000

  local interpolation_count = 0
  local intersection_count = 0
  local hit_count = 0
  local interpolation_time_ms = 0
  local intersection_start_time = 0
  local intersection_time_ms = 0
  local snapshot_time_ms = 0

  -- Start timing for snapshots
  local snapshot_start_time = core.get_us_time()
  for p_name, player_data in pairs(stress_test_simulated_players) do
    player_data.pos.x = player_data.pos.x + player_data.velocity.x * SIMULATED_DTIME
    player_data.pos.y = player_data.pos.y + player_data.velocity.y * SIMULATED_DTIME
    player_data.pos.z = player_data.pos.z + player_data.velocity.z * SIMULATED_DTIME

    local eye_pos = {
      x = player_data.pos.x,
      y = player_data.pos.y + player_data.eye_height,
      z = player_data.pos.z
    }

    local history = weapons_lib.lagcomp.player_history[p_name]
    if not history then
      history = {}
      weapons_lib.lagcomp.player_history[p_name] = history
    end

    -- Create snapshot (exactly like the real add_snapshot function)
    table_insert(history, {
      time = current_time,
      pos = eye_pos,
      velocity = player_data.velocity
    })

    -- Clean up old history entries (exactly like the real function)
    local cutoff_time = current_time - HISTORY_DURATION
    while history[1] and history[1].time < cutoff_time do
      table_remove(history, 1)
    end
  end
  local snapshot_end_time = core.get_us_time()
  snapshot_time_ms = (snapshot_end_time - snapshot_start_time) / 1000

  local interpolation_start_time = core.get_us_time()

  -- This simulates what happens when players shoot - ray-AABB intersection tests
  local fake_ray_length = 40
  local fake_rewind_time = current_time - SIMULATED_PING_MS

  -- Generate shooting decision for each player (like original single loop)
  local shooting_players = {}
  for i = 1, STRESS_TEST_PLAYER_COUNT do
    if math.random() <= STRESS_TEST_SHOOT_CHANCE then
      shooting_players[i] = true
    end
  end

  -- INTERPOLATION PHASE
  for i = 1, STRESS_TEST_PLAYER_COUNT do
    if shooting_players[i] then
      local fake_ray_dir = {
        x = math.random() - RAY_NOISE_FACTOR,
        y = math.random() - RAY_NOISE_FACTOR,
        z = math.random() - RAY_NOISE_FACTOR
      }
      fake_ray_dir = vector.normalize(fake_ray_dir)

      -- Simulate interpolation for each potential target (like get_compensated_hit does)
      for target_name, _ in pairs(stress_test_simulated_players) do
        if target_name ~= ("stress_test_player_" .. i) then
          local target_history = weapons_lib.lagcomp.player_history[target_name]
          if target_history then
            local interpolated_pos = weapons_lib.lagcomp.get_interpolated_position(target_history, fake_rewind_time)
            if interpolated_pos then
              interpolation_count = interpolation_count + 1
            end
          end
        end
      end
    end
  end

  local interpolation_end_time = core.get_us_time()
  interpolation_time_ms = (interpolation_end_time - interpolation_start_time) / 1000

  intersection_start_time = core.get_us_time()

  -- INTERSECTION PHASE - use same shooting players
  for i = 1, STRESS_TEST_PLAYER_COUNT do
    if shooting_players[i] then
      local shooter_name = "stress_test_player_" .. i
      local shooter_data = stress_test_simulated_players[shooter_name]
      if shooter_data then
        local ray_origin = {
          x = shooter_data.pos.x,
          y = shooter_data.pos.y + shooter_data.eye_height,
          z = shooter_data.pos.z
        }

        -- Simulate intersection tests for each potential target
        for target_name, target_data in pairs(stress_test_simulated_players) do
          if target_name ~= shooter_name then
            local target_history = weapons_lib.lagcomp.player_history[target_name]
            if target_history then
              local interpolated_pos = weapons_lib.lagcomp.get_interpolated_position(target_history, fake_rewind_time)

              if interpolated_pos then
                local ray_dir = {
                  x = interpolated_pos.x - ray_origin.x,
                  y = interpolated_pos.y - ray_origin.y,
                  z = interpolated_pos.z - ray_origin.z
                }

                local length = vector.length(ray_dir)
                if length == 0 then
                  -- Skip if same position
                else
                  ray_dir = vector.normalize(ray_dir)

                  -- Add some randomness to simulate aim inaccuracy
                  if math.random() < STRESS_TEST_MISS_CHANCE then
                    ray_dir.x = ray_dir.x + (math.random() - RAY_NOISE_FACTOR) * AIM_INACCURACY_MULTIPLIER
                    ray_dir.y = ray_dir.y + (math.random() - RAY_NOISE_FACTOR) * AIM_INACCURACY_MULTIPLIER
                    ray_dir.z = ray_dir.z + (math.random() - RAY_NOISE_FACTOR) * AIM_INACCURACY_MULTIPLIER
                    ray_dir = vector.normalize(ray_dir)
                  end

                  -- Calculate AABB bounds for the interpolated position
                  local feet_pos = vector_subtract(interpolated_pos, {x = 0, y = target_data.eye_height, z = 0})
                  local box_min = vector_add(feet_pos, {x = DEFAULT_COLLISIONBOX[1], y = DEFAULT_COLLISIONBOX[2], z = DEFAULT_COLLISIONBOX[3]})
                  local box_max = vector_add(feet_pos, {x = DEFAULT_COLLISIONBOX[4], y = DEFAULT_COLLISIONBOX[5], z = DEFAULT_COLLISIONBOX[6]})

                  local distance = weapons_lib.lagcomp.ray_aabb(ray_origin, ray_dir, fake_ray_length, box_min, box_max)
                  intersection_count = intersection_count + 1
                  if distance then
                    hit_count = hit_count + 1
                    break                     -- Early exit when first hit is found (like real grid raycast)
                  end
                end
              end
            end
          end
        end
      end

      local intersection_end_time = core.get_us_time()
      intersection_time_ms = (intersection_end_time - intersection_start_time) / 1000

      local total_end_time = core.get_us_time()
      local total_time_ms = (total_end_time - total_start_time) / 1000

      -- Store timing data for summary output
      stress_test_step_count = stress_test_step_count + 1
      stress_test_timing_data = {
        snapshot_time_ms = snapshot_time_ms,
        interpolation_time_ms = interpolation_time_ms,
        intersection_time_ms = intersection_time_ms,
        total_time_ms = total_time_ms,
        interpolation_count = interpolation_count,
        intersection_count = intersection_count,
        hit_count = hit_count
      }

      -- Summary output every 5 seconds
      if stress_test_step_count % 500 == 0 then
        local data = stress_test_timing_data
        local message = string.format(
          "[STRESS TEST] Step %d: Total: %.2fms | Snapshots: %.2fms | Interpolation: %.2fms (%d calcs) | Intersection: %.2fms (%d tests, %d hits)",
          stress_test_step_count,
          data.total_time_ms or 0,
          data.snapshot_time_ms or 0,
          data.interpolation_time_ms or 0, data.interpolation_count or 0,
          data.intersection_time_ms or 0, data.intersection_count or 0, data.hit_count or 0
        )
        core.chat_send_all(message)
        core.log("action", "[WeaponsLib] " .. message)
      end
    end
  end
end



-- Clean up all stress test data
function weapons_lib.lagcomp.debug_cleanup_stress_test()
  for p_name, _ in pairs(stress_test_simulated_players) do
    if weapons_lib.lagcomp.player_history[p_name] then
      weapons_lib.lagcomp.player_history[p_name] = nil
    end
  end
  stress_test_simulated_players = {}
  weapons_lib.lagcomp.debug_stress_test_active = false
  stress_test_step_count = 0
end
