--[[
	Lag Compensation System - Virtual Raycast
	Virtual raycast functions for lag-compensated hit detection.
]]

local table_insert = table.insert
local vector_add = vector.add
local vector_multiply = vector.multiply
local ipairs = ipairs

local GRID_WIDTH = 0.47
local GRID_SIZE = 3



-- Perform virtual raycast against historical player hitboxes
-- Returns all hits sorted by distance (closest first)
function weapons_lib.lagcomp.perform_virtual_raycast(ray_origin, ray_dir, ray_length, targets_with_hitboxes)
  local all_hits = {}

  for _, target in ipairs(targets_with_hitboxes) do
    local player_obj = target.player

    local hit_result = weapons_lib.lagcomp.test_ray_collision(ray_origin, ray_dir, ray_length, player_obj, target.historical_pos)

    if hit_result then
      table_insert(all_hits, hit_result)
    end
  end

  -- Sort hits by distance (closest first)
  table.sort(all_hits, function(a, b)
    return a.distance < b.distance
  end)

  return all_hits
end



-- Perform 3x3 grid virtual raycast (assisted aim) against historical player hitboxes
-- This mimics the behavior of the standard weapons system for consistency
function weapons_lib.lagcomp.perform_grid_virtual_raycast(shooter, ray_origin, ray_dir, ray_length, targets_with_hitboxes, debug_enabled, shooter_name)
  local all_hits = {}
  local rays_cast = 0

  -- 3x3 grid parameters (matching weapons_utils.lua)
  local grid_width = GRID_WIDTH
  local r_amount = GRID_SIZE
  local left_dir = weapons_lib.get_shooter_left_dir(shooter)
  local head_up_dir = weapons_lib.get_shooter_up_dir(shooter, ray_dir)

  local x_step = vector_multiply(vector_multiply(left_dir, -1), grid_width / r_amount)
  local y_step = vector_multiply(head_up_dir, grid_width / r_amount)

  -- Calculate starting position for the grid
  local start_offset = vector_add(
    vector_multiply(x_step, -(r_amount - 1) / 2),
    vector_multiply(y_step, (r_amount - 1) / 2)
  )
  local ray_pos = vector_add(ray_origin, start_offset)

  weapons_lib.print_debug(debug_enabled, shooter_name,
    "Using 3x3 grid virtual raycast (early exit on first hit)")

  -- Cast rays in 3x3 grid pattern with early exit on first hit
  for row = 1, r_amount do
    for column = 1, r_amount do
      rays_cast = rays_cast + 1

      local current_ray_pos = vector_add(ray_pos,
        vector_add(
          vector_multiply(x_step, column - 1),
          vector_multiply(y_step, -(row - 1))
        )
      )

      local grid_hits = weapons_lib.lagcomp.perform_virtual_raycast(current_ray_pos, ray_dir, ray_length, targets_with_hitboxes)

      if #grid_hits > 0 then
        for _, hit in ipairs(grid_hits) do
          local p_name = hit.object:get_player_name()
          local already_hit = false

          -- Check if this player is already hit
          for _, existing_hit in ipairs(all_hits) do
            if existing_hit.object:get_player_name() == p_name then
              -- Keep the closer hit
              if hit.distance < existing_hit.distance then
                existing_hit.distance = hit.distance
                existing_hit.intersection_point = hit.intersection_point
              end
              already_hit = true
              break
            end
          end

          if not already_hit then
            table_insert(all_hits, hit)
          end
        end

        -- Early exit when first hit is found
        if #all_hits > 0 then
          break
        end
      end
    end

    -- Break if hit found
    if #all_hits > 0 then
      break
    end
  end

  if debug_enabled and #all_hits > 0 then
    weapons_lib.print_debug(debug_enabled, shooter_name,
      string.format("Found hit after %d ray(s)", rays_cast))
  end

  -- Sort hits by distance (closest first)
  table.sort(all_hits, function(a, b)
    return a.distance < b.distance
  end)

  return all_hits
end
