--[[
	Source: https://github.com/minetest/minetest_game/tree/master/mods/default/furnace.lua
	Adapted to work as manual powered grinder
]]

local S = minetest.get_translator("default")

-- List of sound handles for active furnace
local grinder_fire_sounds = {}

--
-- Formspecs
--

function technic_farming.get_compactor_active_formspec(item_percent)
	return "size[8,8.5]"..
		"list[context;src;2.75,0.5;1,1;]"..
		"image[3.75,1.5;1,1;gui_furnace_arrow_bg.png^[lowpart:"..
		(item_percent)..":gui_furnace_arrow_fg.png^[transformR270]"..
		"list[context;dst;4.75,0.96;2,2;]"..
		"list[current_player;main;0,4.25;8,1;]"..
		"list[current_player;main;0,5.5;8,3;8]"..
		"listring[context;dst]"..
		"listring[current_player;main]"..
		"listring[context;src]"..
		"listring[current_player;main]"..
		default.get_hotbar_bg(0, 4.25)
end

function technic_farming.get_compactor_inactive_formspec()
	return "size[8,8.5]"..
		"list[context;src;2.75,0.5;1,1;]"..
		"image[3.75,1.5;1,1;gui_furnace_arrow_bg.png^[transformR270]"..
		"list[context;dst;4.75,0.96;2,2;]"..
		"list[current_player;main;0,4.25;8,1;]"..
		"list[current_player;main;0,5.5;8,3;8]"..
		"listring[context;dst]"..
		"listring[current_player;main]"..
		"listring[context;src]"..
		"listring[current_player;main]"..
		default.get_hotbar_bg(0, 4.25)
end

local function stop_compactor_sound(pos, fadeout_step)
	local hash = minetest.hash_node_position(pos)
	local sound_ids = grinder_fire_sounds[hash]
	if sound_ids then
		for _, sound_id in ipairs(sound_ids) do
			minetest.sound_fade(sound_id, -1, 0)
		end
		grinder_fire_sounds[hash] = nil
	end
end

local function check_node_around_mill(pos)
	local node = minetest.get_node(pos)
	if node.name == "default:water_flowing"
	  or node.name == "default:river_water_flowing" then
		return node.param2 or 0-- returns approx. water flow, if any
	end
	return 0
end

local function check_node_around_windmill(pos)
	local node = minetest.get_node(pos)
	if node.name == "air" then
		return math.random(5,12)-- returns approx. water flow, if any
	end
	return 0
end

local function compactor_node_timer(pos, elapsed)
	--
	-- Initialize metadata
	--
	local meta = minetest.get_meta(pos)

	local src_time = meta:get_float("src_time") or 0

	local inv = meta:get_inventory()
	local srclist
	local dst_full = false
	local full_flow = 120
	local timer_elapsed = meta:get_int("timer_elapsed") or 0
	meta:set_int("timer_elapsed", timer_elapsed + 1)

	-- get flows from water and/or wind mill
	local water_flow = technic_farming.get_water_flow(pos)
	local air_flow = technic_farming.get_air_flow(pos)

	-- ensure that grinding with mp machines are slower than with electric machines
	local water_rel_time = 0.75 * water_flow / full_flow
	local air_rel_time = 0.75 * air_flow / full_flow
	local rel_time = math.min(water_rel_time + air_rel_time, full_flow)
	local update = true
	local grindable
	local grindtime = 0

	while elapsed > 0 and update do
		update = false

		srclist = inv:get_list("src")

		-- Check if we have grindable content
		local grinded = technic.get_recipe("compacting",srclist)
		grindtime = 0
		local grindoutput
		if grinded then
			grindtime = grinded.time or 1
		end
		
		grindable = grindtime ~= 0
		if grindable then 
			src_time = src_time + rel_time

			if src_time >= grindtime then
				-- Place result in dst list if possible
				if inv:room_for_item("dst", grindoutput) then
					inv:add_item("dst", grinded.output)
					inv:set_stack("src",1,grinded.new_input[1])
					src_time = src_time - grindtime
					update = true
				else
					dst_full = true
				end
				-- Play cooling sound
				minetest.sound_play("default_cool_lava",
					{pos = pos, max_hear_distance = 16, gain = 0.07}, true)
			else
				-- Item could not be grinded
				update = true
			end
		
		end

		elapsed = elapsed - 1
	end

	--
	-- Update formspec, infotext and node
	--
	local formspec
	local item_state
	local item_percent = 0
	if grindable then
		item_percent = math.floor(src_time / grindtime * 100)
		if dst_full then
			item_state = S("100% (output full)")
		else
			item_state = S("@1%", item_percent)
		end
	else
		if srclist and not srclist[1]:is_empty() then
			item_state = S("Not grindable")
		else
			item_state = S("Empty")
		end
	end

	local active = true
	if (srclist and srclist[1]:is_empty()) then
		print("empty")
		active = false
	end
	local result = false
	if active then
--		active = true
		formspec = technic_farming.get_compactor_active_formspec(item_percent)
		technic_farming.swap_node(pos, "technic_farming:mp_compactor_active")
		-- make sure timer restarts automatically
		result = true

		-- Play sound every 5 seconds while the furnace is active
		if timer_elapsed == 0 or (timer_elapsed + 1) % 5 == 0 then
			local sound_id = minetest.sound_play("technic_farming:mp_compactor_active",
				{pos = pos, max_hear_distance = 16, gain = 0.25})
			local hash = minetest.hash_node_position(pos)
			compactor_fire_sounds[hash] = compactor_fire_sounds[hash] or {}
			table.insert(compactor_fire_sounds[hash], sound_id)
			-- Only remember the 3 last sound handles
			if #compactor_fire_sounds[hash] > 3 then
				table.remove(compactor_fire_sounds[hash], 1)
			end
			-- Remove the sound ID automatically from table after 11 seconds
			minetest.after(11, function()
				if not compactor_fire_sounds[hash] then
					return
				end
				for f=#compactor_fire_sounds[hash], 1, -1 do
					if compactor_fire_sounds[hash][f] == sound_id then
						table.remove(compactor_fire_sounds[hash], f)
					end
				end
				if #compactor_fire_sounds[hash] == 0 then
					compactor_fire_sounds[hash] = nil
				end
			end)
		end
	else
		formspec = technic_farming.get_compactor_inactive_formspec()
		technic_farming.swap_node(pos, "technic_farming:mp_compactor")
		-- stop timer on the inactive furnace
		minetest.get_node_timer(pos):stop()
		meta:set_int("timer_elapsed", 0)

		stop_compactor_sound(pos)
	end


	local infotext
	if active then
		infotext = S("Compactor active")
	else
		infotext = S("Compactor inactive")
	end
	infotext = infotext .. "\n" .. S("(Item: @1)", item_state)

	--
	-- Set meta values
	--
	meta:set_float("src_time", src_time)
	meta:set_string("formspec", formspec)
	meta:set_string("infotext", infotext)

	return result
end

--
-- Node definitions
--

minetest.register_node("technic_farming:mp_compactor", {
	description = S("MP Compactor"),
	tiles = {
		"technic_farming_mp_compactor_top.png",
		"technic_farming_mp_compactor_bottom.png",
		"technic_farming_mp_compactor_side.png",
		"technic_farming_mp_compactor_side.png",
		"technic_farming_mp_compactor_side.png",
		"technic_farming_mp_compactor_front.png"
	},
	paramtype2 = "facedir",
	groups = {cracky=2},
	legacy_facedir_simple = true,
	is_ground_content = false,
	sounds = default.node_sound_stone_defaults(),

	can_dig = technic_farming.can_dig,

	on_timer = compactor_node_timer,

	on_construct = function(pos)
		local meta = minetest.get_meta(pos)
		local inv = meta:get_inventory()
		inv:set_size('src', 1)
		inv:set_size('dst', 4)
		compactor_node_timer(pos, 0)
	end,

	on_metadata_inventory_move = function(pos)
		minetest.get_node_timer(pos):start(1.0)
	end,
	on_metadata_inventory_put = function(pos)
		minetest.get_node_timer(pos):start(1.0)
	end,
	on_metadata_inventory_take = function(pos)
		minetest.get_node_timer(pos):start(1.0)
	end,
	on_blast = function(pos)
		local drops = {}
		default.get_inventory_drops(pos, "src", drops)
		default.get_inventory_drops(pos, "dst", drops)
		drops[#drops+1] = "technic_farming:mp_compactor"
		minetest.remove_node(pos)
		return drops
	end,

	allow_metadata_inventory_put = technic_farming.allow_metadata_inventory_put,
	allow_metadata_inventory_move = technic_farming.allow_metadata_inventory_move,
	allow_metadata_inventory_take = technic_farming.allow_metadata_inventory_take,
})

minetest.register_node("technic_farming:mp_compactor_active", {
	description = S("Compactor"),
	tiles = {
		"technic_farming_mp_compactor_top.png",
		"technic_farming_mp_compactor_bottom.png",
		"technic_farming_mp_compactor_side.png",
		"technic_farming_mp_compactor_side.png",
		"technic_farming_mp_compactor_side.png",
		"technic_farming_mp_compactor_front.png"
	},
	paramtype2 = "facedir",
	light_source = 8,
	drop = "technic_farming:mp_compactor",
	groups = {cracky=2, not_in_creative_inventory=1},
	legacy_facedir_simple = true,
	is_ground_content = false,
	sounds = default.node_sound_stone_defaults(),
	on_timer = grinder_node_timer,
	on_destruct = function(pos)
		stop_compactor_sound(pos)
	end,

	can_dig = technic_farming.can_dig,

	allow_metadata_inventory_put = technic_farming.allow_metadata_inventory_put,
	allow_metadata_inventory_move = technic_farming.allow_metadata_inventory_move,
	allow_metadata_inventory_take = technic_farming.allow_metadata_inventory_take,
})

minetest.register_craft({
	output = 'technic_farming:mp_compactor',
	recipe = {
		{'technic_farming:wooden_screw', 'technic_farming:wooden_screw','technic_farming:wooden_screw'},
		{'group:wood',                   'technic_farming:stone_plate', 'group:wood'},
		{'group:stone',                  'technic_farming:stone_plate', 'group:stone'},
	}
})
