---------------------------------------------------------------------------------------
-- JAPAN TRAM TLR0600<Toyama Light Rail model>
-- by takaosan
---------------------------------------------------------------------------------------

local S = attrans

local illumination_timer_duration = 10
local illumination_start_time = 18900
local illumination_end_time = 5200

local use_illuminated_appearance = minetest.settings:get_bool("advtrains_train_jpntlr600_illuminated_appearance", true)

-- With the ability for players to now apply custom colors to their trams, it
-- is no longer necessary for there to be trams of specific colors. This
-- option is intended to be enabled for those worlds that do not include the
-- AdvTaiins Livery Designer mod pack and for pre-existing worlds that already
-- have such trams from earlier versions of this mod.
local include_deprecated_trams = minetest.settings:get_bool("advtrains_train_jpntlr600_include_deprecated_trams", false)

local deprecated_tram_colours = {
	"blue",
	"green",
	"red",
	"yellow"
}

-- Begin support code for AdvTrains Livery Designer

local use_advtrains_livery_designer = minetest.get_modpath( "advtrains_livery_designer" ) and advtrains_livery_designer
local mod_name = "advtrains_train_jpntlr600"

local generic_tlr0600_front_unit = mod_name..":".."front_unit"
local generic_tlr0600_back_unit = mod_name..":".."back_unit"

local common_templates = {
	{
		name = "Toyama",
		designer = "Marnack",
		texture_license = "CC-BY-SA-3.0",
		texture_creator = "takaosan, with minor updates by Marnack",
		notes = "This template is loosely based on the livery for the TLR0600 Tram in Toyama, Japan. It supports the four livery schemes defined in the original version of this mod as well as additional customizations that build on the original livery.",
		base_textures = {"advtrains_tlr600_01.png"},
		overlays = {
			[1] = {name = "Exterior Walls",		texture = "advtrains_tlr600_01_exterior_walls.png",				alpha = 228},
			[2] = {name = "Window Trim",		texture = "advtrains_tlr600_01_window_trim.png",				alpha = 228},
			[3] = {name = "Service Color",		texture = "advtrains_tlr600_01_line_coloring.png",				alpha = 228},
			[4] = {name = "Seats",				texture = "advtrains_tlr600_common_seats.png",					alpha = 192},
		},
	},
	{
		name = "Ribbon",
		designer = "Marnack",
		texture_license = "CC-BY-SA-3.0",
		texture_creator = "takaosan, with updates by Marnack",
		notes = "This template features an optional multicolored ribbon that meanders around the exterior of the unit as well as cusomization options for the exterior walls, window trim and seats.",
		base_textures = {"advtrains_tlr600_02.png"},
		overlays = {
			[1] = {name = "Exterior Walls",		texture = "advtrains_tlr600_02_exterior_walls.png",				alpha = 228},
			[2] = {name = "Window Trim",		texture = "advtrains_tlr600_02_window_trim.png",				alpha = 228},
			[3] = {name = "Ribbon",				texture = "advtrains_tlr600_02_ribbon.png",						alpha = 228},
			[4] = {name = "Ribbon Middle",		texture = "advtrains_tlr600_02_ribbon_middle.png",				alpha = 228},
			[5] = {name = "Seats",				texture = "advtrains_tlr600_common_seats.png",					alpha = 192},
		},
	},
	{
		name = "Upper-Lower Split",
		designer = "Marnack",
		texture_license = "CC-BY-SA-3.0",
		texture_creator = "takaosan, with updates by Marnack",
		notes = "This template supports different colors for the upper and lower portions of the exterior walls and an optional service line as well as cusomization options for the window trim and seats.",
		base_textures = {"advtrains_tlr600_03.png"},
		overlays = {
			[1] = {name = "Upper Walls",		texture = "advtrains_tlr600_03_exterior_walls_upper.png",		alpha = 228},
			[2] = {name = "Window Trim",		texture = "advtrains_tlr600_03_window_trim.png",				alpha = 228},
			[3] = {name = "Lower Walls",		texture = "advtrains_tlr600_03_exterior_walls_lower.png",		alpha = 228},
			[4] = {name = "Service Stripe",		texture = "advtrains_tlr600_03_service_stripe.png",				alpha = 228},
			[5] = {name = "Seats",				texture = "advtrains_tlr600_common_seats.png",					alpha = 192},
		},
	},
}

local livery_templates = {
	[generic_tlr0600_front_unit] = common_templates,
	[generic_tlr0600_back_unit]  = common_templates,
}

if include_deprecated_trams then
	-- Include a livery template for each of the deprecated trams so
	-- that players can use the design livery tool to view the trams.
	for _,colour in pairs(deprecated_tram_colours) do
		local deprecated_tram_livery_templates = {
			{
				name = "Default "..colour,
				designer = "takaosan",
				texture_license = "CC-BY-SA-3.0",
				texture_creator = "takaosan",
				notes = "The original livery of the deprecated "..colour.." tram.",
				base_textures = {"advtrains_tlr600_"..colour..".png"},
			}
		}
		livery_templates["advtrains:Tlr600_"..colour.."_front"] = deprecated_tram_livery_templates
		livery_templates["advtrains:Tlr600_"..colour.."_back"]  = deprecated_tram_livery_templates
	end
end

local predefined_liveries = {
	{
		name = "Original Blue",
		notes = "This is the blue color used for the deprecated blue tram",
		livery_design = {
			livery_template_name = "Toyama",
			overlays = {
				[3] = {id = 3,	color = "#2B73B0"},	-- Service Color
			},
		},
	},
	{
		name = "Original Green",
		notes = "This is the green color used for the deprecated green tram",
		livery_design = {
			livery_template_name = "Toyama",
			overlays = {
				[3] = {id = 3,	color = "#7AB02B"},	-- Service Color
			},
		},
	},
	{
		name = "Original Red",
		notes = "This is the red color used for the deprecated red tram",
		livery_design = {
			livery_template_name = "Toyama",
			overlays = {
				[3] = {id = 3,	color = "#EA4524"},	-- Service Color
			},
		},
	},
	{
		name = "Original Yellow",
		notes = "This is the yellow color used for the deprecated yellow tram",
		livery_design = {
			livery_template_name = "Toyama",
			overlays = {
				[3] = {id = 3,	color = "#EBDB24"},	-- Service Color
			},
		},
	},
	{
		name = "Beachfront Express",
		livery_design = {
			livery_template_name = "Toyama",
			overlays = {
				[1] = {id = 1,	color = "#ADD8E6"},	-- Exterior Walls
				[2] = {id = 2,	color = "#FF7F50"},	-- Window Trim
--				[3] = {id = 3,	color = "#000000"},	-- Service Color
				[4] = {id = 4,	color = "#4169E1"},	-- Seats
			},
		},
	},
	{
		name = "Historic District Special",
		livery_design = {
			livery_template_name = "Toyama",
			overlays = {
				[1] = {id = 1,	color = "#F5DEB3"},	-- Exterior Walls
				[2] = {id = 2,	color = "#2E8B57"},	-- Window Trim
--				[3] = {id = 3,	color = "#000000"},	-- Service Color
				[4] = {id = 4,	color = "#2E8B57"},	-- Seats
			},
		},
	},
	{
		name = "Airport Shuttle",
		livery_design = {
			livery_template_name = "Ribbon",
			overlays = {
				[1] = {id = 1,	color = "#87CEEB"},	-- Exterior Walls
				[2] = {id = 2,	color = "#6FB5D1"},	-- Window Trim
				[3] = {id = 3,	color = "#0000FF"},	-- Ribbon
				[4] = {id = 4,	color = "#FAF0E6"},	-- Ribbon Middle
--				[5] = {id = 5,	color = "#000000"},	-- Seats
			},
		},
	},
	{
		name = "Park Service",
		livery_design = {
			livery_template_name = "Ribbon",
			overlays = {
				[1] = {id = 1,	color = "#F0E68C"},	-- Exterior Walls
				[2] = {id = 2,	color = "#F0E68C"},	-- Window Trim
				[3] = {id = 3,	color = "#006400"},	-- Ribbon
				[4] = {id = 4,	color = "#F0E68C"},	-- Ribbon Middle
				[5] = {id = 5,	color = "#D2B48C"},	-- Seats
			},
		},
	},
	{
		name = "Uptown Transit - Red line",
		livery_design = {
			livery_template_name = "Ribbon",
			overlays = {
				[1] = {id = 1,	color = "#708090"},	-- Exterior Walls
				[2] = {id = 2,	color = "#090909"},	-- Window Trim
				[3] = {id = 3,	color = "#800000"},	-- Ribbon
				[4] = {id = 4,	color = "#FF0000"},	-- Ribbon Middle
				[5] = {id = 5,	color = "#800000"},	-- Seats
			},
		},
	},
	{
		name = "Commercial District - Orange line",
		livery_design = {
			livery_template_name = "Ribbon",
			overlays = {
				[1] = {id = 1,	color = "#008080"},	-- Exterior Walls
				[2] = {id = 2,	color = "#800000"},	-- Window Trim
--				[3] = {id = 3,	color = "#000000"},	-- Ribbon
				[4] = {id = 4,	color = "#FF4500"},	-- Ribbon Middle
				[5] = {id = 5,	color = "#008080"},	-- Seats
			},
		},
	},
	{
		name = "Downtown - Green line",
		livery_design = {
			livery_template_name = "Upper-Lower Split",
			overlays = {
				[1] = {id = 1,	color = "#C0C0C0"},	-- Upper Walls
				[2] = {id = 2,	color = "#4682B4"},	-- Window Trim
				[3] = {id = 3,	color = "#708090"},	-- Lower Walls
				[4] = {id = 4,	color = "#00FF00"},	-- Service Stripe
				[5] = {id = 5,	color = "#1B3A5C"},	-- Seats
			},
		},
	},
	{
		name = "Midtown Commuter",
		livery_design = {
			livery_template_name = "Upper-Lower Split",
			overlays = {
				[1] = {id = 1,	color = "#C0C0C0"},	-- Upper Walls
				[2] = {id = 2,	color = "#800000"},	-- Window Trim
				[3] = {id = 3,	color = "#808080"},	-- Lower Walls
--				[4] = {id = 4,	color = "#000000"},	-- Service Stripe
--				[5] = {id = 5,	color = "#000000"},	-- Seats
			},
		},
	},
	{
		name = "Oldtown - Silver line",
		livery_design = {
			livery_template_name = "Upper-Lower Split",
			overlays = {
				[1] = {id = 1,	color = "#F0FFFF"},	-- Upper Walls
				[2] = {id = 2,	color = "#202020"},	-- Window Trim
				[3] = {id = 3,	color = "#1A432C"},	-- Lower Walls
				[4] = {id = 4,	color = "#C0C0C0"},	-- Service Stripe
				[5] = {id = 5,	color = "#1A432C"},	-- Seats
			},
		},
	},
	{
		name = "Suburban Connector - Gold line",
		livery_design = {
			livery_template_name = "Upper-Lower Split",
			overlays = {
				[1] = {id = 1,	color = "#F4F4F4"},	-- Upper Walls
				[2] = {id = 2,	color = "#F4F4F4"},	-- Window Trim
				[3] = {id = 3,	color = "#808080"},	-- Lower Walls
				[4] = {id = 4,	color = "#FFD700"},	-- Service Stripe
				[5] = {id = 5,	color = "#2E8B57"},	-- Seats
			},
		},
	},
	{
		name = "Theater District Shuttle",
		livery_design = {
			livery_template_name = "Upper-Lower Split",
			overlays = {
				[1] = {id = 1,	color = "#C0C0C0"},	-- Upper Walls
				[2] = {id = 2,	color = "#460046"},	-- Window Trim
				[3] = {id = 3,	color = "#8F178F"},	-- Lower Walls
--				[4] = {id = 4,	color = "#000000"},	-- Service Stripe
--				[5] = {id = 5,	color = "#000000"},	-- Seats
			},
		},
	},
}

if use_advtrains_livery_designer then
	-- Notify player if a newer version of AdvTrains Livery Tools is needed.
	if not advtrains_livery_designer.is_compatible_mod_version or
	   not advtrains_livery_designer.is_compatible_mod_version({major = 0, minor = 8, patch = 1}) then
		minetest.log("error", "["..mod_name.."] An old version of AdvTrains Livery Tools was detected. Please update to the latest version to avoid known issues.")
		-- Version 0.8.1 is needed to fix an issue with activating the livery designer tool for a tram that is in the nighttime/illuminated state.
	end

	-- This function is called by the advtrains_livery_designer tool whenever
	-- the player activates the "Apply" button. This implementation is specific
	-- to Tlr600_front and Tlr600_back. A more complex implementation may be
	-- needed if other wagons or livery templates are added.
	local function apply_wagon_livery_textures(player, wagon, textures)
		if wagon and textures and textures[1] then
			local data = advtrains.wagons[wagon.id]
			data.livery = textures[1]
			wagon:set_textures(data)

			-- The following will trigger an update of the wagon if it is
			-- currently nighttime. This is needed so that the illumination and
			-- lighting effect will be applied (if the mod setting is enabled).
			-- Because this is dependant on illumination_timer_duration, there
			-- can be a short delay before the wagon transitions to its
			-- nighttime appearance.
			wagon.object:set_properties({glow = 0,})
		end
	end

	-- Define a callback function that will return a wagon's daytime texture
	-- regardless the the time of day.
	local function on_pre_get_livery_design_from_textures(wagon_type, textures, wagon_id)
		local data = advtrains.wagons[wagon_id]
		if data and data.livery then
			return {data.livery}
		end

		-- Can't find the wagon or its livery has not yet been defined so just
		-- return the textures that were passed in.
		return textures
	end

	local optional_callback_functions = {}
	optional_callback_functions.on_pre_get_livery_design_from_textures = on_pre_get_livery_design_from_textures

	-- Register this mod and its livery function with the advtrains_livery_designer tool.
	advtrains_livery_designer.register_mod(mod_name, apply_wagon_livery_textures, optional_callback_functions)

	-- Register this mod's wagons and livery templates.
	for wagon_type, wagon_livery_templates in pairs(livery_templates) do
		advtrains_livery_database.register_wagon(wagon_type, mod_name)
		for _, livery_template in ipairs(wagon_livery_templates) do
			advtrains_livery_database.add_livery_template(
				wagon_type,
				livery_template.name,
				livery_template.base_textures,
				mod_name,
				(livery_template.overlays and #livery_template.overlays) or 0,
				livery_template.designer,
				livery_template.texture_license,
				livery_template.texture_creator,
				livery_template.notes
			)
			if livery_template.overlays then
				for overlay_id, overlay in ipairs(livery_template.overlays) do
					advtrains_livery_database.add_livery_template_overlay(
						wagon_type,
						livery_template.name,
						overlay_id,
						overlay.name,
						overlay.slot_idx or 1,
						overlay.texture,
						overlay.alpha
					)
				end
			end
		end
	end

	-- Register this mod's predefined wagon liveries.
	for _, predefined_livery in pairs(predefined_liveries) do
		-- Each predefined livery will be defined for each generic TLR0600 unit.
		for _, wagon_type in pairs({generic_tlr0600_front_unit, generic_tlr0600_back_unit}) do
			local livery_design = predefined_livery.livery_design
			livery_design.wagon_type = wagon_type
			advtrains_livery_database.add_predefined_livery(
				predefined_livery.name,
				livery_design,
				mod_name,
				predefined_livery.notes
			)
		end
	end

end

local function set_textures(wagon, data)
	if data.livery then
		wagon.object:set_properties({textures={data.livery}})
	end
end

local function update_livery(wagon, puncher)
	local itemstack = puncher:get_wielded_item()
	local item_name = itemstack:get_name()
	if use_advtrains_livery_designer and item_name == advtrains_livery_designer.tool_name then
		advtrains_livery_designer.activate_tool(puncher, wagon, mod_name)
		return true
	end
	return false
end

-- End of support code for AdvTrains Livery Designer

local function on_custom_on_step(self, dtime, wagon_default_texture_name, tram_end)
	local illumination_timer = (self.illumination_timer or 0) - dtime
	if illumination_timer <= 0 then
		local data = advtrains.wagons[self.id]
		local current_livery = data and data.livery or wagon_default_texture_name
		local time = minetest.get_timeofday() * 24000
		local properties = self.object:get_properties()
		local current_glow = properties.glow or 0
		if (time > illumination_start_time or time < illumination_end_time) then
			-- To minimize server load, only set properties if they have changed.
			if current_glow ~= 5 then
				-- Set nighttime configuration
				self.object:set_properties({textures = {
					current_livery.."^("..current_livery.."^[mask:advtrains_tlr600_common_exterior_mask.png^[multiply:#464646)^advtrains_tlr600_common_illumination.png^advtrains_tlr600_common_lights_"..tram_end..".png"},})
				self.object:set_properties({glow = 5,})
			end
		else
			-- To minimize server load, only set properties if they have changed.
			if current_glow ~= 0 then
				-- Set daytime configuration
				self.object:set_properties({textures = {current_livery},})
				self.object:set_properties({glow = 0,})
			end
		end
		self.illumination_timer = illumination_timer_duration
	else
		self.illumination_timer = illumination_timer
	end
end

local colours = {
	"custom",
}

if include_deprecated_trams then
	for _,colour in pairs(deprecated_tram_colours) do
		table.insert(colours, colour)
	end
end

for _,colour in pairs(colours) do
	local use_predefined_color = colour ~= "custom"

	local wagon_info = {}
	if use_predefined_color then
		-- Register the deprecated trams using their original names in the
		-- advtrains namespace as had been done in earlier version of this mod.
		wagon_info.front = {
			internal_name = "Tlr600_"..colour.."_front",
			display_name = "Tlr600_"..colour.."_front",
			default_texture_name = "advtrains_tlr600_"..colour..".png",
			inventory_image_name = "advtrains_tlr600_"..colour.."_inv.png",
		}
		wagon_info.back = {
			internal_name = "Tlr600_"..colour.."_back",
			display_name = "Tlr600_"..colour.."_back",
			default_texture_name = "advtrains_tlr600_"..colour..".png",
			inventory_image_name = "advtrains_tlr600_"..colour.."_inv.png",
		}
	else
		-- Register the generic TLR0600 trams in this mod's namespace.
		wagon_info.front = {
			internal_name = generic_tlr0600_front_unit,
			display_name = "TLR0600 Front Unit",
			default_texture_name = "advtrains_tlr600_01.png",
			inventory_image_name = "advtrains_tlr600_front_inv.png",
		}
		wagon_info.back = {
			internal_name = generic_tlr0600_back_unit,
			display_name = "TLR0600 Back Unit",
			default_texture_name = "advtrains_tlr600_01.png",
			inventory_image_name = "advtrains_tlr600_back_inv.png",
		}
	end

	-- TLR0600 <FRONT>
	local wagon_def_front = {
		mesh="advtrains_tlr600_front.b3d",
		textures = {wagon_info.front.default_texture_name},
		set_textures = set_textures,
		custom_may_destroy = function(wagon, puncher, time_from_last_punch, tool_capabilities, direction)
			return not update_livery(wagon, puncher)
		end,
		drives_on={default=true},
		max_speed=8,
		seats = {
			{
				name=S("Driver stand"),
				attach_offset={x=0, y=-4, z=20},
				view_offset={x=0, y=-4, z=2},
				driving_ctrl_access=true,
				group="dstand",
			},
			{
				name="1",
				attach_offset={x=-4, y=-4, z=0},
				view_offset={x=0, y=-4, z=0},
				group="pass",
			},
			{
				name="2",
				attach_offset={x=4, y=-4, z=0},
				view_offset={x=0, y=-4, z=0},
				group="pass",
			},
			{
				name="3",
				attach_offset={x=-4, y=-4, z=-8},
				view_offset={x=0, y=-4, z=0},
				group="pass",
			},
			{
				name="4",
				attach_offset={x=4, y=-4, z=-8},
				view_offset={x=0, y=-4, z=0},
				group="pass",
			},
		},
		seat_groups = {
			dstand={
				name = "Driver Stand",
				access_to = {"pass"},
				driving_ctrl_access=true,
			},
			pass={
				name = "Passenger area",
				access_to = {"dstand"},
				require_doors_open=true,
			},
		},
		assign_to_seat_group = {"dstand", "pass"},
		doors={
			open={
				[-1]={frames={x=0, y=40}, time=1},
				[1]={frames={x=80, y=120}, time=1},
				sound = "advtrains_tlr600_dclose",
			},
			close={
				[-1]={frames={x=40, y=80}, time=1},
				[1]={frames={x=120, y=160}, time=1},
				sound = "advtrains_tlr600_dclose",
			}
		},
		door_entry={-1},
		assign_to_seat_group = {"dstand", "pass"},
		visual_size = {x=1, y=1},
		wagon_span=2.3,
		is_locomotive=true,
		collisionbox = {-1.0,-0.5,-1.0, 1.0,2.5,1.0},
		drops={"default:steelblock 4"},
		horn_sound = "advtrains_tlr600_horn",
		custom_on_velocity_change = function(self, velocity, old_velocity, dtime)

			if not velocity or not old_velocity then return end
			
			if velocity > 0 and (self.sound_loop_tmr or 0)<=0 then
				self.sound_loop_handle = minetest.sound_play({name="advtrains_tlr600_loop", gain=0.3}, {object = self.object})
				self.sound_loop_tmr=3
			elseif velocity>0 then
				self.sound_loop_tmr = self.sound_loop_tmr - dtime
			elseif velocity==0 then
				if self.sound_loop_handle then
					minetest.sound_stop(self.sound_loop_handle)
					self.sound_loop_handle = nil
				end
				self.sound_loop_tmr=0
			end
		end,
		custom_on_step = use_illuminated_appearance and function(self, dtime)
			on_custom_on_step(self, dtime, wagon_info.front.default_texture_name, "front")
		end
	}

	advtrains.register_wagon(wagon_info.front.internal_name, wagon_def_front, wagon_info.front.display_name, wagon_info.front.inventory_image_name)

	-- TLR0600 <BACK>
	local wagon_def_back = {
		mesh="advtrains_tlr600_back.b3d",
		textures = {wagon_info.back.default_texture_name},
		set_textures = set_textures,
		custom_may_destroy = function(wagon, puncher, time_from_last_punch, tool_capabilities, direction)
			return not update_livery(wagon, puncher)
		end,
		drives_on={default=true},
		max_speed=8,
		seats = {
			{
				name="1",
				attach_offset={x=-4, y=-4, z=8},
				view_offset={x=0, y=-4, z=0},
				group="pass",
			},
			{
				name="2",
				attach_offset={x=4, y=-4, z=8},
				view_offset={x=0, y=-4, z=0},
				group="pass",
			},
			{
				name="3",
				attach_offset={x=-4, y=-4, z=-8},
				view_offset={x=0, y=-4, z=0},
				group="pass",
			},
			{
				name="4",
				attach_offset={x=4, y=-4, z=-8},
				view_offset={x=0, y=-4, z=0},
				group="pass",
			},
		},
		seat_groups = {
			pass={
				name = "Passenger area",
				access_to = {},
				require_doors_open=true,
			},
		},
		assign_to_seat_group = {"pass"},
		doors={
			open={
				[-1]={frames={x=0, y=40}, time=1},
				[1]={frames={x=80, y=120}, time=1}
			},
			close={
				[-1]={frames={x=40, y=80}, time=1},
				[1]={frames={x=120, y=160}, time=1}
			}
		},
		door_entry={-1, 1},
		visual_size = {x=1, y=1},
		wagon_span=2.1,
		collisionbox = {-1.0,-0.5,-1.0, 1.0,2.5,1.0},
		drops={"default:steelblock 4"},
		custom_on_step = use_illuminated_appearance and function(self, dtime)
			on_custom_on_step(self, dtime, wagon_info.back.default_texture_name, "rear")
		end
	}

	advtrains.register_wagon(wagon_info.back.internal_name, wagon_def_back, wagon_info.back.display_name, wagon_info.back.inventory_image_name)
end

if  minetest.get_modpath("default") then

	local door = "default:steel_ingot"
	if minetest.get_modpath("doors") then
		door = "doors:door_steel"
	end

	local cabin = mod_name..":TLR0600_cabin"
	minetest.register_craftitem(cabin, {
		description = attrans("TLR0600 Cabin"),
		inventory_image = "advtrains_tlr600_cabin.png",
	})

	minetest.register_craft({
		output = cabin,
		recipe = {
			{"default:tinblock", "default:tinblock", "default:tinblock"},
			{door, "wool:white", "default:glass"},
			{"default:steelblock", "default:steelblock", "default:steelblock"},
		}
	})

	minetest.register_craft({
		output = generic_tlr0600_front_unit,
		recipe = {
			{"", cabin, "default:steel_ingot"},
			{"advtrains:wheel", "default:steelblock", "advtrains:wheel"},
		}
	})

	minetest.register_craft({
		output = generic_tlr0600_back_unit,
		recipe = {
			{"", "default:copperblock", ""},
			{"", cabin, "default:steel_ingot"},
			{"advtrains:wheel", "default:steelblock", "advtrains:wheel"},
		}
	})
end
