-- Staff Entity --

local SPEED = 1
local GRAVITY = -9.8

local ANIMS = {
	idle = {range = {x = 0, y = 140}, speed = 15},
	walk = {range = {x = 141, y = 181}, speed = 45},
	carry = {range = {x = 204, y = 244}, speed = 45},
	die = {range = {x = 182, y = 202}, speed = 10, loop = false},
}

local pathfind = dofile(staff.PATH .. "/pathfind.lua")

local entity = {
	animation = "idle",
	destination = nil,
	hostile = -1, -- Docile: -1, Hostile: 0, Target (Hostile against 1 player): >=1
	hp = 100,
	kb_timer = 0,
	last_punch = 0,
	target = nil,
	step = 0,
	velocity = {x = 0, y = 0, z = 0},
}

entity.initial_properties = {
	visual = "mesh",
	mesh = "ikea_staff_member.b3d",
	textures = {"ikea_staff_member.png"},
	collisionbox = {-0.45, 0, -0.45, 0.45, 3.1, 0.45},
	selectionbox = {-0.45, 0, -0.45, 0.45, 3.1, 0.45},
	collide_with_objects = true,
	makes_footstep_sound = true,
	physical = true,
}

entity.animate = function(self, state, speed)
	if self.animation ~= state then
		local anim = ANIMS[state]
		assert(anim.range, "Missing animation range for '" .. state .. "'.")
		self.object:set_animation(anim.range, (anim.speed or 30) * (speed or 1), 0, anim.loop)
		self.animation = state
	end
end

entity.set_velocity = function(self, velocity)
	self.object:set_velocity(velocity)
	self.velocity = velocity
end

-- Interpolated turning
entity.look_at = function(self, target, speed)
	local lookat
	if type(target) == "table" then
		lookat = math.deg(minetest.dir_to_yaw(vector.direction(self.object:get_pos(), target)))
		-- Manual calculation: math.deg(math.atan2(target.z - pos.z, target.x - pos.x)) - 90
	else
		lookat = math.deg(target)
	end

	self.turnspeed = 180
	self.look = lookat
end

entity.look_step = function(self, dtime) -- This isnt perfect yet
	if self.look then
		local yaw = math.deg(self.object:get_yaw())
		if yaw ~= self.look then
			local diff = self.look - yaw

			while diff < -180 do diff = diff + 360 end
			while diff > 180 do diff = diff - 360 end

			if math.abs(diff) < self.turnspeed * dtime then
				self.object:set_yaw(math.rad(self.look))
				self.look = nil
				return
			end

			self.object:set_yaw(math.rad(yaw + math.abs(diff) / diff * self.turnspeed * dtime))
		end
	end
end

entity.go_to = function(self, target)
	self:look_at(target, 10)

	local anim = "walk"
	local speed = SPEED
	if self.hostile >= 0 then
		anim = "carry"
		speed = speed * 1.5
	end
	self:animate(anim, speed)

	local pos = self.object:get_pos()
	local velocity = vector.multiply(vector.direction(pos, target), speed)
	velocity.y = 0
	self:set_velocity(velocity)
end

entity.on_activate = function(self, staticdata)
	self.object:set_armor_groups({fleshy = 100})
	self.object:set_acceleration({x = 0, y = GRAVITY, z = 0})
	self:set_velocity({x = 0, y = 0, z = 0})

	local scale = {x = 1, y = 1, z = 1}
	local box = {-0.45, 0, -0.45, 0.45, 3.1, 0.45}
	if staticdata and staticdata ~= "" then
		local data = minetest.deserialize(staticdata)
		scale = data.scale
		box = data.box
		self.hp = data.hp
	else
		if math.random(1, 4) == 1 then
			scale = {x = 1, y = 0.6, z = 1}
			box = {-0.45, 0, -0.45, 0.45, 1.9, 0.45}
		end
	end

	self.object:set_hp(self.hp)
	self.object:set_properties({visual_size = scale, collisionbox = box, selectionbox = box})

	self:animate("idle")
end

entity.get_staticdata = function(self)
	local props = self.object:get_properties()
	return minetest.serialize({scale = props.visual_size, box = props.collisionbox, hp = self.object:get_hp()})
end

entity.on_punch = function(self, puncher, time, caps, dir, damage)
	if self.hp <= 0 or not puncher or not puncher:is_player() then
		return true
	end

	self.hp = self.hp - damage

	if ikea.is_open() then
		self.hostile = self.hostile + damage * 10
	end

	-- Heart hit effect
	local ppos = puncher:get_pos()
	local eye = table.copy(ppos)
	eye.y = eye.y + puncher:get_properties().eye_height
	local ray = Raycast(eye, vector.add(eye, vector.multiply(puncher:get_look_dir(),
								vector.distance(ppos, self.object:get_pos()))))
	local intersect = vector.add(self.object:get_pos(), {x = 0, y = 1, z = 0})

	for pointed in ray do
		if pointed.type == "object" and not pointed.ref:is_player() then
			intersect = pointed.intersection_point
			break
		end
	end

	local vel = self.object:get_velocity()
	vel.y = 1
	local size = 3 + (damage / 2)
	if size > 10 then
		size = 10
	end

	minetest.add_particle({
		expiration_time = math.random(5, 10) * 0.1,
		pos = intersect,
		velocity = vector.multiply(vel, 2),
		acceleration = {x = 0, y = GRAVITY, z = 0},
		size = size,
		texture = "ikea_staff_damage.png",
		collisiondetection = true,
	})

	-- Damage flash
	self.object:set_texture_mod("^[colorize:#780000:128")
	minetest.after(0.2, function()
		if self and self.object then
			self.object:set_texture_mod("")
		end
	end)

	-- Knockback
	local knockback = math.max(1, math.min(math.ceil(damage / 3), 3))

	self.object:set_velocity(vector.multiply(puncher:get_look_dir(), knockback)) -- Use regular set_velocity
	self.destination = nil
	self.target = puncher:get_player_name()
	self.kb_timer = 0.3

	-- Unalive
	if self.hp <= 0 then
		self:on_death()
		return true
	end

	self.object:set_hp(self.hp)
	return true
end

entity.on_death = function(self)
	self.hp = 0 -- Make sure really dead

	local yaw = math.rad(90 * math.floor((math.deg(self.object:get_yaw()) / 90) + 0.5))
	local pos = self.object:get_pos()

	self:look_at(yaw, 1)
	self:animate("die")
	self.object:set_pos({x = math.floor(pos.x + 0.5), y = pos.y, z = math.floor(pos.z + 0.5)})
	self:set_velocity({x = 0, y = 0, z = 0})

	self.destination = nil
	self.hostile = -1
	self.target = nil

	minetest.after((ANIMS.die.range.y - ANIMS.die.range.x) / ANIMS.die.speed, function()
		minetest.set_node(pos, {name = "staff:corpse", param2 = minetest.dir_to_facedir(minetest.yaw_to_dir(yaw))})
		self.object:remove()
	end)
end

entity.can_see_player = function(self, player)
	local eye = self.object:get_pos()
	eye.y = eye.y + self.object:get_properties().collisionbox[5] - 0.3

	for i = 0, 2 do -- Check at feet, body, and head positions
		for pointed in Raycast(eye, vector.add(player:get_pos(), {x = 0, y = i, z = 0})) do
			if pointed.type == "object" and pointed.ref == player then
				return true
			end
		end
	end

	return false
end

entity.direct_path = function(self, pos)
	if Raycast(vector.add(self.object:get_pos(), {x = 0, y = 0.5, z = 0}), vector.add(pos, {x = 0, y = 0.5, z = 0}), false):next() then
		return false
	end

	return true
end

entity.on_step = function(self, dtime)
	-- Timers
	if self.kb_timer > 0 then
		self.kb_timer = math.max(0, self.kb_timer - dtime)
		if self.kb_timer == 0 then
			self.object:set_velocity({x = 0, y = GRAVITY, z = 0})
		end
	end

	if self.hostile > 0 then
		self.hostile = math.max(0, self.hostile - dtime)
		if self.hostile == 0 then
			self.target = nil
		end
	end

	if ikea.is_open() and self.hostile == 0 then
		self.hostile = -1
	elseif not ikea.is_open() and self.hostile ~= 0 then
		self.hostile = 0
	end

	self:look_step(dtime)

	-- on_step limiter
	self.step = self.step + dtime
	if self.hp <= 0 or self.step < 0.4 then
		return
	else
		self.step = 0
	end

	-- Do all the things
	local pos = self.object:get_pos()

	if self.hostile >= 0 then -- (is hostile)
		if not self.target then -- Look for nearby players TODO: dot product for 180deg fov
			for _, obj in pairs(minetest.get_objects_inside_radius(pos, 8)) do
				if obj:is_player() then
					self.target = obj:get_player_name()
				end
			end
		else -- Follow target
			local player = minetest.get_player_by_name(self.target)
			if player then
				if self:can_see_player(player) then
					local ppos = player:get_pos()
					if self:direct_path(ppos) then
						self.destination = {ppos}
					else
						local success, path = pathfind.tree(pos, player:get_pos())
						if success then
							self.destination = pathfind.trim(path)
						end
					end
				else
					-- Continue looking for 10 seconds
					if self.hostile == 0 then
						self.hostile = 10
					end
				end
			else
				self.target = nil
				self.destination = nil
			end
		end

		-- Attack players
		local rot = minetest.yaw_to_dir(self.object:get_yaw())
		local hitat = {x = pos.x + rot.x, y = pos.y + 1.5, z = pos.z + rot.z}
		for _, o in pairs(minetest.get_objects_inside_radius(hitat, 2)) do
			if o:is_player() and self:can_see_player(o) then
				o:punch(self.object, minetest.get_us_time() / 1000000 - self.last_punch, {
					full_punch_interval = 1.5,
					damage_groups = {fleshy = 2}
				})
				self.last_punch = minetest.get_us_time() / 1000000

				if o:get_hp() == 0 then
					self.target = nil
					self.destination = nil
					self.hostile = 0
					self:set_velocity({x = 0, y = 0, z = 0})
					self:animate("idle")
				end
			end
		end
	end

	-- Move towards destination
	if self.destination then
		-- Turn and move
		self.destination[1].y = pos.y
		if vector.distance(pos, self.destination[1]) <= 0.2 then
			self.object:move_to(self.destination[1])
			table.remove(self.destination, 1)
			if next(self.destination) then
				self:go_to(self.destination[1])
			else -- Reached destination
				self.destination = nil
				self:animate("idle")
				self:set_velocity({x = 0, y = 0, z = 0})
			end
		else
			-- Course-correction
			if not table.equals(self.object:get_velocity(), self.velocity) then
				self:go_to(self.destination[1])
			end
		end
	else -- Wander
		if math.random(1, 10) == 0 then -- 1 in 10 chance of being idle
			self:set_velocity({x = 0, y = 0, z = 0})
			self:animate("idle")
		elseif self.animation == "idle" then -- Once idle, 1 in 10 chance of being active again
			if math.random(1, 10) == 1 then
				-- Find a place to go if we dont have one
				local query = pathfind.find_open_near(staff.randompos(pos, 15))
				if query then
					local success, path = pathfind.tree(pos, query)
					if success then
						self.destination = pathfind.trim(path) -- Only save pivot points
						self:go_to(self.destination[1])
					end
				end
			else
				if math.random(1, 10) == 1 then -- Look in random directions
					local yaw = self.object:get_yaw()
					self:look_at(yaw + math.rad(math.random(-45, 45)), 5)
				end
			end
		end
	end
end

minetest.register_entity("ikea_staff:member", entity)
