-- vim: set ts=4 sw=4 sts=4 noet si: --
-- enchantments_workbench = ew
ew = {}
ew.i18n = core.get_translator("enchantments_workbench")
ew.modpath = core.get_modpath(core.get_current_modname())
local S = ew.i18n

ew.settings = {
	workflow_time = tonumber(core.settings:get("enchantments_workbench.workflow_time")) or 10 * 60 * 2,
	catalysts = {},
}

local function strSplit(str)
	local splt = string.split(str, ",")
	local leList = {}
	for _, x in ipairs(splt) do
		x = ("" .. x):trim()
		local splt2 = string.split(x, " ")
		if nil ~= splt2[1] and nil ~= splt2[2] then
			leList[splt2[1]] = tonumber(splt2[2]) or 10
		end
	end
	return leList
end

local function strClean(str)
	-- clean up insane escape codes from mcl_enchanting
	-- there is some insane escape characters. string.format("%q",name) shows  "\27(TFire Aspect\27E II"
	local output = string.gsub(string.gsub(string.gsub(string.format("%q",str),'.*%@mcl_[%w]+%)',""),"\\27.."," "),'"',"")
	return output
end

local function count_keys(t)
	local count = 0
	for _ in pairs(t) do
		count = count + 1
	end
	return count
end

local function get_wallclock_time()
	return os.time()
end

function ew.swap_node(pos, name)
	local node = core.get_node(pos)
	if node.name == name then
		return
	end
	node.name = name
	core.swap_node(pos, node)
end

local function tables_equals(o1, o2, ignore_mt)
	if o1 == o2 then return true end
	local o1Type = type(o1)
	local o2Type = type(o2)
	if o1Type ~= o2Type then return false end
	if o1Type ~= 'table' then return false end
	if not ignore_mt then
		local mt1 = getmetatable(o1)
		if mt1 and mt1.__eq then
			--compare using built in method
			return o1 == o2
		end
	end
	local keySet = {}
	for key1, value1 in pairs(o1) do
		local value2 = o2[key1]
		if value2 == nil or tables_equals(value1, value2, ignore_mt) == false then
			return false
		end
		keySet[key1] = true
	end
	for key2, _ in pairs(o2) do
		if not keySet[key2] then return false end
	end
	return true
end

local function get_first_free_slot(inv)
	local slots = {"input_book_1","input_book_2","catalyst"}
	for _, v in ipairs(slots) do
		if inv:get_stack(v,1):is_empty() then
			return v
		end
	end
	return "full"
end

function ew.reload_settings()
	ew.settings.workflow_time = tonumber(core.settings:get("enchantments_workbench.workflow_time")) or 10 * 60 * 2
	ew.settings.catalysts = strSplit(core.settings:get("enchantments_workbench.catalysts") or "mcl_core:lapis 5,mcl_core:emerald 10")
end

local max_enchantments = {}
core.register_on_mods_loaded(function()
	-- load max enchantment levels from Mineclonia
	for ench, def in pairs(mcl_enchanting.enchantments) do
		max_enchantments[ench] = def.max_level or 1
	end
	-- load settings
	ew.reload_settings()
end)

local function is_valid_enchantment_level(stack)
	-- purpose: the level of enchantment must be max_level or below.
	local enchantments = mcl_enchanting.get_enchantments(stack)
	for ench, level in pairs(enchantments) do
		local max_level = max_enchantments[ench]
		if level > max_level then
			return false
		end
	end
	return true
end

local function is_valid_first_book(inv, stack)
	local valid = false
	if stack:get_name() == "mcl_enchanting:book_enchanted" and is_valid_enchantment_level(stack) then
		return true
	end
	return false
end

local function is_valid_second_book(inv, stack)
	local book_1 = inv:get_stack("input_book_1",1)
	local stackname = stack:get_name()
	-- second item must be empty book, or identical book to the first
	if stackname == "mcl_books:book" then
		-- Copy (workflow 2): an empty book goes in slot 2
		return true
	elseif stackname == "mcl_enchanting:book_enchanted" then
		-- Enhance (workflow 1): book is identical to the first
		local enchantments_second = mcl_enchanting.get_enchantments(stack)
		local enchantments_first = mcl_enchanting.get_enchantments(book_1)
		-- to print, use core.serialize(enchantments_first)
		if tables_equals(enchantments_second,enchantments_first) then
			return true
		end
	-- any item not mentioned here is not valid
	return false
	end
end

local function is_valid_catalyst(inv, stack)
	local stackname = stack:get_name()
	local current_catalyst = inv:get_stack("catalyst",1)
	if current_catalyst:is_empty() then
		for k, v in pairs(ew.settings.catalysts) do
			if k == stackname then
				return true
			end
		end
	elseif stackname == current_catalyst:get_name() then
		return true
	end
	return false
end

local function calculate_output(inv)
	-- returns is_valid, output, workflow, message
	local book_1 = inv:get_stack("input_book_1",1)
	local book_2 = inv:get_stack("input_book_2",1)
	local is_valid = false
	local output, message
	local workflow = "none" -- or "copy" or "enhance"
	if (not is_valid_first_book(inv,book_1)) or (not is_valid_second_book(inv,book_2)) then
		return false, nil, nil
	end
	if book_2:get_name() == "mcl_books:book" then
		-- Copy (workflow 2)
		output = book_1 -- this is an ItemStack
		is_valid = true
		workflow = S("copy")
	elseif book_2:get_name() == "mcl_enchanting:book_enchanted" then
		-- Enhance (workflow 1)
		-- due to is_valid_second_book, we already know the second book is equivalent to the first.
		local enchantments_first = mcl_enchanting.get_enchantments(book_1)
		local enchantments_second = mcl_enchanting.get_enchantments(book_2)
		local num_enchantments = count_keys(enchantments_second)
		output = book_2
		workflow = S("enhance")
		is_valid = true
		for enchantment, level in pairs(enchantments_second) do
			if max_enchantments[enchantment] < (level + 1) then
				local enchantment_name = strClean(mcl_enchanting.get_enchantment_description(enchantment,level))
				message = "Workbench cannot enhance higher than " .. enchantment_name
				core.log("action",message)
				workflow = "none"
				is_valid = false
			else
				mcl_enchanting.enchant(output, enchantment, level + 1)
			end
		end
	end
	return is_valid, output, workflow, message
end

function ew.get_timer_function(node_normal, node_active)
	return function(pos, elapsed)
		local meta, elapsed_game_time = ew.get_delta_time(pos, elapsed)
		local inv = meta:get_inventory()
		local src_time = meta:get_float("src_time") or 0
		local current_book_1 = meta:get_string("current_book_1") or ""
		local current_book_2 = meta:get_string("current_book_2") or ""
		local book_1 = inv:get_stack("input_book_1",1)
		local book_2 = inv:get_stack("input_book_2",1)
		local catalyst = inv:get_stack("catalyst",1)
		local wallclock_time = get_wallclock_time()
		local work_time = meta:get_float("work_time") or ew.settings.workflow_time
		if book_1:get_name() ~= current_book_1 or book_2:get_name() ~= current_book_2 then
			-- reset workbench progress in this case
			core.log("action","Workbench is resetting progress due to different items being placed")
			src_time = 0
			current_book_1 = book_1:get_name()
			current_book_2 = book_2:get_name()
			work_time = ew.settings.workflow_time
		end
		local active = true
		local is_valid, output_object, workflow, message1 = calculate_output(inv)
		if message1 == nil then message1 = "" end
		local message2 = meta:get_string("message2") or ""
		local message2_lasttime = meta:get_int("message2_lasttime") or get_wallclock_time()
		local message3 = ""
		if not is_valid then
			core.log("action","Workbench has no valid workflows to perform")
			active = false
		else
		end
		local update = true
		local inv_changed = false
		while (elapsed_game_time > 0.00001) and update do
			local el = elapsed_game_time * 1 -- factor; not implemented here
			if is_valid then
				if not inv:room_for_item("output_item", output_object) then
					is_valid = false
				end
			end
			if is_valid then
				el = math.min(el, work_time - src_time)
				inv:set_stack("output_preview",1, output_object)
			end
			if is_valid and not active then
				-- NOP
			elseif active then
				-- NOP
			end
			if is_valid and active then
				core.log("action","Workbench calculates the " .. workflow .. " will produce " .. output_object:to_string())
				if catalyst:get_count() > 0 then
					local used_catalyst = catalyst:take_item()
					inv:set_stack("catalyst", 1, catalyst)
					catalyst = inv:get_stack("catalyst",1)
					local add_percent = (1.0/ew.settings.catalysts[used_catalyst:get_name()])
					local add_seconds = add_percent * ew.settings.workflow_time
					local catalyst_nodename = used_catalyst:get_name()
					local catalyst_name = used_catalyst:get_short_description()
					core.log("action","Using catalyst " .. catalyst_nodename .. " which adds " .. add_percent*100 .. "% (" .. add_seconds .. " seconds)")
					message2 = "Using catalyst " .. catalyst_name .. " which adds " .. add_percent*100 .. "% (" .. add_seconds .. " seconds)"
					message2_lasttime = wallclock_time
					src_time = src_time + add_seconds
				end
				message3 = workflow
				-- do not bother calculating the attributes in text, now that we use an inventory stack for this.
				--[[
				message3 = "The " .. workflow .. " operation will produce"
				local these_enchantments = mcl_enchanting.get_enchantments(output_object)
				local x = 0
				for ench, level in pairs(these_enchantments) do
					x = x + 1
					message3 = message3 .. (x == 1 and " " or ", " ) .. mcl_enchanting.get_enchantment_description(ench,level)
				end
				--]]
				src_time = src_time + el
				-- place result in output_item if done
				if src_time >= work_time then
					inv:add_item("output_item", output_object)
					if workflow == S("enhance") then
						inv:set_stack("input_book_1", 1, ItemStack(""))
					end
					book_2:take_item()
					inv:set_stack("input_book_2", 1, book_2)
					book_1 = inv:get_stack("input_book_1",1)
					book_2 = inv:get_stack("input_book_2",1)
					src_time = 0
					inv_changed = true
					update = false
				end
			end
			elapsed_game_time = elapsed_game_time - el
		end -- end while elapsed_game_time

		if inv_changed then
			mcl_redstone.update_comparators(pos)
		end
		if book_1 and book_1:is_empty() then
			src_time = 0
		end
		local def = core.registered_nodes[node_normal]
		local name = S("Enchantment Workbench")
		if def and def.description then
			name = def._tt_original_description or def.description
		end
		local formspec = ew.get_workbench_inactive_formspec(message1)
		local item_percent = 0
		if is_valid then
			item_percent = math.floor(src_time / work_time * 100)
		end
		local seconds_remaining = math.floor(work_time - src_time)
		if is_valid and active then
			core.log("action","Workbench progress is " .. item_percent .. "%, with " .. seconds_remaining .. " seconds remaining.")
		end
		local result = false
		-- keep the last message2 displayed for 5 seconds
		if wallclock_time - message2_lasttime >= 5 then
			message2 = ""
		end
		if active then
			formspec = ew.get_workbench_active_formspec(item_percent, seconds_remaining, message1, message2, message3)
			-- there is no continued progress.
			ew.swap_node(pos, node_active)
			result = true
		else
			ew.swap_node(pos, node_normal)
			-- stop timer on the inactive furnace
			core.get_node_timer(pos):stop()
		end
		-- set meta values
		meta:set_float("src_time",src_time)
		if book_1 then
			meta:set_string("current_book_1", book_1:get_name())
		else
			meta:set_string("current_book_1", "")
		end
		if book_2 then
			meta:set_string("current_book_2", book_2:get_name())
		else
			meta:set_string("current_book_2", "")
		end
		meta:set_float("work_time",work_time)
		meta:set_string("formspec", formspec)
		--meta:set_float("dialog_el",dialog_el)
		meta:set_float("message2_lasttime",message2_lasttime)
		meta:set_string("message2",message2)
		return result
	end
end

function ew.get_delta_time(pos, elapsed)
	local meta = core.get_meta(pos)
	local time_speed = tonumber(core.settings:get("time_speed")) or 72
	local current_game_time
	if (time_speed < 0.1) then
		return meta, elapsed
	else
		local time_multiplier = 86400 / time_speed
		current_game_time = .0 + ((core.get_day_count() + core.get_timeofday()) * time_multiplier)
	end
	local last_game_time = meta:get_string("last_gametime")
	if last_game_time then
		last_game_time = tonumber(last_game_time)
	end
	if not last_game_time or last_game_time < 1 then
		last_game_time = current_game_time - 0.1
	elseif last_game_time == current_game_time then
		current_game_time = current_game_time + 1.0
	end

	local elapsed_game_time = .0 + current_game_time - last_game_time

	meta:set_string("last_gametime", tostring(current_game_time))

	return meta, elapsed_game_time
end

function ew.workbench_reset_delta_time(pos)
	local meta = core.get_meta(pos)
	local inv = meta:get_inventory()
	local time_speed = tonumber(core.settings:get("time_speed")) or 72
	if (time_speed < 0.1) then
		return
	end
	local time_multiplier = 86400 / time_speed
	local current_game_time = .0 + ((core.get_day_count() + core.get_timeofday()) * time_multiplier)
	local last_game_time = meta:get_string("last_gametime")
	if last_game_time then
		last_game_time = tonumber(last_game_time)
	end
	if not last_game_time or last_game_time < 1 or math.abs(last_game_time - current_game_time) <= 1.5 then
		return
	end
	meta:set_string("last_gametime", tostring(current_game_time))
	meta:set_float("work_time", ew.settings.workflow_time)
end

local workbench = "enchantments_workbench:enchantment_workbench"

function ew.allow_metadata_inventory_put(pos, listname, index, stack, player)
	local meta = core.get_meta(pos)
	local inv = meta:get_inventory()
	local stackname = stack:get_name()
	if listname == "input_book_1" then
		if (not is_valid_first_book(inv, stack)) or inv:get_stack("input_book_1",1):get_count() > 0 then
			return 0
		elseif stackname == "mcl_enchanting:book_enchanted" then
			-- Only operate on one book at a time.
			return 1
		end
	elseif listname == "input_book_2" and is_valid_second_book(inv, stack) then
		return stack:get_count()
	elseif listname == "catalyst" and is_valid_catalyst(inv, stack) then
		local stack1 = ItemStack(stack):take_item()
		if inv:room_for_item(listname, stack) then
			return stack:get_count()
		elseif inv:room_for_item(listname, stack1) then
			local tc = inv:get_stack(listname, 1):get_count()
			return stack:get_stack_max() - tc
		end
	elseif listname == "sorter" then
		local target = get_first_free_slot(inv)
		if stackname == "mcl_books:book" then
			target = "input_book_2"
		elseif is_valid_catalyst(inv,stack) then
			target = "catalyst"
		end
		if target == "input_book_1" then
			return is_valid_first_book(inv, stack) == true and 1 or 0
		elseif (target == "input_book_2" and is_valid_second_book(inv, stack)) or
			(target == "catalyst" and is_valid_catalyst(inv, stack)) then
			local stack1 = ItemStack(stack):take_item()
			if inv:room_for_item(target, stack) then
				return stack:get_count()
			elseif inv:room_for_item(target, stack1) then
				local tc = inv:get_stack(target, 1):get_count()
				return stack:get_stack_max() - tc
			end
		end
	end
	-- By default, nothing else is allowed
	return 0
end

function ew.allow_metadata_inventory_move(pos, from_list, from_index, to_list, to_index, _, player)
	local disallowed = {"sorter","output_preview"}
	for _, v in ipairs(disallowed) do
		if from_list == v or to_list == v then
			return 0
		end
	end
	disallowed = {"input_book_1","catalyst","output_preview"}
	for _, v in ipairs(disallowed) do
		if to_list == "output_item" and from_list == v then
			return 0
		end
	end
	local meta = core.get_meta(pos)
	local inv = meta:get_inventory()
	local stack = inv:get_stack(from_list, from_index)
	return ew.allow_metadata_inventory_put(pos, to_list, to_index, stack, player)
end

function ew.allow_metadata_inventory_take(pos, listname, _, stack, player)
	if listname == "sorter" then return 0 end
	if listname == "output_preview" then return 0 end
	local name = player:get_player_name()
	if core.is_protected(pos, name) then
		core.record_protection_violation(pos, name)
		return 0
	end
	return stack:get_count()
end

function ew.on_metadata_inventory_take(pos, listname, _, stack, player)
	mcl_redstone.update_comparators(pos)
end

function ew.on_metadata_inventory_move(pos, from_list, _, _, _, _, player)
	-- NOP for enchantments_workbench
end

function ew.on_metadata_inventory_put(pos, listname, index, stack, player)
	local meta = core.get_meta(pos)
	local inv = meta:get_inventory()
	local stackname = stack:get_name()
	if listname == "sorter" then
		local target = get_first_free_slot(inv)
		if is_valid_catalyst(inv, stack) then
			target = "catalyst"
		elseif stackname == "mcl_books:book" then
			target = "input_book_2"
		end
		if target == "full" then
			core.log("action","Workbench is full and cannot receive " .. stackname)
			return 0
		else
			inv:add_item(target,stack)
			inv:set_stack("sorter",1,ItemStack(""))
		end
	end
end

ew.tpl_workbench_node = {
	groups = {cracky=3, pickaxey=2, container=4, deco_block=1, enchantments_workbench=1, unmovable_by_piston=1 },
	after_dig_node = mcl_util.drop_items_from_meta_container({"input_book_1","input_book_2","output_item","sorter","catalyst"}),
	on_timer = ew.get_timer_function(workbench, workbench.."_active"),
	allow_metadata_inventory_put = ew.allow_metadata_inventory_put,
	allow_metadata_inventory_move = ew.allow_metadata_inventory_move,
	allow_metadata_inventory_take = ew.allow_metadata_inventory_take,
	on_metadata_inventory_put = ew.on_metadata_inventory_put,
	on_metadata_inventory_move = ew.on_metadata_inventory_move,
	on_metadata_inventory_take = ew.on_metadata_inventory_take,
	_mcl_blast_resistance = 1200,
	_mcl_hardness = 5,
	_tt_help = S("Uses lapis lazuli or emeralds to Enhance or Copy enchanted books"),
	_doc_items_longdesc = S("Enchantment workbenches improve enchanted books by consuming lapis lazuli or emeralds."),
	_doc_items_usagehelp =
		S("Use the enchantment workbench to open the menu.") .. "\n" ..
		S("Place the catalyst (lapis lazuli or emeralds) in the lower slot and the book to copy or enhance in the upper left slot.") .. "\n" ..
		S("Place in the upper right slot an the empty book (to Copy the first book) or a copy of the first book (to Enhance).") .. "\n" ..
		S("Enhancing works by combining 2 of the exact same enchantment into the next level. E.g., Efficiency II+Efficiency II=Efficiency III.") .. "\n" ..
		S("The anvil uses experience to accomplish enhancing, and this workbench uses lapis/emerald to accomplish it."),
	drawtype = "nodebox",
	node_box = {
		type = "fixed",
		fixed = {
			{-0.5, -0.5, -0.5, 0.5, 0.25, 0.5},
			{-0.5+1/16, -0.5, -0.5+1/16, 0.5-1/16, 0.25+1/16, 0.5-1/16},
		},
	},
}

ew.tpl_workbench_node_normal = table.merge(ew.tpl_workbench_node,{
	description = S("Enchantment Workbench"),
	tiles = {
		"mcl_enchanting_table_top.png",
		"mcl_enchanting_table_bottom.png",
		"enchantments_workbench_side.png",
		"enchantments_workbench_side.png",
		"enchantments_workbench_side.png",
		"enchantments_workbench_side.png"
	},
	_doc_items_hidden = false,
	on_metadata_inventory_move = function(pos, from_list, from_index, to_list, to_index, count, player)
		ew.workbench_reset_delta_time(pos)
		core.get_node_timer(pos):start(1.0)
		ew.on_metadata_inventory_move(pos, from_list, from_index, to_list, to_index, count, player)
	end,
	on_metadata_inventory_put = function(pos, listname, index, stack, player)
		-- reset accumulated game time when player changes inventory slots on workbench
		ew.workbench_reset_delta_time(pos)
		-- start timer function: it will sort out whether workbench can operate or not
		core.get_node_timer(pos):start(1.0)
		ew.on_metadata_inventory_put(pos, listname, index, stack, player)
	end,
	on_metadata_inventory_take = function(pos, listname, index, stack, player)
		-- reset accumulated game time when player changes inventory slots on workbench
		ew.workbench_reset_delta_time(pos)
		core.get_node_timer(pos):start(1.0)
		ew.on_metadata_inventory_take(pos, listname, index, stack, player)
	end,
})

ew.tpl_workbench_node_active = table.merge(ew.tpl_workbench_node,{
	description = S("Active Enchantment Workbench"),
	tiles = {
		"mcl_enchanting_table_top.png",
		"mcl_enchanting_table_bottom.png",
		"enchantments_workbench_active_side.png",
		"enchantments_workbench_active_side.png",
		"enchantments_workbench_active_side.png",
		"enchantments_workbench_active_side.png"
	},
	groups = {cracky=3, pickaxey=2, container=4, deco_block=1, enchantments_workbench=1, enchantments_workbench_active=1, not_in_creative_inventory=1, unmovable_by_piston=1 },
	_doc_items_create_entry = false
})

core.register_node(workbench, table.merge(ew.tpl_workbench_node_normal,{
	on_construct = function(pos)
		local meta = core.get_meta(pos)
		local inv = meta:get_inventory()
		inv:set_size("sorter", 1)
		inv:set_size("input_book_1", 1)
		inv:set_size("input_book_2", 1)
		inv:set_size("output_item", 1)
		inv:set_size("catalyst", 1)
		inv:set_size("output_preview", 1)
		local form = ew.get_workbench_inactive_formspec("")
		meta:set_string("formspec", form)
	end,
	on_timer = ew.get_timer_function(workbench, workbench.."_active")
}))
core.register_node(workbench.."_active", table.merge(ew.tpl_workbench_node_active,{
	on_timer = ew.get_timer_function(workbench, workbench.."_active"),
	drop = workbench,
	_mcl_baseitem = workbench
}))

function ew.get_workbench_inactive_formspec(message1)
	-- Define the formspec version and size
	local x1,x2,x3,x4,y1,y2,go
	x1 = 0.875
	x2 = 2.875
	x3 = 4.875
	x4 = 6.875
	x5 = 8.875
	y1 = 1.25
	y2 = 3.25
	go = 0.25 -- graphic offset
	local formspec = {
		"formspec_version[4]",
		"size[11.75,11.425]",

		-- Labels
		"label[0.375,0.375;", core.formspec_escape(S("Enchantment Workbench")) , "]",
		"label["..x1..","..y1..";".. core.formspec_escape(S("Input Book 1")), "]",
		"label["..x2..","..y1..";".. core.formspec_escape(S("Input Book 2")), "]",
		"label["..x4..","..y1..";".. core.formspec_escape(S("Preview")), "]",
		"label["..x5..","..y1..";".. core.formspec_escape(S("Output")), "]",
		"label["..x2..","..y2..";".. core.formspec_escape(S("Catalyst")), "]",
		"label[0.375,5.95;", core.formspec_escape(S("Inventory")), "]",

		-- Slot backgrounds for the node's inventory
		mcl_formspec.get_itemslot_bg_v4(x1, y1+go, 1, 1, 0, "mcl_formspec_itemslot.png^mcl_book_book_empty_slot.png"),
		mcl_formspec.get_itemslot_bg_v4(x2, y1+go, 1, 1, 0, "mcl_formspec_itemslot.png^mcl_book_book_empty_slot.png"),
		mcl_formspec.get_itemslot_bg_v4(x5, y1+go, 1, 1),
		mcl_formspec.get_itemslot_bg_v4(x2, y2+go, 1, 1),

		-- Slot lists for the node's inventory
		"list[context;input_book_1;"  ..x1..","..(y1+go)..";1,1;]",
		"list[context;input_book_2;"  ..x2..","..(y1+go)..";1,1;]",
		"list[context;output_item;"   ..x5..","..(y1+go)..";1,1;]",
		"list[context;catalyst;"      ..x2..","..(y2+go)..";1,1;]",

		"label[0.375,5.0;", core.formspec_escape(message1), "]",

		-- Slot backgrounds for the player's inventory
		mcl_formspec.get_itemslot_bg_v4(0.375, 6.2, 9, 3),
		mcl_formspec.get_itemslot_bg_v4(0.375, 10.15, 9, 1),

		-- Slot lists for the player's inventory
		"list[current_player;main;0.375,6.2;9,3;9]",
		"list[current_player;main;0.375,10.15;9,1;]",

		-- Listrings to allow moving items between lists
		"listring[context;output_item]",
		"listring[current_player;main]",
		"listring[context;sorter]",
		"listring[current_player;main]",
		"listring[context;input_book_1]",
		"listring[current_player;main]",
		"listring[context;input_book_2]",
		"listring[current_player;main]",
		"listring[context;catalyst]",
		"listring[current_player;main]",
	}

	-- Return the concatenated formspec string
	return table.concat(formspec)
end

function ew.get_workbench_active_formspec(item_percent, seconds_remaining, message1, message2, message3)
	-- Define the formspec version and size
	local x1,x2,x3,x4,y1,y2,go,extra
	x1 = 0.875
	x2 = 2.875
	x3 = 4.875
	x4 = 6.875
	x5 = 8.875
	y1 = 1.25
	y2 = 3.25
	go = 0.25 -- graphic offset
	extra = 1.5
	local formspec = {
		"formspec_version[4]",
		"size[11.75,11.425]",

		-- Labels
		"label[0.375,0.375;", core.formspec_escape(S("Enchantment Workbench") .. " " .. S("(in progress)")), "]",
		"label["..x1..","..y1..";".. core.formspec_escape(S("Input Book 1")), "]",
		"label["..x2..","..y1..";".. core.formspec_escape(S("Input Book 2")), "]",
		"label["..x4..","..y1..";".. core.formspec_escape(S("Preview")), "]",
		"label["..x5..","..y1..";".. core.formspec_escape(S("Output")), "]",
		"label["..x2..","..y2..";".. core.formspec_escape(S("Catalyst")), "]",
		"label[0.375,5.95;", core.formspec_escape(S("Inventory")), "]",

		-- Slot backgrounds for the node's inventory
		mcl_formspec.get_itemslot_bg_v4(x1, y1+go, 1, 1, 0, "mcl_formspec_itemslot.png^mcl_book_book_empty_slot.png"),
		mcl_formspec.get_itemslot_bg_v4(x2, y1+go, 1, 1, 0, "mcl_formspec_itemslot.png^mcl_book_book_empty_slot.png"),
		mcl_formspec.get_itemslot_bg_v4(x5, y1+go, 1, 1),
		mcl_formspec.get_itemslot_bg_v4(x2, y2+go, 1, 1),

		-- item workflow percentage
		"image["..x3..","..(y1+0.25)..";1.5,1;gui_furnace_arrow_bg.png^[lowpart:" ..
		(item_percent) .. ":gui_furnace_arrow_fg.png^[transformR270]",
		"label["..x3..","..(y1+go+extra)..";", core.formspec_escape(S("Seconds remaining: ") .. seconds_remaining), "]",

		"label[0.375,4.5;", core.formspec_escape(message1), "]",
		"label[0.375,5.0;", core.formspec_escape(message2), "]",
		"label["..x3..","..(y1+go+0.8)..";", core.formspec_escape(message3), "]",

		-- Slot lists for the node's inventory
		"list[context;input_book_1;"  ..x1..","..(y1+go)..";1,1;]",
		"list[context;input_book_2;"  ..x2..","..(y1+go)..";1,1;]",
		"list[context;output_preview;"..x4..","..(y1+go)..";1,1;]",
		"list[context;output_item;"   ..x5..","..(y1+go)..";1,1;]",
		"list[context;catalyst;"      ..x2..","..(y2+go)..";1,1;]",

		-- Slot backgrounds for the player's inventory
		mcl_formspec.get_itemslot_bg_v4(0.375, 6.2, 9, 3),
		mcl_formspec.get_itemslot_bg_v4(0.375, 10.15, 9, 1),

		-- Slot lists for the player's inventory
		"list[current_player;main;0.375,6.2;9,3;9]",
		"list[current_player;main;0.375,10.15;9,1;]",

		-- Listrings to allow moving items between lists
		"listring[context;output_item]",
		"listring[current_player;main]",
		"listring[context;sorter]",
		"listring[current_player;main]",
		"listring[context;input_book_1]",
		"listring[current_player;main]",
		"listring[context;input_book_2]",
		"listring[current_player;main]",
		"listring[context;catalyst]",
		"listring[current_player;main]",
	}

	-- Return the concatenated formspec string
	return table.concat(formspec)
end

-- Recipes
core.register_craft({
	output = "enchantments_workbench:enchantment_workbench",
	recipe = {
		{"", "mcl_enchanting:table", ""},
		{"mcl_core:diamondblock","mcl_core:obsidian","mcl_core:emeraldblock"},
		{"mcl_core:obsidian", "mcl_core:obsidian", "mcl_core:obsidian"}
	}
})

core.register_craft({
	output = "enchantments_workbench:enchantment_workbench",
	recipe = {
		{"", "mcl_enchanting:table", ""},
		{"mcl_core:emeraldblock","mcl_core:obsidian","mcl_core:diamondblock"},
		{"mcl_core:obsidian", "mcl_core:obsidian", "mcl_core:obsidian"}
	}
})

core.register_lbm({
	label = "Enchantment workbench converts inputs to output",
	name = "enchantments_workbench:update_coolsneak",
	nodenames = { "group:enchantments_workbench", "group:enchantments_workbench_active" },
	run_at_every_load = false,
	action = function(pos, node)
		local m = core.get_meta(pos)
		local inv = m:get_inventory()
		inv:set_size("sorter", 1)
		if core.get_item_group(node.name, "enchantments_workbench_active") == 0 then
			local def = core.registered_nodes[node.name]
			local name = S("Enchantment Workbench")
			if def and def.description then
				name = def._tt_original_description or def.description
			end
			m:set_string("formspec", ew.get_workbench_inactive_formspec())
		end
	end,
})
