lava_furnace = {}
local modname = core.get_current_modname()
local S = core.get_translator(modname)
local C = core.colorize
local F = core.formspec_escape

local LIGHT_ACTIVE_FURNACE = 13

-- Components for MTG
-- List of sound handles for active furnace
local furnace_fire_sounds = {}
local function stop_furnace_sound(pos, fadeout_step)
	local hash = core.hash_node_position(pos)
	local sound_ids = furnace_fire_sounds[hash]
	if sound_ids then
		for _, sound_id in ipairs(sound_ids) do
			core.sound_fade(sound_id, -1, 0)
		end
		furnace_fire_sounds[hash] = nil
	end
end
local function add_item_or_drop(inv, pos, item)
	local leftover = inv:add_item("dst", item)
	if not leftover:is_empty() then
		local above = vector.offset(pos, 0, 1, 0)
		local drop_pos = core.find_node_near(pos, 1, {"air"}) or above
		core.item_drop(leftover, nil, drop_pos)
	end
end
local function apply_logger(def)
	default.set_inventory_action_loggers(def, "furnace")
	return def
end

local gameid = core.get_game_info().id
if gameid ~= "mineclone2" and gameid ~= "VoxeLibre" and gameid ~= "mineclonia" and gameid ~= "minetest" then
	core.log("warning","[" .. modname .. "] expected a gameid of mineclonia or VoxeLibre. Errors might occur.")
end
-- simplify future if statements
if gameid == "mineclone2" or gameid == "VoxeLibre" then
	gameid = "VoxeLibre"
end

local table_merge = nil
if table.merge ~= nil then
	table_merge = table.merge
elseif mcl_util ~= nil and mcl_util.table_merge ~= nil then
	-- voxelibre
	table_merge = mcl_util.table_merge
else
	-- https://codeberg.org/mineclonia/mineclonia/src/branch/main/mods/CORE/mcl_util/table.lua
	-- table.update and table.merge copied
	local table_update = function (t, ...)
		for _, to in ipairs{...} do
			for k, v in pairs(to) do
				t[k] = v
			end
		end
		return t
	end
	table_merge = function(t, ...)
		local t2 = table.copy(t)
		return table_update(t2, ...)
	end
end

-- this needs to be more than 1 for the larger factor numbers to work,
-- probably because the timer/factor resets for each new piece of "fuel"
-- but this means the if you pick up the lava source behind the furnace,
-- you have this much time left to cook with.
local mod_default_lava_fuel_time = 30
-- how fast the lava furnace works, where 1 is same speed as ordinary furnace
local mod_default_lava_factor = 4
local mod_default_fuel_nodes = "mcl_core:lava_source,mcl_core:lava_flowing,default:lava_source,default:lava_flowing"

local adjacents = {
	vector.new(1 , 0, 0),
	vector.new(-1, 0, 0),
	vector.new(0 , 0, 1),
	vector.new(0 , 0,-1),
	vector.new(0 , 1, 0),
	vector.new(0 ,-1, 0)
}  

-- k_ambient_light has <node> <number>,<node> <number> splitting if we need it
local function strSplit(str)
	local splt = string.split(str, ",")
	return splt
	--[]
end

function lava_furnace.get_fuel_nodes()
	local fuel_nodes = core.settings:get("lava_furnace.fuel_nodes")
	if fuel_nodes == nil then fuel_nodes = mod_default_fuel_nodes end
	core.log("verbose","Using fuel_nodes: " .. fuel_nodes)
	local fuel_nodes_table = strSplit(fuel_nodes)
	return fuel_nodes_table
end

--
-- Formspecs
--

local function get_active_formspec(fuel_percent, item_percent, name)
	if gameid == "mineclonia" or gameid == "VoxeLibre" then
		return table.concat({
			"formspec_version[4]",
			"size[11.75,10.425]",
			"label[0.375,0.375;" .. F(C(mcl_formspec.label_color, name)) .. "]",
			mcl_formspec.get_itemslot_bg_v4(3.5, 0.75, 1, 1),
			"list[context;src;3.5,0.75;1,1;]",

			"image[3.5,2;1,1;default_furnace_fire_bg.png^[lowpart:" ..
			(100 - fuel_percent) .. ":default_furnace_fire_fg.png]",

			mcl_formspec.get_itemslot_bg_v4(3.5, 3.25, 1, 1),
			"list[context;fuel;3.5,3.25;1,1;]",

			"image[5.25,2;1.5,1;gui_furnace_arrow_bg.png^[lowpart:" ..
			(item_percent) .. ":gui_furnace_arrow_fg.png^[transformR270]",
			mcl_formspec.get_itemslot_bg_v4(7.875, 2, 1, 1, 0.2),
			"list[context;dst;7.875,2;1,1;]",

			"label[0.375,4.7;" .. F(C(mcl_formspec.label_color, S("Inventory"))) .. "]",
			mcl_formspec.get_itemslot_bg_v4(0.375, 5.1, 9, 3),
			"list[current_player;main;0.375,5.1;9,3;9]",

			mcl_formspec.get_itemslot_bg_v4(0.375, 9.05, 9, 1),
			"list[current_player;main;0.375,9.05;9,1;]",

			"image_button[7.85,0.6;1,1;craftguide_book.png;craftguide;]"..
			"tooltip[craftguide;"..core.formspec_escape(S("Recipe book")).."]"..

			"listring[context;dst]",
			"listring[current_player;main]",
			"listring[context;sorter]",
			"listring[current_player;main]",
			"listring[context;src]",
			"listring[current_player;main]",
			"listring[context;fuel]",
			"listring[current_player;main]",
		})
	elseif gameid == "minetest" then -- minetest_game
		-- https://github.com/luanti-org/minetest_game/blob/master/mods/default/furnace.lua#L14-L30
		return "size[8,8.5]"..
			"list[context;src;2.75,0.5;1,1;]"..
			"list[context;fuel;2.75,2.5;1,1;]"..
			"image[2.75,1.5;1,1;default_furnace_fire_bg.png^[lowpart:"..
			(fuel_percent)..":default_furnace_fire_fg.png]"..
			"image[3.75,1.5;1,1;gui_furnace_arrow_bg.png^[lowpart:"..
			(item_percent)..":gui_furnace_arrow_fg.png^[transformR270]"..
			"list[context;dst;4.75,0.96;2,2;]"..
			"list[current_player;main;0,4.25;8,1;]"..
			"list[current_player;main;0,5.5;8,3;8]"..
			"listring[context;dst]"..
			"listring[current_player;main]"..
			"listring[context;src]"..
			"listring[current_player;main]"..
			"listring[context;fuel]"..
			"listring[current_player;main]"..
			default.get_hotbar_bg(0, 4.25)
	else
		error("[lava_furnace] No formspec for game: "..gameid)
	end
end

local function get_inactive_formspec(name)
	if gameid == "mineclonia" or gameid == "VoxeLibre" then
		return table.concat({
			"formspec_version[4]",
			"size[11.75,10.425]",
			"label[0.375,0.375;" .. F(C(mcl_formspec.label_color, name)) .. "]",
			mcl_formspec.get_itemslot_bg_v4(3.5, 0.75, 1, 1),
			"list[context;src;3.5,0.75;1,1;]",

			"image[3.5,2;1,1;default_furnace_fire_bg.png]",

			mcl_formspec.get_itemslot_bg_v4(3.5, 3.25, 1, 1),
			"list[context;fuel;3.5,3.25;1,1;]",

			"image[5.25,2;1.5,1;gui_furnace_arrow_bg.png^[transformR270]",

			mcl_formspec.get_itemslot_bg_v4(7.875, 2, 1, 1, 0.2),
			"list[context;dst;7.875,2;1,1;]",

			"label[0.375,4.7;" .. F(C(mcl_formspec.label_color, S("Inventory"))) .. "]",
			mcl_formspec.get_itemslot_bg_v4(0.375, 5.1, 9, 3),
			"list[current_player;main;0.375,5.1;9,3;9]",

			mcl_formspec.get_itemslot_bg_v4(0.375, 9.05, 9, 1),
			"list[current_player;main;0.375,9.05;9,1;]",

			"image_button[7.85,0.6;1,1;craftguide_book.png;craftguide;]"..
			"tooltip[craftguide;"..core.formspec_escape(S("Recipe book")).."]"..

			"listring[context;dst]",
			"listring[current_player;main]",
			"listring[context;sorter]",
			"listring[current_player;main]",
			"listring[context;src]",
			"listring[current_player;main]",
			"listring[context;fuel]",
			"listring[current_player;main]",
		})
	elseif gameid == "minetest" then -- minetest_game
		-- https://github.com/luanti-org/minetest_game/blob/master/mods/default/furnace.lua#L34-L48
		return table.concat({
			"size[8,8.5]",
			"list[context;src;2.75,0.5;1,1;]",
			"list[context;fuel;2.75,2.5;1,1;]",
			"image[2.75,1.5;1,1;default_furnace_fire_bg.png]",
			"image[3.75,1.5;1,1;gui_furnace_arrow_bg.png^[transformR270]",
			"list[context;dst;4.75,0.96;2,2;]",
			"list[current_player;main;0,4.25;8,1;]",
			"list[current_player;main;0,5.5;8,3;8]",
			"listring[context;dst]",
			"listring[current_player;main]",
			"listring[context;src]",
			"listring[current_player;main]",
			"listring[context;fuel]",
			"listring[current_player;main]",
			default.get_hotbar_bg(0, 4.25)
		})
	else
		error("[lava_furnace] No formspec for game: " .. gameid)
	end
end


function lava_furnace.receive_fields(_, _, fields, sender)
	if fields.craftguide then
		mcl_craftguide.show(sender:get_player_name())
	end
end

function lava_furnace.give_xp(pos, player)
	if gameid ~= "mineclonia" and gameid ~= "VoxeLibre" then
		return
	end
	local meta = core.get_meta(pos)
	local dir = vector.divide(core.facedir_to_dir(core.get_node(pos).param2), -1.95)
	local xp = meta:get_int("xp")
	if xp > 0 then
		if player then
			mcl_experience.add_xp(player, xp)
		else
			mcl_experience.throw_xp(vector.add(pos, dir), xp)
		end
		meta:set_int("xp", 0)
	end
end

--
-- Node callback functions that are the same for active and inactive furnace
--
function lava_furnace.is_cookable(stack, pos)
	if pos then
		local def = core.registered_nodes[core.get_node(pos).name]
		if def and def._lava_furnace_cook_group and core.get_item_group(stack:get_name(), def._lava_furnace_cook_group) == 0 then return false end
	end
	return core.get_craft_result({method = "cooking", width = 1, items = {stack}}).time ~= 0
end

local function sort_stack(stack, pos)
	if lava_furnace.is_cookable(stack, pos) then
		if mcl_util.is_fuel(stack) and not core.get_meta(pos):get_inventory():room_for_item("src", stack) then
			return "fuel"
		end
		return "src"
	elseif mcl_util.is_fuel(stack) or ( lava_furnace.is_cookable(ItemStack("mcl_sponges:sponge_wet"), pos) and stack:get_name() == "mcl_buckets:bucket_empty" ) then
		return "fuel"
	end
end

function lava_furnace.allow_metadata_inventory_put(pos, listname, index, stack, player)
	local name = player:get_player_name()
	if core.is_protected(pos, name) then
		core.record_protection_violation(pos, name)
		return 0
	end
	local meta = core.get_meta(pos)
	local inv = meta:get_inventory()
	if listname == "fuel" then
		-- Special case: empty bucket (not a fuel, but used for sponge drying)
		if stack:get_name() == "mcl_buckets:bucket_empty" then
			return inv:get_stack(listname, index):get_count() == 0 and 1 or 0
		end
		-- fuel not implemented in lava_furnace
		return 0
	elseif listname == "src" then
		return stack:get_count()
	elseif listname == "dst" then
		return 0
	elseif listname == "sorter" then
		local inv = core.get_meta(pos):get_inventory()
		local trg = sort_stack(stack, pos)
		if trg then
			-- if target inventory would be fuel when shift-clicking a stack to
			-- move it to this lava furnace, then do not allow it
			if trg == "fuel" then return 0 end
			local stack1 = ItemStack(stack):take_item()
			if inv:room_for_item(trg, stack) then
				if stack:get_name() == "mcl_buckets:bucket_empty" then
					return inv:get_stack("fuel", 1):get_count() == 0 and 1 or 0
				end
				return stack:get_count()
			elseif inv:room_for_item(trg, stack1) then
				local tc = inv:get_stack(trg, 1):get_count()
				return stack:get_stack_max() - tc
			end
		end
		return 0
	end
end

function lava_furnace.allow_metadata_inventory_move(pos, from_list, from_index, to_list, to_index, _, player)
	if from_list == "sorter" or to_list == "sorter" then return 0 end
	local meta = core.get_meta(pos)
	local inv = meta:get_inventory()
	local stack = inv:get_stack(from_list, from_index)
	return lava_furnace.allow_metadata_inventory_put(pos, to_list, to_index, stack, player)
end

function lava_furnace.allow_metadata_inventory_take(pos, listname, _, stack, player)
	if listname == "sorter" then return 0 end
	local name = player:get_player_name()
	if core.is_protected(pos, name) then
		core.record_protection_violation(pos, name)
		return 0
	end
	return stack:get_count()
end

function lava_furnace.on_metadata_inventory_take(pos, listname, _, stack, player)
	-- Award smelting achievements
	if listname == "dst" then
		if stack:get_name() == "mcl_core:iron_ingot" then
			awards.unlock(player:get_player_name(), "mcl:acquireIron")
		elseif stack:get_name() == "mcl_fishing:fish_cooked" then
			awards.unlock(player:get_player_name(), "mcl:cookFish")
		elseif stack:get_name() == "mcl_fishing:salmon_cooked" then
			awards.unlock(player:get_player_name(), "mcl:cookFish")
		end
		lava_furnace.give_xp(pos, player)
	end
	if gameid == "mineclonia" then
		mcl_redstone.update_comparators(pos)
	end
end

function lava_furnace.on_metadata_inventory_move(pos, from_list, _, _, _, _, player)
	if from_list == "dst" then
		lava_furnace.give_xp(pos, player)
	end
end

function lava_furnace.on_metadata_inventory_put(pos, listname, _, stack, _)
	if listname == "sorter" then
		local inv = core.get_meta(pos):get_inventory()
		inv:add_item(sort_stack(stack, pos), stack)
		inv:set_stack("sorter", 1, ItemStack(""))
	end
	if gameid == "mineclonia" then
		mcl_redstone.update_comparators(pos)
	end
end

function lava_furnace.swap_node(pos, name)
	local node = core.get_node(pos)
	if node.name == name then
		return
	end
	node.name = name
	core.swap_node(pos, node)
end

function lava_furnace.furnace_reset_delta_time(pos)
	local meta = core.get_meta(pos)
	local time_speed = tonumber(core.settings:get("time_speed")) or 72
	if (time_speed < 0.1) then
		return
	end
	local time_multiplier = 86400 / time_speed
	local current_game_time = .0 + ((core.get_day_count() + core.get_timeofday()) * time_multiplier)

	-- TODO: Change meta:get/set_string() to get/set_float() for "last_gametime".
	-- In Windows *_float() works OK but under Linux it returns rounded unusable values like 449540.000000000
	local last_game_time = meta:get_string("last_gametime")
	if last_game_time then
		last_game_time = tonumber(last_game_time)
	end
	if not last_game_time or last_game_time < 1 or math.abs(last_game_time - current_game_time) <= 1.5 then
		return
	end

	meta:set_string("last_gametime", tostring(current_game_time))
end

function lava_furnace.furnace_get_delta_time(pos, elapsed)
	local meta = core.get_meta(pos)
	local time_speed = tonumber(core.settings:get("time_speed") or 72)
	local current_game_time
	if (time_speed < 0.1) then
		return meta, elapsed
	else
		local time_multiplier = 86400 / time_speed
		current_game_time = .0 + ((core.get_day_count() + core.get_timeofday()) * time_multiplier)
	end

	local last_game_time = meta:get_string("last_gametime")
	if last_game_time then
		last_game_time = tonumber(last_game_time)
	end
	if not last_game_time or last_game_time < 1 then
		last_game_time = current_game_time - 0.1
	elseif last_game_time == current_game_time then
		current_game_time = current_game_time + 1.0
	end

	local elapsed_game_time = .0 + current_game_time - last_game_time

	meta:set_string("last_gametime", tostring(current_game_time))

	return meta, elapsed_game_time
end

function lava_furnace.adjacent_to_lava(pos)
	-- if lava flowing or lava source is within 1 block, then return true
	for _, vv in pairs(adjacents) do
		local pp = vector.add(pos, vv)
		local an = core.get_node(pp)
		for _, fuelname in ipairs(lava_furnace.get_fuel_nodes()) do
			-- short-circuit if we find it
			if an.name == fuelname then return true end
		end
	end
	return false
end

function lava_furnace.get_timer_function(node_normal, node_active, factor, group)
	if gameid == "mineclonia" or gameid == "VoxeLibre" then
		return function(pos, elapsed)
			local meta, elapsed_game_time = lava_furnace.furnace_get_delta_time(pos, elapsed)

			local fuel_time = meta:get_float("fuel_time") or 0
			local fueled = lava_furnace.adjacent_to_lava(pos)
			local src_time = meta:get_float("src_time") or 0
			local src_item = meta:get_string("src_item") or ""
			local fuel_totaltime = meta:get_float("fuel_totaltime") or 0

			local inv = meta:get_inventory()
			local srclist, fuellist

			local cookable, cooked
			local active = true
			local fuel

			srclist = inv:get_list("src")
			fuellist = inv:get_list("fuel")

			-- Check if src item has been changed
			if srclist[1]:get_name() ~= src_item then
				-- Reset cooking progress in this case
				src_time = 0
				src_item = srclist[1]:get_name()
			end

			local update = true
			local inv_changed = false
			while elapsed_game_time > 0.00001 and update do
				--
				-- Cooking
				--

				local el = elapsed_game_time * factor

				-- Check if we have cookable content: cookable
				local aftercooked
				cooked, aftercooked = core.get_craft_result({ method = "cooking", width = 1, items = srclist })
				cookable = cooked.time ~= 0
				if group then
					cookable = cookable and core.get_item_group(inv:get_stack("src", 1):get_name(), group) > 0
				end
				if cookable then
					-- Successful cooking requires space in dst slot and time
					if not inv:room_for_item("dst", cooked.item) then
						cookable = false
					end
				end

				if cookable then -- fuel lasts long enough, adjust el to cooking duration
					el = math.min(el, cooked.time - src_time)
				end

				-- Check if we have enough fuel to burn
				active = fuel_time < fuel_totaltime
				if cookable and not active then
					-- We need to get new fuel
					local afterfuel
					--fuel, afterfuel = core.get_craft_result({ method = "fuel", width = 1, items = fuellist })

					--if fuel.time == 0 then
					if not fueled then
						-- No valid fuel in fuel list -- stop
						fuel_totaltime = 0
						src_time = 0
						update = false
					else
						-- Take fuel from fuel list
						--inv:set_stack("fuel", 1, afterfuel.items[1])
						fuel_time = 0
						--fuel_totaltime = fuel.time
						fuel_totaltime = mod_default_lava_fuel_time
						el = math.min(el, fuel_totaltime)
						active = true
						--fuellist = inv:get_list("fuel")
						inv_changed = true
					end
				elseif active then
					el = math.min(el, fuel_totaltime - fuel_time)
					-- The furnace is currently active and has enough fuel
					fuel_time = fuel_time + el
				end

				-- If there is a cookable item then check if it is ready yet
				if cookable and active then
					src_time = src_time + el
					-- Place result in dst list if done
					if src_time >= cooked.time then
						inv:add_item("dst", cooked.item)
						inv:set_stack("src", 1, aftercooked.items[1])

						-- Unique recipe: Pour water into empty bucket after cooking wet sponge successfully
						if inv:get_stack("fuel", 1):get_name() == "mcl_buckets:bucket_empty" then
							if srclist[1]:get_name() == "mcl_sponges:sponge_wet" then
								inv:set_stack("fuel", 1, "mcl_buckets:bucket_water")
								fuellist = inv:get_list("fuel")
								-- Also for river water
							elseif srclist[1]:get_name() == "mcl_sponges:sponge_wet_river_water" then
								inv:set_stack("fuel", 1, "mcl_buckets:bucket_river_water")
								fuellist = inv:get_list("fuel")
							end
						end

						srclist = inv:get_list("src")
						src_time = 0
						if gameid == "mineclonia" or gameid == "VoxeLibre" then
							meta:set_int("xp", meta:get_int("xp") + 1) -- ToDo give each recipe an idividial XP count
						end
						inv_changed = true
					end
				end

				elapsed_game_time = elapsed_game_time - el
			end

			if inv_changed then
				if gameid == "mineclonia" then
					mcl_redstone.update_comparators(pos)
				end
			end

			if fuel and fuel_totaltime > fuel.time then
				fuel_totaltime = fuel.time
			end
			if srclist and srclist[1]:is_empty() then
				src_time = 0
			end

			local def = core.registered_nodes[node_normal]
			local name = S("Lava Furnace")
			if def and def.description then
				name = def._tt_original_description or def.description
			end

			local formspec
			if def._lava_furnace_get_inactive_formspec then
				formspec = def._lava_furnace_get_inactive_formspec(name)
			else
				formspec = get_inactive_formspec(name)
			end
			local item_percent = 0
			if cookable then
				item_percent = math.floor(src_time / cooked.time * 100)
			end

			local result = false

			if active then
				local fuel_percent = 0
				if fuel_totaltime > 0 then
					fuel_percent = math.floor(fuel_time / fuel_totaltime * 100)
				end
				if def._lava_furnace_get_active_formspec then
					formspec = def._lava_furnace_get_active_formspec(fuel_percent, item_percent, name)
				else
					formspec = get_active_formspec(fuel_percent, item_percent, name)
				end
				lava_furnace.swap_node(pos, node_active)
				-- make sure timer restarts automatically
				result = true
			else
				lava_furnace.swap_node(pos, node_normal)
				-- stop timer on the inactive furnace
				core.get_node_timer(pos):stop()
			end

			--
			-- Set meta values
			--
			meta:set_float("fuel_totaltime", fuel_totaltime)
			meta:set_float("fuel_time", fuel_time)
			meta:set_float("src_time", src_time)
			if srclist then
				meta:set_string("src_item", src_item)
			else
				meta:set_string("src_item", "")
			end
			meta:set_string("formspec", formspec)

			return result
		end
	elseif gameid == "minetest" then
		return function(pos, elapsed)
			--
			-- Initialize metadata
			--
			local meta = core.get_meta(pos)
			local fueled = lava_furnace.adjacent_to_lava(pos)
			local fuel_time = meta:get_float("fuel_time") or 0
			local src_time = meta:get_float("src_time") or 0
			local fuel_totaltime = meta:get_float("fuel_totaltime") or 0

			local inv = meta:get_inventory()
			local srclist, fuellist
			local dst_full = false

			local timer_elapsed = meta:get_int("timer_elapsed") or 0
			meta:set_int("timer_elapsed", timer_elapsed + 1)

			local cookable, cooked
			local fuel
			local update = true
			local items_smelt = 0
			while elapsed > 0 and update do
				update = false

				srclist = inv:get_list("src")
				fuellist = inv:get_list("fuel")

				--
				-- Cooking
				--

				-- Check if we have cookable content
				local aftercooked
				cooked, aftercooked = core.get_craft_result({method = "cooking", width = 1, items = srclist})
				cookable = cooked.time ~= 0

				local el = math.min(elapsed, fuel_totaltime - fuel_time) * factor
				if cookable then -- fuel lasts long enough, adjust el to cooking duration
					el = math.min(el, cooked.time - src_time)
				end

				-- Check if we have enough fuel to burn
				if fuel_time < fuel_totaltime then
					-- The furnace is currently active and has enough fuel
					fuel_time = fuel_time + el
					-- If there is a cookable item then check if it is ready yet
					if cookable then
						src_time = src_time + el
						if src_time >= cooked.time then
							-- Place result in dst list if possible
							if inv:room_for_item("dst", cooked.item) then
								inv:add_item("dst", cooked.item)

								-- stop any final replacement from clogging "src"
								local can_cook = core.get_craft_result({
										method = "cooking", width = 1,
										items = {aftercooked.items[1]:to_string()}})
								can_cook = can_cook.time ~= 0 or not can_cook.item:is_empty()

								if aftercooked.items[1]:is_empty() or can_cook then
									-- cook the final "src" item in the next cycle
									inv:set_stack("src", 1, aftercooked.items[1])
								else
									-- the final "src" item was replaced and is not cookable
									inv:set_stack("src", 1, "")
									add_item_or_drop(inv, pos, aftercooked.items[1])
								end

								src_time = src_time - cooked.time
								update = true
								-- add replacement item to dst so they arent lost
								if cooked.replacements[1] then
									add_item_or_drop(inv, pos, cooked.replacements[1])
								end
							else
								dst_full = true
							end
							items_smelt = items_smelt + 1
						else
							-- Item could not be cooked: probably missing fuel
							update = true
						end
					end
				else
					-- Furnace ran out of fuel
					if cookable then
						-- We need to get new fuel
						--local afterfuel
						--fuel, afterfuel = core.get_craft_result({method = "fuel", width = 1, items = fuellist})

						--if fuel.time == 0 then
						if not fueled then
							-- No valid fuel in fuel list
							fuel_totaltime = 0
							src_time = 0
						else
							-- Take fuel from fuel list
							--inv:set_stack("fuel", 1, afterfuel.items[1])
							fuel_time = 0
							fuel_totaltime = mod_default_lava_fuel_time
							el = math.min(el, fuel_totaltime)
							update = true
							fuel_totaltime = mod_default_lava_fuel_time
						end
					else
						-- We don't need to get new fuel since there is no cookable item
						fuel_totaltime = 0
						src_time = 0
					end
					fuel_time = 0
				end

				elapsed = elapsed - el
			end

			if items_smelt > 0 then
				-- Play cooling sound
				core.sound_play("default_cool_lava",
					{ pos = pos, max_hear_distance = 16, gain = 0.07 * math.min(items_smelt, 7) }, true)
			end
			if fuel and fuel_totaltime > fuel.time then
				fuel_totaltime = fuel.time
			end
			if srclist and srclist[1]:is_empty() then
				src_time = 0
			end

			--
			-- Update formspec, infotext and node
			--
			local formspec
			local item_state
			local item_percent = 0
			if cookable then
				item_percent = math.floor(src_time / cooked.time * 100)
				if dst_full then
					item_state = S("100% (output full)")
				else
					item_state = S("@1%", item_percent)
				end
			else
				if srclist and not srclist[1]:is_empty() then
					item_state = S("Not cookable")
				else
					item_state = S("Empty")
				end
			end

			local fuel_state = S("Empty")
			local active = false
			local result = false

			if fuel_totaltime ~= 0 then
				active = true
				local fuel_percent = 100 - math.floor(fuel_time / fuel_totaltime * 100)
				fuel_state = S("@1%", fuel_percent)
				formspec = default.get_furnace_active_formspec(fuel_percent, item_percent)
				lava_furnace.swap_node(pos, node_active)
				-- make sure timer restarts automatically
				result = true

				-- Play sound every 5 seconds while the furnace is active
				if timer_elapsed == 0 or (timer_elapsed + 1) % 5 == 0 then
					local sound_id = core.sound_play("default_furnace_active",
						{pos = pos, max_hear_distance = 16, gain = 0.25})
					local hash = core.hash_node_position(pos)
					furnace_fire_sounds[hash] = furnace_fire_sounds[hash] or {}
					table.insert(furnace_fire_sounds[hash], sound_id)
					-- Only remember the 3 last sound handles
					if #furnace_fire_sounds[hash] > 3 then
						table.remove(furnace_fire_sounds[hash], 1)
					end
					-- Remove the sound ID automatically from table after 11 seconds
					core.after(11, function()
						if not furnace_fire_sounds[hash] then
							return
						end
						for f=#furnace_fire_sounds[hash], 1, -1 do
							if furnace_fire_sounds[hash][f] == sound_id then
								table.remove(furnace_fire_sounds[hash], f)
							end
						end
						if #furnace_fire_sounds[hash] == 0 then
							furnace_fire_sounds[hash] = nil
						end
					end)
				end
			else
				if fuellist and not fuellist[1]:is_empty() then
					fuel_state = S("@1%", 0)
				end
				formspec = default.get_furnace_inactive_formspec()
				lava_furnace.swap_node(pos, node_normal)
				-- stop timer on the inactive furnace
				core.get_node_timer(pos):stop()
				meta:set_int("timer_elapsed", 0)

				stop_furnace_sound(pos)
			end


			local infotext
			if active then
				infotext = S("Lava Furnace active")
			else
				infotext = S("Lava Furnace inactive")
			end
			infotext = infotext .. "\n" .. S("(Item: @1; Fuel: @2)", item_state, fuel_state)

			--
			-- Set meta values
			--
			meta:set_float("fuel_totaltime", fuel_totaltime)
			meta:set_float("fuel_time", fuel_time)
			meta:set_float("src_time", src_time)
			meta:set_string("formspec", formspec)
			meta:set_string("infotext", infotext)

			return result
		end
	else
		-- FIXME: how to handle a new gameid?
	end
end

lava_furnace.furnace_node_timer = lava_furnace.get_timer_function()

lava_furnace.on_rotate = screwdriver.rotate_simple

-- Returns true if itemstack is fuel, but not for lava bucket if destination already has one
function lava_furnace.is_transferrable_fuel(itemstack, _, _, dst_inventory, dst_list)
	if mcl_util.is_fuel(itemstack) then
		return false
		--[[
		if itemstack:get_name() == "mcl_buckets:bucket_lava" then
			return dst_inventory:is_empty(dst_list)
		else
			return true
		end
		--]]
	else
		return false
	end
end

function lava_furnace.on_hopper_out(uppos, pos)
	local sucked = mcl_util.move_item_container(uppos, pos)

	-- Also suck in non-fuel items from furnace fuel slot
	if not sucked then
		local finv = core.get_inventory({type="node", pos=uppos})
		if finv and not mcl_util.is_fuel(finv:get_stack("fuel", 1)) then
			sucked = mcl_util.move_item_container(uppos, pos, "fuel")
		end
	end
	return sucked
end

-- no on_hopper_in for the fuel for lava_furnace
local stone_sound = nil
if gameid == "mineclonia" or gameid == "VoxeLibre" then
	stone_sound = mcl_sounds.node_sound_stone_defaults
elseif gameid == "minetest" then -- minetest_game
	stone_sound = default.node_sound_stone_defaults
else
	error("[lava_furnace] No sound system for game: " .. gameid)
end
local after_dig_drop = nil
if gameid == "mineclonia" or gameid == "VoxeLibre" then
	after_dig_drop = mcl_util.drop_items_from_meta_container({"src", "dst", "fuel", "sorter"})
end
lava_furnace.tpl_furnace_node = {
	paramtype2 = "facedir",
	paramtype = "light",
	groups = { cracky = 2, pickaxey = 1, container = 4, deco_block = 1, material_stone = 1, furnace = 1, unmovable_by_piston = 1},
	is_ground_content = false,
	sounds = stone_sound(),
	_mcl_hardness = 3.5,

	after_dig_node = after_dig_drop,
	on_destruct = function(pos)
		lava_furnace.give_xp(pos)
	end,

	allow_metadata_inventory_put = lava_furnace.allow_metadata_inventory_put,
	allow_metadata_inventory_move = lava_furnace.allow_metadata_inventory_move,
	allow_metadata_inventory_take = lava_furnace.allow_metadata_inventory_take,
	on_metadata_inventory_move = lava_furnace.on_metadata_inventory_move,
	on_metadata_inventory_take = lava_furnace.on_metadata_inventory_take,
	on_metadata_inventory_put = lava_furnace.on_metadata_inventory_put,
	on_receive_fields = lava_furnace.receive_fields,
	_on_hopper_out = lava_furnace.on_hopper_out,
	on_rotate = lava_furnace.on_rotate,
}

lava_furnace.tpl_furnace_node_normal = table_merge(lava_furnace.tpl_furnace_node,{
	_doc_items_hidden = false,

	on_metadata_inventory_move = function(pos, from_list, from_index, to_list, to_index, count, player)
		-- Reset accumulated game time when player works with furnace:
		lava_furnace.furnace_reset_delta_time(pos)
		core.get_node_timer(pos):start(1.0)

		lava_furnace.on_metadata_inventory_move(pos, from_list, from_index, to_list, to_index, count, player)
	end,
	on_metadata_inventory_put = function(pos, listname, index, stack, player)
		-- Reset accumulated game time when player works with furnace:
		lava_furnace.furnace_reset_delta_time(pos)
		-- start timer function, it will sort out whether furnace can burn or not.
		core.get_node_timer(pos):start(1.0)

		lava_furnace.on_metadata_inventory_put(pos, listname, index, stack, player)
	end,
	on_metadata_inventory_take = function(pos, listname, index, stack, player)
		-- Reset accumulated game time when player works with furnace:
		lava_furnace.furnace_reset_delta_time(pos)
		-- start timer function, it will helpful if player clears dst slot
		core.get_node_timer(pos):start(1.0)

		lava_furnace.on_metadata_inventory_take(pos, listname, index, stack, player)
	end,
})

lava_furnace.tpl_furnace_node_active = table_merge(lava_furnace.tpl_furnace_node,{
	groups = { cracky = 2, pickaxey = 1, container = 4, deco_block = 1, material_stone = 1, furnace = 1, furnace_active = 1, not_in_creative_inventory = 1, unmovable_by_piston = 1 },
	_doc_items_create_entry = false,
	light_source = LIGHT_ACTIVE_FURNACE,
})

function lava_furnace.register_furnace(nodename, def)
	local lava_factor = core.settings:get("lava_furnace.factor") or mod_default_lava_factor
	core.log("info","Using lava_factor of " .. lava_factor)
	local timer_func = lava_furnace.get_timer_function(nodename, nodename.."_active", (def.factor or lava_factor), def.cook_group)
	if gameid == "mineclonia" then
		core.register_node(nodename, table_merge(lava_furnace.tpl_furnace_node_normal,{
			on_construct = function(pos)
				local meta = core.get_meta(pos)
				local name = S("Lava Furnace")
				local def = core.registered_nodes[nodename]
				if def and def.description then
					name = def._tt_original_description or def.description
				end
				if def.get_inactive_formspec then
					meta:set_string("formspec", def.get_inactive_formspec(name))
				else
					meta:set_string("formspec", get_inactive_formspec(name))
				end
				local inv = meta:get_inventory()
				inv:set_size("sorter", 1)
				inv:set_size("src", 1)
				inv:set_size("fuel", 1)
				inv:set_size("dst", 1)
			end,
			on_timer = timer_func,
			_lava_furnace_cook_group = def.cook_group,
			_lava_furnace_get_active_formspec = def.get_active_formspec,
			_lava_furnace_get_inactive_formspec = def.get_inactive_formspec,
		},def.node_normal))
		core.register_node(nodename.."_active", table_merge(lava_furnace.tpl_furnace_node_active,{
			on_timer = timer_func,
			drop = nodename,
			_mcl_baseitem = nodename,
		},def.node_active))
	elseif gameid == "VoxeLibre" then
		minetest.register_node(nodename, {
			description = S("Lava Furnace"),
			_tt_help = S("Smelt or cook items when adjacent to lava"),
			_doc_items_longdesc = S("Furnaces cook or smelt several items, into something else."),
			_doc_items_usagehelp =
				S("Use the furnace to open the furnace menu.") .. "\n" ..
				S("Place this block adjacent to lava source or flowing lava.") .. "\n" ..
				S("The result will be placed into the output slot at the right side.") .. "\n" ..
				S("Use the recipe book to see what you can smelt."),
			_doc_items_hidden = false,
			tiles = {
				"default_furnace_top.png", "default_furnace_bottom.png",
				"default_furnace_side.png", "default_furnace_side.png",
				"default_furnace_side.png", "default_furnace_front.png"
			},
			paramtype2 = "facedir",
			groups = { pickaxey = 1, container = 2, deco_block = 1, material_stone = 1 },
			is_ground_content = false,
			sounds = mcl_sounds.node_sound_stone_defaults(),
			on_timer = timer_func,
			after_dig_node = function(pos, oldnode, oldmetadata, digger)
				local meta = minetest.get_meta(pos)
				local meta2 = meta:to_table()
				meta:from_table(oldmetadata)
				local inv = meta:get_inventory()
				for _, listname in ipairs({ "src", "dst", "fuel" }) do
					local stack = inv:get_stack(listname, 1)
					if not stack:is_empty() then
						local p = { x = pos.x + math.random(0, 10) / 10 - 0.5, y = pos.y,
							z = pos.z + math.random(0, 10) / 10 - 0.5 }
						minetest.add_item(p, stack)
					end
				end
				meta:from_table(meta2)
			end,
			on_construct = function(pos)
				local meta = minetest.get_meta(pos)
				local name = S("Lava Furnace")
				if def.get_inactive_formspec then
					meta:set_string("formspec", def.get_inactive_formspec(name))
				else
					meta:set_string("formspec", get_inactive_formspec(name))
				end
				local inv = meta:get_inventory()
				inv:set_size("src", 1)
				inv:set_size("fuel", 1)
				inv:set_size("dst", 1)
			end,
			on_destruct = function(pos)
				mcl_particles.delete_node_particlespawners(pos)
				lava_furnace.give_xp(pos)
			end,
			on_metadata_inventory_move = function(pos, from_list, from_index, to_list, to_index, count, player)
				-- Reset accumulated game time when player works with furnace:
				lava_furnace.furnace_reset_delta_time(pos)
				core.get_node_timer(pos):start(1.0)

				lava_furnace.on_metadata_inventory_move(pos, from_list, from_index, to_list, to_index, count, player)
			end,
			on_metadata_inventory_put = function(pos, listname, index, stack, player)
				-- Reset accumulated game time when player works with furnace:
				lava_furnace.furnace_reset_delta_time(pos)
				-- start timer function, it will sort out whether furnace can burn or not.
				core.get_node_timer(pos):start(1.0)

				lava_furnace.on_metadata_inventory_put(pos, listname, index, stack, player)
			end,
			on_metadata_inventory_take = function(pos, listname, index, stack, player)
				-- Reset accumulated game time when player works with furnace:
				lava_furnace.furnace_reset_delta_time(pos)
				-- start timer function, it will helpful if player clears dst slot
				core.get_node_timer(pos):start(1.0)

				lava_furnace.on_metadata_inventory_take(pos, listname, index, stack, player)
			end,

			allow_metadata_inventory_put = lava_furnace.allow_metadata_inventory_put,
			allow_metadata_inventory_move = lava_furnace.allow_metadata_inventory_move,
			allow_metadata_inventory_take = lava_furnace.allow_metadata_inventory_take,
			on_receive_fields = lava_furnace.receive_fields,
			_mcl_blast_resistance = 3.5,
			_mcl_hardness = 3.5,
			on_rotate = lava_furnace.on_rotate,
			_mcl_hoppers_on_try_pull = mcl_furnaces.hoppers_on_try_pull,
			_mcl_hoppers_on_try_push = mcl_furnaces.hoppers_on_try_push,
			_mcl_hoppers_on_after_push = function(pos)
				minetest.get_node_timer(pos):start(1.0)
			end,
		})
		minetest.register_node("lava_furnace:furnace_active", {
			description = S("Burning Lava Furnace"),
			_doc_items_create_entry = false,
			tiles = {
				"default_furnace_top.png", "default_furnace_bottom.png",
				"default_furnace_side.png", "default_furnace_side.png",
				"default_furnace_side.png", "default_furnace_front_active.png",
			},
			paramtype2 = "facedir",
			paramtype = "light",
			light_source = LIGHT_ACTIVE_FURNACE,
			drop = "lava_furnace:furnace",
			groups = { pickaxey = 1, container = 2, deco_block = 1, not_in_creative_inventory = 1, material_stone = 1 },
			is_ground_content = false,
			sounds = mcl_sounds.node_sound_stone_defaults(),
			on_timer = timer_func,
			after_dig_node = function(pos, oldnode, oldmetadata, digger)
				local meta = minetest.get_meta(pos)
				local meta2 = meta
				meta:from_table(oldmetadata)
				local inv = meta:get_inventory()
				for _, listname in ipairs({ "src", "dst", "fuel" }) do
					local stack = inv:get_stack(listname, 1)
					if not stack:is_empty() then
						local p = { x = pos.x + math.random(0, 10) / 10 - 0.5, y = pos.y,
							z = pos.z + math.random(0, 10) / 10 - 0.5 }
						minetest.add_item(p, stack)
					end
				end
				meta:from_table(meta2:to_table())
			end,

			on_construct = function(pos)
				local node = minetest.get_node(pos)
				spawn_flames(pos, node.param2)
			end,
			on_destruct = function(pos)
				mcl_particles.delete_node_particlespawners(pos)
				lava_furnace.give_xp(pos)
			end,

			allow_metadata_inventory_put = lava_furnace.allow_metadata_inventory_put,
			allow_metadata_inventory_move = lava_furnace.allow_metadata_inventory_move,
			allow_metadata_inventory_take = lava_furnace.allow_metadata_inventory_take,
			on_metadata_inventory_move = lava_furnace.on_metadata_inventory_move,
			on_metadata_inventory_take = lava_furnace.on_metadata_inventory_take,
			on_metadata_inventory_put = lava_furnace.on_metadata_inventory_put,
			on_receive_fields = lava_furnace.receive_fields,
			_mcl_blast_resistance = 3.5,
			_mcl_hardness = 3.5,
			on_rotate = lava_furnace.on_rotate,
			-- from mineclone2/mods/ITEMS/mcl_furnaces/init.lua#476
			after_rotate = function(pos)
				local node = minetest.get_node(pos)
				mcl_particles.delete_node_particlespawners(pos)
				if node.name == "mcl_furnaces:furnace" then
					return
				end
				spawn_flames(pos, node.param2)
			end,
			_mcl_hoppers_on_try_pull = mcl_furnaces.hoppers_on_try_pull,
			_mcl_hoppers_on_try_push = mcl_furnaces.hoppers_on_try_push,
		})
	elseif gameid == "minetest" then -- minetest_game
		minetest.register_node(nodename, apply_logger(table_merge(lava_furnace.tpl_furnace_node_normal, {
			on_timer = timer_func,
			drop = nodename,
			on_construct = function(pos)
				local meta = minetest.get_meta(pos)
				local name = S("Lava Furnace")
				if def.get_inactive_formspec then
					meta:set_string("formspec", def.get_inactive_formspec(name))
				else
					meta:set_string("formspec", get_inactive_formspec(name))
				end
				local inv = meta:get_inventory()
				inv:set_size('src', 1)
				inv:set_size('fuel', 1)
				inv:set_size('dst', 2*2)
				-- unused
				inv:set_size('sorter', 1)
			end,
			on_blast = function (pos)
				local drops = {}
				default.get_inventory_drops(pos, "src", drops)
				default.get_inventory_drops(pos, "fuel", drops)
				default.get_inventory_drops(pos, "dst", drops)
				-- unused
				default.get_inventory_drops(pos, "sorter", drops)
				drops[#drops+1] = nodename
				return drops
			end
		}, def.node_normal)))
		minetest.register_node(nodename.."_active", apply_logger(table_merge(lava_furnace.tpl_furnace_node_active, {
			-- overwrite tiles textures, so it's displayed properly
			
			on_timer = timer_func,
			drop = nodename,
			on_blast = function (pos)
				local drops = {}
				default.get_inventory_drops(pos, "src", drops)
				default.get_inventory_drops(pos, "fuel", drops)
				default.get_inventory_drops(pos, "dst", drops)
				-- unused
				default.get_inventory_drops(pos, "sorter", drops)
				drops[#drops+1] = nodename
				return drops
			end
		}, def.node_active)))
	end
end

local tiles_active = {
    "default_furnace_top.png", "default_furnace_bottom.png",
    "default_furnace_side.png", "default_furnace_side.png",
    "default_furnace_side.png", "default_furnace_front_active.png",
}
if gameid == "minetest" then
    tiles_active = {
		"default_furnace_top.png", "default_furnace_bottom.png",
		"default_furnace_side.png", "default_furnace_side.png",
		"default_furnace_side.png",
		{
			name = "default_furnace_front_active.png",
			backface_culling = false,
			animation = {
				type = "vertical_frames",
				aspect_w = 16,
				aspect_h = 16,
				length = 1.5
			},
		}
	}
end
lava_furnace.register_furnace("lava_furnace:furnace",{
	node_normal = {
		description = S("Lava Furnace"),
		_tt_help = S("Smelt or cook items when adjacent to lava"),
		_doc_items_longdesc = S("Furnaces cook or smelt several items, into something else."),
		_doc_items_usagehelp =
		S("Use the furnace to open the furnace menu.") .. "\n" ..
		S("Place this block adjacent to lava source or flowing lava.") .. "\n" ..
		S("The result will be placed into the output slot at the right side.") .. "\n" ..
		S("Use the recipe book to see what you can smelt."),
		tiles = {
			"default_furnace_top.png", "default_furnace_bottom.png",
			"default_furnace_side.png", "default_furnace_side.png",
			"default_furnace_side.png", "default_furnace_front.png"
		},
	},
	node_active = {
		description = S("Burning Lava Furnace"),
		tiles = tiles_active,
	},
})

local cobble = "group:cobble"
local goldblock = "mcl_core:goldblock"
if gameid == "minetest" then -- minetest_game
	cobble = "default:cobble"
	goldblock = "default:goldblock"
end

core.register_craft({
	output = "lava_furnace:furnace",
	recipe = {
		{ cobble, cobble, cobble },
		{ cobble, goldblock, cobble },
		{ cobble, cobble, cobble },
	}
})

local normal_furnace = "mcl_furnaces:furnace"
if gameid == "minetest" then -- minetest_game
	normal_furnace = "default:furnace"
end

core.register_craft({
	output = "lava_furnace:furnace",
	type = "shapeless",
	recipe = { normal_furnace, goldblock }
})

if doc ~= nil and doc.add_entry_alias ~= nil then
	doc.add_entry_alias("nodes", "lava_furnace:furnace", "nodes", "lava_furnace:furnace_active")
end

core.register_lbm({
	label = "Update Furnace formspecs and invs to allow new sneak+click behavior",
	name = "lava_furnace:update_coolsneak",
	nodenames = { "group:furnace", "group:furnace_active" },
	run_at_every_load = false,
	action = function(pos, node)
		local m = core.get_meta(pos)
		local inv = m:get_inventory()
		inv:set_size("sorter", 1)

		if core.get_item_group(node.name, "furnace_active") == 0 then --active furnaces update the formspec every second by themselves
			local def = core.registered_nodes[node.name]
			local name = S("Lava Furnace")
			if def and def.description then
				name = def._tt_original_description or def.description
			end
			if def._lava_furnace_get_inactive_formspec then
				m:set_string("formspec", def._lava_furnace_get_inactive_formspec(name))
			else
				m:set_string("formspec", get_inactive_formspec(name))
			end
		end
	end,
})
