local S = farmtools.i18n

local get_node_pointer = function(x, y, z)
	return {
		["under"] = {
			["x"] = x,
			["y"] = y,
			["z"] = z,
		},
		["above"] = {
			["x"] = x,
			["y"] = y+1,
			["z"] = z,
		}
	}
end

local rake_on_use_single_node = function(user, node_ptr)
	-- this function is based on the farming.hoe_on_use from the minetest_game by Luanti
	-- see: https://github.com/luanti-org/minetest_game/blob/master/mods/farming/hoes.lua
	-- original code is licensed under LGPL-2.1

	-- this functions returns a boolean value, that indicates whether it was
	-- able to replace the specified node with soil
	local under = minetest.get_node(node_ptr.under)
	local p = {x=node_ptr.under.x, y=node_ptr.under.y+1, z=node_ptr.under.z}
	local above = minetest.get_node(p)

	-- return if any of the nodes is not registered
	if not minetest.registered_nodes[under.name] then
		return false
	end
	if not minetest.registered_nodes[above.name] then
		return false
	end

	-- check if the node above the pointed thing is air
	if above.name ~= "air" then
		return false
	end

	-- check if pointing at soil
	if minetest.get_item_group(under.name, "soil") ~= 1 then
		return false
	end

	-- check if (wet) soil defined
	local regN = minetest.registered_nodes
	if regN[under.name].soil == nil or regN[under.name].soil.wet == nil or regN[under.name].soil.dry == nil then
		return false
	end

	local player_name = user and user:get_player_name() or ""

	if minetest.is_protected(node_ptr.under, player_name) then
		minetest.record_protection_violation(under, player_name)
		return false
	end
	if minetest.is_protected(node_ptr.above, player_name) then
		minetest.record_protection_violation(above, player_name)
		return false
	end

	-- turn the node into soil and play sound
	minetest.set_node(node_ptr.under, {name = regN[under.name].soil.dry})
	return true
end

farmtools.rake_on_use = function(itemstack, user, pointed_thing, uses)
	-- this function is based on the farming.hoe_on_use from the minetest_game by Luanti
	-- see: https://github.com/luanti-org/minetest_game/blob/master/mods/farming/hoes.lua
	-- original code is licensed under LGPL-2.1
	local pt = pointed_thing

	-- check if pointing at a node
	if not pt then
		return
	end
	if pt.type ~= "node" then
		return
	end

	-- first we check whether the user can use the rake on the pointed node
	-- if they can't, we won't go any further
	-- note that the following function WILL replace the current node with soil.
	-- It will return a boolean value indicating whether it was able to replace
	-- the node with soil
	if not rake_on_use_single_node(user, pointed_thing) then
		return
	end

	-- this will be required for wearing tools depending on the number of nodes turned into soil
	total_nodes_changed = 1

	local under = minetest.get_node(pt.under)
	-- we define the area that will be turned into soil
	-- note that we do not include the pointed node, since it has already been processed
	local nodes_to_transform = {
		{x=pt.under.x + 1, y=pt.under.y, z=pt.under.z},
		{x=pt.under.x - 1, y=pt.under.y, z=pt.under.z},
		{x=pt.under.x, y=pt.under.y, z=pt.under.z + 1},
		{x=pt.under.x, y=pt.under.y, z=pt.under.z - 1},
		{x=pt.under.x + 1, y=pt.under.y, z=pt.under.z + 1},
		{x=pt.under.x - 1, y=pt.under.y, z=pt.under.z - 1},
		{x=pt.under.x + 1, y=pt.under.y, z=pt.under.z - 1},
		{x=pt.under.x - 1, y=pt.under.y, z=pt.under.z + 1},
	}
	-- now for each node, we try to turn it into soil
	for _, p in pairs(nodes_to_transform) do
		local node_ptr = get_node_pointer(p.x, p.y, p.z)
		local success = rake_on_use_single_node(user, node_ptr)
		-- if node was replaced, we increment the count of processed nodes by 1.
		if success then
			total_nodes_changed = total_nodes_changed + 1
		end
	end

	minetest.sound_play("default_dig_crumbly", {
		pos = pt.under,
		gain = 0.3,
	}, true)

	local player_name = user and user:get_player_name() or ""

	if not minetest.is_creative_enabled(player_name) then
		-- wear tool
		-- here we use the value of total_nodes_changed to make the rake break faster the more
		-- nodes were processed
		local wdef = itemstack:get_definition()
		itemstack:add_wear_by_uses(uses / total_nodes_changed)
		-- tool break sound
		if itemstack:get_count() == 0 and wdef.sound and wdef.sound.breaks then
			minetest.sound_play(wdef.sound.breaks, {pos = pt.above,
			gain = 0.5}, true)
		end
	end
	return itemstack
end

-- Register new rakes
-- This is almost a copy-pasted version of the register_hoe function of the farming mod.
farmtools.register_rake = function(name, def)
	-- Check for : prefix (register new rakes in your mod's namespace)
	if name:sub(1,1) ~= ":" then
		name = ":" .. name
	end
	-- Check def table
	if def.description == nil then
		def.description = S("Rake")
	end
	if def.inventory_image == nil then
		def.inventory_image = "unknown_item.png"
	end
	if def.max_uses == nil then
		def.max_uses = 90
	end
	-- Register the tool
	minetest.register_tool(name, {
		description = def.description,
		inventory_image = def.inventory_image,
		on_use = function(itemstack, user, pointed_thing)
			return farmtools.rake_on_use(itemstack, user, pointed_thing, def.max_uses)
		end,
		groups = def.groups,
		sound = {breaks = "default_tool_breaks"},
	})
	-- Register its recipe
	if def.recipe then
		minetest.register_craft({
			output = name:sub(2),
			recipe = def.recipe
		})
	elseif def.material then
		minetest.register_craft({
			output = name:sub(2),
			recipe = {
				{def.material, def.material, def.material},
				{def.material, "group:stick", def.material},
				{"", "group:stick", ""}
			}
		})
	end
end

farmtools.register_rake("farmtools:rake_steel", {
	description = S("Steel Rake"),
	inventory_image = "farmtools_rake_steel.png",
	max_uses = 500,
	material = "default:steel_ingot",
	groups = {hoe = 1, rake = 1}
})
