-- protection effect for player and mobs
-- protection has a timer
-- when the timer is up, then the protection effect is done
-- protection itself is a number, it absorbs hits
-- when it absorbs all the hits, then the entity or player
-- takes damage
--
-- I want to eventually abstract effects
-- but need a few more one offs to get a feel for the pattern

local protection_storage = {}

-- I really want some different visual
-- indicator for protection
local function add_spawner(obj, color)
	local d = 0.2
	local pos = obj:get_pos()
	minetest.add_particlespawner({
		amount = 1,
		time = 1,
		minpos = {x=pos.x-d, y=pos.y+1, z=pos.z-d},
		maxpos = {x=pos.x+d, y=pos.y+2, z=pos.z+d},
		minvel = {x=-0.1, y=0, z=-0.1},
		maxvel = {x=0.1, y=0.1, z=0.1},
		minacc = {x=-0.1, y=0, z=-0.1},
		maxacc = {x=0.1, y=.1, z=0.1},
		minexptime = 0.5,
		maxexptime = 1,
		minsize = 0.5,
		maxsize = 1,
		collisiondetection = false,
		vertical = false,
		texture = "mcl_particles_effect.png^[colorize:"..color..":127",
	})
end


local default_data = {
  damage = 0,
  duration = 0,
  total_timer = 0,
}


local function get_protection_data(target)
  if target:is_player() then
    local name = target:get_player_name()
    return protection_storage[name] or {
      damage = 0,
      duration = 0,
      total_timer = 0,
    }
  end

  local mob = target:get_luaentity()
  return mobkit.recall(mob, 'protection') or {
    damage = 0,
    duration = 0,
    total_timer = 0,
  }
end

local function set_protection_data(target, data)
  if target:is_player() then
    local name = target:get_player_name()
    protection_storage[name] = data
    return
  end

  local mob = target:get_luaentity()
  return mobkit.remember(mob, 'protection', data)
end

local function start_protection(target, damage, duration)
  local data = get_protection_data(target)

  data.damage = damage
  data.duration = duration

  set_protection_data(target, data)
end

local function clear_protection(target)
  local data = get_protection_data(target)
  data.damage = 0
  data.duration = 0
  data.total_timer = 0
  set_protection_data(target, data)
end

local particle_color = "#3A8AC6"

local function protection_on_step(target, dt)
  local data = get_protection_data(target)

  -- means protection is not being applied
  if(not data.damage or data.damage == 0 or data.duration == 0) then
    return
  end

  data.total_timer = data.total_timer + dt

  add_spawner(target, particle_color)

  if data.total_timer > data.duration then
    clear_protection(target)
  end

  set_protection_data(target, data)
end

-- return amount of damage to apply after protection
-- if no protection effect enabled, then all damage is returned
-- if protection absorbs all damage, then will return 0
-- if protection absorbs some of the damage, then returns what part
-- wasn't absorbed
local function apply_protection(target, damage)
  local data = get_protection_data(target)

  if(not data.damage or data.damage == 0 or data.duration == 0) then
    return damage
  end

  local new_damage = data.damage - damage

  if new_damage <= 0 then
    clear_protection(target)
    return new_damage * -1
  end

  data.damage = new_damage
  set_protection_data(target, data)
  return 0
end

core.register_globalstep(function(dt)
	for _, player in pairs(core.get_connected_players()) do
    protection_on_step(player, dt)
  end
end)

core.register_on_joinplayer(function(player)
  local name = player:get_player_name()
	local storage = table.copy(default_data)
	local protection_data = player:get_meta():get_string("mg_protection:data")
	if protection_data ~= "" then
		storage = minetest.deserialize(protection_data) or storage
	end
	protection_storage[name] = storage
end)

local function on_leaveplayer(player)
  local name = player:get_player_name()
	local storage = protection_storage[name]
	if not storage then
		storage = {}
	end
	player:get_meta():set_string("mg_protection:data", minetest.serialize(storage))
	protection_storage[name] = nil
end

core.register_on_leaveplayer(function(player)
	on_leaveplayer(player)
end)

core.register_on_shutdown(function()
	for _,player in ipairs(minetest.get_connected_players()) do
		on_leaveplayer(player)
	end
end)

minetest.register_on_respawnplayer(function(player)
  clear_protection(player)
end)

local protection_api = {
  get_protection_data = get_protection_data,
  start_protection = start_protection,
  protection_on_step = protection_on_step,
  clear_protection = clear_protection,
  apply_protection = apply_protection
}

return protection_api
