local permadeath = core.settings:get_bool("permadeath", true)

local nbDirs = mg_arch.nbDirs
local startY = 0

-- Dungeon generation map
-- To plug this in other games, update this
local lava = "mapgen:magma_source"
local sand = "mapgen:sand"
local stone = "mapgen:stone"
local light = "light:light"
local blackstone = "mapgen:blackstone"
local chasm_edge = "mapgen:chasm_edge"
local stone_block = "mapgen:stone_block"
local door = "doors:door_wood_b"
local secret_door = "doors:door_stone_b"
local cobble = "mapgen:cobble"
local mossycobble = "mapgen:mossycobble"
local fernbottom = "mapgen:double_fern"
local ferntop = "mapgen:double_fern_top"
local grassbottom = "mapgen:double_grass"
local grasstop = "mapgen:double_grass_top"
local grass = "mapgen:grass_5"
local dry_grass = "mapgen:dry_grass"
local air = "air"
local water = "mapgen:water_source"
local wood = "mapgen:wood"
local torch = "mg_torch:torch_wall"
local ladder = "mg_ladder:ladder_wood"
local rubble = "mapgen:rock_"
local glowing_fungus = "mapgen:crimson_roots"
local statue = "mg_mobs:statue"

-- Traps
local fire_trap = "mg_traps:fire_trap"
local descent_trap = "mg_traps:descent_trap"
local flood_trap = "mg_traps:flood_trap"
local net_trap = "mg_traps:net_trap"
local gas_trap = "mg_traps:gas_trap"


-- items for starting chest
local chest = 'mg_chest:chest'
local book_written = 'mg_book:book_written'
local dagger = 'mg_tools:dagger'
local ration = 'mapgen:food_ration'



local function getRubble()
  return rubble .. math.random(1,3)
end

local function save_file(file_name, data)
	local file = io.open(minetest.get_worldpath().."/"..file_name..".mt", "w")
	if file then
		file:write(core.serialize(data))
		file:close()
	end
end

local function load_file(file_name)
	local file = io.open(minetest.get_worldpath().."/"..file_name..".mt", "r")
	if file then
		local table = core.deserialize(file:read("*all"))
		if type(table) == "table" then
			return table
		end
	end
  return {}
end

local spawn_y = 8

local function place_highscores()
  local score_list = load_file("highscores")

  table.sort(score_list, function(a, b)
    return (a.gold or 0) > (b.gold or 0)
  end)

  local pos = {x = math.ceil(mg_arch.DCOLS / 2 - 3), y = spawn_y + 1, z = mg_arch.DROWS}
  for i, score in ipairs(score_list) do
    if (score.gold or 0) > 0 and i <= 10 then
      local p = {x=pos.x, y=pos.y, z=pos.z + i + 3}
      local time = ''
      if score.time then
        time = os.date('%Y-%m-%d %H:%M:%S', score.time) .. '\n'
      end
      local seed_text = ''
      if score.seed then
        seed_text = "\nSeed: "..score.seed
      end
      core.set_node(p, {name="signs:sign_wood_yard", param2=3})
      local text = time .. score.player .. " died on depth " .. score.depth .. " with\n" .. score.gold .. " gold."
      if score.won then
        text = time .. score.player .. " won the game.  " ..
          "They made it to a max depth of  " .. (score.max_depth or "12") ..
          " with\n" .. score.gold .. " gold."
      end
      text = text .. seed_text
      local fields = {
        text = text
      }
      signs_lib.update_sign(p, fields)
    end
  end
end

local air_light = "mg_torch:air_light_3"

local function connectLevels(levels, start_level, end_level, mg_set_node)
  for i=start_level, end_level - 1 do
    local baseY = -11 * (i - 1) + startY
    local c = levels[i]
    local n = levels[i+1]
    local down = c.downStair
    local up = n.upStair

    if down.x == up.x and down.y == up.y then
      return
    end

    local dx
    local dy

    local x = down.x
    local y = down.y

    if down.x < up.x then
      dx = 1
    else
      dx = -1
    end

    if down.y < up.y then
      dy = 1
    else
      dy = -1
    end

    while x ~= up.x do
      x = x + dx
      if y ~= up.y then
        mg_set_node({x=x, y=baseY-2, z=y}, {name=stone})
        mg_set_node({x=x, y=baseY-1, z=y}, {name=air_light})
        mg_set_node({x=x, y=baseY, z=y}, {name=air_light})
      end
    end

    while y ~= up.y do
      y = y + dy
      if y ~= up.y then
        mg_set_node({x=x, y=baseY-2, z=y}, {name=stone})
        mg_set_node({x=x, y=baseY-1, z=y}, {name=air_light})
        mg_set_node({x=x, y=baseY, z=y}, {name=air_light})
      end
    end
  end
end

local function carveTop(grid, mg_set_node)
  for i = 1, mg_arch.DCOLS do
    for j = 1, mg_arch.DROWS do
      local x = i
      local z = j

      local ground = stone
      local middle1 = air
      local middle2 = air
      local top = air
      local finish = air

      local baseY = 8 + startY

      local list = {
        { x = x, y = baseY + 1, z = z },
        { x = x, y = baseY + 2, z = z },
        { x = x, y = baseY + 3, z = z },
        { x = x, y = baseY + 4, z = z },
        { x = x, y = baseY + 5, z = z },
        { x = x, y = baseY + 6, z = z },
        { x = x, y = baseY + 7, z = z },
      }

      local dungeon = grid[i][j].layers.dungeon

      if dungeon == mg_arch.tileType.WALL then
        ground = stone_block
        middle1 = stone_block

        if mg_arch.rand_percent(30) then
          finish = air
          if mg_arch.rand_percent(30) then
            top = air
            if mg_arch.rand_percent(30) then
              middle2 = air
            else
              middle2 = stone_block
            end
          else
            top = stone_block
            middle2 = stone_block
          end
        else
          middle2 = stone_block
          top = stone_block
          finish = stone_block
        end
      end

      mg_set_node(list[1], {name=ground})
      mg_set_node(list[2], {name=middle1})
      mg_set_node(list[3], {name=middle2})
      mg_set_node(list[4], {name=top})
      mg_set_node(list[5], {name=finish})
      mg_set_node(list[6], {name=air})
      mg_set_node(list[7], {name=air})
    end
  end
end

local spawnpoint = {}
spawnpoint.pos = {x = mg_arch.DCOLS / 2, y = spawn_y, z = mg_arch.DROWS + 10}

function spawnpoint.bring(player)
	if player and spawnpoint.pos then
		local pos = spawnpoint.pos
		player:set_pos({x=pos.x, y=pos.y+1.5, z=pos.z})
		player:set_look_horizontal(3.1415)
    return true
	end
end



local function makeEntrance(sx, sy, mg_set_node)
  local baseY = 9 + startY

  local x = sx
  local z = sy
  local list
  list = {
    { x = x - 1, y = baseY + 1, z = z },
    { x = x - 1, y = baseY + 2, z = z },
    { x = x + 1, y = baseY + 1, z = z },
    { x = x + 1, y = baseY + 2, z = z },
  }

  for _, pos in ipairs(list) do
    mg_set_node(pos, {name=light})
  end


  list = {
    { x = x, y = baseY + 1, z = z },
    { x = x, y = baseY + 2, z = z },
    { x = x, y = baseY + 1, z = z + 1 },
    { x = x, y = baseY + 2, z = z + 1 },
    { x = x, y = baseY + 1, z = z + 2 },
    { x = x, y = baseY + 2, z = z + 2 },
  }

  for _, pos in ipairs(list) do
    mg_set_node(pos, {name="air"})
  end

end

local function add_weapon(inv, name)
  local stack = ItemStack(name)
  stack = mg_tools.enchant_weapon(stack, 0)
  inv:add_item('main', stack)
end


local function add_armor(inv, name)
  local stack = ItemStack(name)
  stack = mg_armor.enchant_armor(stack, 0)
  inv:add_item('main', stack)
end


local function makeStartChest(sx, sy)
  local baseY = 9 + startY

  local x = sx
  local z = sy

  local pos = { x = x + 1, y = baseY + 1, z = z + 1 }

  minetest.set_node(pos, {name=chest, param2=2})

  local loc = {type="node", pos=pos}
  local inv = minetest.get_inventory(loc)

  local book = ItemStack(book_written)
  local bookData = {
    owner = "dungeon master",
    title = "The Dungeons of Doom",
    text = 'It is rumored that great treasure lies in the depths of these dungeons. The greatest treasure is the Crystal of Yendor which lies in depth 12 of the dungeons.  The one who brings the crystal to light and place it on the pedistal will win great fame and glory in the world.',
    page = 1,
    page_max = 1
  }
  book:get_meta():from_table({ fields = bookData })

  inv:add_item('main', book)
  inv:add_item('main', ration)
  local stack = ItemStack('bows:arrow')
  stack:set_count(12)
  inv:add_item('main', stack)
  inv:add_item('main', 'bows:bow_wood')

  add_weapon(inv, dagger)
  add_armor(inv, "mg_armor:torso_cloth")

  pos = {x = mg_arch.DCOLS / 2 - 3, y = spawn_y + 1, z = mg_arch.DROWS}
  core.set_node({x=pos.x + 3, y=pos.y, z=pos.z + 13}, {name="mapgen:pedistal"})
end

-- converts the brogue grid into luanti nodes
local function carveLevel(grid, depth, mg_set_node)

  local baseY = -11 * (depth - 1) + startY

  for i = 1, mg_arch.DCOLS do
    for j = 1, mg_arch.DROWS do
      local x = i
      local z = j
      local list = {
        { x = x, y = baseY + 1, z = z },
        { x = x, y = baseY + 2, z = z },
        { x = x, y = baseY + 3, z = z },
        { x = x, y = baseY + 4, z = z },
        { x = x, y = baseY + 5, z = z },
        { x = x, y = baseY + 6, z = z },
        { x = x, y = baseY + 7, z = z },
        { x = x, y = baseY + 8, z = z },
        { x = x, y = baseY + 9, z = z }
      }

      mg_set_node({x=x, y=baseY-1, z=z}, {name=stone})
      mg_set_node({x=x, y=baseY, z=z}, {name=stone})

      local base = blackstone
      local deep1 = blackstone
      local deep2 = blackstone
      local shallow = stone
      local ground = cobble
      local middle1 = stone
      local middle2 = stone
      local top = stone
      local finish = stone

      local dungeon = grid[i][j].layers.dungeon
      local surface = grid[i][j].layers.surface
      local liquid = grid[i][j].layers.liquid
      if dungeon == mg_arch.tileType.FLOOR then
        middle1 = air
        middle2 = air
        top = air

      --Traps
      elseif dungeon == mg_arch.tileType.TRAP_DOOR_HIDDEN then
        ground = descent_trap
        middle1 = air
        middle2 = air
        top = air
      elseif dungeon == mg_arch.tileType.TRAP_DOOR then
        ground = descent_trap
        middle1 = air
        middle2 = air
        top = air
      elseif dungeon == mg_arch.tileType.FLAMETHROWER_HIDDEN then
        ground = fire_trap
        middle1 = air
        middle2 = air
        top = air
      elseif dungeon == mg_arch.tileType.FLAMETHROWER then
        ground = fire_trap
        middle1 = air
        middle2 = air
        top = air

      elseif dungeon == mg_arch.tileType.GAS_TRAP_POISON then
        ground = gas_trap
        middle1 = air
        middle2 = air
        top = air
      elseif dungeon == mg_arch.tileType.GAS_TRAP_POISON_HIDDEN then
        ground = gas_trap
        middle1 = air
        middle2 = air
        top = air

      elseif dungeon == mg_arch.tileType.NET_TRAP_HIDDEN then
        ground = net_trap
        middle1 = air
        middle2 = air
        top = air
      elseif dungeon == mg_arch.tileType.NET_TRAP then
        ground = net_trap
        middle1 = air
        middle2 = air
        top = air
      elseif dungeon == mg_arch.tileType.FLOOD_TRAP_HIDDEN then
        ground = flood_trap
        middle1 = air
        middle2 = air
        top = air
      elseif dungeon == mg_arch.tileType.FLOOD_TRAP then
        ground = flood_trap
        middle1 = air
        middle2 = air
        top = air


      elseif dungeon == mg_arch.tileType.WALL then
        middle1 = stone_block
        middle2 = stone_block
        top = stone_block
      elseif dungeon == mg_arch.tileType.TORCH_WALL then
        middle1 = stone_block
        middle2 = light
        top = stone_block
      elseif dungeon == mg_arch.tileType.STATUE_INERT then
        middle1 = statue
        middle2 = air
        top = air
      elseif dungeon == mg_arch.tileType.DOOR then
        -- I add and rotate the doors later
        middle1 = door
        middle2 = air
        top = stone_block
      elseif dungeon == mg_arch.tileType.SECRET_DOOR then
        middle1 = secret_door
        middle2 = air
        top = stone_block
      elseif dungeon == mg_arch.tileType.UP_STAIRS then
        middle1 = air
        middle2 = air
        top = air
        finish = air
        mg_set_node({x=x, y=baseY+10, z=z}, {name=air})
        mg_set_node({x=x, y=baseY+11, z=z}, {name=air})
      elseif dungeon == mg_arch.tileType.DOWN_STAIRS then
        mg_set_node({x=x, y=baseY-1, z=z}, {name=air})
        mg_set_node({x=x, y=baseY, z=z}, {name=air})
        base = air
        deep2 = air
        deep1 = air
        shallow = air
        ground = air
        middle1 = air
        middle2 = air
        top = air
      end

      if surface == mg_arch.df.DF_GRASS then
        if dungeon == mg_arch.tileType.FLOOR then
          ground = mossycobble
        else
          -- To preserve the traps
          ground = ground
        end
        middle1 = grass
      elseif surface == mg_arch.df.DF_FOLIAGE then
        if dungeon == mg_arch.tileType.FLOOR then
          ground = mossycobble
        else
          -- To preserve the traps
          ground = ground
        end
        middle1 = fernbottom
        middle2 = ferntop
      elseif surface == mg_arch.df.DF_DEAD_GRASS then
        middle1 = dry_grass
      --TODO need to set the param2 color of the dead foliage to be dead
      elseif surface == mg_arch.df.DF_DEAD_FOLIAGE then
        middle1 = grassbottom
        middle2 = grasstop
      elseif surface == mg_arch.df.DF_RUBBLE then
        middle1 = getRubble()
      elseif surface == mg_arch.df.DF_LUMINESCENT_FUNGUS then
        middle1 = glowing_fungus
      elseif surface == mg_arch.df.DF_GRANITE_COLUMN then
        ground = stone
        middle1 = stone
        middle2 = stone
        top = stone
      end
      -- let walls take priority
      if dungeon == mg_arch.tileType.WALL then
        middle1 = stone_block
        middle2 = stone_block
        top = stone_block
      elseif dungeon == mg_arch.tileType.GRANITE then
        middle1 = stone
        middle2 = stone
        top = stone
      elseif liquid == mg_arch.tileType.CHASM then
        mg_set_node({x=x, y=baseY-1, z=z}, {name=air})
        mg_set_node({x=x, y=baseY, z=z}, {name=air})
        base = air
        deep1 = air
        deep2 = air
        shallow = air
        ground = air
        middle1 = air
        middle2 = air
        top = air
      elseif
        liquid == mg_arch.tileType.BRIDGE or
        liquid == mg_arch.tileType.BRIDGE_EDGE
      then
        mg_set_node({x=x, y=baseY-1, z=z}, {name=air})
        mg_set_node({x=x, y=baseY, z=z}, {name=air})
        base = air
        deep1 = air
        deep2 = air
        shallow = air
        ground = wood
        middle1 = air
        middle2 = air
        top = air
      elseif surface == mg_arch.tileType.BRIDGE_EDGE or
             surface == mg_arch.tileType.BRIDGE
      then
        base = blackstone
        deep1 = blackstone
        deep2 = blackstone
        shallow = chasm_edge
        ground = wood
        middle1 = air
        middle2 = air
        top = air


      --Liquids
      elseif liquid == mg_arch.tileType.CHASM_EDGE then
        base = blackstone
        deep1 = blackstone
        deep2 = blackstone
        shallow = chasm_edge
        ground = air
        middle1 = air
        middle2 = air
        top = air
      elseif liquid == mg_arch.tileType.DEEP_WATER then
        base = sand
        deep1 = water
        deep2 = water
        shallow = water
        ground = water
        middle1 = air
        middle2 = air
        top = air
      elseif liquid == mg_arch.tileType.SHALLOW_WATER then
        base = sand
        deep1 = sand
        deep2 = sand
        shallow = sand
        ground = water
        middle1 = air
        middle2 = air
        top = air
      elseif liquid == mg_arch.tileType.LAVA then
        base = stone
        deep1 = lava
        deep2 = lava
        shallow = lava
        ground = lava
        middle1 = air
        middle2 = air
        top = air
      end

      if liquid == mg_arch.df.DF_SUNLIGHT then
        finish = 'mg_torch:air_light_14'
      end

      mg_set_node(list[1], {name=base})
      mg_set_node(list[2], {name=deep1})
      mg_set_node(list[3], {name=deep2})
      mg_set_node(list[4], {name=shallow})
      mg_set_node(list[5], {name=ground})
      mg_set_node(list[6], {name=middle1})
      mg_set_node(list[7], {name=middle2})
      mg_set_node(list[8], {name=top})
      mg_set_node(list[9], {name=finish})
    end
  end
end

-- Digs out a little more to connect the chasm with the level under it
local function carveChasm(grid, depth)
  if grid == nil then
    return
  end
  local baseY = -11 * (depth - 1) + startY
  for i = 1, mg_arch.DCOLS do
    for j = 1, mg_arch.DROWS do
      local x = i
      local z = j

      local liquid = grid[i][j].layers.liquid

      if liquid == mg_arch.tileType.CHASM or
         liquid == mg_arch.tileType.BRIDGE or
         liquid == mg_arch.tileType.BRIDGE_EDGE
      then
        minetest.set_node({x=x, y=baseY-2, z=z}, {name=air})
        minetest.set_node({x=x, y=baseY-1, z=z}, {name=air})
        minetest.set_node({x=x, y=baseY, z=z}, {name=air})
      end
    end
  end
end

-- This adjusts nodes with param2
-- and adds some lighting
local function placeItems(grid, depth)

  local dir2param2 = {
    2,
    0,
    3,
    1
  }

  local baseY = -11 * (depth - 1) + startY
  -- place the torches after we place all the other nodes
  -- Place ladders connecting the levels to get up and down
  for i = 1, mg_arch.DCOLS do
    for j = 1, mg_arch.DROWS do
      local x = i
      local z = j
      local list = {
        { x = x, y = baseY +-1, z = z },
        { x = x, y = baseY + 0, z = z },
        { x = x, y = baseY + 1, z = z },
        { x = x, y = baseY + 2, z = z },
        { x = x, y = baseY + 3, z = z },
        { x = x, y = baseY + 4, z = z },
        { x = x, y = baseY + 5, z = z },
        { x = x, y = baseY + 6, z = z },
        { x = x, y = baseY + 7, z = z },
        { x = x, y = baseY + 8, z = z },
        { x = x, y = baseY + 9, z = z }
      }

      local showlight = i % 5 == (j % 5 )

      local dungeon = grid[i][j].layers.dungeon
      local liquid = grid[i][j].layers.liquid

      if dungeon == mg_arch.tileType.TRAP_DOOR_HIDDEN or
         dungeon == mg_arch.tileType.TRAP_DOOR or
         dungeon == mg_arch.tileType.FLAMETHROWER_HIDDEN or
         dungeon == mg_arch.tileType.FLAMETHROWER or
         dungeon == mg_arch.tileType.NET_TRAP_HIDDEN or
         dungeon == mg_arch.tileType.NET_TRAP or
         dungeon == mg_arch.tileType.GAS_TRAP_POISON or
         dungeon == mg_arch.tileType.GAS_TRAP_POISON_HIDDEN or
         dungeon == mg_arch.tileType.FLOOD_TRAP_HIDDEN or
         dungeon == mg_arch.tileType.FLOOD_TRAP then

         -- voxel manip doesn't start timers - so I need to start trap timers
         local ground_pos = list[7]
         core.get_node_timer(ground_pos):start(0.1)
       end


      if liquid == mg_arch.tileType.CHASM or
         liquid == mg_arch.tileType.BRIDGE or
         liquid == mg_arch.tileType.BRIDGE_EDGE
      then
        minetest.set_node({x=x, y=baseY-2, z=z}, {name=air})
        minetest.set_node({x=x, y=baseY-1, z=z}, {name=air})
        minetest.set_node({x=x, y=baseY, z=z}, {name=air})
      end

      if dungeon == mg_arch.tileType.UP_STAIRS or dungeon == mg_arch.tileType.DOWN_STAIRS then
        local sn
        local en
        if dungeon == mg_arch.tileType.UP_STAIRS then
          sn=8
          en=13
        else
          sn=1
          en=8
        end
        for dir=1, 4 do
          local x1 = i + nbDirs[dir][1]
          local y1 = j + nbDirs[dir][2]
          local isInMap = mg_arch.coordinatesAreInMap(x1, y1)
          if isInMap and (grid[x1][y1].layers.dungeon == mg_arch.tileType.WALL or dir == 4) then
            for n=sn, en do
              local nodePos = { x = i, y = baseY + n - 2, z = j }
              local wallPos = { x = x1, y = baseY + n - 2, z = y1 }

              local wdir = minetest.dir_to_wallmounted(vector.subtract(wallPos, nodePos))
              minetest.set_node(nodePos, {name=ladder, param2=wdir})
            end
            break
          end
        end
      end

      -- Rotate the statues to face the floor not the wall
      if dungeon == mg_arch.tileType.STATUE_INERT then
        for dir=1, 4 do
          local x1 = i + nbDirs[dir][1]
          local y1 = j + nbDirs[dir][2]
          local isInMap = mg_arch.coordinatesAreInMap(x1, y1)
          if isInMap and (grid[x1][y1].layers.dungeon == mg_arch.tileType.FLOOR) then
            local nodePos = { x = x, y = baseY + 6, z = z }
            local param2 = dir2param2[dir]

            minetest.set_node(nodePos, {name=statue, param2=param2})
            break
          end
        end
      end

      -- add the doors and rotate them the right way
      if dungeon == mg_arch.tileType.DOOR or dungeon == mg_arch.tileType.SECRET_DOOR then
        for dir=1, 4 do
          local x1 = i + nbDirs[dir][1]
          local y1 = j + nbDirs[dir][2]
          local isInMap = mg_arch.coordinatesAreInMap(x1, y1)
          if isInMap and (grid[x1][y1].layers.dungeon == mg_arch.tileType.FLOOR) then
            local doorNodePos = { x = x, y = baseY + 6, z = z }
            local param2 = 2
            if dir == 3 or dir == 4 then
              param2 = 1
            end

            local door_name = door
            if dungeon == mg_arch.tileType.SECRET_DOOR then
              door_name = secret_door
            end

            minetest.set_node(doorNodePos, {name=door_name, param2=param2})
            break
          end
        end
      end

      -- Hack to turn this off for now
      local use_torches = false
      if dungeon == mg_arch.tileType.WALL and use_torches then
        -- Dynamically update the light level
        -- to lower the lower the depth
        -- also do the distribution better
        -- do this and then go back and check all walls
        -- if there's a light with in 4 blocks of it
        if showlight then
          for dir=1, 4 do
            local x1 = i + nbDirs[dir][1]
            local y1 = j + nbDirs[dir][2]
            local isInMap = mg_arch.coordinatesAreInMap(x1, y1)
            if isInMap and grid[x1][y1].layers.liquid == mg_arch.df.DF_DARKNESS then
              break
            end
            if isInMap and grid[x1][y1].layers.dungeon == mg_arch.tileType.FLOOR then
              local pos = { x = x1, y = baseY + 7, z = y1 }

              local light_level = math.ceil((20-depth + 1)/20 * 12);

              local wdir = minetest.dir_to_wallmounted(vector.subtract(list[9], pos))
              minetest.set_node(pos, {name=torch..light_level, param=1, param2=wdir})
              break
            end
          end
        end
      end
    end
  end
end


local function placeOneItem(item, pos)
  local cat = mg_arch.item_api.getItemCategory(item.category)
  item = cat.on_item_spawn(item, pos)
  print('spawned item: ' .. (item.quantity or 1) .. ' (' .. item.enchant1 .. ') ' ..
        item.category .. " of " .. item.kind .. ' as ' .. item.node_name)
end

local function placeRealItems(depth, items, pmap)
  local baseY = -11 * (depth - 1) + startY
  if depth == 0 then
    baseY = 4
  end

  for _,item in pairs(items) do
    local y = baseY + 6
    local ll = pmap[item.x][item.y].layers.liquid

    -- Lower the item if it's in shallow water
    -- or chasm edge so it's not floating
    if ll == mg_arch.tileType.SHALLOW_WATER
      or ll == mg_arch.tileType.CHASM_EDGE
    then
      y = y - 1
    end

    local pos = { x = item.x, y = y, z = item.y }
    placeOneItem(item, pos)
  end
end


local function carveFirstLevels()
  local seed = os.time()
  --seed = 1738630348
  mg_arch.dungeon = table.copy(mg_arch.default_dungeon)
  mg_arch.dungeon.seed = seed


  local players = core.get_connected_players()
  for _, player in pairs(players) do
    local meta = player:get_meta()
    meta:set_int("dungeon_seed", seed)
  end

  core.handle_async(function(inner_seed)
    local MAX_LEVELS = 2
    math.randomseed(inner_seed)
    local worldData = mg_arch.generateWorldData()
    local levels = mg_arch.makeLevels()
    local pmaps = {}
    for i=1, MAX_LEVELS do
      local calc_seed = inner_seed + i - 1
      math.randomseed(calc_seed)
      print('depth '..i..' seed ' .. calc_seed)
      local grid = mg_arch.digDungeon(i, levels)
      local xLoc = levels[i].upStair.x
      local yLoc = levels[i].upStair.y
      mg_arch.mark_upstairs_fov(grid, xLoc, yLoc)
      pmaps[i] = grid
      local levelItems = mg_arch.populateItems(
        grid, worldData, i, xLoc, yLoc, false)
      levels[i].items = levelItems
      local monsters = mg_arch.monsters.populateMonsters(i, grid)
      levels[i].monsters = monsters
      mg_arch.printPmap(grid, false, levelItems, monsters)
    end

    return {
      worldData = worldData,
      NUMBER_GENERATED_LEVELS = MAX_LEVELS,
      pmaps = pmaps,
      levels = levels,
    }
  end,
  function(result)
    local MAX_LEVELS = 2
    print('async all done: ' .. result.NUMBER_GENERATED_LEVELS)
    mg_arch.dungeon.worldData = result.worldData
    mg_arch.dungeon.NUMBER_GENERATED_LEVELS = result.NUMBER_GENERATED_LEVELS
    mg_arch.dungeon.pmaps = result.pmaps
    mg_arch.dungeon.levels = result.levels

    local pmaps = result.pmaps
    local levels = result.levels

    -- TODO generate some top levels
    local top = mg_arch.makeTop(2)
    local topItems = mg_arch.populateTopItems(top, result.worldData, 1)

    -- write data to file
    mg_arch.save_dungeon(mg_arch.dungeon)

    -- Minetest specific functionality
    -- clear previous entities
    minetest.clear_objects({mode = "quick"})

    -- Set up voxel manip
    local p1 = { x = 1, y = -11 * (MAX_LEVELS - 1)  + startY - 1, z = 1}
    local p2 = { x = mg_arch.DCOLS, y = startY + 15, z = mg_arch.DROWS}
    local vm = core.get_voxel_manip()
    local emin, emax = vm:read_from_map(p1, p2)
    local data = vm:get_data()
    local a = VoxelArea:new{
      MinEdge = emin,
      MaxEdge = emax
    }

    local function mg_set_node(pos, node_data)
      local node_id = core.get_content_id(node_data.name)
      data[a:index(pos.x, pos.y, pos.z)] = node_id
    end

    carveTop(top, mg_set_node)

    for i, pmap in ipairs(pmaps) do
      carveLevel(pmap, i, mg_set_node)
    end


    makeEntrance(levels[1].upStair.x, levels[1].upStair.y, mg_set_node)

    vm:set_data(data)
    vm:write_to_map(true)

    for i, pmap in ipairs(pmaps) do
      placeItems(pmap, i)
    end

    for i, grid in ipairs(pmaps) do
      placeRealItems(i, levels[i].items, grid)
      levels[i].items = nil
    end

    placeRealItems(0, topItems, top)

    for i=1, MAX_LEVELS do
      mg_mobs.spawn_monsters(i, levels[i].monsters)
      levels[i].monsters = nil
    end

    makeStartChest(levels[1].upStair.x, levels[1].upStair.y)

    connectLevels(levels, 1, 2, core.set_node)

    place_highscores()
  end, seed)
end


local function carveOneLevel(depth)
  if depth > mg_arch.dungeon.MAX_LEVELS then
    return
  end

  if mg_arch.dungeon.NUMBER_GENERATED_LEVELS == 0 then
    -- handled by on player join
    return
  end

  local levels = mg_arch.dungeon.levels
  local worldData = mg_arch.dungeon.worldData
  local seed = mg_arch.dungeon.seed

  core.handle_async(function(inner_seed, depth, AMULET_LEVEL, levels, worldData)
    local i = depth
    local calc_seed = inner_seed + i - 1
    math.randomseed(calc_seed)
    print('depth '..i..' seed ' .. calc_seed)
    local grid = mg_arch.digDungeon(i, levels)
    local xLoc = levels[depth].upStair.x
    local yLoc = levels[depth].upStair.y
    mg_arch.mark_upstairs_fov(grid, xLoc, yLoc)
    local items = mg_arch.populateItems(
        grid, worldData, i, xLoc, yLoc, i == AMULET_LEVEL)
    local monsters = mg_arch.monsters.populateMonsters(i, grid)
    mg_arch.printPmap(grid, false, items, monsters)
    return {
      grid = grid,
      items = items,
      monsters = monsters,
      -- Levels and world data are mutated
      -- so need to pass them back in the result
      worldData = worldData,
      levels = levels
    }
  end,
  function(result)
    local i = depth
    local grid = result.grid

    mg_arch.dungeon.worldData = result.worldData
    mg_arch.dungeon.pmaps[i] = result.grid
    mg_arch.dungeon.levels = result.levels
    mg_arch.dungeon.NUMBER_GENERATED_LEVELS = i

    local p1 = { x = 1, y = -11 * (depth - 1)  + startY - 1, z = 1}
    local p2 = { x = mg_arch.DCOLS, y = p1.y + 11, z = mg_arch.DROWS}
    local vm = core.get_voxel_manip()
    local emin, emax = vm:read_from_map(p1, p2)
    local data = vm:get_data()
    local a = VoxelArea:new{
      MinEdge = emin,
      MaxEdge = emax
    }

    local function mg_set_node(pos, node_data)
      local node_id = core.get_content_id(node_data.name)
      data[a:index(pos.x, pos.y, pos.z)] = node_id
    end

    -- actually generate it in minetest
    carveLevel(grid, i, mg_set_node)

    vm:set_data(data)
    vm:write_to_map(true)

    -- Figure out where items go
    placeItems(grid, i)
    placeRealItems(i, result.items, grid)
    mg_mobs.spawn_monsters(i, result.monsters)

    carveChasm(mg_arch.dungeon.pmaps[depth-1], depth - 1)

    connectLevels(result.levels, depth - 1, depth, core.set_node)

    mg_arch.save_dungeon(mg_arch.dungeon)
  end, seed, depth, mg_arch.dungeon.AMULET_LEVEL, levels, worldData)
end


core.register_chatcommand("carve", {
  func = function(name, param)
    carveFirstLevels()

    return true, "Reset dungeon"
  end,
})

core.register_on_joinplayer(function(player)
  local meta = player:get_meta()
  local player_seed = meta:get_int("dungeon_seed")
  if mg_arch.dungeon.NUMBER_GENERATED_LEVELS == 0 then
    print('no levels generated yet, generate first levels')
    -- Need to set a small time out for this to work
    minetest.after(0.5, carveFirstLevels)
  elseif player_seed == 0 then
    meta:set_int("dungeon_seed", mg_arch.dungeon.seed or 0)
  -- Means player joined since the game has been reset
  -- bring them back to the beginning
  elseif player_seed ~= mg_arch.dungeon.seed then
    meta:set_int("dungeon_seed", mg_arch.dungeon.seed)
    spawnpoint.bring(player)
  end
end)


local function save_highscore(player, did_win)
  local seed = mg_arch.dungeon.seed
  local meta = player:get_meta()
  local gold = meta:get_int("gold")
  local depth = meta:get_int("current_level")
  local max_level = meta:get_int("max_level")
  local score_list = load_file("highscores")
  local time = os.time()
  local item = {
    gold = gold,
    depth = depth,
    max_depth = max_level,
    player = player:get_player_name(),
    won = did_win,
    time = time,
    seed = seed,
  }
  table.insert(score_list, item)
  save_file("highscores", score_list)

  -- Reset the players gold now
  meta:set_int("gold", 0)
end



core.register_on_respawnplayer(function(player)
  -- This isn't working for some reason
  -- It's being called, but play still spawns somewhere else
  -- need to set a small timeout to get this to work... :-(

  -- Reset map if single player is playing
  -- need to figure out how to handle multiple
  -- players -- maybe respawn player at the beginning
  -- of the level?
  local players = core.get_connected_players()
  local meta = player:get_meta()
  local win_status = meta:get_int("win_status") or 0
  --local is_singleplayer = core.is_singleplayer()

  -- If single player or if in multiplayer someone won the game
  -- reset the game and take everyone to the spawn point
  if( permadeath or win_status == 1) then
    -- Only save highscores if didn't already win the game
    -- the game file is saved at win condition
    -- right now player restarts game by killing themselves
    if win_status < 1 then
      -- TODO: would be cool to save cause of death in there too
      if permadeath then
        save_highscore(player, false)
      end
    end
    -- if the player did happen to win
    -- reset their status to 0
    meta:set_int("win_status", 0)


    carveFirstLevels()

    -- There's a weird bug that will spawn player at 0,0,0
    -- Hacky timeout gets around that - i'm not sure how
    -- to fix that
    core.after(0.1, function()
      -- should reset all players
      -- to starting point
      for _, next_player in pairs(players) do
        spawnpoint.bring(next_player)
      end
    end)

  else
    -- No peradeath - helpful when playing with others
    -- respawn at the start of the last level
    local current_level = meta:get_int("current_level")
    core.after(0.1, function()
      if current_level > 0  then
        local us = mg_arch.dungeon.levels[current_level].upStair
        local y = -11 * (current_level - 1) + startY + 6
        local x = us.x or 0
        local z = us.y or 0
        local pos = {
          x = x,
          y = y,
          z = z
        }
        player:set_pos(pos)
      else
        spawnpoint.bring(player)
      end
    end)
  end
end)

-- [register] On New Player
core.register_on_newplayer(function(player)
	spawnpoint.bring(player)
end)


-- gets called for each player
-- double players means double monsters :-)
-- assumes this get called every second
local function process_monster_spawn_timer(player)
  local meta = player:get_meta()
  local monster_spawn_timer = meta:get_int("monster_spawn_timer")

  -- not set yet - set it
  if monster_spawn_timer == 0 then
    monster_spawn_timer = math.random(125, 175)
    meta:set_int("monster_spawn_timer", monster_spawn_timer)
    return
  end

  --print('monster timer ' .. monster_spawn_timer)
  monster_spawn_timer = monster_spawn_timer - 1

  -- was a 1, now at 0, spawn horde
  if monster_spawn_timer <= 0 then
    local depth = meta:get_int("current_level")
    if depth <=0 then
      -- player is probably flying around
      -- don't want to crash the game...
      -- try again in 1 second
      meta:set_int("monster_spawn_timer", 1)
      return
    end
    local pmap = mg_arch.dungeon.pmaps[depth]
    -- no pmap - retry in a few seconds
    if not pmap then
      meta:set_int("monster_spawn_timer", 1)
      return
    end
    monster_spawn_timer = math.random(125, 175)
    meta:set_int("monster_spawn_timer", monster_spawn_timer)
    local forbiddenFlags = {hord_is_summoned = 1, horde_machine_only = 1}


    -- Mark current player FOV in pmap
    -- so monsters dont spawn there
    local pos = player:get_pos()
    local x = math.floor(pos.x + 0.5)
    local y = math.floor(pos.z + 0.5)
    mg_arch.clear_fov(pmap)
    mg_arch.mark_upstairs_fov(pmap, x, y)

    local horde = mg_arch.monsters.spawn_horde(depth, pmap, 0, -1, -1, forbiddenFlags, {})
    mg_mobs.spawn_monsters(depth, horde)
    return
  end

  meta:set_int("monster_spawn_timer", monster_spawn_timer)
end

-- This might be a performance killer
-- expecially on higher depths
-- only update 1 / second for now
-- might need to load this in a voxmanip object
-- for better performance
-- also need to figure out how to turn off the lights
-- of a previous level - they stay on when you go down or up
local function update_player_lighting (player)
  local player_pos = player:get_pos()
  local x = math.floor(player_pos.x + 0.5)
  local z = math.floor(player_pos.z + 0.5)

  local meta = player:get_meta()
  local depth = meta:get_int("current_level")
  local baseY = -11 * (depth - 1) + startY

  local radius = mg_arch.DCOLS - 1
  for _ = 1, depth do
    -- radius = radius * 85 / 100
    -- since goal is level 10 - have it get darker sooner
    radius = radius * 70 / 100
  end

  radius = math.ceil(radius + 225 / 100)

  --print('light radius is ' .. radius)

  -- Have a little bigger than radius to turn off
  -- lights outside of raduis
  local startx = x - math.ceil(radius*1.25)
  local endx = x + math.ceil(radius*1.25)
  local startz = z - math.ceil(radius*1.25)
  local endz = z + math.ceil(radius*1.25)

  local grid = mg_arch.dungeon.pmaps[depth]

  for i=startx, endx do
    for j=startz, endz do
      local pos = { x = i, y = baseY + 8, z = j }
      local node = core.get_node(pos)
      if (x-i)*(x-i) + (z-j)*(z-j) <= radius*radius then
        local isInMap = mg_arch.coordinatesAreInMap(i, j)
        local is_darkness = grid and isInMap and grid[i][j].layers.liquid == mg_arch.df.DF_DARKNESS
        if node.name == 'air' and not is_darkness then
          core.set_node(pos, {name="mg_torch:air_light_8"})
        end
      else
        if node.name == 'mg_torch:air_light_8' then
          core.set_node(pos, {name="air"})
        end
      end
    end
  end
end

local timer = 0

core.register_globalstep(function(dtime)
  timer = timer + dtime
  if timer < 1 then
    return
  end
  timer = 0
  -- ignore if we haven't generated anything levels yet
  -- they will get generated a little bit after
  -- the player has joined
  if mg_arch.dungeon.NUMBER_GENERATED_LEVELS == 0 then
    return
  end

	for _, player in pairs(core.get_connected_players()) do
    -- needs to be called every second
    process_monster_spawn_timer(player)
    update_player_lighting(player)

    -- Not sure if this is performant or not
    local meta = player:get_meta()
    local player_seed = meta:get_int("dungeon_seed")
    local current_level = meta:get_int("current_level")
    local pos = player:get_pos()
    local depth = math.floor((pos.y - startY - 10) / (-11) + 1)
    -- update new level if they are not the same and the player
    -- is in the same dungeon as last time - multiplayer check here
    -- like if a player left between dungeon sessions and another player won and
    -- regenerated the dungeon
    if current_level ~= depth and player_seed == mg_arch.dungeon.seed then
      local username = player:get_player_name()
      minetest.chat_send_player(username, 'You are now at depth ' .. depth)
      meta:set_int("current_level", depth)
      print('player '..username..' is now at depth ' .. depth)

      if depth + 1 > mg_arch.dungeon.NUMBER_GENERATED_LEVELS then
        carveOneLevel(depth + 1)
        minetest.chat_send_player(username, 'created a dungeon level ' .. depth + 1)
        end
    end

    local max_level = meta:get_int("max_level")
    if depth > max_level then
      max_level = depth
      meta:set_int("max_level", max_level)
      print('set player max level at ' .. max_level)
    end
  end
end)


local esc = minetest.formspec_escape

local function formspec_read(text)
	return "textarea[0.5,1.5;7.5,7;;" .. esc(text) .. ";]"
end

local formspec_size = "size[8,8]"

core.register_node("mg_game:crystal", {
  description = "Crystal",
  drawtype = "mesh",
  inventory_image = "regulus_crystal.png",
  pointable = true,
  walkable = false,
  groups = {crumbly = 3},
  mesh = "regulus_crystal.obj",
  paramtype = "light",
  paramtype2 = "facedir",
  sunlight_propagates = true,
  tiles = {"regulus_crystal.png"},
  light_source = 14,
  after_place_node = function(pos, placer, _itemstack, _pointed_thing)
    local node_below = core.get_node({x=pos.x, y=pos.y-1, z=pos.z})
    local username = placer:get_player_name()

    local meta = placer:get_meta()
    local win_status = meta:get_int("win_status") or 0

    -- Win the game by placing on pedistal
    if node_below.name == "mapgen:pedistal" and win_status ~= 1 then
      local win_text = ''
      if permadeath then
        save_highscore(placer, true)
        win_text = 'You have won the game.  Your name is forever eched in glory.   \n\n(To replay a new dungeon, respawn yourself by dying)'
      else
        win_text = 'You have won the game.  However, because you turned off permadeath, you highscore will not be saved.  To reset your stats, turn on permadeath, then find a way to die.'
      end


      local formspec = formspec_size .. formspec_read(win_text)
      minetest.show_formspec(username, "mg_game:end_game", formspec)

      meta:set_int("win_status", 1)

      -- Bye bye Crystal
      -- The crystal won't reset when a new dungeon is generated
      -- so remove it now to prevent the player form triggering
      -- a win after the dungeon respawns
      minetest.after(2, function()
        core.set_node(pos, {name= 'air'})
      end)
    end
  end
})
