local function has_line_of_sight(mob_pos, target_pos)
  -- only nodes - no object or liques
  local cast = core.raycast(mob_pos, target_pos, false, false)
  local thing = cast:next()
  while thing do
    if thing.type == "node" then
      local node = core.get_node(thing.intersection_point)
      local def = minetest.registered_nodes[node.name]
      -- A walkable node is a solid node
      if def and def.walkable then
        return false
      end
    end
    thing = cast:next()
  end
  return true
end

-- https://www.gamedeveloper.com/programming/shooting-a-moving-target
local function aim_ahead(pos, vel, speed)
  local a = vector.dot(vel, vel) - speed*speed
  local b = 2*vector.dot(vel, pos)
  local c = vector.dot(pos, pos)

  local desc = b*b - 4*a*c;

  -- If the discriminant is negative, then there is no solution
  if desc > 0 then
    return 2*c/(math.sqrt(desc) - b)
  else
    return -1
  end
end

local function aim_point(tpos, tvel, spos, speed)
  local dt = aim_ahead(vector.subtract(tpos, spos), tvel, speed)
  if dt == -1 then
    return nil
  end
  local point = vector.add(tpos, vector.multiply(tvel, dt))
  return point
end

local function shuffle_list_in_place(list)
  for i=1, #list do
    local r = math.random(i, #list)
    if i~= r then
      local buf = list[r]
      list[r] = list[i]
      list[i] = buf
    end
  end
end


-- protection is a bolt you shoot on allies to protect them
local function shoot_protection(self)
  local nearby_group_members = {}
	for _,obj in ipairs(self.nearby_objects) do
    local mob = obj:get_luaentity()
    if mob and
       mob.leader_id == self.leader_id and
       mob.mg_id ~= self.mg_id
    then
      local p = mg_protection.get_protection_data(obj)
      -- not already protected
      if p.duration == 0 then
        table.insert(nearby_group_members, mob)
      end
    end
  end

  if #nearby_group_members == 0 then
    return
  end

  shuffle_list_in_place(nearby_group_members)

  local middle_pos = mobkit.get_middle_pos(self)
  local target = nil

	for _, mob in ipairs(nearby_group_members) do
    local target_middle_pos = mobkit.get_middle_pos(mob)
    local can_see = has_line_of_sight(middle_pos, target_middle_pos)
    if can_see then
      target = mob
      break
    end
  end

  if not target then
    return
  end

  local target_middle_pos = mobkit.get_middle_pos(target)
  local vel = target.object:get_velocity()
  local future_target_pos = aim_point(target_middle_pos, vel, middle_pos, 10)

  if future_target_pos then
    mg_bolts.mob_shoot("mg_bolts:protection", self.object, future_target_pos, 1, 0)
    return true
  end
end

-- webs shoot at players
local function shoot_web(self)
  local nearby_players = {}
	for _,obj in ipairs(self.nearby_objects) do
    if obj:is_player() and mobkit.is_alive(obj) then
      table.insert(nearby_players, obj)
    end
  end

  if #nearby_players == 0 then
    return
  end

  shuffle_list_in_place(nearby_players)

  local middle_pos = mobkit.get_middle_pos(self)
  local target = nil

	for _, player in ipairs(nearby_players) do
    local target_middle_pos = mobkit.get_middle_pos(player)
    local can_see = has_line_of_sight(middle_pos, target_middle_pos)
    if can_see then
      target = player
      break
    end
  end

  if not target then
    return
  end

  local target_middle_pos = mobkit.get_middle_pos(target)
  local vel = target:get_velocity()
  local future_target_pos = aim_point(target_middle_pos, vel, middle_pos, 10)

  if future_target_pos then
    mg_bolts.mob_shoot("mg_bolts:web", self.object, future_target_pos, 1, 0)
    return true
  end
end

-- TODO - this can likely be distilled into two routines
-- ones for shooting allies and another for shooting enemies
-- arrows shoot at players
local function shoot_arrow(self)
  local nearby_players = {}
	for _,obj in ipairs(self.nearby_objects) do
    if obj:is_player() and mobkit.is_alive(obj) then
      table.insert(nearby_players, obj)
    end
  end

  if #nearby_players == 0 then
    return
  end

  shuffle_list_in_place(nearby_players)

  local middle_pos = mobkit.get_middle_pos(self)
  local target = nil

	for _, player in ipairs(nearby_players) do
    local target_middle_pos = mobkit.get_middle_pos(player)
    local can_see = has_line_of_sight(middle_pos, target_middle_pos)
    if can_see then
      target = player
      break
    end
  end

  if not target then
    return
  end

  local target_middle_pos = mobkit.get_middle_pos(target)
  local vel = target:get_velocity()
  local future_target_pos = aim_point(target_middle_pos, vel, middle_pos, 20)
  future_target_pos.y = future_target_pos.y + 1

  if future_target_pos then
    bows.mob_shoot("bows:arrow", self.object, future_target_pos)
    return true
  end
end


local function shoot_bolts(self)
  if not self.bolts or #self.bolts == 0 then
    return
  end

  local i = math.random(1, #self.bolts)
  local bolt = self.bolts[i]

  if bolt == 'protection' then
    return shoot_protection(self)
  elseif bolt == 'spiderweb' then
    return shoot_web(self)
  elseif bolt == 'arrow' then
    shoot_arrow(self)
  end

end


return {
  shoot_bolts = shoot_bolts
}
