local function get_charm_enchant(itemstack)
  local def = itemstack:get_definition()
  local meta = itemstack:get_meta()
  local enchant = meta:get_int('enchant')
  if enchant <= 0 then
    return def._enchant or 0
  end
  return enchant
end

local function set_desc(stack, charge_level)
  local def =  stack:get_definition()
  local meta = stack:get_meta()
  local max_charges = meta:get_int("recharge_max_charges")
  local enchant = get_charm_enchant(stack)
  local enchant_desc = '+'..enchant..' '
  local name = stack:get_name()
  local is_wand = string.find(name, 'wand')
  local is_charm = string.find(name, 'charm')

  local new_desc = enchant_desc .. def.description ..
    " (".. charge_level  ..
    " of ".. max_charges .." charges)"

  if is_wand then
    new_desc = def.description .. " (".. charge_level .. " charges)"
  end
  if is_charm then
    local charge_desc = " (ready)"
    if charge_level < 1 then
      charge_desc = " (recharging)"
    end
    new_desc = enchant_desc .. def.description .. charge_desc
  end
  meta:set_string("description",  new_desc)
end

local function charge_one(player, inv, i, stack)
  local player_name = player:get_player_name()
  local meta = stack:get_meta()
  local wear = stack:get_wear() or 0
  local max_charges = meta:get_int("recharge_max_charges")

  -- charge the item once
  local new_wear = wear - 1
  if new_wear >= 0 then
    stack:set_wear(new_wear)
    set_desc(stack, max_charges - new_wear)
  end

  local desc = meta:get_string("description")

  -- if not fully charged then restart the charge
  if new_wear > 0 then
    local time = core.get_gametime()
    meta:set_int("recharge_start", time)
    core.chat_send_player(player_name, desc .. ' gained a charge.')
  -- if full charge then set recharge start time to 0
  elseif new_wear == 0 then
    meta:set_int("recharge_start", 0)
    core.chat_send_player(player_name,  desc .. ' is fully charged at ' .. max_charges .. ' charges.')
  end
  inv:set_stack("main", i, stack)
end

-- Apply the effects of a staff of wisdom
-- for now just do this every second
-- if this is too much, then I'll need to do it
-- on use of the staff, on equip or uneqip of the ring
local function calc_staff_duration(player, item)
  local def = item:get_definition()
  local base_duration = def._base_duration or 500
  local enchant = get_charm_enchant(item)
  local wisdom_rate = mg_rings.get_player_wisdom_rate(player) or 0
  local duration = base_duration / enchant / (1.3^wisdom_rate)
  return duration
end

local cb_timer = 0
core.register_globalstep(function(dtime)
  cb_timer = cb_timer + dtime
  -- check every second
  if cb_timer < 1 then
    return
  end
  cb_timer = 0

  local now = core.get_gametime()

	for _, player in pairs(core.get_connected_players()) do
    local inv = player:get_inventory()
    local size = inv:get_size("main")
    for i=1, size do
      local stack = inv:get_stack("main", i)
      local name = stack:get_name()
      local meta = stack:get_meta()
      local start = meta:get_int("recharge_start")
      local duration = meta:get_int("recharge_duration")

      if start > 0 and string.find(name, 'staff') then
        duration = calc_staff_duration(player, stack)
      end

      local time_left = duration - (now - start)
      local wear = stack:get_wear() or 0

      if start > 0 and time_left <= 0 and wear > 0 then
        charge_one(player, inv, i, stack)
      end
    end
  end
end)

local function apply_player_reaping(player, dmg)
  if not player:is_player() then
    return
  end

  local reap_rate = mg_rings.get_player_reaping_rate(player, dmg)

  print('speed up recharge by ' .. reap_rate .. ' seconds.')

  if reap_rate == 0 then
    return
  end

  local inv = player:get_inventory()
  local size = inv:get_size("main")
  for i=1, size do
    local stack = inv:get_stack("main", i)
    local wear = stack:get_wear() or 0
    local meta = stack:get_meta()
    local start = meta:get_int("recharge_start")
    if start > 0 and wear > 0 then
      print('add to recharge start to speed up')
      meta:set_int("recharge_start", start + reap_rate)
    end
  end
end


local function init_recharge_if_none(itemstack, duration)
  local meta = itemstack:get_meta()
  if meta:get_int("recharge_start") <= 0 then
    local time = core.get_gametime()
    meta:set_int("recharge_start", time)
    meta:set_int("recharge_duration", duration)
    return itemstack
  end
  return itemstack
end

local function use_charge(
  max_charges,
  recharge_duration,
  itemstack,
  user,
  cb
)
  local wear = itemstack:get_wear()
  local meta = itemstack:get_meta()
  local name = itemstack:get_name()

  local is_wand = string.find(name, 'wand')

  if wear < max_charges then
    local new_wear = wear + 1
    itemstack:set_wear(new_wear)
    set_desc(itemstack, max_charges - new_wear)
    cb()
  elseif is_wand then
    local username = user:get_player_name()
    local desc = meta:get_string("description")
    minetest.chat_send_player(username, desc .. ' fizzles.  It is out of charges.')
  else
    local now = core.get_gametime()
    local start = meta:get_int("recharge_start")
    local time_left = math.ceil(recharge_duration) - (now - start)
    --TODO if timeleft is negative, then need to charge the item and use it
    -- chat message that has seconds left to recharge
    local username = user:get_player_name()
    local desc = meta:get_string("description")
    minetest.chat_send_player(username, desc .. ' is out of charges. It will recharge in ' .. time_left .. ' seconds')
  end

  -- wands don't recharge
  if is_wand then
    meta:set_int("recharge_start", 0)
    return itemstack
  end

  return init_recharge_if_none(itemstack, recharge_duration)
end

local function recharge_item(stack)
  local meta = stack:get_meta()
  local new_wear = 0
  local max_charges = meta:get_int("recharge_max_charges")

  stack:set_wear(new_wear)
  meta:set_int("recharge_start", 0)
  set_desc(stack, max_charges)
  return stack
end

local function recharge_items(user, category)
  local username = user:get_player_name()
  local inv =  user:get_inventory()
  local size = inv:get_size("main")
  for i=1, size do
    local stack = inv:get_stack("main", i)
    local meta = stack:get_meta()
    local name = stack:get_name()
    local def = stack:get_definition()
    if string.find(name, category) then
      local new_wear = 0
      local max_charges = meta:get_int("recharge_max_charges")
      stack:set_wear(new_wear)
      set_desc(stack, max_charges - new_wear)
      meta:set_int("recharge_start", 0)
      minetest.chat_send_player(username, def.description .. ' is fully charged at ' .. max_charges .. ' charges.')
      inv:set_stack("main", i, stack)
    end
  end
end

local function recharge_staves(user)
  -- This will get a quarter staff too :-(
  -- should be mg_bolts:staff_
  recharge_items(user, 'staff')
end

local function recharge_charms(user)
  -- should be mg_bolts:charm_
  -- but then I'd need to rename everything...
  -- I'll need to do wands too.. boo hoo
  recharge_items(user, 'charm')
end

return {
  get_charm_enchant = get_charm_enchant,
  use_charge = use_charge,
  recharge_item = recharge_item,
  recharge_items = recharge_items,
  recharge_staves = recharge_staves,
  recharge_charms = recharge_charms,
  apply_player_reaping = apply_player_reaping,
  calc_staff_duration = calc_staff_duration,
  set_desc = set_desc
}
