-- protection effect for player and mobs
-- protection has a timer
-- when the timer is up, then the protection effect is done
-- protection itself is a number, it absorbs hits
-- when it absorbs all the hits, then the entity or player
-- takes damage
--
-- I want to eventually abstract effects
-- but need a few more one offs to get a feel for the pattern

local effect_storage = {}


local default_data = {
  cool_down = 0,
  duration = 0,
  total_timer = 0,
  range = 0,
}


local function get_effect_data(target)
  if target:is_player() then
    local name = target:get_player_name()
    return effect_storage[name] or {
      cool_down = 0,
      duration = 0,
      total_timer = 0,
      range = 0,
    }
  end

  local mob = target:get_luaentity()
  return mobkit.recall(mob, 'darkness') or {
    cool_down = 0,
    duration = 0,
    total_timer = 0,
    range = 0,
  }
end

local function set_effect_data(target, data)
  if target:is_player() then
    local name = target:get_player_name()
    effect_storage[name] = data
    return
  end

  local mob = target:get_luaentity()
  return mobkit.remember(mob, 'darkness', data)
end

local function start_effect(target, duration, range)
  local data = get_effect_data(target)

  data.range = range
  data.duration = duration
  data.total_timer = 0
  data.cool_down = 0

  set_effect_data(target, data)

  if target:is_player() then
    target:set_sky({fog = {
      fog_distance = data.total_timer / data.duration * (data.range or 40),
      fog_start = 0.75,
      fog_color = "#000000"
    }})
  end
end

local function clear_effect(target)
  local data = get_effect_data(target)
  data.range = 0
  data.cool_down = 0
  data.duration = 0
  data.total_timer = 0
  set_effect_data(target, data)
  if target:is_player() then
    -- set back to client control
    target:set_sky({fog = {
      fog_distance = -1,
      fog_start = -1,
      fog_color = "#00000000"
    }})
  end
end

local function effect_on_step(target, dt)
  local data = get_effect_data(target)

  -- means effect is not being applied
  if(not data.duration or data.duration == 0) then
    return
  end

  data.total_timer = data.total_timer + dt
  data.cool_down = data.cool_down + dt

  if data.cool_down > 0.25 then
    data.cool_down = 0
    if target:is_player() then
      target:set_sky({fog = {
        fog_distance = 2 + data.total_timer / data.duration * (data.range or 40),
        fog_start = 0.75,
        fog_color = "#000000"
      }})
    end
  end


  if data.total_timer > data.duration then
    clear_effect(target)
  end

  set_effect_data(target, data)
end


core.register_globalstep(function(dt)
	for _, player in pairs(core.get_connected_players()) do
    effect_on_step(player, dt)
  end
end)

core.register_on_joinplayer(function(player)
  local name = player:get_player_name()
	local storage = table.copy(default_data)
	local effect_data = player:get_meta():get_string("mg_darkness:data")
	if effect_data ~= "" then
		storage = minetest.deserialize(effect_data) or storage
	end
	effect_storage[name] = storage
end)

local function on_leaveplayer(player)
  local name = player:get_player_name()
	local storage = effect_storage[name]
	if not storage then
		storage = {}
	end
	player:get_meta():set_string("mg_darkness:data", minetest.serialize(storage))
	effect_storage[name] = nil
end

core.register_on_leaveplayer(function(player)
	on_leaveplayer(player)
end)

core.register_on_shutdown(function()
	for _,player in ipairs(minetest.get_connected_players()) do
		on_leaveplayer(player)
	end
end)

minetest.register_on_respawnplayer(function(player)
  clear_effect(player)
end)

local effect_api = {
  start_darkness = start_effect,
  clear_darkness = clear_effect,
}

return effect_api
