
local function map(tbl, f)
  local t = {}
  for k,v in pairs(tbl) do
    t[k] = f(v)
  end
  return t
end

local function filter(tbl, f)
  local t = {}
  for _,v in pairs(tbl) do
    if f(v) then
      table.insert(t, v)
    end
  end
  return t
end

local function some(tbl, f)
  for _,v in pairs(tbl) do
    if f(v) then
      return true
    end
  end
  return false
end

local FP_FACTOR = 65536
local allocGrid = mg_arch.allocGrid
local fillGrid = mg_arch.fillGrid
local DROWS = mg_arch.DROWS
local DCOLS = mg_arch.DCOLS
local AMULET_LEVEL = mg_arch.AMULET_LEVEL
local rand_percent = mg_arch.rand_percent
local rand_range = mg_arch.rand_range
local passableArcCount = mg_arch.passableArcCount

local min = mg_arch.min
local tt = mg_arch.tileType
local nbDirs = mg_arch.nbDirs
local coordinatesAreInMap = mg_arch.coordinatesAreInMap
local cellHasTerrainFlag = mg_arch.cellHasTerrainFlag
local tf = mg_arch.terrainFlags
local bor = mg_arch.bit.bor


-- length = DEEPEST LEVEL - AMULET LEVEL
local lumenstoneDistribution = {3, 3, 3, 2, 2, 2, 2, 2, 1, 1, 1, 1, 1, 1};

local function generateWorldData()
  return {
    lifePotionFrequency = 0,
    strengthPotionFrequency = 40,
    enchantScrollFrequency = 60,
    goldGenerated = 0,
    foodSpawned = 0,
    lifePotionsSpawned = 0
  }
end

local categories = {};

local function noop() end

local function registerItemCategory (name, data)

  -- let mods override the categories
  for _, cat in ipairs(categories) do
    if cat.name == name then
      cat.prob = data.prob or cat.prob or 0
      cat.on_item_spawn = data.on_item_spawn or cat.on_item_spawn or noop
      return
    end
  end

  -- if not exist, insert in the table
  table.insert(categories, {
    name = name,
    prob = data.prob,
    on_item_spawn = data.on_item_spawn
  })
end

local function getItemCategory(name)
  for _, cat in ipairs(categories) do
    if cat.name == name then
      return cat
    end
  end
end


local function getCategoryNames ()
  return map(categories, function(c) return c.name end)
end

local itemTable = {}

local function registerMgItem(
  name, node_name, category, frequency, marketValue, strengthRequired, range,
  identified, called, magicPolarity, magicPolarityRevealed, description
)
  local item = {
    name = name,
    node_name = node_name,
    category = category,
    frequency = frequency,
    marketValue = marketValue,
    strengthRequired = strengthRequired,
    range = range,
    identified = identified,
    called = called,
    magicPolarity = magicPolarity,
    magicPolarityRevealed = magicPolarityRevealed,
    description = description
  }

  table.insert(itemTable, item)
  return item
end

local function getItem(kind, category)
  for _, item in ipairs(itemTable) do
    if item.name == kind and item.category == category then
      return item
    end
  end
end


local function allocItem()
  return {
    category = "",
    kind = "",
    node_name = "",
    flags = {},
    armor = 0,
    strengthRequired = 0,
    enchant1 = 0,
    enchant2 = 0,
    timesEnchanted = 0,
    vorpalEnemy = 0,
    charges = 0,
    quantity = 1,
    quiverNumber = 0,
    originDepth = 0,
    inscription = "",
    lastUsed1 = 0,
    lastUsed2 = 0,
    lastUsed3 = 0,
  }
end

local POW_GOLD = {
  -- b^3.05, with b from 0 to 25:
  0, 1, 8, 28, 68, 135, 236, 378, 568, 813, 1122, 1500, 1956, 2497, 3131,
  3864, 4705, 5660, 6738, 7946, 9292, 10783, 12427, 14232, 16204, 18353
};

local POW_FOOD = {
  -- b^1.35 fixed point, with b from 1 to 50 (for future-proofing):
  65536, 167059, 288797, 425854, 575558, 736180, 906488, 1085553, 1272645,
  1467168, 1668630, 1876612, 2090756, 2310749, 2536314, 2767208, 3003211,
  3244126, 3489773, 3739989, 3994624, 4253540, 4516609, 4783712, 5054741,
  5329591, 5608167, 5890379, 6176141, 6465373, 6758000, 7053950, 7353155,
  7655551, 7961076, 8269672, 8581283, 8895856, 9213341, 9533687, 9856849,
  10182782, 10511443, 10842789, 11176783, 11513384, 11852556, 12194264,
  12538472, 12885148
};


local function pickItemCategory(theCategories)
  local sum = 0

  local filteredCatProb = filter(categories,
    function(c)
      return some(theCategories,
        function (tc) return tc == c.name end
      )
    end
  )

  for _,cat in pairs(filteredCatProb) do
    sum = sum + cat.prob
  end

  if sum == 0 then
    return theCategories[1]
  end
  local randIndex = rand_range(1, sum)

  for _,cat in pairs(filteredCatProb) do
    if randIndex <= cat.prob then
      return cat.name
    end
    randIndex = randIndex - cat.prob
  end
  return theCategories[0]
end

local function chooseKind(items, category)
  local tbl = filter(items, function(i) return i.category == category end)
  local totalFrequencies = 0
  for _,item in pairs(tbl) do
    totalFrequencies = totalFrequencies + math.max(0, item.frequency)
  end
  local randomFrequency = rand_range(1, totalFrequencies)

  for _,item in pairs(tbl) do
    if randomFrequency <= item.frequency then
      return item
    end
    randomFrequency = randomFrequency - math.max(0, item.frequency)
  end
end



local function generateItem(depth, worldData, theCategory, theKind)
  local itemDef
  if #theCategory == 0 then
    theCategory = getCategoryNames()
  end
  local c = pickItemCategory(theCategory)
  local item = allocItem()
  item.kind = theKind
  item.category = c

  -- enchanting the item is done in mg_dungeon_registration now
  if c == "scroll" or
     c == "food" or
     c == "armor" or
     c == "weapon" or
     c == "ring" or
     c == "wand" or
     c == "staff" or
     c == "charm" or
     c == 'potion' then
    if theKind == '' then
      itemDef = chooseKind(itemTable, c)
      item.kind = itemDef.name
      item.node_name = itemDef.node_name
    else
     itemDef = getItem(theKind, c)
     item.node_name = itemDef.node_name
    end
  elseif c == "gold" then
    item.kind = 'gold'
    item.quantity = rand_range(50 + depth*10, 100+depth*15)
  elseif c == "amulet" then
    item.kind = 'amulet'
    item.category = 'amulet'
  elseif c == "gem" then
    item.category = 'gem'
    item.kind = 'gem'
  end

  return item
end


local function aggregateGoldLowerBound(d)
  return (POW_GOLD[d] + 320 * (d))
end

local function aggregateGoldUpperBound(d)
  return (POW_GOLD[d] + 420 * (d))
end

-- I want to redo the flag system
-- to match what minetest does with groups
local function hasDeepLakeOrStairsFlags(pmap, x, y)
  return pmap[x][y].layers.dungeon == tt.UP_STAIRS or
         pmap[x][y].layers.dungeon == tt.DOWN_ARROW or
         pmap[x][y].layers.liquid == tt.DEEP_WATER or
         pmap[x][y].layers.liquid == tt.LAVA or
         pmap[x][y].layers.liquid == tt.CHASM
end

local function isPassableOrSecretDoor(pmap, x, y)
  return cellHasTerrainFlag(pmap, x, y, tf.T_OBSTRUCTS_PASSABILITY) == false or
    -- So is a secret door -- I don't have cellHasTMFlag yet
    pmap[x][y].layers.dungeon == tt.SECRET_DOOR
          -- TODO add secret door code (no secret doors yet)
          -- not sure what do do with the below
          --|| (cellHasTMFlag(x, y, TM_IS_SECRET) && !(discoveredTerrainFlagsAtLoc(x, y) & T_OBSTRUCTS_PASSABILITY)));
end


local function fillItemSpawnHeatMap(pmap, heatMap, heatLevel, x, y)
  if pmap[x][y].layers.dungeon == tt.DOOR then
    heatLevel = heatLevel + 10;
  elseif (pmap[x][y].layers.dungeon == tt.SECRET_DOOR) then
    heatLevel = heatLevel + 3000;
  end

  if (heatMap[x][y] > heatLevel) then
    heatMap[x][y] = heatLevel
  end

  for dir = 1, 4 do
    local newX = x + nbDirs[dir][1]
    local newY = y + nbDirs[dir][2]
    if coordinatesAreInMap(newX, newY)
      -- i'll just see if it's lava, water, chasm, or stairs
      --and false == cellHasTerrainFlag(newX, newY, T_IS_DEEP_WATER | T_LAVA_INSTA_DEATH | T_AUTO_DESCENT)
      and false == hasDeepLakeOrStairsFlags(pmap, newX, newY)
      and isPassableOrSecretDoor(pmap, newX, newY)
      and heatLevel < heatMap[newX][newY] then
        fillItemSpawnHeatMap(pmap, heatMap, heatLevel, newX, newY);
    end
  end
end


local function getItemSpawnLoc(heatMap, totalHeat)
  if totalHeat < 0 then
    return {
      x = 0,
      y = 0,
      success = false
    }
  end

  local randIndex = rand_range(1, totalHeat)
  for j=1, DROWS do
    for i=1, DCOLS do
      local currentHeat =  heatMap[i][j]
      if randIndex <= currentHeat then
        return {
          x = i,
          y = j,
          success = true
        }
      end
      randIndex = randIndex - currentHeat;
    end
  end

  return {
    x = 0,
    y = 0,
    success = false
  }
end

local function coolHeatMapAt(heatMap, x, y, totalHeat)
  -- For some reason I got a nil error here one time
  if coordinatesAreInMap(x, y) == false then
    return totalHeat
  end

  local currentHeat = heatMap[x][y]
  if currentHeat == 0 then
    return totalHeat
  end
  totalHeat = totalHeat - currentHeat
  heatMap[x][y] = 0

  for k=-5, 5 do
    for l=-5, 5 do
      if coordinatesAreInMap(x+k, y+l) and heatMap[x+k][y+l] == currentHeat then
        local coolBy = math.max(1, math.floor(heatMap[x+k][y+l]/10))
        heatMap[x+k][y+l] = coolBy
        totalHeat = totalHeat - (currentHeat - coolBy)
      end
    end
  end
  return totalHeat
end

local function populateItems(pmap, worldData, depth, upstairsX, upstairsY, generate_amulet)
  local itemSpawnHeatMap = allocGrid()
  local numberOfItems
  local numberOfGoldPiles

  if depth > AMULET_LEVEL then
    numberOfItems = lumenstoneDistribution[depth - AMULET_LEVEL - 1]
    numberOfGoldPiles = 0
    --TODO - I think there's more to be done here...
  else
    worldData.lifePotionFrequency = worldData.lifePotionFrequency + 34
    worldData.strengthPotionFrequency = worldData.strengthPotionFrequency + 17
    worldData.enchantScrollFrequency = worldData.enchantScrollFrequency + 30

    local enchantScroll = getItem('enchanting', 'scroll')
    local strengthPotion = getItem('strength', 'potion')
    local lifePotion = getItem('life', 'potion')

    enchantScroll.frequency = worldData.enchantScrollFrequency
    lifePotion.frequency = worldData.lifePotionFrequency
    strengthPotion.frequency = worldData.strengthPotionFrequency

    numberOfItems = 3

    while rand_percent(60) do
      numberOfItems = numberOfItems + 1
    end

    if depth <= 2 then
      numberOfItems = numberOfItems + 2
    elseif depth <= 4 then
      numberOfItems = numberOfItems + 1
    end

    print('generate ' .. numberOfItems .. ' items')

    numberOfGoldPiles = min(5, depth / 4)

    local goldBonusProbability = 60
    while rand_percent(goldBonusProbability) and numberOfGoldPiles <= 10 do
      numberOfGoldPiles = numberOfGoldPiles + 1
      goldBonusProbability = goldBonusProbability - 15
    end

    if depth > 5 then
      if worldData.goldGenerated < aggregateGoldLowerBound(depth - 1) then
        numberOfGoldPiles = numberOfGoldPiles + 2
      elseif worldData.goldGenerated > aggregateGoldUpperBound(depth + 1) then
        numberOfGoldPiles = numberOfGoldPiles - 2
      end
    end

    fillGrid(itemSpawnHeatMap, 50000)
    fillItemSpawnHeatMap(pmap, itemSpawnHeatMap, 5, upstairsX, upstairsY)

    -- initial heatmap

    local totalHeat = 0

    for j=1, DROWS do
      for i=1, DCOLS do
        if cellHasTerrainFlag(pmap, i, j, bor(tf.T_OBSTRUCTS_ITEMS, tf.T_PATHING_BLOCKER))
          or pmap[i][j].flags.inLoop > 0
          or pmap[i][j].flags.isChokepoint > 0
          -- TODO pmap[i][j].flags.isInMachine > 0 ||
          or passableArcCount(pmap, i, j) > 1
        then
          itemSpawnHeatMap[i][j] = 0
        elseif itemSpawnHeatMap[i][j] == 50000 then
          itemSpawnHeatMap[i][j] = 0
        end
        totalHeat = totalHeat + itemSpawnHeatMap[i][j]
      end
    end

    --print('item heat map')
    --mg_arch.printGrid(itemSpawnHeatMap, true)

    local randomDepthOffset = 0
    if depth > 2 then
      randomDepthOffset = rand_range(-1, 1) + rand_range(-1,1)
    end

    local filteredCat = filter(getCategoryNames(), function(c) return c ~= 'gold'end)

    local itemList = {}

    -- Spawn amulate first - I hope this gets it away from the ladder
    if generate_amulet then
      local theItem = generateItem(depth, worldData, {'amulet'}, '')
      local loc = getItemSpawnLoc(itemSpawnHeatMap, totalHeat)
      theItem.originDepth = depth
      theItem.x = loc.x
      theItem.y = loc.y
      table.insert(itemList, theItem)
      print('amulet at x: '..loc.x..', y: '.. loc.y)
    end

    local foodItem = getItem('ration of food', 'food')

    for _=1, numberOfItems do
      local theCategory = filteredCat;
      local theKind = ''
      if (worldData.foodSpawned + foodItem.strengthRequired / 3) * 4 * FP_FACTOR
         <= (POW_FOOD[depth] + randomDepthOffset * FP_FACTOR) * foodItem.strengthRequired * 45/100
      then
        theCategory = {'food'}
        if depth > AMULET_LEVEL then
          numberOfItems = numberOfItems + 1
        end
      elseif depth > AMULET_LEVEL then
        theCategory = {'gem'}
      elseif worldData.lifePotionsSpawned * 4 + 3 < depth + randomDepthOffset then
        theCategory = {'potion'}
        theKind = "life"
      end
      local theItem = generateItem(depth, worldData, theCategory, theKind)
      theItem.originDepth = depth
      if theItem.category == 'food' then
        local item = getItem(theItem.kind, 'food')
        worldData.foodSpawned = worldData.foodSpawned + item.strengthRequired
      end

      local x
      local y
      if theItem.category == 'food'
         or (theItem.category == 'potion' and theItem.kind == 'strength')
      then
        local loc
        repeat
          loc = mg_arch.randomMatchingLocation(pmap, tt.FLOOR, tt.NOTHING, -1)
          x = loc.x
          y = loc.y
        until passableArcCount(pmap, loc.x, loc.y) <= 1 -- not in a hallway
      else
        local loc = getItemSpawnLoc(itemSpawnHeatMap, totalHeat)
        x = loc.x
        y = loc.y
      end

      local sign = ((theItem.enchant1 or 0) >= 0 and '+') or ''
      local quantity = theItem.quantity or 1

      print('generated ' .. quantity .. ' ' .. sign .. theItem.enchant1 .. ' ' .. theItem.category .. ' of ' .. theItem.kind .. ' at '..x..', '.. y..'. totalheat: '..totalHeat)
      totalHeat = coolHeatMapAt(itemSpawnHeatMap, x, y, totalHeat)

      if theItem.category == 'scroll' and theItem.kind == "enchanting" then
        worldData.enchantScrollFrequency = worldData.enchantScrollFrequency - 50
      elseif theItem.category == 'potion' and theItem.kind == "life" then
        worldData.lifePotionFrequency = worldData.lifePotionFrequency - 150
        worldData.lifePotionsSpawned = worldData.lifePotionsSpawned + 1
      elseif theItem.category == 'potion' and theItem.kind == "strength" then
        worldData.strengthPotionFrequency = worldData.strengthPotionFrequency - 50
      end

      theItem.x = x
      theItem.y = y
      if theItem.x > 0 and theItem.y > 0 then
        table.insert(itemList, theItem)
      end
    end

    for _=1, numberOfGoldPiles do
      local theItem = generateItem(depth, worldData, {'gold'}, '')
      local loc = getItemSpawnLoc(itemSpawnHeatMap, totalHeat)
      totalHeat = coolHeatMapAt(itemSpawnHeatMap, loc.x, loc.y, totalHeat)
      worldData.goldGenerated = worldData.goldGenerated + theItem.quantity
      theItem.originDepth = depth
      theItem.x = loc.x
      theItem.y = loc.y

      table.insert(itemList, theItem)
      print('gold at x: '..loc.x..', y: '.. loc.y..' gold: '..theItem.quantity)
    end


    return itemList
  end
end


-- generate some comsumables for the top
-- TODO probably just add some flags to the items table
-- instead of having this new table...
local kindsForTop = {
  {kind = 'incendiary dart', cat = 'weapon'},
  {kind = 'recharging', cat = 'scroll'},
  {kind = 'shattering', cat = 'scroll'},
  {kind = 'speed', cat = 'potion'},
  {kind = 'levitation', cat = 'potion'},
  {kind = 'incineration', cat = 'potion'},
  {kind = 'descent', cat = 'potion'},
  {kind = 'creeping death', cat = 'potion'}
}

local function generateTopItem()
  local item = allocItem()
  local idx = rand_range(1, #kindsForTop)
  local topItem = kindsForTop[idx]
  item.kind = topItem.kind
  item.category = topItem.cat
  local itemDef = getItem(item.kind, item.category)
  if itemDef then
    item.node_name = itemDef.node_name
  end
  return item
end




local function populateTopItems(pmap, worldData, depth)
  local stairs = mg_arch.randomMatchingLocation(pmap, tt.FLOOR, tt.NOTHING, -1)
  local itemSpawnHeatMap = allocGrid()
  local numberOfItems

    numberOfItems = rand_range(3,4)
    local numberOfGoldPiles = rand_range(1,2)

    fillGrid(itemSpawnHeatMap, 50000)
    fillItemSpawnHeatMap(pmap, itemSpawnHeatMap, 5, stairs.x, stairs.y)

    -- initial heatmap

    local totalHeat = 0

    for j=1, DROWS do
      for i=1, DCOLS do
        if cellHasTerrainFlag(pmap, i, j, bor(tf.T_OBSTRUCTS_ITEMS, tf.T_PATHING_BLOCKER))
          or pmap[i][j].flags.inLoop > 0
          or pmap[i][j].flags.isChokepoint > 0
          or passableArcCount(pmap, i, j) > 1
        then
          itemSpawnHeatMap[i][j] = 0
        elseif itemSpawnHeatMap[i][j] == 50000 then
          itemSpawnHeatMap[i][j] = 0
        end
        totalHeat = totalHeat + itemSpawnHeatMap[i][j]
      end
    end

    --print('item heat map')
    --mg_arch.printGrid(itemSpawnHeatMap, true)

    local itemList = {}

    for _=1, numberOfItems do
      local theItem = generateTopItem()
      local loc = getItemSpawnLoc(itemSpawnHeatMap, totalHeat)

      theItem.x = loc.x
      theItem.y = loc.y

      print('top item: generated a ' .. theItem.category .. ' of ' .. theItem.kind .. ' at '..loc.x..', '.. loc.y..'. totalheat: '..totalHeat)
      totalHeat = coolHeatMapAt(itemSpawnHeatMap, loc.x, loc.y, totalHeat)

      if theItem.x > 0 and theItem.y > 0 then
        table.insert(itemList, theItem)
      end
    end

    for _=1, numberOfGoldPiles do
      local theItem = generateItem(depth, worldData, {'gold'}, '')
      local loc = getItemSpawnLoc(itemSpawnHeatMap, totalHeat)
      totalHeat = coolHeatMapAt(itemSpawnHeatMap, loc.x, loc.y, totalHeat)
      worldData.goldGenerated = worldData.goldGenerated + theItem.quantity
      theItem.originDepth = depth
      theItem.x = loc.x
      theItem.y = loc.y

      table.insert(itemList, theItem)
      print('gold at x: '..loc.x..', y: '.. loc.y..' gold: '..theItem.quantity)
    end


    return itemList
end


mg_arch.generateWorldData = generateWorldData
mg_arch.populateItems = populateItems
mg_arch.populateTopItems = populateTopItems

mg_arch.item_api = {}
mg_arch.item_api.registerItemCategory = registerItemCategory
mg_arch.item_api.registerMgItem = registerMgItem
mg_arch.item_api.getItemCategory = getItemCategory
