-- armor is a global variable referencing the 3darmor mod

local get_equiped_armor_element = function( player, match_element)
  local name, inv = armor:get_valid_player(player, "[get_weared_armor]")
  if not name then
    return nil
  end
  for i=1, inv:get_size("armor") do
    local item_name = inv:get_stack("armor", i):get_name()
    local element = armor:get_element(item_name)

    if element == match_element then
      return inv:get_stack("armor", i)
    end
  end
  return nil
end

local function replace_armor (player, itemstack)
  local self = armor
  local name, armor_inv = self:get_valid_player(player, "[equip]")
  local armor_element = self:get_element(itemstack:get_name())
  if name and armor_element then
    local index
    for i=1, armor_inv:get_size("armor") do
      local stack = armor_inv:get_stack("armor", i)
      if self:get_element(stack:get_name()) == armor_element then
        index = i
        break
      elseif not index and stack:is_empty() then
        index = i
      end
    end
    local stack = itemstack:take_item()
    armor_inv:set_stack("armor", index, stack)
    self:run_callbacks("on_equip", player, index, stack)
    self:set_player_armor(player)
    self:save_armor_inventory(player)
  end
  return itemstack
end

local function calc_net_protection(
  item_protection,
  item_strength_required,
  item_enchantment,
  player_strength,
  player_weakness
  )
  local enchantment = mg_strength.netEnchant(item_enchantment, item_strength_required, player_strength, player_weakness)

  local net_protection = item_protection + enchantment * 10
  if net_protection < 0 then
    return 0
  end
  return net_protection
end

local function calc_defense_fraction(net_protection)
  return 0.877347^((net_protection)/10)
end

local function get_armor_prop(itemstack, prop, default)
  local def = itemstack:get_definition()
  local meta = itemstack:get_meta()
  local m = meta:get_int(prop)
  if m > 0 or m < 0 then
    return m
  end
  return def[prop] or default
end

local function get_armor_details(itemstack)
  local strength_required = get_armor_prop(itemstack, '_str_requirement', 10)
  local enchant = get_armor_prop(itemstack, '_enchant', 0)
  local player_enchant = get_armor_prop(itemstack, 'player_enchant', 0)
  -- protected from acid
  local protected = get_armor_prop(itemstack, 'protected', 0)
  -- based level of protection from hits
  local protection = get_armor_prop(itemstack, '_protection', 0)

  -- cursed, not or blessed
  local magic_polarity = get_armor_prop(itemstack, 'magic_polarity', 0)
  local magic_polarity_revealed = get_armor_prop(itemstack, "magic_polarity_revealed", 0)

  local identified = get_armor_prop(itemstack, 'identified', 0)

  local worn_duration = get_armor_prop(itemstack, 'worn_duration', 0)
  local worn_start = get_armor_prop(itemstack, 'worn_start', 0)

  return {
    strength_required = strength_required,
    enchant = enchant,
    protected = protected,
    protection = protection,
    magic_polarity = magic_polarity,
    magic_polarity_revealed = magic_polarity_revealed,
    identified = identified,
    player_enchant = player_enchant,
    worn_duration = worn_duration,
    worn_start = worn_start,
  }
end

local function update_desc(stack)
  local data = get_armor_details(stack)
  local meta = stack:get_meta()
  local def = stack:get_definition()

  local enchant = data.enchant
  local unknown = ''

  if data.identified == 0 then
    enchant = (data.player_enchant or 0)
    unknown = '?'
  end

  local magic_desc = ''
  if data.magic_polarity == -1 and (data.identified or data.magic_polarity_revealed) then
    magic_desc = 'cursed '
  end

  local sign = (enchant >= 0 and '+') or ''
  local show_protect = (data.protected > 0 and ' (protected)') or ''
  local description = sign..enchant..unknown..' '..magic_desc..def._base_description .. show_protect

  meta:set_string("description", description)
  return description
end

local function enchant_armor(stack, amount, from_player)
  local meta = stack:get_meta()
  amount = amount or 1

  local details = get_armor_details(stack)

  local strength_required = details.strength_required - amount
  if amount < 0 then
    strength_required = details.strength_required
  end
  local enchant = details.enchant + amount

  if from_player then
    local player_enchant = details.player_enchant + amount
    meta:set_int("player_enchant", player_enchant)
  end

  meta:set_int("_str_requirement", strength_required)
  meta:set_int("_enchant", enchant)

  update_desc(stack)

  return stack
end

local function degrade_armor(stack, amount)
  local meta = stack:get_meta()
  local details = get_armor_details(stack)

  if details.protected > 0 then
    return
  end

  amount = amount or 1
  local enchant = details.enchant - amount
  meta:set_int("_enchant", enchant)

  -- Always take away from player enchant
  local player_enchant = details.player_enchant - amount
  meta:set_int("player_enchant", player_enchant)

  local description = update_desc(stack)

  return stack, description
end

local function degrade_player_armor(player, amount)
  local torso_armor = get_equiped_armor_element(player, 'torso')
  if torso_armor then
    local stack, description = degrade_armor(torso_armor, amount)
    if stack then
      replace_armor(player, stack)

      local username = player:get_player_name()
      local message = "Your ".. description .." is degraded by 1."
      core.chat_send_player(username, message)
    end
  end
end

local function protect_armor(itemstack)
  local meta = itemstack:get_meta()
  meta:set_int("protected", 1)
  update_desc(itemstack)
  return itemstack
end

local player_ac_cache = {}

local function get_player_ac(player, as_identified)
  local torso_armor = get_equiped_armor_element(player, 'torso')
  local player_str = mg_strength.get_strength_raw(player)
  local player_weakness = mg_strength.get_weakness(player)
  local protection = 20
  local armor_strength_required = 10
  local enchant = 0
  if torso_armor then
    local details = get_armor_details(torso_armor)

    -- if dont want true stats and armor is not identified
    -- then only show based on player enchants
    if not as_identified and details.identified ~= 1 then
      details.strength_required = details.strength_required + details.enchant - details.player_enchant
      details.enchant = details.player_enchant
    end

    armor_strength_required = details.strength_required
    enchant = details.enchant
    protection = details.protection
  end

  local net_protection = calc_net_protection(
    protection,
    armor_strength_required,
    enchant,
    player_str,
    player_weakness
  )

  local username = player:get_player_name()
  player_ac_cache[username] = net_protection

  return net_protection
end

local function get_player_ac_cached (player_name)
  return player_ac_cache[player_name] or 0
end

local function register_armor(name, desc, str_requirement, protection, info, groups)
  info = info or "!!!NO TEXT FOUND!!!"

  groups = groups or {}
  groups.armor_torso = 1
  groups.armor = 1
  groups.upgradable = 1

  armor:register_armor("mg_armor:torso_"..name, {
    description = desc,
    inventory_image = "mg_armor_"..name.."_inv.png",

    -- Not relying on armor mod to apply resistance to damage
    armor_groups = {fleshy = 0},

    groups = groups,

    _info = info,
    _base_description = desc,
    _str_requirement = str_requirement,
    _protection = protection,
    _enchant = 0,

    -- TODO - armor should autoid if worn for more than 600 seconds...
    on_equip = function(player, _index, itemstack)
      local armor_penalty = str_requirement - 12
      if armor_penalty < 0 then
        armor_penalty = 0
      end

      local username = player:get_player_name()

      -- update stealth
      mg_stealth.set_stealth_prop(username, 'armor_penalty', armor_penalty)


      local ac = get_player_ac(player, false)
      local message = 'Your AC is ' .. ac .. '.'

      -- Message if armor needs more player strength
      local details = get_armor_details(itemstack)

      -- don't reveal true strength requirement if not identified
      if details.identified ~= 1 then
        details.strength_required = details.strength_required + details.enchant - details.player_enchant
        details.enchant = details.player_enchant
      end

      local strdiff = mg_strength.player_strength_diff(player, details.strength_required)
      if strdiff < 0 then
        message = message .. ' You stagger under the weight of your armor.  ' .. math.abs(strdiff) .. ' more strength would be ideal.'
      end

      core.chat_send_player(username, message)


      -- Check for total time worn to auto id
      local time = core.get_gametime()
      local meta = itemstack:get_meta()
      print('equip armor - set worn start to ' .. time)
      meta:set_int('worn_start', time)
    end,

    on_unequip = function(player, _index, itemstack)
      local username = player:get_player_name()
      mg_stealth.set_stealth_prop(username, 'armor_penalty', 0)
      get_player_ac(player, false)

      -- Check for total time worn to auto id
      local details = get_armor_details(itemstack)
      local time = core.get_gametime()
      local worn_duration = time - details.worn_start + details.worn_duration
      local meta = itemstack:get_meta()
      meta:set_int('worn_duration', worn_duration)
    end,
  })
end

local cb_timer = 0
core.register_globalstep(function(dtime)
  cb_timer = cb_timer + dtime
  if cb_timer < 10 then
    return
  end
  cb_timer = 0

  local now = core.get_gametime()

	for _, player in pairs(core.get_connected_players()) do
    -- Auto id armor
    local stack = get_equiped_armor_element(player, 'torso')
    if stack then
      local details = get_armor_details(stack)
      local worn_duration = now - details.worn_start + details.worn_duration
      if worn_duration >= 500 and details.identified == 0 then
        local playername = player:get_player_name()
        print('id armor for player ' .. playername)
        local meta = stack:get_meta()
        meta:set_int('identified', 1)
        local description = update_desc(stack)
        replace_armor(player, stack)
        core.chat_send_player(playername, "You are now more familiar with your " .. description)
      end
    end
  end
end)


-- used for combat, so want real value
local function get_player_defense_fraction(player)
  return calc_defense_fraction(get_player_ac(player, true))
end


register_armor("cloth", "Cloth Armor", 10, 30, "This lightweight armor offers basic protection.", {flammable = 1})
register_armor("leather", "Leather Armor", 12, 40, "Armor made from tanned monster hide. Not as light as cloth armor but provides better protection.", {flammable = 1})
register_armor("wood", "Wooden Armor", 13, 50, "Armor made from wood.", {flammable = 1})
register_armor("mail", "Mail Armor", 15, 70, "Interlocking metal links make for a tough but flexible suit of armor.")
register_armor("scale", "Scale Armor", 17, 90, "Metal scales are sewn onto a leather vest create a flexible, yet protective armor.")
register_armor("plate", "Plate Armor", 19, 110, "Enormous plates of metal are joined together into a suit that provides unmatched "..
"protection to any adventurer strong enough to bear its staggering weight.")


mg_armor = {}
mg_armor.enchant_armor = enchant_armor
mg_armor.get_player_ac = get_player_ac
mg_armor.get_player_ac_cached = get_player_ac_cached
mg_armor.get_player_defense_fraction = get_player_defense_fraction
mg_armor.calc_defense_fraction = calc_defense_fraction

mg_armor.degrade_player_armor = degrade_player_armor
mg_armor.protect_armor = protect_armor
mg_armor.update_desc = update_desc

mg_armor.get_equiped_armor_element = get_equiped_armor_element
mg_armor.get_armor_details = get_armor_details
mg_armor.replace_armor = replace_armor

