-- Need some bitwise operators
-- from https://github.com/AlberTajuelo/bitop-lua/blob/master/src/bitop/funcs.lua

if mg_arch == nil then
  mg_arch = {}
end

local MOD = 2^32
local MODM = MOD-1

local function memoize(f)

  local mt = {}
  local t = setmetatable({}, mt)

  function mt:__index(k)
    local v = f(k)
    t[k] = v
    return v
  end

  return t
end

local function make_bitop_uncached(t, m)
  local function bitop(a, b)
    local res,p = 0,1
    while a ~= 0 and b ~= 0 do
      local am, bm = a%m, b%m
      res = res + t[am][bm]*p
      a = (a - am) / m
      b = (b - bm) / m
      p = p*m
    end
    res = res + (a+b) * p
    return res
  end
  return bitop
end

local function make_bitop(t)
  local op1 = make_bitop_uncached(t, 2^1)
  local op2 = memoize(function(a)
    return memoize(function(b)
      return op1(a, b)
    end)
  end)
  return make_bitop_uncached(op2, 2^(t.n or 1))
end

function toBits(num)
    -- returns a table of bits, least significant first.
    local t={} -- will contain the bits
    while num>0 do
        rest=math.fmod(num,2)
        t[#t+1]=rest
        num=(num-rest)/2
    end
    return table.concat(t)
end

local bxor = make_bitop {[0]={[0]=0,[1]=1},[1]={[0]=1,[1]=0}, n=4}

local function lshift(a,disp)
  return (a * 2^disp) % 2^32
end
local function bnot(a)   return MODM - a end

local function band(a,b) return ((a+b) - bxor(a,b))/2 end

local function bor(a,b)  return MODM - band(MODM - a, MODM - b) end

local function bit32_bxor(a, b, c, ...)
  local z
  if b then
    a = a % MOD
    b = b % MOD
    z = bxor(a, b)
    if c then
      z = bit32_bxor(z, c, ...)
    end
    return z
  elseif a then
    return a % MOD
  else
    return 0
  end
end

local function bit32_band(a, b, c, ...)
  local z
  if b then
    a = a % MOD
    b = b % MOD
    z = ((a+b) - bxor(a,b)) / 2
    if c then
      z = bit32_band(z, c, ...)
    end
    return z
  elseif a then
    return a % MOD
  else
    return MODM
  end
end

local function bit32_bor(a, b, c, ...)
  local z
  if b then
    a = a % MOD
    b = b % MOD
    z = MODM - band(MODM - a, MODM - b)
    if c then
      z = bit32_bor(z, c, ...)
    end
    return z
  elseif a then
    return a % MOD
  else
    return 0
  end
end

local function Fl (n)
  return lshift(1, n)
end

-- End of bitwise operators

-- Terrain flags
local terrainFlags = {
  T_OBSTRUCTS_PASSABILITY         = Fl(0), --cannot be walked through
  T_OBSTRUCTS_VISION              = Fl(1), --blocks line of sight
  T_OBSTRUCTS_ITEMS               = Fl(2), --items can't be on this tile
  T_OBSTRUCTS_SURFACE_EFFECTS     = Fl(3), --grass, blood, etc. cannot exist on this tile
  T_OBSTRUCTS_GAS                 = Fl(4), --blocks the permeation of gas
  T_OBSTRUCTS_DIAGONAL_MOVEMENT   = Fl(5), --can't step diagonally around this tile
  T_SPONTANEOUSLY_IGNITES         = Fl(6), --monsters avoid unless chasing player or immune to fire
  T_AUTO_DESCENT                  = Fl(7), --automatically drops creatures down a depth level and does some damage (2d6)
  T_LAVA_INSTA_DEATH              = Fl(8), --kills any non-levitating non-fire-immune creature instantly
  T_CAUSES_POISON                 = Fl(9), --any non-levitating creature gets 10 poison
  T_IS_FLAMMABLE                  = Fl(10), --terrain can catch fire
  T_IS_FIRE                       = Fl(11), --terrain is a type of fire; ignites neighboring flammable cells
  T_ENTANGLES                     = Fl(12), --entangles players and monsters like a spiderweb
  T_IS_DEEP_WATER                 = Fl(13), --steals items 50% of the time and moves them around randomly
  T_CAUSES_DAMAGE                 = Fl(14), --anything on the tile takes max(1-2, 10%) damage per turn
  T_CAUSES_NAUSEA                 = Fl(15), --any creature on the tile becomes nauseous
  T_CAUSES_PARALYSIS              = Fl(16), --anything caught on this tile is paralyzed
  T_CAUSES_CONFUSION              = Fl(17), --causes creatures on this tile to become confused
  T_CAUSES_HEALING                = Fl(18), --heals 20% max HP per turn for any player or non-inanimate monsters
  T_IS_DF_TRAP                    = Fl(19), --spews gas of type specified in fireType when stepped on
  T_CAUSES_EXPLOSIVE_DAMAGE       = Fl(20), --is an explosion; deals higher of 15-20 or 50% damage instantly, but not again for five turns
  T_SACRED                        = Fl(21), --monsters that aren't allies of the player will avoid stepping here
}
local tf = terrainFlags;

tf.T_CAN_BE_BRIDGED = tf.T_AUTO_DESCENT

terrainFlags.T_PATHING_BLOCKER = bit32_bor(
  tf.T_OBSTRUCTS_PASSABILITY,
  tf.T_OBSTRUCTS_VISION,
  tf.T_AUTO_DESCENT,
  tf.T_LAVA_INSTA_DEATH,
  tf.T_IS_DEEP_WATER,
  tf.T_SPONTANEOUSLY_IGNITES
)
terrainFlags.T_OBSTRUCTS_EVERYTHING = bit32_bor(
  tf.T_OBSTRUCTS_PASSABILITY,
  tf.T_OBSTRUCTS_VISION,
  tf.T_OBSTRUCTS_ITEMS,
  tf.T_OBSTRUCTS_GAS,
  tf.T_OBSTRUCTS_SURFACE_EFFECTS,
  tf.T_OBSTRUCTS_DIAGONAL_MOVEMENT
)

terrainFlags.T_LAKE_PATHING_BLOCKER = bit32_bor(
  tf.T_AUTO_DESCENT,
  tf.T_LAVA_INSTA_DEATH,
  tf.T_IS_DEEP_WATER,
  tf.T_SPONTANEOUSLY_IGNITES
)

mg_arch.terrainFlags = terrainFlags

local function hasAnyFlags(flags1, flags2)
  for k1,v1 in pairs(flags1) do
    for k2,v2 in pairs(flags2) do
      if k1 == k2 and v1 > 0 and v2 > 0 then
        return true
      end
    end
  end
  return false
end

-- hasAllFlags({f1 = 1, f2 = 1}, {f1 = 1, f2 = 1, f3 = 1}) is true
-- hasAllFlags({f1 = 1, f2 = 1, f3 = 1}, {f1 = 1, f2 = 1}) is false
-- basically flags1 is a subset of flags2
local function hasAllFlags(flags1, flags2)
  for k1,v1 in pairs(flags1) do
    local hasFlag = false
    for k2,v2 in pairs(flags2) do
      if k1 == k2 and v1 > 0 and v2 > 0 then
        hasFlag = true
      end
    end
    if hasFlag == false then
      return false
    end
  end
  return true
end



mg_arch.bit = {}
mg_arch.bit.Fl = Fl
mg_arch.bit.bor = bor
mg_arch.bit.band = band
mg_arch.bit.bnot = bnot
mg_arch.bit.bit32_band = bit32_band
mg_arch.bit.bit32_bor = bit32_bor
mg_arch.hasAnyFlags = hasAnyFlags
mg_arch.hasAllFlags = hasAllFlags
