local function printTable (table)
  -- print(json.encode(table))
end


if mg_arch == nil then
  mg_arch = {}
end

-- depends on grid.lua
local allocGrid = mg_arch.allocGrid
local copyNewGrid = mg_arch.copyNewGrid
local fillGrid = mg_arch.fillGrid
local DROWS = mg_arch.DROWS
local DCOLS = mg_arch.DCOLS

-- depends on globals.lua
local terrainFlags = mg_arch.terrainFlags
local tf = mg_arch.terrainFlags
local Fl = mg_arch.bit.Fl
local bor = mg_arch.bit.bor
local band = mg_arch.bit.band
local bnot = mg_arch.bit.bnot
local bit32_bor = mg_arch.bit.bit32_bor


local DEPTH_ACCELERATOR = 1

-- consts
local dungeonFeatureType = {
  DF_GRANITE_COLUMN = 1,
  DF_CRYSTAL_WALL = 2,
  DF_LUMINESCENT_FUNGUS = 3,
  DF_GRASS = 4,
  DF_DEAD_GRASS = 5,
  DF_BONES = 6,
  DF_RUBBLE = 7,
  DF_FOLIAGE = 8,
  DF_FUNGUS_FOREST = 9,
  DF_DEAD_FOLIAGE = 10,
  DF_SUNLIGHT = 11,
  DF_DARKNESS = 12,
  DF_SHOW_DOOR = 13,
  DF_SHOW_POISON_GAS_TRAP = 14,
  DF_SHOW_PARALYSIS_GAS_TRAP = 15,
  DF_SHOW_TRAPDOOR_HALO = 16,
  DF_SHOW_TRAPDOOR = 17,
  DF_SHOW_CONFUSION_GAS_TRAP = 18,
  DF_SHOW_FLAMETHROWER_TRAP = 19,
  DF_SHOW_FLOOD_TRAP = 20,
  DF_SHOW_NET_TRAP = 21,
  DF_SHOW_ALARM_TRAP = 22,
  DF_RED_BLOOD = 23,
  DF_GREEN_BLOOD = 24,
  DF_PURPLE_BLOOD = 25,
  DF_WORM_BLOOD = 26,
  DF_ACID_BLOOD = 27,
  DF_ASH_BLOOD = 28,
  DF_EMBER_BLOOD = 29,
  DF_ECTOPLASM_BLOOD = 30,
  DF_RUBBLE_BLOOD = 31,
  DF_ROT_GAS_BLOOD = 32,
  DF_VOMIT = 33,
  DF_BLOAT_DEATH = 34,
  DF_BLOAT_EXPLOSION = 35,
  DF_BLOOD_EXPLOSION = 36,
  DF_FLAMEDANCER_CORONA = 37,
  DF_MUTATION_EXPLOSION = 38,
  DF_MUTATION_LICHEN = 39,
  DF_REPEL_CREATURES = 40,
  DF_ROT_GAS_PUFF = 41,
  DF_STEAM_PUFF = 42,
  DF_STEAM_ACCUMULATION = 43,
  DF_METHANE_GAS_PUFF = 44,
  DF_SALAMANDER_FLAME = 45,
  DF_URINE = 46,
  DF_UNICORN_POOP = 47,
  DF_PUDDLE = 48,
  DF_ASH = 49,
  DF_ECTOPLASM_DROPLET = 50,
  DF_FORCEFIELD = 51,
  DF_FORCEFIELD_MELT = 52,
  DF_SACRED_GLYPHS = 53,
  DF_LICHEN_GROW = 54,
  DF_TUNNELIZE = 55,
  DF_SHATTERING_SPELL = 56,
  DF_WEB_SMALL = 57,
  DF_WEB_LARGE = 58,
  DF_ANCIENT_SPIRIT_VINES = 59,
  DF_ANCIENT_SPIRIT_GRASS = 60,
  DF_TRAMPLED_FOLIAGE = 61,
  DF_SMALL_DEAD_GRASS = 62,
  DF_FOLIAGE_REGROW = 63,
  DF_TRAMPLED_FUNGUS_FOREST = 64,
  DF_FUNGUS_FOREST_REGROW = 65,
  DF_ACTIVE_BRIMSTONE = 66,
  DF_INERT_BRIMSTONE = 67,
  DF_BLOODFLOWER_PODS_GROW_INITIAL = 68,
  DF_BLOODFLOWER_PODS_GROW = 69,
  DF_BLOODFLOWER_POD_BURST = 70,
  DF_DEWAR_CAUSTIC = 71,
  DF_DEWAR_CONFUSION = 72,
  DF_DEWAR_PARALYSIS = 73,
  DF_DEWAR_METHANE = 74,
  DF_DEWAR_GLASS = 75,
  DF_CARPET_AREA = 76,
  DF_BUILD_ALGAE_WELL = 77,
  DF_ALGAE_1 = 78,
  DF_ALGAE_2 = 79,
  DF_ALGAE_REVERT = 80,
  DF_OPEN_DOOR = 81,
  DF_CLOSED_DOOR = 82,
  DF_OPEN_IRON_DOOR_INERT = 83,
  DF_ITEM_CAGE_OPEN = 84,
  DF_ITEM_CAGE_CLOSE = 85,
  DF_ALTAR_INERT = 86,
  DF_ALTAR_RETRACT = 87,
  DF_PORTAL_ACTIVATE = 88,
  DF_INACTIVE_GLYPH = 89,
  DF_ACTIVE_GLYPH = 90,
  DF_SILENT_GLYPH_GLOW = 91,
  DF_GUARDIAN_STEP = 92,
  DF_MIRROR_TOTEM_STEP = 93,
  DF_GLYPH_CIRCLE = 94,
  DF_REVEAL_LEVER = 95,
  DF_PULL_LEVER = 96,
  DF_CREATE_LEVER = 97,
  DF_BRIDGE_FALL_PREP = 98,
  DF_BRIDGE_FALL = 99,
  DF_PLAIN_FIRE = 100,
  DF_GAS_FIRE = 101,
  DF_EXPLOSION_FIRE = 102,
  DF_DART_EXPLOSION = 103,
  DF_BRIMSTONE_FIRE = 104,
  DF_BRIDGE_FIRE = 105,
  DF_FLAMETHROWER = 106,
  DF_EMBERS = 107,
  DF_EMBERS_PATCH = 108,
  DF_OBSIDIAN = 109,
  DF_ITEM_FIRE = 110,
  DF_CREATURE_FIRE = 111,
  DF_FLOOD = 112,
  DF_FLOOD_2 = 113,
  DF_FLOOD_DRAIN = 114,
  DF_HOLE_2 = 115,
  DF_HOLE_DRAIN = 116,
  DF_DEEP_WATER_FREEZE = 117,
  DF_ALGAE_1_FREEZE = 118,
  DF_ALGAE_2_FREEZE = 119,
  DF_DEEP_WATER_MELTING = 120,
  DF_DEEP_WATER_THAW = 121,
  DF_SHALLOW_WATER_FREEZE = 122,
  DF_SHALLOW_WATER_MELTING = 123,
  DF_SHALLOW_WATER_THAW = 124,
  DF_POISON_GAS_CLOUD = 125,
  DF_CONFUSION_GAS_TRAP_CLOUD = 126,
  DF_NET = 127,
  DF_AGGRAVATE_TRAP = 128,
  DF_METHANE_GAS_ARMAGEDDON = 129,
  DF_POISON_GAS_CLOUD_POTION = 130,
  DF_PARALYSIS_GAS_CLOUD_POTION = 131,
  DF_CONFUSION_GAS_CLOUD_POTION = 132,
  DF_INCINERATION_POTION = 133,
  DF_DARKNESS_POTION = 134,
  DF_HOLE_POTION = 135,
  DF_LICHEN_PLANTED = 136,
  DF_ARMOR_IMMOLATION = 137,
  DF_STAFF_HOLE = 138,
  DF_STAFF_HOLE_EDGE = 139,
  DF_ALTAR_COMMUTE = 140,
  DF_MAGIC_PIPING = 141,
  DF_INERT_PIPE = 142,
  DF_ALTAR_RESURRECT = 143,
  DF_MACHINE_FLOOR_TRIGGER_REPEATING = 144,
  DF_SACRIFICE_ALTAR = 145,
  DF_SACRIFICE_COMPLETE = 146,
  DF_SACRIFICE_CAGE_ACTIVE = 147,
  DF_COFFIN_BURSTS = 148,
  DF_COFFIN_BURNS = 149,
  DF_TRIGGER_AREA = 150,
  DF_CAGE_DISAPPEARS = 151,
  DF_MEDIUM_HOLE = 152,
  DF_MEDIUM_LAVA_POND = 153,
  DF_MACHINE_PRESSURE_PLATE_USED = 154,
  DF_WALL_CRACK = 155,
  DF_WOODEN_BARRICADE_BURN = 156,
  DF_SURROUND_WOODEN_BARRICADE = 157,
  DF_SPREADABLE_WATER = 158,
  DF_SHALLOW_WATER = 159,
  DF_WATER_SPREADS = 160,
  DF_SPREADABLE_WATER_POOL = 161,
  DF_SPREADABLE_DEEP_WATER_POOL = 162,
  DF_SPREADABLE_COLLAPSE = 163,
  DF_COLLAPSE = 164,
  DF_COLLAPSE_SPREADS = 165,
  DF_ADD_MACHINE_COLLAPSE_EDGE_DORMANT = 166,
  DF_BRIDGE_ACTIVATE = 167,
  DF_BRIDGE_ACTIVATE_ANNOUNCE = 168,
  DF_BRIDGE_APPEARS = 169,
  DF_ADD_DORMANT_CHASM_HALO = 170,
  DF_LAVA_RETRACTABLE = 171,
  DF_RETRACTING_LAVA = 172,
  DF_OBSIDIAN_WITH_STEAM = 173,
  DF_SHOW_POISON_GAS_VENT = 174,
  DF_POISON_GAS_VENT_OPEN = 175,
  DF_ACTIVATE_PORTCULLIS = 176,
  DF_OPEN_PORTCULLIS = 177,
  DF_VENT_SPEW_POISON_GAS = 178,
  DF_SHOW_METHANE_VENT = 179,
  DF_METHANE_VENT_OPEN = 180,
  DF_VENT_SPEW_METHANE = 181,
  DF_PILOT_LIGHT = 182,
  DF_DISCOVER_PARALYSIS_VENT = 183,
  DF_PARALYSIS_VENT_SPEW = 184,
  DF_REVEAL_PARALYSIS_VENT_SILENTLY = 185,
  DF_AMBIENT_BLOOD = 186,
  DF_CRACKING_STATUE = 187,
  DF_STATUE_SHATTER = 188,
  DF_TURRET_EMERGE = 189,
  DF_WORM_TUNNEL_MARKER_DORMANT = 190,
  DF_WORM_TUNNEL_MARKER_ACTIVE = 191,
  DF_GRANITE_CRUMBLES = 192,
  DF_WALL_OPEN = 193,
  DF_DARKENING_FLOOR = 194,
  DF_DARK_FLOOR = 195,
  DF_HAUNTED_TORCH_TRANSITION = 196,
  DF_HAUNTED_TORCH = 197,
  DF_MUD_DORMANT = 198,
  DF_MUD_ACTIVATE = 199,
  DF_ELECTRIC_CRYSTAL_ON = 200,
  DF_TURRET_LEVER = 201,
  DF_SHALLOW_WATER_POOL = 202,
  DF_DEEP_WATER_POOL = 203,
  DF_SWAMP_WATER = 204,
  DF_SWAMP = 205,
  DF_SWAMP_MUD = 206,
  DF_HAY = 207,
  DF_JUNK = 208,
  DF_REMNANT = 209,
  DF_REMNANT_ASH = 210,
  DF_CHASM_HOLE = 211,
  DF_CATWALK_BRIDGE = 212,
  DF_LAKE_CELL = 213,
  DF_LAKE_HALO = 214,
  DF_WALL_SHATTER = 215,
  DF_MONSTER_CAGE_OPENS = 216,
  DF_STENCH_BURN = 217,
  DF_STENCH_SMOLDER = 218,
}

local df = dungeonFeatureType

-- there's a big enum for all the tile types
-- This is just some of them to get me going
local tileType = {
  NOTHING = 0,
  GRANITE = 1,
  FLOOR = 2,
  FLOOR_FLOODABLE = 3,
  CARPET = 4,
  MARBLE_FLOOR = 5,
  WALL = 6,
  DOOR = 7,
  OPEN_DOOR = 8,
  SECRET_DOOR = 9,
  LOCKED_DOOR = 10,
  OPEN_IRON_DOOR_INERT = 11,
  DOWN_STAIRS = 12,
  UP_STAIRS = 13,
  DUNGEON_EXIT = 14,
  DUNGEON_PORTAL = 15,
  TORCH_WALL = 16,
  CRYSTAL_WALL = 17,
  PORTCULLIS_CLOSED = 18,
  PORTCULLIS_DORMANT = 19,
  WOODEN_BARRICADE = 20,
  PILOT_LIGHT_DORMANT = 21,
  PILOT_LIGHT = 22,
  HAUNTED_TORCH_DORMANT = 23,
  HAUNTED_TORCH_TRANSITIONING = 24,
  HAUNTED_TORCH = 25,
  WALL_LEVER_HIDDEN = 26,
  WALL_LEVER = 27,
  WALL_LEVER_PULLED = 28,
  WALL_LEVER_HIDDEN_DORMANT = 29,
  STATUE_INERT = 30,
  STATUE_DORMANT = 31,
  STATUE_CRACKING = 32,
  STATUE_INSTACRACK = 33,
  PORTAL = 34,
  TURRET_DORMANT = 35,
  WALL_MONSTER_DORMANT = 36,
  DARK_FLOOR_DORMANT = 37,
  DARK_FLOOR_DARKENING = 38,
  DARK_FLOOR = 39,
  MACHINE_TRIGGER_FLOOR = 40,
  ALTAR_INERT = 41,
  ALTAR_KEYHOLE = 42,
  ALTAR_CAGE_OPEN = 43,
  ALTAR_CAGE_CLOSED = 44,
  ALTAR_SWITCH = 45,
  ALTAR_SWITCH_RETRACTING = 46,
  ALTAR_CAGE_RETRACTABLE = 47,
  PEDESTAL = 48,
  MONSTER_CAGE_OPEN = 49,
  MONSTER_CAGE_CLOSED = 50,
  COFFIN_CLOSED = 51,
  COFFIN_OPEN = 52,

  GAS_TRAP_POISON_HIDDEN = 53,
  GAS_TRAP_POISON = 54,
  TRAP_DOOR_HIDDEN = 55,
  TRAP_DOOR = 56,
  GAS_TRAP_PARALYSIS_HIDDEN = 57,
  GAS_TRAP_PARALYSIS = 58,
  MACHINE_PARALYSIS_VENT_HIDDEN = 59,
  MACHINE_PARALYSIS_VENT = 60,
  GAS_TRAP_CONFUSION_HIDDEN = 61,
  GAS_TRAP_CONFUSION = 62,
  FLAMETHROWER_HIDDEN = 63,
  FLAMETHROWER = 64,
  FLOOD_TRAP_HIDDEN = 65,
  FLOOD_TRAP = 66,
  NET_TRAP_HIDDEN = 67,
  NET_TRAP = 68,
  ALARM_TRAP_HIDDEN = 69,
  ALARM_TRAP = 70,
  MACHINE_POISON_GAS_VENT_HIDDEN = 71,
  MACHINE_METHANE_VENT_DORMANT = 72,
  MACHINE_METHANE_VENT = 73,
  STEAM_VENT = 74,
  MACHINE_PRESSURE_PLATE = 75,
  MACHINE_PRESSURE_PLATE_USED = 76,
  MACHINE_GLYPH = 77,
  MACHINE_GLYPH_INACTIVE = 78,
  DEWAR_CAUSTIC_GAS = 79,
  DEWAR_CONFUSION_GAS = 80,
  DEWAR_PARALYSIS_GAS = 81,
  DEWAR_METHANE_GAS = 82,

  DEEP_WATER = 83,
  SHALLOW_WATER = 84,
  MUD = 85,
  CHASM = 86,
  CHASM_EDGE = 87,
  MACHINE_COLLAPSE_EDGE_DORMANT = 88,
  MACHINE_COLLAPSE_EDGE_SPREADING = 89,
  LAVA = 90,
  LAVA_RETRACTABLE = 91,
  LAVA_RETRACTING = 92,
  SUNLIGHT_POOL = 93,
  DARKNESS_PATCH = 94,
  ACTIVE_BRIMSTONE = 95,
  INERT_BRIMSTONE = 96,
  OBSIDIAN = 97,
  BRIDGE = 98,
  BRIDGE_FALLING = 99,
  BRIDGE_EDGE = 100,
  STONE_BRIDGE = 101,
  MACHINE_FLOOD_WATER_DORMANT = 102,
  MACHINE_FLOOD_WATER_SPREADING = 103,
  MACHINE_MUD_DORMANT = 104,
  ICE_DEEP = 105,
  ICE_DEEP_MELT = 106,
  ICE_SHALLOW = 107,
  ICE_SHALLOW_MELT = 108,

  HOLE = 109,
  HOLE_GLOW = 110,
  HOLE_EDGE = 111,
  FLOOD_WATER_DEEP = 112,
  FLOOD_WATER_SHALLOW = 113,
  GRASS = 114,
  DEAD_GRASS = 115,
  GRAY_FUNGUS = 116,
  LUMINESCENT_FUNGUS = 116,
  LICHEN = 117,
  HAY = 118,

}

local tt = tileType

-- Create a function to find the name by value
local function getKeyByValue(tbl, value)
  for name, val in pairs(tbl) do
    if val == value then
      return name
    end
  end
  return value  -- Return nil if no matching value is found
end

-- there's a lot more - I just put what I needed for now
local PDS_OBSTRUCTION = -2
local PDS_FORBIDDEN = -1

local MINIMUM_LAVA_LEVEL = 4           -- how deep before lava can be generated
local MINIMUM_BRIMSTONE_LEVEL = 17          -- how deep before brimstone can be generated
local DEEPEST_LEVEL = 40
local HORIZONTAL_CORRIDOR_MIN_LENGTH = 5
local HORIZONTAL_CORRIDOR_MAX_LENGTH = 15
local VERTICAL_CORRIDOR_MIN_LENGTH = 2
local VERTICAL_CORRIDOR_MAX_LENGTH = 9

local AMULET_LEVEL = 23;


local NO_DIRECTION    = -1
-- Cardinal directions; must be 1-4:
-- lua is 1 indexed arrays - dumb
local UP              = 1
local DOWN            = 2
local LEFT            = 3
local RIGHT           = 4
-- Secondary directions; must be 5-8:
local UPLEFT          = 5
local DOWNLEFT        = 6
local UPRIGHT         = 7
local DOWNRIGHT       = 8
local DIRECTION_COUNT = 8
local nbDirs = {{0,-1}, {0,1}, {-1,0}, {1, 0}, {-1,-1}, {-1,1}, {1,-1}, {1,1}};
local cDirs =  {{0, 1}, {1,1}, { 1,0}, {1,-1},{ 0,-1}, {-1,-1}, {-1,0}, {-1,1}};

local CAVE_MIN_WIDTH = 50
local CAVE_MIN_HEIGHT = 20

local dungeonProfileTypes = {
  BASIC = 1,
  BASIC_FIRST_ROOM = 2,
  GOBLIN_WARREN = 3,
  SENTINEL_SANCTUARY = 4,
  NUMBER_DUNGEON_PROFILES = 4
}

local room_type_count = 8;

local dungeonProfileCatalog = {
  {roomFrequencies = {2, 1,1,1,7,1, 0, 0}, corridorChance = 10}, -- basic
  {roomFrequencies = {10,0,0,3,7,10,10,0}, corridorChance = 0},  -- first room
  {roomFrequencies = {0, 0,1,0,0,0, 0, 0}, corridorChance = 0}, -- goblin warrens
  {roomFrequencies = {0, 5,0,1,0,0, 0, 0}, corridorChance = 0}, -- sentiel
}



-- math functions

-- no tern (exp ? true : false) in lua so use function
local function tern ( cond , T , F )
    if cond then return T else return F end
end

local floor = math.floor




local terrainGroupNames = {
  OBSTRUCTS_PASSABILITY         = "obstructsPassability",
  OBSTRUCTS_VISION              = "obstructsVision",
  OBSTRUCTS_ITEMS               = "obstructsItems",
  OBSTRUCTS_SURFACE_EFFECTS     = "obstructsSurfaceEffects",
  OBSTRUCTS_GAS                 = "obstructsGas",
  OBSTRUCTS_DIAGONAL_MOVEMENT   = "obstructsDiagonalMovement",
  SPONTANEOUSLY_IGNITES         = "spontaneouslyIgnits",
  AUTO_DESCENT                  = "autoDescent", --automatically drops creatures down a depth level and does some damage (2d6)
  LAVA_INSTA_DEATH              = "lavaInstaDeath", --kills any non-levitating non-fire-immune creature instantly
  CAUSES_POISON                 = "causesPoison", --any non-levitating creature gets 10 poison
  IS_FLAMMABLE                  = "isFlammable", --terrain can catch fire
  IS_FIRE                       = "isFire", --terrain is a type of fire; ignites neighboring flammable cells
  ENTANGLES                     = "entangles", --entangles players and monsters like a spiderweb
  IS_DEEP_WATER                 = "isDeepWater", --steals items 50% of the time and moves them around randomly
  CAUSES_DAMAGE                 = "causesDamage", --anything on the tile takes max(1-2, 10%) damage per turn
  CAUSES_NAUSEA                 = "causesNausea", --any creature on the tile becomes nauseous
  CAUSES_PARALYSIS              = "causesParalysis", --anything caught on this tile is paralyzed
  CAUSES_CONFUSION              = "causesConfustion", --causes creatures on this tile to become confused
  CAUSES_HEALING                = "causesHealing", --heals 20% max HP per turn for any player or non-inanimate monsters
  IS_DF_TRAP                    = "isDFTrap", --spews gas of type specified in fireType when stepped on
  CAUSES_EXPLOSIVE_DAMAGE       = "causesExplosiveDamage", --is an explosion; deals higher of 15-20 or 50% damage instantly, but not again for five turns
  SACRED                        = "scared", --monsters that aren't allies of the player will avoid stepping here
}

local tgn = terrainGroupNames

local pathingBlockerGroup = {
  [tgn.OBSTRUCTS_PASSABILITY] = 1,
  [tgn.OBSTRUCTS_VISION] = 1,
  [tgn.AUTO_DESCENT] = 1,
  [tgn.LAVA_INSTA_DEATH] = 1,
  [tgn.IS_DEEP_WATER] = 1,
  [tgn.SPONTANEOUSLY_IGNITES] = 1
}

local obstructsEverythingGroup = {
  [tgn.OBSTRUCTS_PASSABILITY] = 1,
  [tgn.OBSTRUCTS_VISION] = 1,
  [tgn.OBSTRUCTS_ITEMS] = 1,
  [tgn.OBSTRUCTS_GAS] = 1,
  [tgn.OBSTRUCTS_SURFACE_EFFECTS] = 1,
  [tgn.OBSTRUCTS_DIAGONAL_MOVEMENT] = 1
}




local DFFlags = {
    DFF_EVACUATE_CREATURES_FIRST    = Fl(0),    -- Creatures in the DF area get moved outside of it
    DFF_SUBSEQ_EVERYWHERE           = Fl(1),    -- Subsequent DF spawns in every cell that this DF spawns in, instead of only the origin
    DFF_TREAT_AS_BLOCKING           = Fl(2),    -- If filling the footprint of this DF with walls would disrupt level connectivity, then abort.
    DFF_PERMIT_BLOCKING             = Fl(3),    -- Generate this DF without regard to level connectivity.
    DFF_ACTIVATE_DORMANT_MONSTER    = Fl(4),    -- Dormant monsters on this tile will appear -- e.g. when a statue bursts to reveal a monster.
    DFF_CLEAR_OTHER_TERRAIN         = Fl(5),    -- Erase other terrain in the footprint of this DF.
    DFF_BLOCKED_BY_OTHER_LAYERS     = Fl(6),    -- Will not propagate into a cell if any layer in that cell has a superior priority.
    DFF_SUPERPRIORITY               = Fl(7),    -- Will overwrite terrain of a superior priority.
    DFF_AGGRAVATES_MONSTERS         = Fl(8),    -- Will act as though an aggravate monster scroll of effectRadius radius had been read at that point.
    DFF_RESURRECT_ALLY              = Fl(9),    -- Will bring back to life your most recently deceased ally.
};

local dff = DFFlags


-- small part of catelog
local function makeCatEntry(tType, flags, priority)
  return {
    tType = tType,
    flags = flags,
    drawPriority = priority
  }
end

-- can't have nothing because lua is 1 based array
-- stupid stupid
local nt = makeCatEntry(tileType.NOTHING, 0, 100)

local tileCatalog = {
  makeCatEntry(tileType.GRANITE, tf.T_OBSTRUCTS_EVERYTHING, 0),
  makeCatEntry(tileType.FLOOR, 0, 95),
  makeCatEntry(tileType.FLOOR_FLOODABLE, 0, 95),
  makeCatEntry(tileType.CARPET, tf.T_IS_FLAMMABLE, 85),
  makeCatEntry(tileType.MARBLE_FLOOR, 0, 85),
  makeCatEntry(tileType.WALL, tf.T_OBSTRUCTS_EVERYTHING, 0),
  makeCatEntry(tileType.DOOR, tf.T_IS_FLAMMABLE, 25),
  makeCatEntry(tileType.OPEN_DOOR, tf.T_IS_FLAMMABLE, 25),
  makeCatEntry(tileType.SECRET_DOOR, bit32_bor(tf.T_OBSTRUCTS_EVERYTHING, tf.T_IS_FLAMMABLE), 0),
  makeCatEntry(tileType.LOCKED_DOOR, tf.T_OBSTRUCTS_EVERYTHING, 15),
  makeCatEntry(tileType.OPEN_IRON_DOOR_INERT, tf.T_OBSTRUCTS_SURFACE_EFFECTS, 90),
  makeCatEntry(tileType.DOWN_STAIRS, tf.T_OBSTRUCTS_ITEMS, 30),
  makeCatEntry(tileType.UP_STAIRS, tf.T_OBSTRUCTS_ITEMS, 30),
  makeCatEntry(tileType.DUNGEON_EXIT, tf.T_OBSTRUCTS_ITEMS, 30),
  makeCatEntry(tileType.DUNGEON_PORTAL, tf.T_OBSTRUCTS_ITEMS, 30),
  makeCatEntry(tileType.TORCH_WALL, tf.T_OBSTRUCTS_EVERYTHING, 0),
  makeCatEntry(tileType.CRYSTAL_WALL, bor(tf.T_OBSTRUCTS_PASSABILITY, tf.T_OBSTRUCTS_ITEMS), 0),
}

tileCatalog[tileType.STATUE_INERT] = makeCatEntry(tileType.STATUE_INERT, bor(tf.T_OBSTRUCTS_PASSABILITY, tf.T_OBSTRUCTS_ITEMS), 0)

-- BRIDGE and BRIDGE EDGE



tileCatalog[53] = makeCatEntry(tileType.GAS_TRAP_POISON_HIDDEN, tf.T_IS_DF_TRAP, 95)
tileCatalog[54] = makeCatEntry(tileType.GAS_TRAP_POISON, tf.T_IS_DF_TRAP, 30)

tileCatalog[55] = makeCatEntry(tileType.TRAP_DOOR_HIDDEN, tf.T_IS_DF_TRAP, 95)
tileCatalog[56] = makeCatEntry(tileType.TRAP_DOOR, tf.T_IS_DF_TRAP, 30)

tileCatalog[57] = makeCatEntry(tileType.GAS_TRAP_PARALYSIS_HIDDEN, tf.T_IS_DF_TRAP, 95)
tileCatalog[58] = makeCatEntry(tileType.GAS_TRAP_PARALYSIS, tf.T_IS_DF_TRAP, 30)

tileCatalog[61] = makeCatEntry(tileType.GAS_TRAP_CONFUSION_HIDDEN, tf.T_IS_DF_TRAP, 95)
tileCatalog[62] = makeCatEntry(tileType.GAS_TRAP_CONFUSION, tf.T_IS_DF_TRAP, 30)

tileCatalog[63] = makeCatEntry(tileType.FLAMETHROWER_HIDDEN, tf.T_IS_DF_TRAP, 95)
tileCatalog[64] = makeCatEntry(tileType.FLAMETHROWER, tf.T_IS_DF_TRAP, 30)

tileCatalog[65] = makeCatEntry(tileType.FLOOD_TRAP_HIDDEN, tf.T_IS_DF_TRAP, 95)
tileCatalog[66] = makeCatEntry(tileType.FLOOD_TRAP, tf.T_IS_DF_TRAP, 30)

tileCatalog[67] = makeCatEntry(tileType.NET_TRAP_HIDDEN, tf.T_IS_DF_TRAP, 95)
tileCatalog[68] = makeCatEntry(tileType.NET_TRAP, tf.T_IS_DF_TRAP, 30)

tileCatalog[69] = makeCatEntry(tileType.ALARM_TRAP_HIDDEN, tf.T_IS_DF_TRAP, 95)
tileCatalog[70] = makeCatEntry(tileType.ALARM_TRAP, tf.T_IS_DF_TRAP, 30)

tileCatalog[83] = makeCatEntry(tileType.DEEP_WATER, bor(tf.T_IS_DEEP_WATER, tf.T_IS_FLAMMABLE), 40)
tileCatalog[84] = makeCatEntry(tileType.SHALLOW_WATER, 0, 55)
tileCatalog[85] = makeCatEntry(tileType.MUD, 0, 55)
tileCatalog[86] = makeCatEntry(tileType.CHASM, tf.T_AUTO_DESCENT, 40)
tileCatalog[87] = makeCatEntry(tileType.CHASM_EDGE, 0, 40)
tileCatalog[90] = makeCatEntry(tileType.LAVA, tf.T_LAVA_INSTA_DEATH, 40)
tileCatalog[96] = makeCatEntry(tileType.INERT_BRIMSTONE, tf.T_SPONTANEOUSLY_IGNITES, 0)
tileCatalog[97] = makeCatEntry(tileType.OBSIDIAN, 0, 0)

tileCatalog[tt.BRIDGE] = makeCatEntry(tileType.BRIDGE, tf.T_IS_FLAMMABLE, 45)
tileCatalog[tt.BRIDGE_EDGE] = makeCatEntry(tileType.BRIDGE_EDGE, tf.T_IS_FLAMMABLE, 45)

tileCatalog[tt.FLOOD_WATER_SHALLOW] = makeCatEntry(tt.FLOOD_WATER_SHALLOW, 0, 45)
tileCatalog[tt.FLOOD_WATER_DEEP] = makeCatEntry(tt.FLOOD_WATER_DEEP, 0, 45)

local function createAG (
  terrain,
  layer,
  DFType,
  machine,
  requiredDungeonFoundationType,
  requiredLiquidFoundationType,
  minDepth,
  maxDepth,
  frequency,
  minNumberIntercept,
  minNumberSlope,
  maxNumber
)
  return {
    terrain = terrain,
    layer = layer,
    DFType = DFType,
    machine = machine,
    requiredDungeonFoundationType = requiredDungeonFoundationType,
    requiredLiquidFoundationType = requiredLiquidFoundationType,
    minDepth = minDepth,
    maxDepth = maxDepth,
    frequency = frequency,
    minNumberIntercept = minNumberIntercept,
    minNumberSlope = minNumberSlope,
    maxNumber = maxNumber
  }
end

local DUNGEON = 'dungeon'
local SURFACE = 'surface'
local GAS = 'gas'
local LIQUID = 'liquid'

local autoGeneratorCatalog = {
  createAG(0,                            0,      df.DF_GRANITE_COLUMN,       0, tt.FLOOR, tt.NOTHING,    1,      DEEPEST_LEVEL, 60,   100, 0,  4),
  createAG(0,                            0,      df.DF_CRYSTAL_WALL,         0, tt.WALL,  tt.NOTHING,    15,     DEEPEST_LEVEL, 15,  -325, 24, 5),
  createAG(0,                            0,      df.DF_LUMINESCENT_FUNGUS,   0, tt.FLOOR, tt.NOTHING,    7,      DEEPEST_LEVEL, 15,  -300, 75, 14),
  createAG(0,                            0,      df.DF_GRASS,                0, tt.FLOOR, tt.NOTHING,    0,      10,             0,  1000, -80, 10),
  createAG(0,                            0,      df.DF_DEAD_GRASS,           0, tt.FLOOR, tt.NOTHING,    4,      9,              0,  -200,   80,         10),
  createAG(0,                            0,      df.DF_DEAD_GRASS,           0, tt.FLOOR, tt.NOTHING,    9,      14,             0,  1200,   -80,        10),
  createAG(0,                            0,      df.DF_BONES,                0, tt.FLOOR, tt.NOTHING,    12,     DEEPEST_LEVEL-1,30, 0,      0,          4),
  createAG(0,                            0,      df.DF_RUBBLE,               0, tt.FLOOR, tt.NOTHING,    0,      DEEPEST_LEVEL-1,30, 0,      0,          4),
  createAG(0,                            0,      df.DF_FOLIAGE,              0, tt.FLOOR, tt.NOTHING,    0,      8,              15, 1000,   -333,       10),
  createAG(0,                            0,      df.DF_FUNGUS_FOREST,        0, tt.FLOOR, tt.NOTHING,    13,     DEEPEST_LEVEL,  30, -600,   50,         12),
  createAG(0,                            0,      df.DF_BUILD_ALGAE_WELL,     0, tt.FLOOR, tt.DEEP_WATER, 10,     DEEPEST_LEVEL,  50, 0,      0,          2),
  createAG(tt.STATUE_INERT,              DUNGEON,0,                          0, tt.WALL,  tt.NOTHING,    6,      DEEPEST_LEVEL-1,5,  -100,   35,         3),
  createAG(tt.STATUE_INERT,              DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    10,     DEEPEST_LEVEL-1,50, 0,      0,          3),
  createAG(tt.TORCH_WALL,                DUNGEON,0,                          0, tt.WALL,  tt.NOTHING,    6,      DEEPEST_LEVEL-1,5,  -200,   70,         12),

  -- Pre-revealed traps
  createAG(tt.GAS_TRAP_POISON,           DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    2,      4,              20,     0,      0,          1),
  createAG(tt.NET_TRAP,                  DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    2,      5,              20,     0,      0,          1),
  --createAG(tt.ALARM_TRAP,                DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    4,      7,              20,     0,      0,          1),
  --createAG(tt.GAS_TRAP_CONFUSION,        DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    2,      10,             20,     0,      0,          1),
  createAG(tt.FLAMETHROWER,              DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    4,      12,             20,     0,      0,          1),
  createAG(tt.FLOOD_TRAP,                DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    5,     14,             20,     0,      0,          1),
  --createAG(0,                DUNGEON,df.DF_FLOOD,                          0, tt.FLOOR, tt.NOTHING,    5,     14,             20,     0,      0,          1),

  -- Hidden traps
  createAG(tt.GAS_TRAP_POISON_HIDDEN,    DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    5,      DEEPEST_LEVEL-1,20,     100,    0,          3),
  createAG(tt.NET_TRAP_HIDDEN,           DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    6,      DEEPEST_LEVEL-1,20,     100,    0,          3),
  --createAG(tt.ALARM_TRAP_HIDDEN,         DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    8,      DEEPEST_LEVEL-1,20,     100,    0,          2),
  createAG(tt.TRAP_DOOR_HIDDEN,          DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    5,      DEEPEST_LEVEL-1,20,     100,    0,          2),
  --createAG(tt.GAS_TRAP_CONFUSION_HIDDEN, DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    11,     DEEPEST_LEVEL-1,20,     100,    0,          3),
  createAG(tt.FLAMETHROWER_HIDDEN,       DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    13,     DEEPEST_LEVEL-1,20,     100,    0,          3),
  createAG(tt.FLOOD_TRAP_HIDDEN,         DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    15,     DEEPEST_LEVEL-1,20,     100,    0,          3),
  createAG(0,                            0,      df.DF_SUNLIGHT,             0, tt.FLOOR, tt.NOTHING,    0,      5,              15,     500,    -150,       10),
  createAG(0,                            0,      df.DF_DARKNESS,             0, tt.FLOOR, tt.NOTHING,    1,      15,             15,     500,    -50,        10),
  --createAG(tt.STEAM_VENT,                DUNGEON,0,                          0, tt.FLOOR, tt.NOTHING,    16,     DEEPEST_LEVEL-1,30,     100,    0,          3),
  --createAG(tt.CRYSTAL_WALL,              DUNGEON,0,                          0, tt.WALL,  tt.NOTHING,    DEEPEST_LEVEL,DEEPEST_LEVEL,100,0,      0,          600),

  -- Dewars
  -- createAG(tt.DEWAR_CAUSTIC_GAS,         DUNGEON,df.DF_CARPET_AREA,          0, tt.FLOOR, tt.NOTHING,    8,      DEEPEST_LEVEL-1,2,      0,      0,          2),
  -- createAG(tt.DEWAR_CONFUSION_GAS,       DUNGEON,df.DF_CARPET_AREA,          0, tt.FLOOR, tt.NOTHING,    8,      DEEPEST_LEVEL-1,2,      0,      0,          2),
  -- createAG(tt.DEWAR_PARALYSIS_GAS,       DUNGEON,df.DF_CARPET_AREA,          0, tt.FLOOR, tt.NOTHING,    8,      DEEPEST_LEVEL-1,2,      0,      0,          2),
  -- createAG(tt.DEWAR_METHANE_GAS,         DUNGEON,df.DF_CARPET_AREA,          0, tt.FLOOR, tt.NOTHING,    8,      DEEPEST_LEVEL-1,2,      0,      0,          2),

  -- Flavor machines
  createAG(0,                            0,      df.DF_LUMINESCENT_FUNGUS,      0, tt.FLOOR, tt.NOTHING,    DEEPEST_LEVEL,DEEPEST_LEVEL,100,0,      0,          200),

}

local function createDF(tile, layer, startProbability, probabilityDecrement, flags, description, effectRadius, propagationTerrain, subsequentDF)
  return {
    tile = tile,
    layer = layer,
    startProbability = startProbability,
    probabilityDecrement = probabilityDecrement,
    flags = flags,
    description = description,
    effectRadius = effectRadius,
    propagationTerrain = propagationTerrain,
    subsequentDF = subsequentDF
  }
end

-- I messed up on this (or maybe I had something less in mind when I ported this?) 
-- the first item to createDF is suppose to be a tile type not a dungeon feature :-/
-- Also most dungeon features in this catalog are spawned during game time, not map creation 
-- and I may not actually need them...
local dungeonFeatureCatalog = {
    createDF(df.DF_GRANITE_COLUMN,            SURFACE,    80,     70,     dff.DFF_CLEAR_OTHER_TERRAIN),
    createDF(df.DF_CRYSTAL_WALL,              DUNGEON,    200,    50,     dff.DFF_CLEAR_OTHER_TERRAIN),
    createDF(df.DF_LUMINESCENT_FUNGUS,        SURFACE,    60,     8,      dff.DFF_BLOCKED_BY_OTHER_LAYERS),
    createDF(df.DF_GRASS,                     SURFACE,    75,     5,      dff.DFF_BLOCKED_BY_OTHER_LAYERS),
    createDF(df.DF_DEAD_GRASS,                SURFACE,    75,     5,      dff.DFF_BLOCKED_BY_OTHER_LAYERS,  "", 0,  nil, df.DF_DEAD_FOLIAGE),
    createDF(df.DF_BONES,                     SURFACE,    75,     23,     0),
    createDF(df.DF_RUBBLE,                    SURFACE,    45,     23,     0),
    createDF(df.DF_FOLIAGE,                   SURFACE,    100,    33,     (dff.DFF_BLOCKED_BY_OTHER_LAYERS)),
    createDF(df.DF_FUNGUS_FOREST,             SURFACE,    100,    45,     (dff.DFF_BLOCKED_BY_OTHER_LAYERS)),
    createDF(df.DF_DEAD_FOLIAGE,              SURFACE,    50,     30,     (dff.DFF_BLOCKED_BY_OTHER_LAYERS)),

    -- misc. liqDF_uids
    createDF(df.DF_SUNLIGHT,             LIQUID,     65,     6,      0),
    createDF(df.DF_DARKNESS,            LIQUID,     65,     11,     0),

    -- Dungeon features spawned during gameplay:
    -- revealed secrets
    createDF(df.DF_DOOR,                      DUNGEON,    0,      0,      0, ""),
    createDF(df.DF_GAS_TRAP_POISON,           DUNGEON,    0,      0,      0, ""),
    createDF(df.DF_GAS_TRAP_PARALYSIS,        DUNGEON,    0,      0,      0, ""),
    createDF(df.DF_CHASM_EDGE,                LIQUID,     100,    100,    0, ""),
    createDF(df.DF_TRAP_DOOR,                 LIQUID,     0,      0,      df.DFF_CLEAR_OTHER_TERRAIN, "", 0, 0, DF_SHOW_TRAPDOOR_HALO),
    createDF(df.DF_GAS_TRAP_CONFUSION,        DUNGEON,    0,      0,      0, "", GENERIC_FLASH_LIGHT),
    createDF(df.DF_FLAMETHROWER,              DUNGEON,    0,      0,      0, "", GENERIC_FLASH_LIGHT),
    createDF(df.DF_FLOOD_TRAP,                DUNGEON,    0,      0,      0, "", GENERIC_FLASH_LIGHT),
    createDF(df.DF_NET_TRAP,                  DUNGEON,    0,      0,      0, "", GENERIC_FLASH_LIGHT),
    createDF(df.DF_ALARM_TRAP,                DUNGEON,    0,      0,      0, "", GENERIC_FLASH_LIGHT),

    -- bloods
    -- Start probability is actually a percentage for bloods.
    -- Base probability is 15 + (damage * 2/3), and then take the given percentage of that.
    -- If it's a gas, we multiply the base by an additional 100.
    -- Thus to get a starting gas volume of a poison potion (1000), with a hit for 10 damage, use a starting probability of 48.
    createDF(df.DF_RED_BLOOD,                 SURFACE,    100,    25,     0),
    createDF(df.DF_GREEN_BLOOD,               SURFACE,    100,    25,     0),
    createDF(df.DF_PURPLE_BLOOD,              SURFACE,    100,    25,     0),
    createDF(df.DF_WORM_BLOOD,                SURFACE,    100,    25,     0),
    createDF(df.DF_ACID_SPLATTER,             SURFACE,    200,    25,     0),
    createDF(df.DF_ASH,                       SURFACE,    50,     25,     0),
    createDF(df.DF_EMBERS,                    SURFACE,    125,    25,     0),
    createDF(df.DF_ECTOPLASM,                 SURFACE,    110,    25,     0),
    createDF(df.DF_RUBBLE,                    SURFACE,    33,     25,     0),
    createDF(df.DF_ROT_GAS,                   GAS,        12,     0,      0),

    -- monstedf.DF_r effects
    createDF(df.DF_VOMIT,                     SURFACE,    30,     10,     0),
    createDF(df.DF_POISON_GAS,                GAS,        2000,   0,      0),
    createDF(df.DF_GAS_EXPLOSION,             SURFACE,    350,    100,    0,  ""),
    createDF(df.DF_RED_BLOOD,                 SURFACE,    150,    30,     0),
    createDF(df.DF_FLAMEDANCER_FIRE,          SURFACE,    200,    75,     0),

    -- mutatidf.DF_on effects
    createDF(df.DF_GAS_EXPLOSION,             SURFACE,    350,    100,    0,  "The corpse detonates with terrifying force!"),
    createDF(df.DF_LICHEN,                    SURFACE,    70,     60,     0,  "Poisonous spores burst from the corpse!"),

    -- misc
    createDF(df.DF_NOTHING,                   GAS,        0,      0,      dff.DFF_EVACUATE_CREATURES_FIRST),
    createDF(df.DF_ROT_GAS,                   GAS,        15,     0,      0),
    createDF(df.DF_STEAM,                     GAS,        325,    0,      0),
    createDF(df.DF_STEAM,                     GAS,        15,     0,      0),
    createDF(df.DF_METHANE_GAS,               GAS,        2,      0,      0),
    createDF(df.DF_EMBERS,                    SURFACE,    0,      0,      0),
    createDF(df.DF_URINE,                     SURFACE,    65,     25,     0),
    createDF(df.DF_UNICORN_POOP,              SURFACE,    65,     40,     0),
    createDF(df.DF_PUDDLE,                    SURFACE,    13,     25,     0),
    createDF(df.DF_ASH,                       SURFACE,    0,      0,      0),
    createDF(df.DF_ECTOPLASM,                 SURFACE,    0,      0,      0),
    createDF(df.DF_FORCEFIELD,                SURFACE,    100,    50,     0),
    createDF(df.DF_FORCEFIELD_MELT,           SURFACE,    0,      0,      0),
    createDF(df.DF_SACRED_GLYPH,              SURFACE,    100,    100,    0,  ""),
    createDF(df.DF_LICHEN,                    SURFACE,    2,      100,    (dff.DFF_BLOCKED_BY_OTHER_LAYERS)), -- Lichen won't spread through lava.
    createDF(df.DF_RUBBLE,                    SURFACE,    45,     23,     (dff.DFF_ACTIVATE_DORMANT_MONSTER)),
    createDF(df.DF_RUBBLE,                    SURFACE,    0,      0,      (dff.DFF_ACTIVATE_DORMANT_MONSTER)),

    createDF(df.DF_SPIDERWEB,                 SURFACE,    15,     12,     0),
    createDF(df.DF_SPIDERWEB,                 SURFACE,    100,    39,     0),

    createDF(df.DF_ANCIENT_SPIRIT_VINES,      SURFACE,    75,     70,     0),
    createDF(df.DF_ANCIENT_SPIRIT_GRASS,      SURFACE,    50,     47,     0),

    -- foliage
    createDF(df.DF_TRAMPLED_FOLIAGE,          SURFACE,    0,      0,      0),
    createDF(df.DF_DEAD_GRASS,                SURFACE,    75,     75,     0),
    createDF(df.DF_FOLIAGE,                   SURFACE,    0,      0,      (dff.DFF_BLOCKED_BY_OTHER_LAYERS)),
    createDF(df.DF_TRAMPLED_FUNGUS_FOREST,    SURFACE,    0,      0,      0),
    createDF(df.DF_FUNGUS_FOREST,             SURFACE,    0,      0,      (dff.DFF_BLOCKED_BY_OTHER_LAYERS)),

    -- brimstoneDF_
    createDF(df.DF_ACTIVE_BRIMSTONE,          LIQUID,     0,      0,      0),
    createDF(df.DF_INERT_BRIMSTONE,           LIQUID,     0,      0,      0,  "", 0,  0,  df.DF_BRIMSTONE_FIRE),

    -- bloodwortDF_
    createDF(df.DF_BLOODFLOWER_POD,           SURFACE,    60,     60,     dff.DFF_EVACUATE_CREATURES_FIRST),
    createDF(df.DF_BLOODFLOWER_POD,           SURFACE,    10,     10,     dff.DFF_EVACUATE_CREATURES_FIRST),
    createDF(df.DF_HEALING_CLOUD,             GAS,        350,    0,      0),

    -- dewars
    createDF(df.DF_POISON_GAS,                GAS,        20000,  0,      0, "the dewar shatters and pressurized caustic gas explodes outward!", 4, 0, df.DF_DEWAR_GLASS),
    createDF(df.DF_CONFUSION_GAS,             GAS,        20000,  0,      0, "the dewar shatters and pressurized confusion gas explodes outward!", 4, 0, df.DF_DEWAR_GLASS),
    createDF(df.DF_PARALYSIS_GAS,             GAS,        20000,  0,      0, "the dewar shatters and pressurized paralytic gas explodes outward!", 4, 0, df.DF_DEWAR_GLASS),
    createDF(df.DF_METHANE_GAS,               GAS,        20000,  0,      0, "the dewar shatters and pressurized methane gas explodes outward!", 4, 0, df.DF_DEWAR_GLASS),
    createDF(df.DF_BROKEN_GLASS,              SURFACE,    100,    70,     0),
    createDF(df.DF_CARPET,                    DUNGEON,    120,    20,     0),

    -- algae
    createDF(df.DF_DEEP_WATER_ALGAE_WELL,     DUNGEON,    0,      0,      dff.DFF_SUPERPRIORITY),
    createDF(df.DF_DEEP_WATER_ALGAE_1,        LIQUID,     50,     100,    0,  "",  0,     tt.DEEP_WATER, df.DF_ALGAE_2),
    createDF(df.DF_DEEP_WATER_ALGAE_2,        LIQUID,     0,      0,      0),
    createDF(df.DF_DEEP_WATER,                LIQUID,     0,      0,      dff.DFF_SUPERPRIORITY),

    -- doors, itDF_em cages, altars, glyphs, guardians -- reusable machine components
    createDF(df.DF_OPEN_DOOR,                 DUNGEON,    0,      0,      0),
    createDF(df.DF_DOOR,                      DUNGEON,    0,      0,      0),
    createDF(df.DF_OPEN_IRON_DOOR_INERT,      DUNGEON,    0,      0,      0,  ""),
    createDF(df.DF_ALTAR_CAGE_OPEN,           DUNGEON,    0,      0,      0,  "the cages lift off of the altars as you approach."),
    createDF(df.DF_ALTAR_CAGE_CLOSED,         DUNGEON,    0,      0,      (DFF_EVACUATE_CREATURES_FIRST), "the cages lower to cover the altars."),
    createDF(df.DF_ALTAR_INERT,               DUNGEON,    0,      0,      0),
    createDF(df.DF_FLOOR_FLOODABLE,           DUNGEON,    0,      0,      0,  "the altar retracts into the ground with a grinding sound."),
    createDF(df.DF_PORTAL_LIGHT,              SURFACE,    0,      0,      bor(dff.DFF_EVACUATE_CREATURES_FIRST, dff.DFF_ACTIVATE_DORMANT_MONSTER), "the archway flashes, and you catch a glimpse of another world!"),
    createDF(df.DF_MACHINE_GLYPH_INACTIVE,    DUNGEON,    0,      0,      0),
    createDF(df.DF_MACHINE_GLYPH,             DUNGEON,    0,      0,      0),
    createDF(df.DF_GUARDIAN_GLOW,             SURFACE,    0,      0,      0,  ""),
    createDF(df.DF_GUARDIAN_GLOW,             SURFACE,    0,      0,      0,  "the glyph beneath you glows, and the guardians take a step!"),
    createDF(df.DF_GUARDIAN_GLOW,             SURFACE,    0,      0,      0,  "the mirrored totem flashes, reflecting the red glow of the glyph beneath you."),
    createDF(df.DF_MACHINE_GLYPH,             DUNGEON,    200,    95,     dff.DFF_BLOCKED_BY_OTHER_LAYERS),
    createDF(df.DF_WALL_LEVER,                DUNGEON,    0,      0,      0,  "you notice a lever hidden behind a loose stone in the wall."),
    createDF(df.DF_WALL_LEVER_PULLED,         DUNGEON,    0,      0,      0),
    createDF(df.DF_WALL_LEVER_HIDDEN,         DUNGEON,    0,      0,      0),

    createDF(df.DF_BRIDGE_FALLING,            LIQUID,     200,    100,    0, "", 0, tt.BRIDGE),
    createDF(df.DF_CHASM,                     LIQUID,     0,      0,      0, "", 0, 0, df.DF_BRIDGE_FALL_PREP),

    -- fire
    createDF(df.DF_PLAIN_FIRE,                SURFACE,    0,      0,      0),
    createDF(df.DF_GAS_FIRE,                  SURFACE,    0,      0,      0),
    createDF(df.DF_GAS_EXPLOSION,             SURFACE,    60,     17,     0),
    createDF(df.DF_DART_EXPLOSION,            SURFACE,    0,      0,      0),
    createDF(df.DF_BRIMSTONE_FIRE,            SURFACE,    0,      0,      0),
    createDF(0,                               0,          0,      0,      0,  "the rope bridge snaps from the heat and plunges into the chasm!",  0,      0, df.DF_BRIDGE_FALL),
    createDF(df.DF_PLAIN_FIRE,                SURFACE,    100,    37,     0),
    createDF(df.DF_EMBERS,                    SURFACE,    0,      0,      0),
    createDF(df.DF_EMBERS,                    SURFACE,    100,    94,     0),
    createDF(df.DF_OBSIDIAN,                  SURFACE,    0,      0,      0),
    createDF(df.DF_ITEM_FIRE,                 SURFACE,    0,      0,      0,  ""),
    createDF(df.DF_CREATURE_FIRE,             SURFACE,    0,      0,      0,  ""),

    -- ice effecDF_ts
    createDF(df.DF_ICE_DEEP,                  LIQUID,     150,    50,     dff.DFF_EVACUATE_CREATURES_FIRST,   "",  0,      tt.DEEP_WATER,         df.DF_ALGAE_1_FREEZE),
    createDF(df.DF_ICE_DEEP,                  LIQUID,     150,    50,     dff.DFF_EVACUATE_CREATURES_FIRST,   "",  0,      tt.DEEP_WATER_ALGAE_1, df.DF_ALGAE_2_FREEZE),
    createDF(df.DF_ICE_DEEP,                  LIQUID,     150,    50,     dff.DFF_EVACUATE_CREATURES_FIRST,   "",  0,      tt.DEEP_WATER_ALGAE_2, df.DF_SHALLOW_WATER_FREEZE),
    createDF(df.DF_ICE_DEEP_MELT,             LIQUID,     0,      0,      0),
    createDF(df.DF_DEEP_WATER,                LIQUID,     0,      0,      0),
    createDF(df.DF_ICE_SHALLOW,               LIQUID,     100,    50,     dff.DFF_EVACUATE_CREATURES_FIRST,   "",  0,      tt.SHALLOW_WATER),
    createDF(df.DF_ICE_SHALLOW_MELT,          LIQUID,     0,      0,      0),
    createDF(df.DF_SHALLOW_WATER,             LIQUID,     0,      0,      0),

    -- gas trap DF_effects
    createDF(df.DF_POISON_GAS,                GAS,        1000,   0,      0,  "a cloud of caustic gas sprays upward from the floor!"),
    createDF(df.DF_CONFUSION_GAS,             GAS,        300,    0,      0,  "a sparkling cloud of confusion gas sprays upward from the floor!"),
    createDF(df.DF_NETTING,                   SURFACE,    300,    90,     0,  "a net falls from the ceiling!"),
    createDF(0,                               0,          0,      0,      dff.DFF_AGGRAVATES_MONSTERS, "a piercing shriek echoes through the nearby rooms!", 0, 0, DCOLS/2),
    createDF(df.DF_METHANE_GAS,               GAS,        10000,  0,      0), -- debugging toy

    -- potions
    createDF(df.DF_POISON_GAS,                GAS,        1000,   0,      0,  "", 4),
    createDF(df.DF_PARALYSIS_GAS,             GAS,        1000,   0,      0,  "", 4),
    createDF(df.DF_CONFUSION_GAS,             GAS,        1000,   0,      0,  "", 4),
    createDF(df.DF_PLAIN_FIRE,                SURFACE,    100,    37,     0,  ""),
    createDF(df.DF_DARKNESS_CLOUD,            GAS,        200,    0,      0),
    createDF(df.DF_HOLE_EDGE,                 SURFACE,    300,    100,    0,  "", 3,0,          df.DF_HOLE_2),
    createDF(df.DF_LICHEN,                    SURFACE,    70,     60,     0),

    -- other iteDF_ms
    createDF(df.DF_PLAIN_FIRE,                SURFACE,    100,    45,     0,  "", 3),
    createDF(df.DF_HOLE_GLOW,                 SURFACE,    200,    100,    dff.DFF_SUBSEQ_EVERYWHERE,  "", 3,0, df.DF_STAFF_HOLE_EDGE),
    createDF(df.DF_HOLE_EDGE,                 SURFACE,    100,    100,    0),

    -- machine cDF_omponents

    -- commutatiDF_on altars
    createDF(df.DF_COMMUTATION_ALTAR_INERT,   DUNGEON,    0,      0,      0,  "the items on the two altars flash with a brilliant light!"),
    createDF(df.DF_PIPE_GLOWING,              SURFACE,    90,     60,     0),
    createDF(df.DF_PIPE_INERT,                SURFACE,    0,      0,      0,  ""),

    -- resurrdf.DF_ection altars
    createDF(df.DF_RESURRECTION_ALTAR_INERT,  DUNGEON,    0,      0,      dff.DFF_RESURRECT_ALLY, "An old friend emerges from a bloom of sacred light!", EMPOWERMENT_LIGHT),
    createDF(df.DF_MACHINE_TRIGGER_FLOOR_REPEATING, LIQUID, 300,  100,    dff.DFF_SUPERPRIORITY, "", 0, tt.CARPET),

    -- sacrifdf.DF_ice altars
    createDF(df.DF_SACRIFICE_ALTAR,           DUNGEON,    0,      0,      0,  "a demonic presence whispers its demand: \"Bring to me the marked sacrifice!\""),
    createDF(df.DF_SACRIFICE_LAVA,            DUNGEON,    0,      0,      0,  "demonic cackling echoes through the room as the altar plunges downward!"),
    createDF(df.DF_ALTAR_CAGE_RETRACTABLE,    DUNGEON,    0,      0,      0),

    -- coffindf.DF_ bursts open to reveal vampire:
    createDF(df.DF_COFFIN_OPEN,               DUNGEON,    0,      0,      dff.DFF_ACTIVATE_DORMANT_MONSTER,   "the coffin opens and a dark figure rises!", 3),
    createDF(df.DF_PLAIN_FIRE,                SURFACE,    0,      0,      dff.DFF_ACTIVATE_DORMANT_MONSTER,   "as flames begin to lick the coffin, its tenant bursts forth!", 0, 0, df.DF_EMBERS_PATCH),
    createDF(df.DF_MACHINE_TRIGGER_FLOOR,     DUNGEON,    200,    100,    0),

    -- throwidf.DF_ng tutorial:
    createDF(df.DF_ALTAR_INERT,               DUNGEON,    0,      0,      0,  "the cage lifts off of the altar."),
    createDF(df.DF_TRAP_DOOR,                 LIQUID,     225,    100,    bor(dff.DFF_CLEAR_OTHER_TERRAIN, dff.DFF_SUBSEQ_EVERYWHERE), "", 0, 0, DF_SHOW_TRAPDOOR_HALO),
    createDF(df.DF_LAVA,                      LIQUID,     225,    100,    (dff.DFF_CLEAR_OTHER_TERRAIN)),
    createDF(df.DF_MACHINE_PRESSURE_PLATE_USED,DUNGEON,   0,      0,      0),

    -- rat trdf.DF_ap:
    createDF(df.DF_RAT_TRAP_WALL_CRACKING,    DUNGEON,    0,      0,      0,  "a scratching sound emanates from the nearby walls!",  0, 0, df.DF_RUBBLE),

    -- woodendf.DF_ barricade at entrance:
    createDF(df.DF_PLAIN_FIRE,                SURFACE,    0,      0,      0,  "flames quickly consume the wooden barricade."),

    -- woodendf.DF_ barricade around altar:
    createDF(df.DF_WOODEN_BARRICADE,          DUNGEON,    220,    100,    bor(dff.DFF_TREAT_AS_BLOCKING, dff.DFF_SUBSEQ_EVERYWHERE), "", 0, 0, df.DF_SMALL_DEAD_GRASS),

    -- here
    -- shallodf.DF_w water flood machine:
    createDF(df.DF_MACHINE_FLOOD_WATER_SPREADING, LIQUID, 0,      0,      0,  "you hear a heavy click, and the nearby water begins flooding the area!"),
    createDF(df.DF_SHALLOW_WATER,             LIQUID,     0,      0,      0),
    createDF(df.DF_MACHINE_FLOOD_WATER_SPREADING,LIQUID,  100,    100,    0,  "",  0,      tt.FLOOR_FLOODABLE,            df.DF_SHALLOW_WATER),
    createDF(df.DF_MACHINE_FLOOD_WATER_DORMANT,LIQUID,    250,    100,    (dff.DFF_TREAT_AS_BLOCKING), "", 0, 0,            df.DF_SPREADABLE_DEEP_WATER_POOL),
    createDF(df.DF_DEEP_WATER,                LIQUID,     90,     100,    bor(dff.DFF_CLEAR_OTHER_TERRAIN, dff.DFF_PERMIT_BLOCKING)),

    -- unstabdf.DF_le floor machine:
    createDF(df.DF_MACHINE_COLLAPSE_EDGE_SPREADING,LIQUID,0,      0,      0,  "you hear a deep rumbling noise as the floor begins to collapse!"),
    createDF(df.DF_CHASM,                     LIQUID,     0,      0,      dff.DFF_CLEAR_OTHER_TERRAIN, "", 0, 0, df.DF_SHOW_TRAPDOOR_HALO),
    createDF(df.DF_MACHINE_COLLAPSE_EDGE_SPREADING,LIQUID,100,    100,    0,  "", 0,  tt.FLOOR_FLOODABLE,    df.DF_COLLAPSE),
    createDF(df.DF_MACHINE_COLLAPSE_EDGE_DORMANT,LIQUID,  0,      0,      0),

    -- levitadf.DF_tion bridge machine:
    createDF(df.DF_CHASM_WITH_HIDDEN_BRIDGE_ACTIVE,LIQUID,100,    100,    0,  "",  0,  tt.CHASM_WITH_HIDDEN_BRIDGE,  df.DF_BRIDGE_APPEARS),
    createDF(df.DF_CHASM_WITH_HIDDEN_BRIDGE_ACTIVE,LIQUID,100,    100,    0,  "a stone bridge extends from the floor with a grinding sound.", 0,  tt.CHASM_WITH_HIDDEN_BRIDGE,  df.DF_BRIDGE_APPEARS),
    createDF(df.DF_STONE_BRIDGE,              LIQUID,     0,      0,      0),
    createDF(df.DF_MACHINE_CHASM_EDGE,        LIQUID,     100,    100,    0),

    -- retracdf.DF_ting lava pool:
    createDF(df.DF_LAVA_RETRACTABLE,          LIQUID,     100,    100,    0,  "",  0,  tt.LAVA),
    createDF(df.DF_LAVA_RETRACTING,           LIQUID,     0,      0,      0,  "hissing fills the air as the lava begins to cool."),
    createDF(df.DF_OBSIDIAN,                  SURFACE,    0,      0,      0,  "", 0,      0,         df.DF_STEAM_ACCUMULATION),

    -- hiddendf.DF_ poison vent machine:
    createDF(df.DF_MACHINE_POISON_GAS_VENT_DORMANT,DUNGEON,0,     0,      0,  "you notice an inactive gas vent hidden in a crevice of the floor."),
    createDF(df.DF_MACHINE_POISON_GAS_VENT,   DUNGEON,    0,      0,      0,  "deadly purple gas starts wafting out of hidden vents in the floor!"),
    createDF(df.DF_PORTCULLIS_CLOSED,         DUNGEON,    0,      0,      DFF_EVACUATE_CREATURES_FIRST,   "with a heavy mechanical sound, an iron portcullis falls from the ceiling!", GENERIC_FLASH_LIGHT),
    createDF(df.DF_PORTCULLIS_DORMANT,        DUNGEON,    0,      0,      0,  "the portcullis slowly rises from the ground into a slot in the ceiling."),
    createDF(df.DF_POISON_GAS,                GAS,        25,     0,      0),

    -- hiddendf.DF_ methane vent machine:
    createDF(df.DF_MACHINE_METHANE_VENT_DORMANT,DUNGEON,0,        0,      0,  "you notice an inactive gas vent hidden in a crevice of the floor."),
    createDF(df.DF_MACHINE_METHANE_VENT,      DUNGEON,    0,      0,      0,  "explosive methane gas starts wafting out of hidden vents in the floor!", 0, 0, df.DF_VENT_SPEW_METHANE),
    createDF(df.DF_METHANE_GAS,               GAS,        60,     0,      0),
    createDF(df.DF_PILOT_LIGHT,               DUNGEON,    0,      0,      0,  "a torch falls from its mount and lies sputtering on the floor."),

    -- paralydf.DF_sis trap:
    createDF(df.DF_MACHINE_PARALYSIS_VENT,    DUNGEON,    0,      0,      0,  "you notice an inactive gas vent hidden in a crevice of the floor."),
    createDF(df.DF_PARALYSIS_GAS,             GAS,        350,    0,      0,  "paralytic gas sprays upward from hidden vents in the floor!", 0, 0, df.DF_REVEAL_PARALYSIS_VENT_SILENTLY),
    createDF(df.DF_MACHINE_PARALYSIS_VENT,    DUNGEON,    0,      0,      0),

    -- thematdf.DF_ic dungeon:
    createDF(df.DF_RED_BLOOD,                 SURFACE,    75,     25,     0),

    -- statuadf.DF_ry:
    createDF(df.DF_STATUE_CRACKING,           DUNGEON,    0,      0,      0,  "cracks begin snaking across the marble surface of the statue!", 0, 0, df.DF_RUBBLE),
    createDF(df.DF_RUBBLE,                    SURFACE,    120,    100,    dff.DFF_ACTIVATE_DORMANT_MONSTER,   "the statue shatters!", 3, 0, df.DF_RUBBLE),

    -- hiddendf.DF_ turrets:
    createDF(df.DF_WALL,                      DUNGEON,    0,      0,      dff.DFF_ACTIVATE_DORMANT_MONSTER,   "you hear a click, and the stones in the wall shift to reveal turrets!", 0, 0, df.DF_RUBBLE),

    -- worm tunnDF_els:
    createDF(df.DF_WORM_TUNNEL_MARKER_DORMANT,LIQUID,     5,      5,      0,  "", 0,  0,  tt.GRANITE),
    createDF(df.DF_WORM_TUNNEL_MARKER_ACTIVE, LIQUID,     0,      0,      0),
    createDF(df.DF_FLOOR,                     DUNGEON,    0,      0,      bor(dff.DFF_SUPERPRIORITY, dff.DFF_ACTIVATE_DORMANT_MONSTER),  "", 0, 0,  0,  0,  df.DF_TUNNELIZE),
    createDF(df.DF_FLOOR,                     DUNGEON,    0,      0,      0,  "the nearby wall cracks and collapses in a cloud of dust!", 5,  0,  DF_TUNNELIZE),

    -- haunted rDF_oom:
    createDF(df.DF_DARK_FLOOR_DARKENING,      DUNGEON,    0,      0,      0,  "the light in the room flickers and you feel a chill in the air."),
    createDF(df.DF_DARK_FLOOR,                DUNGEON,    0,      0,      dff.DFF_ACTIVATE_DORMANT_MONSTER,   "", 0, 0, df.DF_ECTOPLASM_DROPLET),
    createDF(df.DF_HAUNTED_TORCH_TRANSITIONING,DUNGEON,   0,      0,      0),
    createDF(df.DF_HAUNTED_TORCH,             DUNGEON,    0,      0,      0),

    -- mud pit:
    createDF(df.DF_MACHINE_MUD_DORMANT,       LIQUID,     100,    100,    0),
    createDF(df.DF_MUD,                       LIQUID,     0,      0,      dff.DFF_ACTIVATE_DORMANT_MONSTER,   "across the bog, bubbles rise ominously from the mud."),

    -- electric DF_crystals:
    createDF(df.DF_ELECTRIC_CRYSTAL_ON,       DUNGEON,    0,      0,      0, "the crystal absorbs the electricity and begins to glow."),
    createDF(df.DF_WALL,                      DUNGEON,    0,      0,      dff.DFF_ACTIVATE_DORMANT_MONSTER,   "the wall above the lever shifts to reveal a spark turret!"),

    -- idyll:
    createDF(df.DF_SHALLOW_WATER,             LIQUID,     150,    100,    (dff.DFF_PERMIT_BLOCKING)),
    createDF(df.DF_DEEP_WATER,                LIQUID,     90,     100,    bor(bor(dff.DFF_TREAT_AS_BLOCKING, dff.DFF_CLEAR_OTHER_TERRAIN),  dff.DFF_SUBSEQ_EVERYWHERE), "", 0, 0, DF_SHALLOW_WATER_POOL),

    -- swamp:df.DF_
    createDF(df.DF_SHALLOW_WATER,             LIQUID,     30,     100,    0),
    createDF(df.DF_GRAY_FUNGUS,               SURFACE,    80,     50,     0,  "", 0, 0, df.DF_SWAMP_MUD),
    createDF(df.DF_MUD,                       LIQUID,     75,     5,      0,  "", 0, 0, df.DF_SWAMP_WATER),

    -- camp:
    createDF(df.DF_HAY,                       SURFACE,    90,     87,     0),
    createDF(df.DF_JUNK,                      SURFACE,    20,     20,     0),

    -- remnants:DF_
    createDF(df.DF_CARPET,                    DUNGEON,    110,    20,     dff.DFF_SUBSEQ_EVERYWHERE,  "", 0, 0, df.DF_REMNANT_ASH),
    createDF(df.DF_BURNED_CARPET,             SURFACE,    120,    100,    0),

    -- chasm catDF_walk:
    createDF(df.DF_CHASM,                     LIQUID,     0,      0,      dff.DFF_CLEAR_OTHER_TERRAIN, "", 0, 0, df.DF_SHOW_TRAPDOOR_HALO),
    createDF(df.DF_STONE_BRIDGE,              LIQUID,     0,      0,      dff.DFF_CLEAR_OTHER_TERRAIN),

    -- lake catwDF_alk:
    createDF(df.DF_DEEP_WATER,                LIQUID,     0,      0,      dff.DFF_CLEAR_OTHER_TERRAIN, "", 0, 0, df.DF_LAKE_HALO),
    createDF(df.DF_SHALLOW_WATER,             LIQUID,     160,    100,    0),

    -- worms df.DF_pop out of walls:
    createDF(df.DF_RUBBLE,                    SURFACE,    120,    100,    dff.DFF_ACTIVATE_DORMANT_MONSTER,   "the nearby wall explodes in a shower of stone fragments!", 3, 0, df.DF_RUBBLE),

    -- monster cDF_ages open:
    createDF(df.DF_MONSTER_CAGE_OPEN,         DUNGEON,    0,      0,      0),

    -- goblin waDF_rren:
    createDF(df.DF_STENCH_SMOKE_GAS,          GAS,        50,     0,      0, "",  0, 0, df.DF_PLAIN_FIRE),
    createDF(df.DF_STENCH_SMOKE_GAS,          GAS,        50,     0,      0, "",  0, 0, df.DF_EMBERS),
}

-- FLOOD TRAP
dungeonFeatureCatalog[df.DF_FLOOD] = createDF(
  tt.FLOOD_WATER_SHALLOW,       SURFACE,    225,    37,     0,  "",  0,      nil,          df.DF_FLOOD_2
)

dungeonFeatureCatalog[df.DF_FLOOD_2] = createDF(tt.FLOOD_WATER_DEEP,          SURFACE,    175,    37,     0,  "the area is flooded as water rises through imperceptible holes in the ground.")
dungeonFeatureCatalog[df.DF_FLOOD_DRAIN] = createDF(tt.FLOOD_WATER_SHALLOW,       SURFACE,    10,     25,     0)
dungeonFeatureCatalog[df.DF_HOLE_2] = createDF(tt.HOLE,                      SURFACE,    200,    100,    0)
dungeonFeatureCatalog[df.DF_HOLE_DRAIN] = createDF(tt.HOLE_EDGE,                 SURFACE,    0,      0,      0)




--TODO need to addd in the dungeonFeatureCatalog

-- more math funcs

local function min(x, y)
  if x < y then
    return x
  else
    return y
  end
end

local function max(x, y)
  if x > y then
    return x
  else
    return y
  end
end

local function clamp(x, low, high)
  return min(high, max(x, low))
end

local function rand_percent(percent)
  return math.random(0,99) < clamp(percent, 0, 100)
end

local function rand_range(x,y)
  return math.random(x,y)
end

local abs = math.abs

local function distanceBetween(x1, y1, x2, y2)
  return max(abs(x1 - x2), abs(y1 - y2))
end

local function fillSequentialList(list, length)
  if list == nil then
    list = {}
  end
  for i=1, length do
    list[i] = i
  end
end

local function shuffleList(list)
  for i=1, #list do
    local r = rand_range(i, #list)
    if i~= r then
      local buf = list[r]
      list[r] = list[i]
      list[i] = buf
    end
  end
end


local function coordinatesAreInMap(x, y)
  return x > 0 and x <= DCOLS and y > 0 and y <= DROWS
end

local function cellHasTerrainType(pmap, x, y, terrain)
  return pmap[x][y].layers.dungeon == terrain or
         pmap[x][y].layers.liquid == terrain or
         pmap[x][y].layers.surface == terrain or
         pmap[x][y].layers.gas == terrain
end

local function terrainFlag(pmap, x, y)
  local d = pmap[x][y].layers.dungeon
  local l = pmap[x][y].layers.liquid
  local s = pmap[x][y].layers.surface

  local dfs = 0
  local lfs = 0
  local sfs = 0

  if d > 0 then
    dfs = tileCatalog[d].flags
  end
  if l > 0 then
    lfs = tileCatalog[l].flags
  end
  if s > 0 then
    sfs = tileCatalog[s].flags
  end

  return bor(dfs, bor(lfs, sfs))
end

local function cellHasTerrainFlag(pmap, x, y, flag)
  local tile = terrainFlag(pmap, x, y);
  -- print('d, flag, tile, result')
  -- print(d)
  --print(flag .. ' ' .. toBits(flag))
  --print(tile .. ' ' .. toBits(tile))
  --print(band(flag, tile))
  return band(flag, tile) > 0
end


--

-- probably should move these
--
local function createBlock ()
  return {
    layers = {
      -- defaults to GRANITE
      dungeon = tileType.GRANITE,
      liquid = tileType.NOTHING,
      gas = tileType.NOTHING,
      surface = tileType.NOTHING,
    },
    machineNumber = 0,
    rememberedTerrain = tileType.NOTHING,
    rememberedTerrainFlags = tileType.NOTHING,
    rememberedTMFlags = 0,
    rememberedCellFlags = 0,
    rememberedItemCategory = 0,
    rememberedItemKind = 0,
    rememberedItemQuantity = 0,
    rememberedItemOriginDepth = 0,
    flags = {
      inLoop = 0,
      isChokepoint = 0,
      isGateSite = 0,
      isInRoomMachine = 0,
      hasMonster = 0
    },
    volume = 0,
  };
end

local function allocPmap ()
  local pmap = {}
  for i=1, DCOLS do
    pmap[i] = {}
    for j=1, DROWS do
      pmap[i][j] = createBlock()
    end
  end
  return pmap
end



local function getQualifyingGridLocNear(loc, x, y, grid, deterministic)
  local candidateLocs = 0
  local k = 1
  while k < max(DROWS, DCOLS) and candidateLocs == 0 do
    for i=x-k, x+k do
      for j=y-k, y+k do
        if coordinatesAreInMap(i, j) and
           (i == x-k or i == x+k or j == y-k or j == y+k) and
           grid[i][j] > 0
         then
           candidateLocs = candidateLocs + 1
         end
      end
    end
    k = k+1
  end


  if candidateLocs == 0 then
    return false
  end

  local randIndex
  if deterministic then
    randIndex = 1 + candidateLocs / 2
  else
    randIndex = rand_range(1, candidateLocs)
  end

  --print('can locs: '..candidateLocs)
  --print('rand index: '..randIndex)

  for k = 1, max(DROWS, DCOLS) do
    for i=x-k, x+k do
      for j=y-k, y+k do
        if coordinatesAreInMap(i, j) and
           (i == x-k or i == x+k or j == y-k or j == y+k) and
           grid[i][j] > 0 then
           randIndex = randIndex - 1
           if randIndex == 0 then
             loc.x = i
             loc.y = j
             return true
           end
         end
      end
    end
  end

  return false
end

local function findItem(x, y, items)
  for _, item in ipairs(items) do
    if x == item.x and y == item.y then
      return item
    end
  end
end

local function getItemChar(item)
  if item.category == 'potion' then
    return '!'
  elseif item.category == 'wand' then
    return '~'
  elseif item.category == 'food' then
    return '%'
  elseif item.category == 'staff' then
    return '/'
  elseif item.category == 'weapon' then
    return '↑'
  elseif item.category == 'gold' then
    return '*'
  elseif item.category == 'scroll' then
    return '𝅘𝅥𝅮'
  elseif item.category == 'armor' then
    return '['
  elseif item.category == 'charm' then
    return '🗲'
  elseif item.category == 'ring' then
    return 'o'
  end
  return ')'
end

local function printPmap(pmap, showNum, items, monsters)
  items = items or {}
  monsters = monsters or {}
  for j=1, DROWS do
    local row = "";
    for i=1, DCOLS do
      local item = findItem(i, j, items)
      local monster = findItem(i, j, monsters)
      local char = ' '
      if pmap[i][j].layers[DUNGEON] == tileType.GRANITE then
        char = '\27[40m\27[37mG'
      end
      if pmap[i][j].layers.dungeon == tileType.TORCH_WALL then
        char = '\27[103m\27[37m!'
      end
      if pmap[i][j].layers.dungeon == tileType.FLOOR then
        char = '\27[40m\27[40m '
      end
      if pmap[i][j].layers.dungeon == tileType.DOOR  then
        char = '\27[40m\27[33m+'
      end
      if pmap[i][j].layers.dungeon == tileType.SECRET_DOOR  then
        char = '\27[40m\27[90m+'
      end
      if pmap[i][j].layers.surface == df.DF_GRASS then
        char = '\27[40m\27[92m"'
      end
      if pmap[i][j].layers.surface == df.DF_DEAD_GRASS then
        char = '\27[40m\27[93m"'
      end
      if pmap[i][j].layers.surface == df.DF_FOLIAGE then
        char = '\27[40m\27[92mȹ'
      end
      if pmap[i][j].layers.surface == df.DF_DEAD_FOLIAGE then
        char = '\27[40m\27[93mȹ'
      end
      if pmap[i][j].layers.dungeon == tileType.NET_TRAP_HIDDEN then
        char = '\27[40m\27[37m◇'
      end
      if pmap[i][j].layers.dungeon == tileType.NET_TRAP then
        char = '\27[40m\27[37m◇'
      end
      if pmap[i][j].layers.dungeon == tileType.TRAP_DOOR_HIDDEN then
        char = '\27[40m\27[37m◇'
      end
      if pmap[i][j].layers.dungeon == tileType.TRAP_DOOR then
        char = '\27[40m\27[37m◇'
      end
      if pmap[i][j].layers.dungeon == tileType.FLAMETHROWER then
        char = '\27[40m\27[37m◇'
      end
      if pmap[i][j].layers.dungeon == tileType.FLOOD_TRAP then
        char = '\27[40m\27[37m◇'
      end
      if pmap[i][j].layers.liquid == tileType.DEEP_WATER  then
        char = '\27[44m\27[37m~'
      end
      if pmap[i][j].layers.liquid == tileType.SHALLOW_WATER  then
        char = '\27[44m\27[37m.'
      end
      if pmap[i][j].layers.liquid == tileType.CHASM  then
        char = '\27[40m\27[90m`'
      end
      if pmap[i][j].layers.liquid == tileType.CHASM_EDGE  then
        char = '\27[40m\27[90m,'
      end
      if pmap[i][j].layers.liquid == tileType.BRIDGE  then
        char = '\27[43m\27[97m='
      end
      if pmap[i][j].layers.surface == tileType.BRIDGE_EDGE  then
        char = '\27[43m\27[97m-'
      end
      if pmap[i][j].layers.liquid == tileType.LAVA  then
        char = '\27[41m\27[37m~'
      end
      if pmap[i][j].layers.liquid == tileType.INERT_BRIMSTONE  then
        char = '%'
      end
      if pmap[i][j].layers.liquid == tileType.OBSIDIAN  then
        char = '*'
      end
      if pmap[i][j].layers[DUNGEON] == tileType.DOWN_STAIRS then
        char = '\27[40m\27[93m>'
      end
      if pmap[i][j].layers[DUNGEON] == tileType.UP_STAIRS then
        char = '\27[40m\27[93m<'
      end
      -- Wall always wins
      if pmap[i][j].layers.dungeon == tileType.WALL then
        char = '\27[30m\27[47m#'
      end
      if pmap[i][j].layers.surface == df.DF_GRANITE_COLUMN then
        char = '\27[30m\27[47mC'
      end
      if pmap[i][j].layers.liquid == tileType.BRIDGE_EDGE  then
        char = '\27[43m\27[97m='
      end
      if pmap[i][j].layers[DUNGEON] == tileType.GRANITE then
        char = '\27[40m\27[37mG'
      end
      if pmap[i][j].layers.surface == tileType.FLOOD_WATER_SHALLOW then
        char = '\27[44m\27[37m.'
      end
      if pmap[i][j].layers.surface == tileType.FLOOD_WATER_DEEP then
        char = '\27[44m\27[37m~'
      end
      if item then
        local itemChar = getItemChar(item)
        char = '\27[40m\27[93m' .. itemChar
      end
      if monster then
        local itemChar = "M"
        char = '\27[40m\27[93m' .. itemChar
      end
      if showNum then
        row = row .. pmap[i][j].layers.surface
      else
        row = row .. char
      end
    end
    print(row ..'\27[0m')
  end
  return pmap
end


local function printGrid(grid, showNum)
  for j=1, DROWS do
    local row = "";
    for i=1, DCOLS do
      local char = '#'
      local val = grid[i][j]
      if grid[i][j] == 1 then
        char = '.'
      end
      if grid[i][j] >  1 then
        char = '+'
      end
      if showNum and val < 10 then
        row = row .. grid[i][j]
      elseif showNum and val >= 10 and val < 15 then
        row = row .. '-'
      elseif showNum and val >= 15 and val < 20 then
        row = row .. '+'
      elseif showNum and val >= 20 and val < 40 then
        row = row .. '*'
      elseif showNum and val >= 40 and val < 80 then
        row = row .. '%'
      elseif showNum and val >= 80 then
        row = row .. '#'
      else
        row = row .. char
      end
    end
    print(row)
  end
  return grid
end


local function drawRectangleOnGrid(grid, x, y, width, height, value)
  local i,j;
  for i=math.floor(x), x+width do
    for j=math.floor(y), y+height do
      grid[i][j] = value
    end
  end
end

function drawCircleOnGrid(grid, x, y, radius, value)
  local startX = max(0, x - radius - 1) + 1
  local endX = max(DCOLS, x + radius)
  local startY = max(0, y - radius - 1) + 1
  local endY = max(DROWS, y + radius)

  for i=startX, endX do
    for j=startY, endY do
      if ((i-x)*(i-x) + (j-y)*(j-y) < radius * radius + radius) then
        grid[i][j] = value
      end
    end
  end
end

local function cellularAutomataRound(grid, birthParams, survivalParams)
  local buffer2 = copyNewGrid(grid)
  local i, j, nbCount, newX, newY, dir

  for i=1, DCOLS do
    for j=1, DROWS do
      nbCount = 1
      for dir = 1, DIRECTION_COUNT do
        -- need to find nbDirs
        newX = i + nbDirs[dir][1];
        newY = j + nbDirs[dir][2];
        if coordinatesAreInMap(newX, newY) and buffer2[newX][newY] > 0 then
          nbCount = nbCount + 1
        end
      end

      if buffer2[i][j] == 0 and birthParams:sub(nbCount, nbCount) == 't' then
        grid[i][j] = 1 -- birth
      elseif buffer2[i][j] > 0 and survivalParams:sub(nbCount, nbCount) == 't' then
        -- survival
      else
        grid[i][j] = 0 -- death
      end
    end
  end
end

local function fillContiguousRegion(grid, x, y, fillValue)
  local newX, newY
  local numberOfCells = 1

  grid[x][y] = fillValue

  for dir = 1, 4 do
    newX = x + nbDirs[dir][1]
    newY = y + nbDirs[dir][2]
    if not coordinatesAreInMap(newX, newY) then
      break
    end
    if grid[newX][newY] == 1 then
      numberOfCells = numberOfCells + fillContiguousRegion(grid, newX, newY, fillValue)
    end
  end
  return numberOfCells
end

local function createBlobOnGrid(grid,
  roundCount,
  minBlobWidth, minBlobHeight,
  maxBlobWidth, maxBlobHeight,
  percentSeeded, birthParameters, survivalParameters)
  local loopNumber = 1
  local topBlobNumber = 1 -- there's a bug somewhere - need to set this to 1 since it'll never be 1, setting to 0 causes a bug
  local blobWidth = 0
  local blobHeight = 0
  local topBlobSize = 0
  local topBlobMinX = maxBlobWidth
  local topBlobMaxX = 0
  local topBlobMinY = maxBlobHeight
  local topBlobMaxY = 0

  repeat
    fillGrid(grid, 0)
    -- print('looking for cave')
    for i=2, maxBlobWidth do
      for j=2, maxBlobHeight do
        grid[i][j] = tern(rand_percent(percentSeeded), 1, 0)
      end
    end
    for k=1, roundCount do
      cellularAutomataRound(grid, birthParameters, survivalParameters)
    end

    local blobNumber = 2
    local blobSize = 0

    -- Fill each blob with its own number, starting with 2 (since 1 means floor), and keeping track of the biggest??
    for i=1, DCOLS do
      for j=1, DROWS do
        if grid[i][j] == 1 then
          blobSize = fillContiguousRegion(grid, i, j, blobNumber)
          if blobSize > topBlobSize then
            topBlobSize = blobSize
            topBlobNumber = blobNumber
          end
          blobNumber = blobNumber + 1
        end
      end
    end

    -- get top blob's height and width
    -- min and max x
    local foundACellThisLine = false
    for i=1, DCOLS do
      foundACellThisLine = false
      for j=1, DROWS do
        if grid[i][j] == topBlobNumber then
          foundACellThisLine = true
          break
        end
      end
      if foundACellThisLine then
        if i < topBlobMinX then
          topBlobMinX = i
        end
        if i > topBlobMaxX then
          topBlobMaxX = i
        end
      end
    end

    for j=1, DROWS do
      foundACellThisLine = false
      for i=1, DCOLS do
        if grid[i][j] == topBlobNumber then
          foundACellThisLine = true
          break
        end
      end
      if foundACellThisLine then
        if j < topBlobMinY then
          topBlobMinY = j
        end
        if j > topBlobMaxY then
          topBlobMaxY = j
        end
      end
    end

    blobWidth = (topBlobMaxX - topBlobMinX) + 2
    blobHeight = (topBlobMaxY - topBlobMinY) + 2

  until (blobWidth >= minBlobWidth and
         blobHeight >= minBlobHeight and
         topBlobNumber > 0)

  for i=1, DCOLS do
    for j=0, DROWS do
      if grid[i][j] == topBlobNumber then
        grid[i][j] = 1;
      else
        grid[i][j] = 0;
      end
    end
  end

  return {
    minX = topBlobMinX,
    minY = topBlobMinY,
    width = blobWidth,
    height = blobHeight
  }
end

local function validLocationCount(grid, validValue)
  local count = 0
  for i = 1, DCOLS do
    for j = 1, DROWS do
      if grid[i][j] == validValue then
        count = count + 1
      end
    end
  end
  return count;
end

local function randomLocationInGrid(grid, validValue)
  local locationCount = validLocationCount(grid, validValue)
  if locationCount <= 0 then
    return {
      x = -1,
      y = -1
    }
  end
  local index = rand_range(0, locationCount - 1)
  for i = 1, DCOLS do
    for j = 1, DROWS do
      if grid[i][j] == validValue then
        if index == 0 then
          return {
            x = i,
            y = j
          }
        end
        index = index - 1
      end
    end
  end
end



-- end grid functions
--


local function adjustProfileForDepth (profile, depth)
  local descentPercent = clamp(100 * (depth - 1) / (AMULET_LEVEL - 1), 0, 100)
  return {
    roomFrequencies = {
      profile.roomFrequencies[1] + 20 * (100 - descentPercent) / 100,
      profile.roomFrequencies[2] + 10 * (100 - descentPercent) / 100,
      profile.roomFrequencies[3],
      profile.roomFrequencies[4] +  7 * (100 - descentPercent) / 100,
      profile.roomFrequencies[5],
      profile.roomFrequencies[6] +  10 * descentPercent / 100,
      profile.roomFrequencies[7],
      profile.roomFrequencies[8],
    },
    corridorChance = profile.corridorChance + 80 * (100 - descentPercent) / 100
  };
end

local function adjustProfileForDepthFirstRoom (profile, depth)
  local descentPercent = clamp(100 * (depth - 1) / (AMULET_LEVEL - 1), 0, 100)
  if depth == 1 then
    return {
      roomFrequencies = {0,0,0,0,0,0,0,1},
      corridorChance = profile.corridorChance
    };
  else
    return {
      roomFrequencies = {
        profile.roomFrequencies[1],
        profile.roomFrequencies[2],
        profile.roomFrequencies[3],
        profile.roomFrequencies[4],
        profile.roomFrequencies[5],
        profile.roomFrequencies[6],
        profile.roomFrequencies[7] + 50 * descentPercent / 100,
        profile.roomFrequencies[8],
      },
      corridorChance = profile.corridorChance
    }
  end
end

local function insertRoomAt (grid, room, roomToGridX, roomToGridY, xRoom, yRoom)
  local newX, newY;

  grid[xRoom + roomToGridX][yRoom + roomToGridY] = 1;

  for dir = 1, 4 do
    newX = xRoom + nbDirs[dir][1]
    newY = yRoom + nbDirs[dir][2]
    if coordinatesAreInMap(newX, newY) and
       room[newX][newY] > 0 and
       coordinatesAreInMap(newX + roomToGridX, newY + roomToGridY) and
       grid[newX + roomToGridX][newY + roomToGridY] == 0 then
      insertRoomAt(grid, room, roomToGridX, roomToGridY, newX, newY)
    end
  end
end

local function oppositeDirection(dir)
  if dir == UP then
    return DOWN
  elseif dir == DOWN then
    return UP
  elseif dir == LEFT then
    return RIGHT
  elseif dir == RIGHT then
    return LEFT
  elseif dir == UPRIGHT then
    return DOWNLEFT
  elseif dir == DOWNLEFT then
    return UPRIGHT
  elseif dir == DOWNRIGHT then
    return UPLEFT
  elseif dir == UPLEFT then
    return DOWNRIGHT
  else
    return NO_DIRECTION
  end
end

local function directionOfDoorSite(grid, x, y)
  if grid[x][y] > 0 then
    return NO_DIRECTION
  end
  local solutionDir = NO_DIRECTION
  for dir=1, 4 do
    local newX = x + nbDirs[dir][1]
    local newY = y + nbDirs[dir][2]
    local oppX = x - nbDirs[dir][1]
    local oppY = y - nbDirs[dir][2]
    if coordinatesAreInMap(oppX, oppY) and
       coordinatesAreInMap(newX, newY) and
       grid[oppX][oppY] == 1 then
      if solutionDir ~= NO_DIRECTION then
        return NO_DIRECTION
      end
      solutionDir = dir;
    end
  end
  return solutionDir
end

local function chooseRandomDoorSites(roomMap, doorSites)
  local grid = copyNewGrid(roomMap);
  for i = 1, DCOLS do
    for j = 1, DROWS do
      if grid[i][j] == 0 then
        local dir = directionOfDoorSite(roomMap, i, j)
        if dir ~= NO_DIRECTION then
          -- trace a ray - blabla
          local newX = i + nbDirs[dir][1]
          local newY = j + nbDirs[dir][2]
          local doorSiteFailed = false
          local k = 1
          while k <= 10 and coordinatesAreInMap(newX, newY) and doorSiteFailed == false do
            if grid[newX][newY] > 0 then
              doorSiteFailed = true
            end
            newX = newX + nbDirs[dir][1]
            newY = newY + nbDirs[dir][2]
            k = k + 1
          end
          if doorSiteFailed == false then
            grid[i][j] = dir + 1
          end
        end
      end
    end
  end
  for dir = 1, 4 do
    local pos = randomLocationInGrid(grid, dir + 1)
    doorSites[dir].x = pos.x
    doorSites[dir].y = pos.y
  end
end

local function designEntranceRoom(grid)
  --print("design entrance room")
  fillGrid(grid, 0)
  local roomWidth = 8
  local roomHeight = 10
  local roomWidth2 = 20
  local roomHeight2 = 5
  local roomX = DCOLS/2 - roomWidth/2 - 1
  local roomY = DROWS - roomHeight - 2
  local roomX2 = DCOLS/2 - roomWidth2/2 - 1
  local roomY2 = DROWS - roomHeight2 - 2

  drawRectangleOnGrid(grid, roomX, roomY, roomWidth, roomHeight, 1)
  drawRectangleOnGrid(grid, roomX2, roomY2, roomWidth2, roomHeight2, 1)
end

local function designCavern(grid, minWidth, maxWidth, minHeight, maxHeight)
  local blobGrid = allocGrid()
  fillGrid(grid, 0)

  local blobMeta = createBlobOnGrid(
    blobGrid, 5, minWidth, minHeight, maxWidth, maxHeight, 55,
    'ffffffttt', 'ffffttttt')

  local fillX = 1
  local fillY = 1
  local foundFillPoint = false

  for i=1, DCOLS do
    for j=1, DROWS do
      if blobGrid[i][j] > 0 and foundFillPoint == false then
        foundFillPoint = true
        fillX = i
        fillY = j
      end
    end
  end

  local destX = math.ceil((DCOLS - blobMeta.width + 2) / 2);
  local destY = math.ceil((DROWS - blobMeta.height + 2) /2);

  insertRoomAt(grid, blobGrid, destX - blobMeta.minX, destY - blobMeta.minY, fillX, fillY)
end

local function designCrossRoom(grid)
    local roomWidth, roomHeight, roomWidth2, roomHeight2, roomX, roomY, roomX2, roomY2;

    fillGrid(grid, 0);

    roomWidth = rand_range(3, 12);
    roomX = rand_range(max(0, DCOLS/2 - (roomWidth - 1)), min(DCOLS, DCOLS/2));
    roomWidth2 = rand_range(4, 20);
    roomX2 = (roomX + (roomWidth / 2) + rand_range(0, 2) + rand_range(0, 2) - 3) - (roomWidth2 / 2);

    roomHeight = rand_range(3, 7);
    roomY = (DROWS/2 - roomHeight);

    roomHeight2 = rand_range(2, 5);
    roomY2 = (DROWS/2 - roomHeight2 - (rand_range(0, 2) + rand_range(0, 1)));

    drawRectangleOnGrid(grid, roomX - 5, roomY + 5, roomWidth, roomHeight, 1);
    drawRectangleOnGrid(grid, roomX2 - 5, roomY2 + 5, roomWidth2, roomHeight2, 1);
end

local function designSymmetricalCrossRoom(grid)
    local majorWidth, majorHeight, minorWidth, minorHeight;

    fillGrid(grid, 0);

    majorWidth = rand_range(4, 8);
    majorHeight = rand_range(4, 5);

    minorWidth = rand_range(3, 4);
    if (majorHeight % 2 == 0) then
        minorWidth = minorWidth - 1;
    end
    minorHeight = 3;
    if (majorWidth % 2 == 0) then
        minorHeight = minorHeight - 1;
    end

    drawRectangleOnGrid(grid, (DCOLS - majorWidth)/2, (DROWS - minorHeight)/2, majorWidth, minorHeight, 1);
    drawRectangleOnGrid(grid, (DCOLS - minorWidth)/2, (DROWS - majorHeight)/2, minorWidth, majorHeight, 1);
end

local function designSmallRoom(grid)
    local width, height;

    fillGrid(grid, 0);
    width = rand_range(3, 6);
    height = rand_range(2, 4);
    drawRectangleOnGrid(grid, (DCOLS - width) / 2, (DROWS - height) / 2, width, height, 1);
end

local function designCircularRoom(grid)
  local radius;

  if (rand_percent(5)) then
    radius = rand_range(4, 10);
  else
    radius = rand_range(2, 4);
  end

  fillGrid(grid, 0);
  drawCircleOnGrid(grid, (DCOLS + 1)/2, (DROWS + 1)/2, radius, 1);

  if (radius > 6 and rand_percent(50)) then
    drawCircleOnGrid(grid, (DCOLS + 1)/2, (DROWS + 1)/2, rand_range(3, radius - 3), 0);
  end
end

local function designChunkyRoom(grid)
  local i, x, y;
  local minX, maxX, minY, maxY;
  local chunkCount = rand_range(2, 8);

  fillGrid(grid, 0);
  drawCircleOnGrid(grid, (DCOLS + 1)/2, (DROWS+1)/2, 2, 1);
  minX = (DCOLS + 1)/2 - 3;
  maxX = (DCOLS + 1)/2 + 3;
  minY = (DROWS + 1)/2 - 3;
  maxY = (DROWS + 1)/2 + 3;

  for i=1, chunkCount do
    x = rand_range(minX, maxX);
    y = rand_range(minY, maxY);
    if (grid[x][y]) then

      drawCircleOnGrid(grid, x, y, 2, 1);
      i = i + 1;
      minX = max(1, min(x - 3, minX));
      maxX = min(DCOLS - 2, max(x + 3, maxX));
      minY = max(1, min(y - 3, minY));
      maxY = min(DROWS - 2, max(y + 3, maxY));
    end
  end
end

local function attachHallwayTo(grid, doorSites)
  local dirs = {}
  fillSequentialList(dirs, 4)
  shuffleList(dirs)
  local hallDirIdx = 1
  local dir
  for i, _dir in ipairs(dirs) do
    dir = _dir
    local dx = doorSites[_dir].x
    local dy = doorSites[_dir].y
    if dx ~= -1 and
       dy ~= -1 and
       coordinatesAreInMap(
         dx + nbDirs[_dir][1] * HORIZONTAL_CORRIDOR_MAX_LENGTH,
         dy + nbDirs[_dir][2] * VERTICAL_CORRIDOR_MAX_LENGTH
       ) then
       break
     end
     hallDirIdx = hallDirIdx + 1
  end

  if hallDirIdx > 4 then
    print('no valid hall dir')
    return
  end
  local length = 1
  if dir == UP or dir == DOWN then
    length = rand_range(VERTICAL_CORRIDOR_MIN_LENGTH, VERTICAL_CORRIDOR_MAX_LENGTH)
  else
    length = rand_range(HORIZONTAL_CORRIDOR_MIN_LENGTH, HORIZONTAL_CORRIDOR_MAX_LENGTH)
  end

  local x = doorSites[dir].x
  local y = doorSites[dir].y
  for i = 1, length do
    if coordinatesAreInMap(x, y) then
      grid[x][y] = 1
    end
    x = x + nbDirs[dir][1]
    y = y + nbDirs[dir][2]
  end

  x = clamp(x - nbDirs[dir][1], 0, DCOLS)
  y = clamp(y - nbDirs[dir][2], 0, DROWS)

  local allowObliqueHallwayExit = rand_percent(15)
  for dir2 = 1, 4 do
    local newX = x + nbDirs[dir][1]
    local newY = y + nbDirs[dir][2]
    if (dir2 ~= dir and allowObliqueHallwayExit == false) or
       coordinatesAreInMap(newX, newY) == false or
       grid[newX][newY] > 0 then
       doorSites[dir2].x = -1
       doorSites[dir2].y = -1
     else
       doorSites[dir2].x = newX
       doorSites[dir2].y = newY
     end
  end
end

local function designRandomRoom (grid, attachHallway, doorSites, roomTypeFrequencies)
  local sum = 0
  for _, freq in ipairs(roomTypeFrequencies) do
    sum = sum + freq
  end

  local randIndex = math.random(0, sum - 1)

  local roomType = 0
  for i, freq in ipairs(roomTypeFrequencies) do
    if randIndex < freq then
      roomType = i
      break
    else
      randIndex = randIndex - freq
    end
  end

  -- print("design room type "..roomType)
  if roomType == 1 then
    designCrossRoom(grid)
  elseif roomType == 2 then
    designSymmetricalCrossRoom(grid)
  elseif roomType == 3 then
    designSmallRoom(grid)
  elseif roomType == 4 then
    designCircularRoom(grid)
  elseif roomType == 5 then
    designChunkyRoom(grid)
  elseif roomType == 6 then
    local r = rand_range(0,2);
    if r == 0 then
      designCavern(grid, 3, 12, 4, 8); -- Compact cave room.
    elseif r == 1 then
      designCavern(grid, 3, 12, 15, DROWS-2); -- Large north-south cave room.
    else
      designCavern(grid, 20, DROWS-2, 4, 8); --  Large east-west cave room.
    end
  elseif roomType == 7 then
    designCavern(grid, CAVE_MIN_WIDTH, DCOLS - 1, CAVE_MIN_HEIGHT, DROWS - 1)
  elseif roomType == 8 then
    designEntranceRoom(grid)
  end

  if doorSites ~= nil then
    chooseRandomDoorSites(grid, doorSites)
    if attachHallway then
      attachHallwayTo(grid, doorSites)
    end
  end
end

local function createPos()
  return {
    x = 0,
    y = 0
  }
end

local function roomFitsAt(dungeonMap, roomMap, deltaX, deltaY)
  --delta X = roomToDungeonX
  --delta Y = roomToDungeonY
  for xRoom = 1, DCOLS do
    for yRoom = 1, DROWS do
      if roomMap[xRoom][yRoom] > 0 then
        local xDungeon = xRoom + deltaX;
        local yDungeon = yRoom + deltaY;
        for i = xDungeon - 1, xDungeon + 1 do
          for j = yDungeon - 1, yDungeon + 1 do
            if coordinatesAreInMap(i, j) == false or dungeonMap[i][j] > 0 then
              return false
            end
          end
        end
      end
    end
  end
  return true
end

local function attachRooms(grid, theDP, attempts, maxRoomCount)
  local doorSites = {}
  local doorSites = {
    createPos(),
    createPos(),
    createPos(),
    createPos()
  }

  local sCoord = {}
  fillSequentialList(sCoord, DCOLS*DROWS)
  shuffleList(sCoord)

  local roomMap = allocGrid()
  local roomsBuilt = 0
  local roomsAttempted = 0
  while roomsBuilt < maxRoomCount and roomsAttempted < attempts do
    fillGrid(roomMap, 0)
    -- TODO still need to implement hallways
    local attachHallway = roomsAttempted <= attempts - 5 and rand_percent(theDP.corridorChance)
    designRandomRoom(roomMap, attachHallway, doorSites, theDP.roomFrequencies)

    --print("random room " .. roomsAttempted)

    for i=1, DCOLS * DROWS do
      local x = math.ceil(sCoord[i] / DROWS)
      local y = sCoord[i] % DROWS + 1
      local dir = directionOfDoorSite(grid, x, y)
      local oppDir = oppositeDirection(dir)

      if dir ~= NO_DIRECTION then
        local deltaX = x - doorSites[oppDir].x
        local deltaY = y - doorSites[oppDir].y
        local roomFit = roomFitsAt(grid, roomMap, deltaX, deltaY)
        local foo = doorSites[oppDir].x ~= -1
        if roomFit and foo then
          insertRoomAt(grid, roomMap,
            deltaX, deltaY,
            doorSites[oppDir].x,
            doorSites[oppDir].y)
          grid[x][y] = 2;
          roomsBuilt = roomsBuilt + 1
          break;
        end
      end
    end


    roomsAttempted = roomsAttempted + 1
  end

end

-- All the stuff for carve dungeon is above
-- The rest is for digDungeon
--
local function checkLoopiness(pmap, x, y)
  if pmap[x][y].flags.inLoop == 0 then
    return false
  end
  local c = 0
  for sdir = 1, 8 do
    local newX = x + cDirs[sdir][1]
    local newY = y + cDirs[sdir][2]
    if false == coordinatesAreInMap(newX, newY)
       or pmap[newX][newY].flags.inLoop == 0
    then
      break
    else
      c = c + 1
    end
  end
  if c == 8 then
    return false
  end

  local newX = x + cDirs[c+1][1]
  local newY = y + cDirs[c+1][2]

  local numStrings = 0
  local maxStringLength = 0
  local currentStringLength = 0
  local inString = false

  local dir = c + 1
  local max = dir + 8
  while dir < max do
    local newX = x + cDirs[((dir - 1) % 8) + 1][1]
    local newY = y + cDirs[((dir - 1) % 8) + 1][2]
    if coordinatesAreInMap(newX, newY)
      and pmap[newX][newY].flags.inLoop > 0
    then
      currentStringLength = currentStringLength + 1
      if inString == false then
        if numStrings > 0 then
          return false
        end
        numStrings = numStrings + 1
        inString = true
      end
    elseif inString then
      if currentStringLength > maxStringLength then maxStringLength = currentStringLength end
      currentStringLength = 0
      inString = false
    end
    dir = dir + 1
  end
  if inString and currentStringLength > maxStringLength then
    maxStringLength = currentStringLength
  end

  if numStrings == 1 and maxStringLength <= 4 then
    pmap[x][y].flags.inLoop = 0
    for dir=1, 8 do
      local newX = x + cDirs[dir][1]
      local newY = y + cDirs[dir][2]
      if coordinatesAreInMap(newX, newY) then
        checkLoopiness(pmap, newX, newY)
      end
    end
    return true
  end
  return false
end

local function auditLoop(pmap, x, y, grid)
  if coordinatesAreInMap(x, y)
    and grid[x][y] == 0
    and pmap[x][y].flags.inLoop == 0
  then
    grid[x][y] = 1
    for dir=1, 8 do
      local newX = x + nbDirs[dir][1]
      local newY = y + nbDirs[dir][2]
      if coordinatesAreInMap(newX, newY) then
        auditLoop(pmap, newX, newY, grid)
      end
    end
  end
end

local function analyzeMap(pmap, calculateChokeMap)
  local grid = allocGrid()
  local passMap = allocGrid()
  for i=1, DCOLS do
    for j=1, DROWS do
      if cellHasTerrainFlag(pmap, i, j, tf.T_PATHING_BLOCKER)
        -- and not cellhasTMFlag TM_IS_SECRET
      then
        pmap[i][j].flags.inLoop = 0
        passMap[i][j] = 0
      else
        pmap[i][j].flags.inLoop = 1
        passMap[i][j] = 1
      end
    end
  end

  for i=1, DCOLS do
    for j=1, DROWS do
      checkLoopiness(pmap, i, j)
    end
  end


  fillGrid(grid, 0)
  auditLoop(pmap, 1, 1, grid)

  for i=1, DCOLS do
    for j=1, DROWS do
      if pmap[i][j].flags.inLoop > 0 then
        local designationSurvives = false
        for dir=1, 8 do
          local newX = i + nbDirs[dir][1]
          local newY = j + nbDirs[dir][2]
          if coordinatesAreInMap(newX, newY)
            and grid[newX][newY] == 0
            and pmap[newX][newY].flags.inLoop == 0
          then
            designationSurvives = true
            break
          end
        end
        if false == designationSurvives then
          grid[i][j] = 1
          pmap[i][j].flags.inLoop = 0
        end
      end
    end
  end

  --find chokepoints and mark them
  for i=1, DCOLS do
    for j=1, DROWS do
      pmap[i][j].flags.isChokepoint = 0
      if passMap[i][j] and pmap[i][j].flags.inLoop == 0 then
        local passableArcCount = 0
        for dir = 1, 8 do
          local oldX = i + cDirs[((dir + 6) % 8 + 1)][1]
          local oldY = j + cDirs[((dir + 6) % 8 + 1)][2]
          local newX = i + cDirs[dir][1]
          local newY = j + cDirs[dir][2]
          if (coordinatesAreInMap(newX, newY) and passMap[newX][newY] > 0)
            ~= (coordinatesAreInMap(oldX, oldY) and passMap[oldX][oldY] > 0)
          then
            passableArcCount = passableArcCount + 1
            if passableArcCount > 2 then
              if (i > 1 and passMap[i-1][j] == 0 and i < DCOLS and passMap[i+1][j] == 0) or
                 (j > 1 and passMap[i][j-1] == 0 and j < DROWS and passMap[i][j+1] == 0)
              then
                pmap[i][j].flags.isChokepoint = 1
              end
              break
            end
          end
        end
      end
    end
  end

  --TODO calculateChokeMap
end

local function randomMatchingLocation(pmap, dungeonType, liquidType, terrainType)
  local failsafeCount = 1
  local x, y
  repeat
    failsafeCount = failsafeCount + 1
    -- TODO try the shuffle alo here to reduce dups?
    x = rand_range(1, DCOLS)
    y = rand_range(1, DROWS)
  until(failsafeCount >= 500 or (
       (terrainType >= 0 and cellHasTerrainType(pmap, x, y, terrainType))
       or (terrainType < 0 and ((dungeonType < 0 or pmap[x][y].layers.dungeon == dungeonType) and
                                 (liquidType < 0 or pmap[x][y].layers.liquid == liquidType))) ))
       --TODO need to figure this line out --
       --or (terrainType < 0 and band(tileCatalog[dungeonType].flags, tf.T_OBSTRUCTS_ITEMS) == 0 and false == cellHasTerrainFlag(pmap, x, y, tf.T_OBSTRUCTS_ITEMS))))

  if failsafeCount >= 500 then
    return {
      match = false,
      x = 0,
      y = 0,
    }
  end
  return {
    match = true,
    x = x,
    y = y
  }
end

local function cellIsPassableOrDoor (pmap, x, y)
  return false == cellHasTerrainFlag(pmap, x, y, tf.T_PATHING_BLOCKER) or
    pmap[x][y].layers.dungeon == tileType.DOOR or 
    pmap[x][y].layers.dungeon == tileType.SECRET_DOOR

  -- TODO add the rest of the code
  -- not steps
  -- not door
  -- Need to implement terrain mechanics for this to work
end

local function connectCell(pmap, x, y, zoneLabel, blockingMap, zoneMap)
  zoneMap[x][y] = zoneLabel
  local size = 1

  for dir=1, 4 do
    local newX = x + nbDirs[dir][1]
    local newY = y + nbDirs[dir][2]
    if coordinatesAreInMap(newX, newY)
      and zoneMap[newX][newY] == 0
      and (blockingMap == nil or blockingMap[newX][newY] == 0)
      and cellIsPassableOrDoor(pmap, newX, newY)
    then
      size = size + connectCell(pmap, newX, newY, zoneLabel, blockingMap, zoneMap)
    end
  end
  return size
end


-- returns 0 if not (in lua not the same as false)
local function levelIsDisconnectedWithBlockingMap(pmap, blockingMap, countRegionSize, debug)
  local zoneMap = allocGrid();
  local zoneCount = 0
  local zoneSizes = {}

  -- Map out the zones with the blocking area blocked
  for i=1, DCOLS do
    for j=1, DROWS do
      if cellIsPassableOrDoor(pmap, i, j)
        and zoneMap[i][j] == 0
        and blockingMap[i][j] == 0
      then
        for dir=1, 4 do
          local newX = i + nbDirs[dir][1]
          local newY = j + nbDirs[dir][2]
          if coordinatesAreInMap(newX, newY) and blockingMap and blockingMap[newX][newY] and blockingMap[newX][newY] > 0 then
            zoneCount = zoneCount + 1
            table.insert(zoneSizes, connectCell(pmap, i, j, zoneCount, blockingMap, zoneMap))
            break
          end
        end
      end
    end
  end

  if debug then
    print('blocking map: ')
    printGrid(blockingMap, true)

    print('zone map 1: ')
    printGrid(zoneMap, true)
  end

  -- Expand the zones into the blocking area
  for i=1, DCOLS do
    for j=1, DROWS do
      if blockingMap[i][j] > 0
        and zoneMap[i][j] == 0
        and cellIsPassableOrDoor(pmap, i, j)
      then
        for dir=1, 4 do
          local newX = i + nbDirs[dir][1]
          local newY = j + nbDirs[dir][2]
          local borderingZone = zoneMap[newX][newY]
          if borderingZone ~= 0 then
            connectCell(pmap, i, j, borderingZone, nil, zoneMap)
            break
          end
        end
      end
    end
  end

  if debug then
    print('zone map 2: ')
    printGrid(zoneMap, true)
  end

  local smallestQualifyingZoneSize = 10000

  -- Figure out which zones touch
  for i=1, DCOLS do
    for j=1, DROWS do
      if zoneMap[i][j] ~= 0 then
        for dir=1, 4 do
          local newX = i + nbDirs[dir][1]
          local newY = j + nbDirs[dir][2]
          local borderingZone = coordinatesAreInMap(newX, newY) and zoneMap[newX][newY]
          if borderingZone ~= false
            and zoneMap[i][j] ~= borderingZone
            and borderingZone ~= 0
          then
            if false == countRegionSize then
              print('  zones touch - zoneMap '..zoneMap[i][j]..' border zone: '..borderingZone)
              return 1
            end
            smallestQualifyingZoneSize = min(smallestQualifyingZoneSize, zoneSizes[zoneMap[i][j]])
            smallestQualifyingZoneSize = min(smallestQualifyingZoneSize, zoneSizes[borderingZone])
            break
          end
        end
      end
    end
  end

  if smallestQualifyingZoneSize < 10000 then
    return smallestQualifyingZoneSize
  end
  return 0
end

local function spawnMapDF(pmap, x, y, propagationTerrain, requirePropTerrain, startProb, probDec, spawnMap)
  local t = 1
  spawnMap[x][y] = t
  local madeChange = true
  local x2, y2
  while madeChange and startProb > 0 do
    madeChange = false
    t = t + 1
    for i=1, DCOLS do
      for j=1, DROWS do
        if spawnMap[i][j] == t - 1 then
          for dir = 1, 4 do
            x2 = i + nbDirs[dir][1]
            y2 = j + nbDirs[dir][2]
            if coordinatesAreInMap(x2, y2)
               and (requirePropTerrain == false or (propagationTerrain > 0 and cellHasTerrainType(pmap, x2, y2, propagationTerrain)))
               and (cellHasTerrainFlag(pmap, x2, y2, tf.T_OBSTRUCTS_SURFACE_EFFECTS) == false or (requirePropTerrain and cellHasTerrainType(pmap, x2, y2, propagationTerrain)))
               and rand_percent(startProb) then
              spawnMap[x2][y2] = t
              madeChange = true
            end
          end
        end
      end
    end

    startProb = startProb - probDec
    if t > 100 then
      for i=1, DCOLS do
        for j=1, DROWS do
          if spawnMap[i][j] == t then
            spawnMap[i][j] = 2
          elseif spawnMap[i][j] > 0 then
            spawnMap[i][j] = 1
          end
        end
      end
      t = 2
    end

  end
  if requirePropTerrain and cellHasTerrainType(pmap, x, y, propagationTerrain) == false then
    spawnMap[x][y] = 0
  end
end

local function fillSpawnMap(pmap, layer, surfaceTileType, spawnMap, blockedByOtherLayers, refresh, superpriority)
  local accomplishedSomething = false
  --print('fill layer: ' .. layer .. ' with ' .. surfaceTileType)
  --printGrid(spawnMap, true)

  for i = 1, DCOLS do
    for j = 1, DROWS do
      if spawnMap[i][j] > 0
        and pmap[i][j].layers[layer] ~= surfaceTileType
        and (superpriority or tileCatalog[pmap[i][j].layers[layer]].drawPriority >= tileCatalog[surfaceTileType].drawPriority)
        and false == (layer == SURFACE and cellHasTerrainFlag(pmap, i, j, tf.T_OBSTRUCTS_SURFACE_EFFECTS))
      then
        pmap[i][j].layers[layer] = surfaceTileType
        accomplishedSomething = true
      else
        spawnMap[i][j] = 0
      end
    end
  end
  return accomplishedSomething
end

local function spawnDungeonFeature(pmap, x, y, feat, refreshCell, abortIfBlocking)
  local succeeded = false
  local blockingMap = allocGrid();

  local blocking = abortIfBlocking and
    band(feat.flags, dff.DFF_PERMIT_BLOCKING) == 0 and
    ((feat.tile and band(tileCatalog[feat.tile].flags, tf.T_PATHING_BLOCKER) > 0) or
     band(feat.flags, dff.DFF_TREAT_AS_BLOCKING) > 0)

  --print('    feature tile: ' .. (feat.tile or 'nil'))

  if feat.tile then
    if feat.layer == GAS then
      pmap[x][y].volume = pmap[x][y].volume + feat.startProbability
      pmap[x][y].layers.gas = feat.tile
      --if refreshCall refreshDungeonCell(x,y)
      succeeded = true
      --print('    spawn GAS success')
    else
      --printTable(feat)
      spawnMapDF(pmap,
        x, y,
        feat.propagationTerrain,
        feat.propagationTerrain ~= nil ,
        feat.startProbability,
        feat.probabilityDecrement,
        blockingMap)

      local isDisconnected = 0
      --print('    DF blocking flag?: ' .. tern(blocking, 'true', 'false'))
      if blocking then
        isDisconnected = levelIsDisconnectedWithBlockingMap(pmap, blockingMap, false)
        --print('    Does DF disconnect level? '.. isDisconnected)
      end
      if blocking == false or isDisconnected == 0 then
        fillSpawnMap(
          pmap,
          feat.layer,
          feat.tile,
          blockingMap,
          band(feat.flags, dff.DFF_BLOCKED_BY_OTHER_LAYERS),
          refreshCell,
          band(feat.flags, dff.DFF_SUPERPRIORITY))
        succeeded = true
        --print('    spawn df success')
      end
      if succeeded and band(feat.flags, dff.DFF_CLEAR_OTHER_TERRAIN) then
        --print('    TODO: clear other terrain!')
      end
    end
  end
  return succeeded
end

local function createGenerator(depth, pmap, buildAreaMachines, gen)
  if depth < gen.minDepth or depth > gen.maxDepth then
    return
  end
  local count = math.ceil(min((gen.minNumberIntercept + depth * gen.minNumberSlope) / 100, gen.maxNumber))
  while rand_percent(gen.frequency) and count < gen.maxNumber do
    count = count + 1
  end
  -- print('try to spawn feature ' .. gen.DFType .. ' ' .. getKeyByValue(df, gen.DFType) .. ' or terrain: ' .. getKeyByValue(tt, gen.terrain) .. ' with count ' .. count)
  for _ = 1, count do
    local l = randomMatchingLocation(pmap, gen.requiredDungeonFoundationType, gen.requiredLiquidFoundationType, -1)
    if l.match then
      -- print('  found location match')
      if gen.DFType > 0 then
        -- print('  spawn dungeon feature at x ' .. l.x .. ', y: ' .. l.y)
        local feat = dungeonFeatureCatalog[gen.DFType]
        spawnDungeonFeature(pmap, l.x, l.y, feat, false, true)
        if feat.subsequentDF then
          local subDF = dungeonFeatureCatalog[feat.subsequentDF]
          spawnDungeonFeature(pmap, l.x, l.y, subDF, false, true)
        end

      end
      if gen.terrain > 0 then
        -- print('  spawn terrain maybe? at x ' .. l.x .. ', y: ' .. l.y)
        local currentPriority = tileCatalog[pmap[l.x][l.y].layers[gen.layer]].drawPriority
        local newPriority = tileCatalog[gen.terrain].drawPriority
        if currentPriority >= newPriority then
          local grid = allocGrid()
          grid[l.x][l.y] = 1
          local blocks = band(tileCatalog[gen.terrain].flags, tf.T_PATHING_BLOCKER)
          local disconnects = levelIsDisconnectedWithBlockingMap(pmap, grid, false)
          -- print('  has higher priority.  blocks ' .. tern(blocks > 0, 'yes', 'no') ..
                -- ' disconnects ' .. tern(disconnects > 0, 'yes', 'no'))
          if blocks == false
            or disconnects == 0
          then
            -- print('  placed terrain')
            pmap[l.x][l.y].layers[gen.layer] = gen.terrain
          end
        end
      end
    end
  end
  if buildAreaMachines then
    -- print('TOOD - build a machine')
    -- print('Machines look really tricky')
  end
end

local function runAutogenerators(depth, pmap, buildAreaMachines)
  for _, gen in ipairs(autoGeneratorCatalog) do
    createGenerator(depth, pmap, buildAreaMachines, gen)
  end
end


local function findReplaceGrid(grid, findValueMin, findValueMax, fillValue)
  for i = 1, DCOLS do
    for j = 1, DROWS do
      if grid[i][j] >= findValueMin and grid[i][j] <= findValueMax then
        grid[i][j] = fillValue;
      end
    end
  end
end

local function addLoops(grid, minPathDist)
  local dirCoords = {{1,0},{0,1}}
  local sCoord = {}
  fillSequentialList(sCoord, DCOLS*DROWS)
  shuffleList(sCoord)

  local pathMap = allocGrid()
  local costMap = copyNewGrid(grid);

  findReplaceGrid(costMap, 0, 0, PDS_OBSTRUCTION)
  findReplaceGrid(costMap, 1, 30000, 1)

  for i=1, DCOLS * DROWS do
    local x = math.ceil(sCoord[i] / DROWS)
    local y = sCoord[i] % DROWS + 1
    -- 0 means stone
    if grid[x][y] == 0 then
      for d=1, 2 do
        local newX = x + dirCoords[d][1]
        local oppX = x - dirCoords[d][1]
        local newY = y + dirCoords[d][2]
        local oppY = y - dirCoords[d][2]
        if coordinatesAreInMap(newX, newY)
          and coordinatesAreInMap(oppX, oppY)
          and grid[newX][newY] == 1
          and grid[oppX][oppY] == 1
        then
           fillGrid(pathMap, 30000)
           pathMap[newX][newY] = 0

           mg_arch.dijkstra.scan(pathMap, costMap, false)

           if pathMap[oppX][oppY] > minPathDist then
             grid[x][y] = 2
             costMap[x][y] = 1
             break;
           end

         end
      end
    end
  end
end

local function finishDoors(pmap, depth)
  local secretDoorChance = clamp((depth - 1) * 67 / (AMULET_LEVEL - 1), 0, 67)
  -- print('finish doors - secret door chance:  ' .. secretDoorChance)
  for i=2, DCOLS -1 do
    for j=2, DROWS -1 do
      if pmap[i][j].layers.dungeon == tt.DOOR
        and pmap[i][j].machineNumber == 0
      then
        if (false == cellHasTerrainFlag(pmap, i+1, j, tf.T_OBSTRUCTS_PASSABILITY) or false == cellHasTerrainFlag(pmap, i-1, j, tf.T_OBSTRUCTS_PASSABILITY))
          and (false == cellHasTerrainFlag(pmap, i, j+1, tf.T_OBSTRUCTS_PASSABILITY) or false == cellHasTerrainFlag(pmap, i, j-1, tf.T_OBSTRUCTS_PASSABILITY))
        then
          -- If there's passable terrain to the left or right, and there's passable terrain
          -- above or below, then the door is orphaned and must be removed.
          -- print('finish doors - remove door because its orphaned - senario 1')
          pmap[i][j].layers.dungeon = tt.FLOOR
        elseif tern(cellHasTerrainFlag(pmap, i+1, j, tf.T_PATHING_BLOCKER), 1, 0)
             + tern(cellHasTerrainFlag(pmap, i-1, j, tf.T_PATHING_BLOCKER), 1, 0)
             + tern(cellHasTerrainFlag(pmap, i, j+1, tf.T_PATHING_BLOCKER), 1, 0)
             + tern(cellHasTerrainFlag(pmap, i, j-1, tf.T_PATHING_BLOCKER), 1, 0) >= 3
        then
          -- print('finish doors - remove door because its orphaned - senario 2')
          pmap[i][j].layers.dungeon = tt.FLOOR
        elseif rand_percent(secretDoorChance) then
          -- print('finish doors - upgrade this door to a secret door :-)')
          pmap[i][j].layers.dungeon = tt.SECRET_DOOR
        end
      end
    end
  end
end

local function getDungeonFlags(pmap, x, y)
  local tile = pmap[x][y].layers.dungeon
  return tileCatalog[tile].flags
end

local function removeDiagonalOpenings(pmap)
  local diagonalCornerRemoved
  repeat
    diagonalCornerRemoved = false
    for i=1, DCOLS - 1 do
      for j=1, DROWS - 1 do
        for k=0, 1 do
          if band(getDungeonFlags(pmap, i + k,       j    ), tf.T_OBSTRUCTS_PASSABILITY) == 0 and
             band(getDungeonFlags(pmap, i + (1 - k), j    ), tf.T_OBSTRUCTS_PASSABILITY) > 0 and
             band(getDungeonFlags(pmap, i + (1 - k), j    ), tf.T_OBSTRUCTS_DIAGONAL_MOVEMENT) > 0 and
             band(getDungeonFlags(pmap, i + k,       j + 1), tf.T_OBSTRUCTS_PASSABILITY) > 0 and
             band(getDungeonFlags(pmap, i + k,       j + 1), tf.T_OBSTRUCTS_DIAGONAL_MOVEMENT) > 0 and
             band(getDungeonFlags(pmap, i + (1 - k), j + 1), tf.T_OBSTRUCTS_PASSABILITY) == 0
          then
            local x1, x2, y1
            if rand_percent(50) then
              x1 = i + (1-k)
              x2 = i + k
              y1 = j
            else
              x1 = i + k
              x2 = i + (1-k)
              y1 = j
            end
            -- Brogue code checks there's no monster and no machines
            -- TODO - when implement machines - check that there's no machines here

            print('remove diagonal corner')
            diagonalCornerRemoved = true

            pmap[x1][y1].layers.dungeon = pmap[x2][y1].layers.dungeon
            pmap[x1][y1].layers.liquid = pmap[x2][y1].layers.liquid
            pmap[x1][y1].layers.surface = pmap[x2][y1].layers.surface
            pmap[x1][y1].layers.gas = pmap[x2][y1].layers.gas

          end
        end
      end
    end
  until diagonalCornerRemoved == false
end

local function buildABridge(pmap, depth)
  local nCols = {}
  local nRows = {}

  local bridgeRatioX = 100 + (100 + 100 * depth * DEPTH_ACCELERATOR / 9) * rand_range(10, 20) / 10
  local bridgeRatioY = 100 + (400 + 100 * depth * DEPTH_ACCELERATOR / 18) * rand_range(10, 20) / 10


  fillSequentialList(nCols, DCOLS)
  shuffleList(nCols)
  fillSequentialList(nRows, DROWS)
  shuffleList(nRows)

  for i2=1, DCOLS  do
    local i = nCols[i2]
    for j2=1, DROWS do
      local j = nRows[j2]
      -- TODO add machine check here
      if cellHasTerrainFlag(pmap, i, j, bor(tf.T_CAN_BE_BRIDGED, tf.T_PATHING_BLOCKER)) == false then

        -- try a horizontal bridge
        local foundExposure = false
        local _k = 0
        for k = i + 1, DCOLS do
          if cellHasTerrainFlag(pmap, k, j, tf.T_CAN_BE_BRIDGED)
            and false == cellHasTerrainFlag(pmap, k, j, tf.T_OBSTRUCTS_PASSABILITY)
            and cellHasTerrainFlag(pmap, k, j - 1, bor(tf.T_CAN_BE_BRIDGED, tf.T_OBSTRUCTS_PASSABILITY))
            and cellHasTerrainFlag(pmap, k, j + 1, bor(tf.T_CAN_BE_BRIDGED, tf.T_OBSTRUCTS_PASSABILITY))
          then
            if false == cellHasTerrainFlag(pmap, k, j-1, tf.T_OBSTRUCTS_PASSABILITY)
              and false == cellHasTerrainFlag(pmap, k, j+1, tf.T_OBSTRUCTS_PASSABILITY)
            then
              _k = k + 1
              foundExposure = true
            end
          else
            -- break out of the loop if any of those are false
            break
          end
        end

        if _k < DCOLS and _k > 0
          and _k - i > 3
          and foundExposure
          and false == cellHasTerrainFlag(pmap, _k, j, bor(tf.T_PATHING_BLOCKER, tf.T_CAN_BE_BRIDGED))
          -- Prevents duplicate bridges
          -- TODO 100 * pathingDistance(i,j,k,j,
        then
          print('bridge - horizontal try to build from ' .. i .. ' to ' .. _k .. ', ' .. j)
          local distance = mg_arch.dijkstra.pathingDistance(pmap, i, j, _k, j, tf.T_PATHING_BLOCKER)
          local realBridgeRatio = 100 * distance / (_k - i)
          if realBridgeRatio > bridgeRatioX then
            print('distance is good - build for real')
            for l=i+1, _k - 1 do
              pmap[l][j].layers.liquid = tt.BRIDGE
            end
            pmap[i][j].layers.surface = tt.BRIDGE_EDGE
            pmap[_k][j].layers.surface = tt.BRIDGE_EDGE
            return true
          end
        end

        -- Now try vertical bridge
        foundExposure = false
        _k = 0
        for k = j + 1, DCOLS do
          if coordinatesAreInMap(i, k)
            and cellHasTerrainFlag(pmap, i, k, tf.T_CAN_BE_BRIDGED)
            and false == cellHasTerrainFlag(pmap, i, k, tf.T_OBSTRUCTS_PASSABILITY)
            and cellHasTerrainFlag(pmap, i-1, k, bor(tf.T_CAN_BE_BRIDGED, tf.T_OBSTRUCTS_PASSABILITY))
            and cellHasTerrainFlag(pmap, i+1, k, bor(tf.T_CAN_BE_BRIDGED, tf.T_OBSTRUCTS_PASSABILITY))
          then
            if false == cellHasTerrainFlag(pmap, i-1, k, tf.T_OBSTRUCTS_PASSABILITY)
              and false == cellHasTerrainFlag(pmap, i+1, k, tf.T_OBSTRUCTS_PASSABILITY)
            then
              _k = k + 1
              foundExposure = true
            end
          else
            -- break out of the loop if any of those are false
            break
          end
        end

        if _k < DROWS and _k > 0
          and _k - j > 3
          and foundExposure
          and false == cellHasTerrainFlag(pmap, i, _k, bor(tf.T_PATHING_BLOCKER, tf.T_CAN_BE_BRIDGED))
          -- Prevents duplicate bridges
          -- TODO 100 * pathingDistance(i,j,k,j,
        then
          print('bridge - vertical try to build ' .. i .. ', from '.. j ..' to ' .. _k)

          local distance = mg_arch.dijkstra.pathingDistance(pmap, i, j, i, _k, tf.T_PATHING_BLOCKER)
          local realBridgeRatio = 100 * distance / (_k - j)
          if realBridgeRatio > bridgeRatioY then
            print('distance is good - build for real')
            for l=j+1, _k - 1 do
              pmap[i][l].layers.liquid = tt.BRIDGE
            end
            pmap[i][j].layers.surface = tt.BRIDGE_EDGE
            pmap[i][_k].layers.surface = tt.BRIDGE_EDGE
            return true
          end
        end


      end
    end
  end

  return false
end

local function iterate(iterator)
  for i=1, DCOLS do
    for j=1, DROWS do
      iterator(i, j)
    end
  end
end

local function iterateReverse(iterator)
  for i=1, DCOLS do
    for j=1, DROWS do
      iterator(DCOLS + 1 - i, DROWS + 1 - j)
    end
  end
end

local function cleanUpLakeBoundaries(pmap)
  local reverse = true
  local failsafe = 100
  local madeChange
  local tfLakeObstruct = bor(tf.T_LAKE_PATHING_BLOCKER, tf.T_OBSTRUCTS_PASSABILITY)
  repeat
    madeChange = false
    reverse = not reverse
    failsafe = failsafe - 1
    local iterator = function (i, j)
      if cellHasTerrainFlag(pmap, i, j, tfLakeObstruct) and i > 1 and j > 1 and i < DCOLS and j < DROWS then
        local subjectFlags = band(terrainFlag(pmap, i, j), tfLakeObstruct)
        local x
        local y
        local tfiPrev = band(terrainFlag(pmap, i - 1, j), band(tf.T_LAKE_PATHING_BLOCKER, bnot(subjectFlags)))
        local tfiNext = band(terrainFlag(pmap, i + 1, j), band(tf.T_LAKE_PATHING_BLOCKER, bnot(subjectFlags)))
        local tfjPrev = band(terrainFlag(pmap, i, j - 1), band(tf.T_LAKE_PATHING_BLOCKER, bnot(subjectFlags)))
        local tfjNext = band(terrainFlag(pmap, i, j + 1), band(tf.T_LAKE_PATHING_BLOCKER, bnot(subjectFlags)))
        if tfiPrev > 0 and tfiPrev == tfiNext then
          x = i + 1
          y = j
        elseif tfjPrev > 0 and tfjPrev == tfjNext then
          x = i
          y = j + 1
        end
        if x ~= nil then
          madeChange = true
          print('clean up lake boundary here: ' .. i .. ', ' .. j)
          pmap[i][j].layers.dungeon = pmap[x][y].layers.dungeon
          pmap[i][j].layers.liquid =  pmap[x][y].layers.liquid
          pmap[i][j].layers.surface = pmap[x][y].layers.surface
          pmap[i][j].layers.gas =     pmap[x][y].layers.gas
        end
      end
    end
    if reverse == false then
      iterate(iterator)
    else
      iterateReverse(iterator)
    end
  until madeChange == false or failsafe <= 0
end

local function lakeFloodFill(pmap, x, y, floodMap, grid, lakeMap, dungeonToGridX, dungeonToGridY)
  floodMap[x][y] = 1
  for dir=1, 4 do
    local newX = x + nbDirs[dir][1]
    local newY = y + nbDirs[dir][2]
    if coordinatesAreInMap(newX, newY) and
       floodMap[newX][newY] == 0 and
       cellHasTerrainFlag(pmap, newX, newY, tf.T_OBSTRUCTS_EVERYTHING) == false and
       lakeMap[newX][newY] == 0 and
       ( coordinatesAreInMap(newX + dungeonToGridX, newY + dungeonToGridY) == false or
         grid[newX+dungeonToGridX][newY+dungeonToGridY] == 0) then
      lakeFloodFill(pmap, newX, newY, floodMap, grid, lakeMap, dungeonToGridX, dungeonToGridY)
    end
  end
end

local function lakeDisruptsPassibility(pmap, grid, lakeMap, dungeonToGridX, dungeonToGridY)
  local floodMap = allocGrid()
  fillGrid(floodMap, 0)

  local x = -1
  local y = -1
  for i = 1, DCOLS do
    for j = 1, DROWS do
      local isBlocker = cellHasTerrainFlag(pmap, i, j, tf.T_PATHING_BLOCKER)
      local isLakeMapEmpty = lakeMap[i][j] == 0
      local isInMap = coordinatesAreInMap(i + dungeonToGridX, j + dungeonToGridY)
      local isCurrentLakeEmpty = isInMap == false or grid[i+dungeonToGridX][j+dungeonToGridY] == 0
      if isBlocker == false and
         isLakeMapEmpty and
         isCurrentLakeEmpty
      then
        x = i
        y = j
        break
      end
    end
    if x > -1 then
      break
    end
  end

  lakeFloodFill(pmap, x, y, floodMap, grid, lakeMap, dungeonToGridX, dungeonToGridY)

  local result = false

  for i=1, DCOLS do
    for j=1, DROWS do
      local isInMap = coordinatesAreInMap(i+dungeonToGridX, j+dungeonToGridY)
      if cellHasTerrainFlag(pmap, i, j, tf.T_PATHING_BLOCKER) == false and
         lakeMap[i][j] == 0 and
         floodMap[i][j] == 0 and
         (isInMap == false or grid[i+dungeonToGridX][j+dungeonToGridY] == 0)
      then
        result = true
        break
      end
    end
    if result == true then
      break
    end
  end
  return result
end

local function designLakes(pmap, lakeMap)
  local grid = allocGrid()
  fillGrid(lakeMap, 0)
  local lakeMaxHeight = 15
  local lakeMaxWidth = 30


  while lakeMaxHeight >= 10 do
    fillGrid(grid, 0)

    -- puts a blob on grid
    -- it's in the upper right
    local l = createBlobOnGrid(grid, 5, 4, 4, lakeMaxHeight, lakeMaxHeight, 55, 'ffffftttt', 'ffffttttt')

    local lakeX = l.minX
    local lakeY = l.minY
    local lakeWidth = l.width
    local lakeHeight = l.height

    local xStart = 3 - lakeX
    local xEnd = DCOLS - lakeWidth - lakeX - 2
    local yStart = 3 - lakeY
    local yEnd = DROWS - lakeHeight - lakeY - 2

    -- 20 tries to fit that blob somewhere on the map
    for _ = 1, 20 do
      local x = rand_range(xStart, xEnd)
      local y = rand_range(yStart, yEnd)
      local isBad = lakeDisruptsPassibility(pmap, grid, lakeMap, -x, -y)
      if isBad == false then
        for i = 1, lakeWidth do
          for j = 1, lakeHeight do
            if grid[i + lakeX][j + lakeY] > 0 then
              lakeMap[i + lakeX + x][j + lakeY + y] = 1
              pmap[i + lakeX + x][j + lakeY + y].layers.dungeon = tileType.FLOOR
            end
          end
        end
        break
      end
    end

    lakeMaxHeight = lakeMaxHeight - 1
    lakeMaxWidth = lakeMaxWidth - 2
  end
end

local function liquidType(depth)
  local randMin = tern(depth < MINIMUM_LAVA_LEVEL, 1 , 0)
  local randMax = tern(depth < MINIMUM_BRIMSTONE_LEVEL, 2 , 3)
  local rand = rand_range(randMin, randMax)
  if depth == DEEPEST_LEVEL then
    rand = 1
  end

  if rand == 0 then
    return {
      deep = tileType.LAVA,
      shallow = tileType.NOTHING,
      shallowWidth = 0
    }
  elseif rand == 1 then
    return {
      deep = tileType.DEEP_WATER,
      shallow = tileType.SHALLOW_WATER,
      shallowWidth = 2
    }
  elseif rand == 2 then
    return {
      deep = tileType.CHASM,
      shallow = tileType.CHASM_EDGE,
      shallowWidth = 1
    }
  elseif rand == 3 then
    return {
      deep = tileType.INERT_BRIMSTONE,
      shallow = tileType.OBSIDIAN,
      shallowWidth = 2
    }
  end
end

local function fillLake(pmap, x, y, liquid, scanWidth, wreathMap, lakeMap)
  for i=x-scanWidth, x + scanWidth do
    for j=y-scanWidth, y + scanWidth do
      if coordinatesAreInMap(i, j) and lakeMap[i][j] > 0 then
        lakeMap[i][j] = 0
        pmap[i][j].layers.liquid = liquid
        wreathMap[i][j] = 1
        fillLake(pmap, i, j, liquid, scanWidth, wreathMap, lakeMap)
      end
    end
  end
end

local function createWreath(pmap, shallow, width, wreathMap)
  for i = 1, DCOLS do
    for j = 1, DROWS do
      if wreathMap[i][j] > 0 then
        for k = i - width, i + width + 1 do
          for l = j - width, j + width + 1 do
            if coordinatesAreInMap(k,l) and pmap[k][l].layers.liquid == tileType.NOTHING
              and (i-k)*(i-k) + (j-l)*(j-l) <= width*width then
              pmap[k][l].layers.liquid = shallow
              if pmap[k][l].layers.dungeon == tileType.DOOR then
                pmap[k][l].layers.dungeon = tileType.FLOOR
              end
            end
          end
        end
      end
    end
  end
end

local function fillLakes(pmap, lakeMap, depth)
  local wreathMap = allocGrid()
  for i=1, DCOLS do
    for j=1, DROWS do
      if lakeMap[i][j] > 0 then
        local l = liquidType(depth)
        fillGrid(wreathMap, 0)
        fillLake(pmap, i, j, l.deep, 4, wreathMap, lakeMap)
        createWreath(pmap, l.shallow, l.shallowWidth, wreathMap)
      end
    end
  end
end

local function finishWalls (pmap, includingDiagonals)
  local foundExposure = false
  local dirs = tern(includingDiagonals, 8, 4)
  for i=1, DCOLS do
    for j=1, DROWS do
      if pmap[i][j].layers.dungeon == tileType.GRANITE then
        foundExposure = false
        for dir=1, dirs do
          local x1 = i + nbDirs[dir][1]
          local y1 = j + nbDirs[dir][2]
          if coordinatesAreInMap(x1, y1) and
            --pmap[x1][y1].layers.dungeon == tileType.FLOOR
            (cellHasTerrainFlag(pmap, x1, y1, tf.T_OBSTRUCTS_VISION) == false or cellHasTerrainFlag(pmap, x1, y1, tf.T_OBSTRUCTS_PASSABILITY) == false)
            then
              pmap[i][j].layers.dungeon = tileType.WALL
              foundExposure = true
              break;
            end
        end
      elseif pmap[i][j].layers.dungeon == tileType.WALL then
        foundExposure = false
        for dir=1, dirs do
          local x1 = i + nbDirs[dir][1]
          local y1 = j + nbDirs[dir][2]
          if coordinatesAreInMap(x1, y1) and
            (cellHasTerrainFlag(pmap, x1, y1, tf.T_OBSTRUCTS_VISION) == false or cellHasTerrainFlag(pmap, x1, y1, tf.T_OBSTRUCTS_PASSABILITY) == false)
            then
              foundExposure = true
              break;
            end
        end
        if foundExposure == false then
          pmap[i][j].layers.dungeon = tileType.GRANITE
        end
      end
    end
  end
end

local function prepareForStairs(pmap, x, y, grid)
  local newX
  local newY
  for dir=1, 4 do
    if false == cellHasTerrainFlag(pmap, x + nbDirs[dir][1], y + nbDirs[dir][2], tf.T_OBSTRUCTS_PASSABILITY) then
      newX = x + nbDirs[dir][2]
      newY = y + nbDirs[dir][1]
      pmap[newX][newY].layers.dungeon = tt.TORCH_WALL

      newX = x - nbDirs[dir][2]
      newY = y - nbDirs[dir][1]
      pmap[newX][newY].layers.dungeon = tt.TORCH_WALL
    end
  end

  for dir=1, 8 do
    newX = x + nbDirs[dir][1]
    newY = y + nbDirs[dir][2]
    if pmap[newX][newY].layers.dungeon == tt.GRANITE then
      pmap[newX][newY].layers.dungeon = tt.WALL
    end
  end
  for newX = max(0, x-5), min(DCOLS, x + 5) do
    for newY = max(0, y-5), min(DROWS, y + 5) do
      if coordinatesAreInMap(newX, newY) then
        grid[newX][newY] = 0
      end
    end
  end
end

local function passableArcCount(pmap, x, y)
  local arcCount = 0
  for dir=1, 8 do
    local oldX = x + cDirs[((dir + 6) % 8) + 1][1]
    local oldY = y + cDirs[((dir + 6) % 8) + 1][2]
    local newX = x + cDirs[dir][1]
    local newY = y + cDirs[dir][2]
    if (coordinatesAreInMap(newX, newY) and cellIsPassableOrDoor(pmap, newX, newY))
       ~= (coordinatesAreInMap(oldX, oldY) and cellIsPassableOrDoor(pmap, oldX, oldY)) then
      arcCount = arcCount + 1
    end
  end
  return arcCount / 2
end


local function validStairLoc(pmap, x, y)
  if x <= 1 or x >= DCOLS or y <= 1 or y >= DROWS or pmap[x][y].layers.dungeon ~= tileType.WALL then
    return 0
  end
  local neighborWallCount = 0
  for dir=1, 4 do
    local newX = x + nbDirs[dir][1]
    local newY = y + nbDirs[dir][2]
    if cellHasTerrainFlag(pmap, newX, newY, tf.T_OBSTRUCTS_PASSABILITY) then
      neighborWallCount = neighborWallCount + 1
    else
      if cellHasTerrainFlag(pmap, newX, newY, tf.T_PATHING_BLOCKER)
        or passableArcCount(pmap, newX, newY) >= 2
      then
        return 0
      end
      newX = x - nbDirs[dir][1] + nbDirs[dir][1]
      newY = y - nbDirs[dir][2] + nbDirs[dir][2]
      if cellHasTerrainFlag(pmap, newX, newY, tf.T_OBSTRUCTS_PASSABILITY) == false then
        return 0
      end

      newX = x - nbDirs[dir][1] - nbDirs[dir][1]
      newY = y - nbDirs[dir][2] - nbDirs[dir][2]
      if coordinatesAreInMap(newX, newY) and cellHasTerrainFlag(pmap, newX, newY, tf.T_OBSTRUCTS_PASSABILITY) == false then
        return 0
      end
    end
  end
  if neighborWallCount ~= 3 then
    return 0
  end
  return 1
end

local function placeStairs(pmap, depth, levels)
  local grid = allocGrid();
  for i = 1, DCOLS do
    for j=1, DROWS do
      grid[i][j] = validStairLoc(pmap, i, j)
    end
  end

  local upLoc = {x = 0, y = 0}
  local downLoc = {x = 0, y = 0}

  -- sets stair location
  -- print('get down loc')
  local isGood = getQualifyingGridLocNear(downLoc, levels[depth].downStair.x, levels[depth].downStair.y, grid, false)

  prepareForStairs(pmap, downLoc.x, downLoc.y, grid)

  pmap[downLoc.x][downLoc.y].layers.dungeon = tt.DOWN_STAIRS
  pmap[downLoc.x][downLoc.y].layers.liquid = tt.NOTHING
  pmap[downLoc.x][downLoc.y].layers.surface = tt.NOTHING
  levels[depth].downStair = downLoc

  -- I think I have a bug in copying the brogue code over
  -- this fixes stair placement so water doesn't go down it
  for dir=1, 8 do
    local newX = downLoc.x + nbDirs[dir][1]
    local newY = downLoc.y + nbDirs[dir][2]
    if pmap[newX][newY].layers.liquid ~= tt.NOTHING then
      pmap[newX][newY].layers.dungeon = tt.FLOOR
      pmap[newX][newY].layers.liquid = tt.NOTHING
      pmap[newX][newY].layers.surface = tt.NOTHING
    end
  end

  -- Upstairs
  --
  -- print('get up loc')
  local isGood = getQualifyingGridLocNear(upLoc, levels[depth].upStair.x, levels[depth].upStair.y, grid, false)
  prepareForStairs(pmap, upLoc.x, upLoc.y, grid)

  pmap[upLoc.x][upLoc.y].layers.dungeon = tt.UP_STAIRS
  pmap[upLoc.x][upLoc.y].layers.liquid = tt.NOTHING
  pmap[upLoc.x][upLoc.y].layers.surface = tt.NOTHING
  levels[depth].upStair = upLoc


end

-- on the grid
-- 0 granite,
-- 1 floor
-- 2 possible door site
-- -1 off limits

local function carveDungeon(depth)
  local theDP = adjustProfileForDepth(dungeonProfileCatalog[dungeonProfileTypes.BASIC], depth)
  --printTable(theDP);
  local theFirstRoomDP = adjustProfileForDepthFirstRoom(dungeonProfileCatalog[dungeonProfileTypes.BASIC_FIRST_ROOM], depth)
  --printTable(theFirstRoomDP);

  local grid = allocGrid()
  designRandomRoom(grid, false, nil, theFirstRoomDP.roomFrequencies)

  attachRooms(grid, theDP, 35, 35)

  -- printGrid(grid);
  -- print("Attach rooms run")
  return grid;
end


local function digDungeon(depth, levels)
  print('-----------------------')
  print('-----------------------')
  print('-----------------------')
  print('make level ' .. depth)

  local grid = carveDungeon(depth)
  addLoops(grid, 20)

  local pmap = allocPmap();
  for i = 1, DCOLS do
    for j = 1, DROWS do
      if grid[i][j] == 1 then
        pmap[i][j].layers.dungeon = tileType.FLOOR
      elseif grid[i][j] == 2 then
        pmap[i][j].layers.dungeon = tern(
          rand_percent(60) and depth < DEEPEST_LEVEL,
          tileType.DOOR,
          tileType.FLOOR
        )
      end
    end
  end

  -- This actually does to something becuase the default is granite
  finishWalls(pmap, false)
  -- printPmap(pmap)

  local lakeMap = allocGrid()
  designLakes(pmap, lakeMap)
  fillLakes(pmap, lakeMap, depth)

  finishWalls(pmap, false)

  runAutogenerators(depth, pmap, false)

  removeDiagonalOpenings(pmap)

  cleanUpLakeBoundaries(pmap)

  while buildABridge(pmap, depth) do
    -- nothing
  end

  finishDoors(pmap, depth)

  placeStairs(pmap, depth, levels)

  -- mark loops and choke points
  -- useful for item placement
  analyzeMap(pmap, false)

  --printPmap(pmap)



  return pmap;
end


local function makeLevels ()
  local levels = {}
  for i = 1, DEEPEST_LEVEL do
    local level = {}
    local downStair = {x = 0, y = 0}
    local upStair = {x = 0, y = 0}
    if i == 1 then
      upStair.x = (DCOLS - 1) / 2
      upStair.y = DROWS - 1
    end
    level.downStair = downStair
    level.upStair = upStair
    levels[i] = level
  end
  for i = 1, DEEPEST_LEVEL do
    local level = levels[i]
    repeat
      level.downStair.x = rand_range(2, DCOLS - 2)
      level.downStair.y = rand_range(2, DROWS - 2)
    until (distanceBetween(level.upStair.x, level.upStair.y,
                           level.downStair.x, level.downStair.y) > DCOLS / 3)
    if i < DEEPEST_LEVEL then
      levels[i + 1].upStair.x = levels[i].downStair.x
      levels[i + 1].upStair.y = levels[i].downStair.y
    end
  end
  return levels
end

local function makeTop(depth)
  local grid = carveDungeon(depth)
  addLoops(grid, 20)
  local pmap = allocPmap();
  for i = 1, DCOLS do
    for j = 1, DROWS do
      pmap[i][j].layers.liquid = tileType.NOTHING
      if grid[i][j] == 1 then
        pmap[i][j].layers.dungeon = tileType.FLOOR
      elseif grid[i][j] == 2 then
        pmap[i][j].layers.dungeon = tern(
          rand_percent(60) and depth < DEEPEST_LEVEL,
          tileType.DOOR,
          tileType.FLOOR
        )
      end
    end
  end
  finishWalls(pmap, false)
  return pmap
end



-- Big funcs
mg_arch.makeTop = makeTop
mg_arch.carveDungeon = carveDungeon
mg_arch.digDungeon = digDungeon
mg_arch.makeLevels = makeLevels

-- utilities for other files

mg_arch.rand_percent = rand_percent
mg_arch.tern = tern
mg_arch.rand_range = rand_range
mg_arch.tileType = tileType
mg_arch.df = df
mg_arch.AMULET_LEVEL = AMULET_LEVEL
mg_arch.min = min
mg_arch.max = max
mg_arch.cellHasTerrainFlag = cellHasTerrainFlag
mg_arch.cellHasTerrainType = cellHasTerrainType
mg_arch.randomMatchingLocation = randomMatchingLocation
mg_arch.passableArcCount = passableArcCount


mg_arch.nbDirs = nbDirs
mg_arch.cDirs = cDirs

mg_arch.coordinatesAreInMap = coordinatesAreInMap

mg_arch.printGrid = printGrid
mg_arch.printPmap = printPmap


