--[[
-- brogue marks the grid spaces around the stairs as in field of view
-- so that monsters don't spawn there
-- i need to do something similar
  zeroOutGrid(grid);
  getFOVMask(grid, upLoc.x, upLoc.y, max(DCOLS, DROWS) * FP_FACTOR, (T_OBSTRUCTS_VISION), 0, false);
  for (i=0; i<DCOLS; i++) {
      for (j=0; j<DROWS; j++) {
          if (grid[i][j]) {
              pmap[i][j].flags |= IN_FIELD_OF_VIEW;
          }
      }
  }

//         Octants:      //
//          \7|8/        //
//          6\|/1        //
//          --@--        //
//          5/|\2        //
//          /4|3\        //

void betweenOctant1andN(short *x, short *y, short x0, short y0, short n) {
    short x1 = *x, y1 = *y;
    short dx = x1 - x0, dy = y1 - y0;
    switch (n) {
        case 1:
            return;
        case 2:
            *y = y0 - dy;
            return;
        case 5:
            *x = x0 - dx;
            *y = y0 - dy;
            return;
        case 6:
            *x = x0 - dx;
            return;
        case 8:
            *x = x0 - dy;
            *y = y0 - dx;
            return;
        case 3:
            *x = x0 - dy;
            *y = y0 + dx;
            return;
        case 7:
            *x = x0 + dy;
            *y = y0 - dx;
            return;
        case 4:
            *x = x0 + dy;
            *y = y0 + dx;
            return;
    }
}

// Returns a boolean grid indicating whether each square is in the field of view of (loc.x, loc.y).
// forbiddenTerrain is the set of terrain flags that will block vision (but the blocking cell itself is
// illuminated); forbiddenFlags is the set of map flags that will block vision.
// If cautiousOnWalls is set, we will not illuminate blocking tiles unless the tile one space closer to the origin
// is visible to the player; this is to prevent lights from illuminating a wall when the player is on the other
// side of the wall.
void getFOVMask(char grid[DCOLS][DROWS], short xLoc, short yLoc, fixpt maxRadius,
                unsigned long forbiddenTerrain, unsigned long forbiddenFlags, boolean cautiousOnWalls) {
    pos loc = { xLoc, yLoc };

    for (int i=1; i<=8; i++) {
        scanOctantFOV(grid, loc.x, loc.y, i, maxRadius, 1, LOS_SLOPE_GRANULARITY * -1, 0,
                      forbiddenTerrain, forbiddenFlags, cautiousOnWalls);
    }
}

// This is a custom implementation of recursive shadowcasting.
void scanOctantFOV(char grid[DCOLS][DROWS], short xLoc, short yLoc, short octant, fixpt maxRadius,
                   short columnsRightFromOrigin, long startSlope, long endSlope, unsigned long forbiddenTerrain,
                   unsigned long forbiddenFlags, boolean cautiousOnWalls) {
    const pos loc = { xLoc, yLoc };

    if (columnsRightFromOrigin * FP_FACTOR >= maxRadius) return;

    short i, a, b, iStart, iEnd, x, y, x2, y2; // x and y are temporary variables on which we do the octant transform
    long newStartSlope, newEndSlope;
    boolean cellObstructed;

    newStartSlope = startSlope;

    a = ((LOS_SLOPE_GRANULARITY / -2 + 1) + startSlope * columnsRightFromOrigin) / LOS_SLOPE_GRANULARITY;
    b = ((LOS_SLOPE_GRANULARITY / -2 + 1) + endSlope * columnsRightFromOrigin) / LOS_SLOPE_GRANULARITY;

    iStart = min(a, b);
    iEnd = max(a, b);

    // restrict vision to a circle of radius maxRadius
    if ((columnsRightFromOrigin*columnsRightFromOrigin + iEnd*iEnd) >= maxRadius*maxRadius / FP_FACTOR / FP_FACTOR) {
        return;
    }
    if ((columnsRightFromOrigin*columnsRightFromOrigin + iStart*iStart) >= maxRadius*maxRadius / FP_FACTOR / FP_FACTOR) {
        iStart = (int) (-1 * fp_sqrt((maxRadius*maxRadius / FP_FACTOR) - (columnsRightFromOrigin*columnsRightFromOrigin * FP_FACTOR)) / FP_FACTOR);
    }

    x = loc.x + columnsRightFromOrigin;
    y = loc.y + iStart;
    betweenOctant1andN(&x, &y, loc.x, loc.y, octant);
    boolean currentlyLit = coordinatesAreInMap(x, y) && !(cellHasTerrainFlag(x, y, forbiddenTerrain) ||
                                                          (pmap[x][y].flags & forbiddenFlags));
    for (i = iStart; i <= iEnd; i++) {
        x = loc.x + columnsRightFromOrigin;
        y = loc.y + i;
        betweenOctant1andN(&x, &y, loc.x, loc.y, octant);
        if (!coordinatesAreInMap(x, y)) {
            // We're off the map -- here there be memory corruption.
            continue;
        }
        cellObstructed = (cellHasTerrainFlag(x, y, forbiddenTerrain) || (pmap[x][y].flags & forbiddenFlags));
        // if we're cautious on walls and this is a wall:
        if (cautiousOnWalls && cellObstructed) {
            // (x2, y2) is the tile one space closer to the origin from the tile we're on:
            x2 = loc.x + columnsRightFromOrigin - 1;
            y2 = loc.y + i;
            if (i < 0) {
                y2++;
            } else if (i > 0) {
                y2--;
            }
            betweenOctant1andN(&x2, &y2, loc.x, loc.y, octant);

            if (pmap[x2][y2].flags & IN_FIELD_OF_VIEW) {
                // previous tile is visible, so illuminate
                grid[x][y] = 1;
            }
        } else {
            // illuminate
            grid[x][y] = 1;
        }
        if (!cellObstructed && !currentlyLit) { // next column slope starts here
            newStartSlope = (long int) ((LOS_SLOPE_GRANULARITY * (i) - LOS_SLOPE_GRANULARITY / 2) / (columnsRightFromOrigin * 2 + 1) * 2);
            currentlyLit = true;
        } else if (cellObstructed && currentlyLit) { // next column slope ends here
            newEndSlope = (long int) ((LOS_SLOPE_GRANULARITY * (i) - LOS_SLOPE_GRANULARITY / 2)
                            / (columnsRightFromOrigin * 2 - 1) * 2);
            if (newStartSlope <= newEndSlope) {
                // run next column
                scanOctantFOV(grid, loc.x, loc.y, octant, maxRadius, columnsRightFromOrigin + 1, newStartSlope, newEndSlope,
                              forbiddenTerrain, forbiddenFlags, cautiousOnWalls);
            }
            currentlyLit = false;
        }
    }
    if (currentlyLit) { // got to the bottom of the scan while lit
        newEndSlope = endSlope;
        if (newStartSlope <= newEndSlope) {
            // run next column
            scanOctantFOV(grid, loc.x, loc.y, octant, maxRadius, columnsRightFromOrigin + 1, newStartSlope, newEndSlope,
                          forbiddenTerrain, forbiddenFlags, cautiousOnWalls);
        }
    }
}
]]--

local coordinatesAreInMap = mg_arch.coordinatesAreInMap
local cellHasTerrainFlag = mg_arch.cellHasTerrainFlag
local hasAnyFlags = mg_arch.hasAnyFlags

local function printTable (table)
  print(json.encode(table))
end



local function betweenOctant1andN(x, y, x0, y0, n)
  local x1 = x
  local y1 = y
  local dx = x1 - x0
  local dy = y1 - y0
  if n == 1 then
    return x, y
  elseif n == 2 then
    return x, y0 - dy
  elseif n == 5 then
    return x0 - dx, y0 - dy
  elseif n == 6 then
    return x0 - dx, y
  elseif n == 8 then
    return x0 - dy, y0 - dx
  elseif n == 3 then
    return x0 - dy, y0 + dx
  elseif n == 7 then
    return x0 + dy, y0 - dx
  elseif n == 4 then
    return x0 + dy, y0 + dx
  end
end

local LOS_SLOPE_GRANULARITY = 32768
--local LOS_SLOPE_GRANULARITY = 0.5

local function scanOctantFOV(
  pmap, grid, xLoc, yLoc, octant, maxRadius,
  columnsRightFromOrigin, startSlope, endSlope,
  forbiddenTerrain, forbiddenFlags,
  cautiousOnWalls
)
  local loc = {x = xLoc, y = yLoc}
  if columnsRightFromOrigin >= maxRadius then
    return
  end

  local newStartSlope = startSlope
  local a = ((LOS_SLOPE_GRANULARITY / -2 + 1) + startSlope * columnsRightFromOrigin) / LOS_SLOPE_GRANULARITY
  local b = ((LOS_SLOPE_GRANULARITY / -2 + 1) + endSlope * columnsRightFromOrigin) / LOS_SLOPE_GRANULARITY
  local iStart = math.min(a, b)
  local iEnd = math.max(a, b)

  if (columnsRightFromOrigin*columnsRightFromOrigin + iEnd*iEnd) >= maxRadius*maxRadius then
    return
  end

  if (columnsRightFromOrigin*columnsRightFromOrigin + iStart*iStart) >= maxRadius*maxRadius then
    iStart = -1* math.sqrt(maxRadius*maxRadius - columnsRightFromOrigin*columnsRightFromOrigin)
  end

  local x = math.ceil(loc.x  + columnsRightFromOrigin)
  local y = math.ceil(loc.y + iStart)
  x, y = betweenOctant1andN(x, y, loc.x, loc.y, octant)

  local currentlyLit = coordinatesAreInMap(x, y) and
    not (cellHasTerrainFlag(pmap, x, y, forbiddenTerrain)
         or hasAnyFlags(pmap[x][y].flags, forbiddenFlags))

  for i=iStart, iEnd do
    x = math.ceil(loc.x + columnsRightFromOrigin)
    y = math.ceil(loc.y + i)
    x, y = betweenOctant1andN(x, y, loc.x, loc.y, octant)
    if coordinatesAreInMap(x, y) then
      local cellObstructed = cellHasTerrainFlag(pmap, x, y, forbiddenTerrain) or hasAnyFlags(pmap[x][y].flags, forbiddenFlags)
      grid[x][y] = 1
      if not cellObstructed and not currentlyLit then
        newStartSlope = (LOS_SLOPE_GRANULARITY * i - LOS_SLOPE_GRANULARITY / 2) / (columnsRightFromOrigin * 2 + 1) * 2
        currentlyLit = true
      elseif cellObstructed and currentlyLit then
        newEndSlope = (LOS_SLOPE_GRANULARITY * i - LOS_SLOPE_GRANULARITY / 2) / (columnsRightFromOrigin * 2 - 1) * 2
        if newStartSlope <= newEndSlope then
          scanOctantFOV(pmap, grid, loc.x, loc.y, octant, maxRadius, columnsRightFromOrigin + 1, newStartSlope, newEndSlope, forbiddenTerrain, forbiddenFlags, cautiousOnWalls)
        end
        currentlyLit = false
      end
    end
  end
  if currentlyLit then
    local newEndSlope = endSlope
    if newStartSlope <= newEndSlope then
        scanOctantFOV(pmap, grid, loc.x, loc.y, octant, maxRadius, columnsRightFromOrigin + 1, newStartSlope, newEndSlope, forbiddenTerrain, forbiddenFlags, cautiousOnWalls)
    end
  end
end

local function getFOVMask(pmap, grid, xLoc, yLoc, maxRadius, forbiddenTerrain, forbiddenFlags, cautiousOnWalls)
  for i=1, 8 do
    scanOctantFOV(
      pmap, grid, xLoc, yLoc, i, maxRadius,
      1, LOS_SLOPE_GRANULARITY * -1, 0,
      forbiddenTerrain, forbiddenFlags,
      cautiousOnWalls
    )
  end
end

local tf = mg_arch.terrainFlags
local function mark_upstairs_fov(pmap, xLoc, yLoc)
  local fovGrid = mg_arch.allocGrid()
  mg_arch.getFOVMask(pmap, fovGrid, xLoc, yLoc, mg_arch.DCOLS, tf.T_OBSTRUCTS_VISION, {}, false)

  for i = 1, mg_arch.DCOLS do
    for j = 1, mg_arch.DROWS do
      if fovGrid[i][j] > 0 then
        pmap[i][j].flags.in_field_of_view = 1
      end
    end
  end
end

local function clear_fov(pmap)
  for i = 1, mg_arch.DCOLS do
    for j = 1, mg_arch.DROWS do
      pmap[i][j].flags.in_field_of_view = nil
    end
  end
end

mg_arch.getFOVMask = getFOVMask
mg_arch.mark_upstairs_fov = mark_upstairs_fov
mg_arch.clear_fov = clear_fov
