
table.insert(armor.elements, "ring")

local function noop()
end

-- probably should have a utils mod somewhere
local function uuid()
  local template ='xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'
  return string.gsub(template, '[xy]', function (c)
    local v = (c == 'x') and math.random(0, 0xf) or math.random(8, 0xb)
    return string.format('%x', v)
  end)
end

local function get_ring_prop(itemstack, prop, default)
  local def = itemstack:get_definition()
  local meta = itemstack:get_meta()
  local m = meta:get_int(prop)
  if m > 0 then
    return m
  end
  return def["_"..prop] or default
end

local function get_ring_enchant(itemstack)
  local enchant = get_ring_prop(itemstack, 'enchant', 0)
  return enchant
end

local function get_ring_enchant_player(item)
  local player_enchant = get_ring_prop(item, 'player_enchant', 0)
  local identified = get_ring_prop(item, 'identified', 0)
  local enchant = get_ring_prop(item, 'enchant', 0)
  if identified == 0 then
    return player_enchant
  end
  return enchant
end

local function update_desc(stack)
  if stack == nil then
    return nil
  end

  local meta = stack:get_meta()
  local def = stack:get_definition()
  local enchant = get_ring_prop(stack, 'enchant', 0)
  local player_enchant = get_ring_prop(stack, 'player_enchant', 0)
  local identified = get_ring_prop(stack, 'identified', 0)

  local unknown = ''
  if identified == 0 then
    enchant = (player_enchant or 0)
    unknown = '?'
  end

  local sign = (enchant >= 0 and '+') or ''
  local description = sign..enchant..unknown..' '..def.description

  meta:set_string("description", description)
  return stack
end

local function enchant_ring(stack, amount, from_player)
  local meta = stack:get_meta()
  amount = amount or 1

  local enchant = get_ring_prop(stack, 'enchant', 0) + amount

  if from_player then
    local player_enchant = get_ring_prop(stack, 'player_enchant', 0)
    player_enchant = player_enchant + amount
    meta:set_int("player_enchant", player_enchant)
  end

  meta:set_int("enchant", enchant)
  update_desc(stack)

  return stack
end

local function common_equip (_player, _index, itemstack)
    local time = core.get_gametime()
    local meta = itemstack:get_meta()
    meta:set_int('worn_start', time)
    meta:set_string('equip_id', uuid())
end

local function common_unequip (_player, _index, itemstack)
    local time = core.get_gametime()
    local meta = itemstack:get_meta()
    local worn_start = meta:get_int('worn_start')
    local worn_duration = meta:get_int('worn_duration')
    worn_duration = time - worn_start + worn_duration
    meta:set_int('worn_duration', worn_duration)
end

local function register_ring(def)
  local model = def.model
  local level = def.level

  local name = "mg_rings:ring_"..model..level
  armor:register_armor(name, {
    description     = model:gsub("^%l",string.upper).." Ring",
    inventory_image = "goops_"..model.."_ring"..level..".png",
    texture         = "goops_ring"..level..".png",
    preview         = "goops_ring"..level.."_preview.png",
    groups          = {
      armor_ring = level,
    },
    mg_help = def.mg_help,
    mg_help_player_stats = def.mg_help_player_stats,
    wield_scale = {x=.25, y=.25, z=.25},
		on_equip = function(player, index, itemstack)
      common_equip(player, index, itemstack)
      if def.on_equip then
        def.on_equip(player, index, itemstack)
      end
    end,
    on_unequip = function(player, index, itemstack)
      -- I cannot seem to modify the itemstack put in here
      -- I'm going to look through the players main inventory and see if I can
      -- find the right stack
      local meta = itemstack:get_meta()
      local equip_id = meta:get_string('equip_id');

      local listname = "main"
      local inv = player:get_inventory()
      local size = inv:get_size(listname)
      for i=1, size do
        local pstack = inv:get_stack(listname, i)
        local pmeta = pstack:get_meta()
        local pequip_id = pmeta:get_string('equip_id');
        if pequip_id == equip_id then
          common_unequip(player, index, pstack)
          if def.on_unequip then
            def.on_unequip(player, index, pstack)
          end
          inv:set_stack(listname, i, pstack)
        end
      end
    end
  })
end

local cb_timer = 0
core.register_globalstep(function(dtime)
  cb_timer = cb_timer + dtime
  if cb_timer < 10 then
    return
  end
  cb_timer = 0

  local now = core.get_gametime()

	for _, player in pairs(core.get_connected_players()) do
    -- Auto id armor
    local stack = mg_armor.get_equiped_armor_element(player, 'ring')
    if stack then
      local meta = stack:get_meta()
      local worn_start = meta:get_int('worn_start')
      local worn_duration = meta:get_int('worn_duration')
      local identified = meta:get_int('identified')
      worn_duration = now - worn_start + worn_duration
      print('ring worn for ' .. worn_duration)
      if worn_duration >= 500 and identified == 0 then
        local playername = player:get_player_name()
        print('id ring for player ' .. playername)
        meta:set_int('identified', 1)
        update_desc(stack)
        local description = meta:get_string("description")
        mg_armor.replace_armor(player, stack)
        core.chat_send_player(playername, "You are now more familiar with your " .. description)
      end
    end
  end
end)

local function mg_ring_id_help(item)
  local meta = item:get_meta()
  local worn_duration = meta:get_int('worn_duration')
  local identified = meta:get_int('identified')

  print('worn duration is ' .. worn_duration)

  if identified == 0 then
    local id_time = 500 - worn_duration
    return "This ring has not been identified yet.  It will look like a +1 ring.  " ..
      "You have worn this ring for " .. worn_duration .. " seconds.  "..
      "If you wear it for "..id_time.." more seconds, it will identify itself.\n"
  end
  return ""
end


-- Regeneration
-- if it takes 15 seconds to regenerate 1 HP then at
-- 20 HP it'll take 300 turns to regenerate
-- SECONDS / 1 HP = TOTAL REGEN TIME / MAX HP
-- when wearing ring of regeneration
-- TOTAL REGEN TIME = 300 * (0.75 ^ enchantment bonus) + 1
-- in memory table cache
local regen_rate = {}

local function calc_regen_ring_time(enchant)
  return 300 * 0.75^enchant + 1
end

local function on_regen_equip(player, _index, itemstack)
  local enchant = get_ring_enchant(itemstack)
  local props = player:get_properties()
  local max_hp = props.hp_max
  local player_name = player:get_player_name()
  local regen_time = calc_regen_ring_time(enchant)
  -- how many seconds need to pass before health
  -- goes up one hp
  local regen_tick = regen_time / max_hp
  print('equip ring')
  print('enchant: '..enchant)
  print('time to regen 1 HP is '..regen_tick)
  regen_rate[player_name] = regen_tick
end

local function on_regen_unequip(player, _index, _itemstack)
  local props = player:get_properties()
  local max_hp = props.hp_max
  local player_name = player:get_player_name()
  local regen_time = 301
  local regen_tick = regen_time / max_hp
  print('time to regen 1 HP is '..regen_tick)
  regen_rate[player_name] = regen_tick
end

local function get_player_regen_rate(player)
  local player_name = player:get_player_name()
  local rate = regen_rate[player_name]
  if rate then
    return rate
  end

  local props = player:get_properties()
  local max_hp = props.hp_max
  local calc_rate = 300 / max_hp
  -- Add to cache
  regen_rate[player_name] = calc_rate
  return calc_rate
end

register_ring({
  model = 'regeneration',
  level = 2,
  on_equip = on_regen_equip,
  on_unequip = on_regen_unequip,
  mg_help = "This ring will increase the rate at which you regenerate HP.  The higher its enchanted the quicker your HP will regenerate.  A cursed ring will slow your HP regeneration.",
  mg_help_player_stats = function(item)
    local def = item:get_definition()
    local meta = item:get_meta()
    local desc = meta:get_string("description") or def.description
    local id_message = mg_ring_id_help(item)
    local player_enchant = get_ring_enchant_player(item)
    local regen_time = math.ceil(calc_regen_ring_time(player_enchant))
    local regen_time_new = math.ceil(calc_regen_ring_time(player_enchant + 1))

    local message0 = desc .. ". " .. def.mg_help .. "\n"
      .. id_message ..
      "Wearing this ring, your health will fully recharge in " .. regen_time .. " seconds.\n" ..
      "If you enchant this ring, that time will decrease to "..regen_time_new .. " seconds."

    return message0
  end
})




-- Tranference
--
--
local transference_rate = {}

local function calc_transference_rate(rate, dmg)
  if rate > 0 then
    return math.ceil(rate / 100 * dmg)
  elseif rate < 0 then
    return math.floor(rate / 100 * dmg)
  end
  return 0
end

local function apply_player_transference(player, dmg)
  -- Only applies to players not mobs
  if not player:is_player() then
    return
  end

  local player_name = player:get_player_name()
  local rate = transference_rate[player_name] or 0
  local value = calc_transference_rate(rate, dmg)
  --print('transference {damage: '..dmg..', rate: '.. rate ..', value: ' .. value ..'}')
  if value > 0 then
    local hp = player:get_hp()
    player:set_hp(hp + value, {_mg_type = "transference"})
  end
end

local function on_trans_unequip(player, _index, _itemstack)
  local player_name = player:get_player_name()
  transference_rate[player_name] = 0
end

local function on_trans_equip(player, _index, itemstack)
  local player_name = player:get_player_name()
  local enchant = get_ring_enchant(itemstack)
  transference_rate[player_name] = enchant * 5
end

register_ring({
  model = 'transference',
  level = 1,
  on_equip = on_trans_equip,
  on_unequip = on_trans_unequip,
  mg_help = "While wearing this ring, when you hit a monster a percentage of the damage dealt to it will be transfered to your HP.  A cursed ring will deal damage to you whenever you deal damage to a monster.",
  mg_help_player_stats = function(item)
    local def = item:get_definition()
    local meta = item:get_meta()
    local desc = meta:get_string("description") or def.description
    local id_message = mg_ring_id_help(item)
    local player_enchant = get_ring_enchant_player(item)
    local trans_rate = math.ceil(calc_transference_rate(player_enchant*5, 100))
    local trans_rate_new = math.ceil(calc_transference_rate((player_enchant+1)*5, 100))

    local trans_message
    if trans_rate > 0 then
      trans_message = "Wearing this ring, whenever you hit a monster, you will increase your health by " .. trans_rate .. " percent of the damage you dealt.\n"
    else
      trans_message = "Wearing this ring, whenever you hit a monster, you will decrease your health by " .. trans_rate .. " percent of the damage you dealt.\n"
    end

    if trans_rate_new > 0 then
      trans_message = trans_message .. "If you enchant this ring, that will increase to "..trans_rate_new .. " percent."
    else
      trans_message = trans_message .. "If you enchant this ring, the new rate will be "..trans_rate_new .. " percent."
    end

    local message0 = desc .. ". " .. def.mg_help .. "\n"
      .. id_message .. trans_message

    return message0
  end
})


-- A ring of reaping will shorten the time to recharge staves and charms
-- it will remove rand_range(0, damage*enchant) seconds from the time for
-- all those to recharge, the function to apply that is going to be in the
-- mg_bolts mod though
local reaping_rate = {}

local function get_player_reaping_rate(player, dmg)
  -- Only applies to players not mobs
  if not player:is_player() then
    return 0
  end

  local player_name = player:get_player_name()
  local rate = reaping_rate[player_name] or 0
  if rate == 0 then
    return 0
  end

  return math.random(0, dmg * rate)
end

local function on_reaping_unequip(player, _index, _itemstack)
  local player_name = player:get_player_name()
  reaping_rate[player_name] = 0
end

local function on_reaping_equip(player, _index, itemstack)
  local player_name = player:get_player_name()
  local enchant = get_ring_enchant(itemstack)
  reaping_rate[player_name] = enchant
end

register_ring({
  model = 'reaping',
  level = 3,
  on_equip = on_reaping_equip,
  on_unequip = on_reaping_unequip,
  mg_help = "While wearing this ring, when you hit a monster, it reduces some of the time for your all your staves and charms to recharge.  A cursed ring will add time for all your staves and charms to recharge whenever you deal damage to a monster."
})


-- wisdom ring
local wisdom_rate = {}

local function get_player_wisdom_rate(player)
  -- Only applies to players not mobs
  if not player:is_player() then
    return 0
  end

  local player_name = player:get_player_name()
  local rate = wisdom_rate[player_name] or 0
  return rate
end

local function on_wisdom_unequip(player, _index, _itemstack)
  local player_name = player:get_player_name()
  wisdom_rate[player_name] = 0
end

local function on_wisdom_equip(player, _index, itemstack)
  local player_name = player:get_player_name()
  local enchant = get_ring_enchant(itemstack)
  wisdom_rate[player_name] = enchant
end

register_ring({
  model = 'wisdom',
  level = 1,
  on_equip = on_wisdom_equip,
  on_unequip = on_wisdom_unequip,
  mg_help = "While wearing this ring, your staves recharge at a faster rate.  A cursed ring will increase the time it takes for your staves to recharge."
})


local function on_stealth_unequip(player, _index, _itemstack)
  local player_name = player:get_player_name()
  mg_stealth.set_stealth_prop(player_name, 'ring_bonus', 0)
end

local function on_stealth_equip(player, _index, itemstack)
  local player_name = player:get_player_name()
  local enchant = get_ring_enchant(itemstack)
  mg_stealth.set_stealth_prop(player_name, 'ring_bonus', enchant)
end

register_ring({
  model = 'stealth',
  level = 3,
  on_equip = on_stealth_equip,
  on_unequip = on_stealth_unequip,
  mg_help = "Wearing this ring will reduce your stealth range.  Each enchantment point will reduce your stealth range by one point.  The lowest this ring can reduce your stealth range is 2.  A cursed ring will increase your stealth range."
})

local function get_ring_help(itemname)
    local def = core.registered_items[itemname]

    if not def then return "" end

    local help = ""

    if def.mg_help and def.mg_help ~= "" then
      local img = "<img name=".. def.inventory_image .." float=left width=64 height=64> "
      help = img .. "<b>" .. def.description .. ".</b> " .. def.mg_help
    end

    return help
end

local function get_ring_help_stats(item, player)
  local def = item:get_definition()
  if def.mg_help_player_stats then
    return def.mg_help_player_stats(item, player)
  end
  local meta = item:get_meta()
  local desc = meta:get_string("description") or def.description
  local id_message = mg_ring_id_help(item)
  return desc .. ". " .. def.mg_help .. "\n"
      .. id_message
end


mg_rings = {}
mg_rings.enchant_ring = enchant_ring
mg_rings.get_player_regen_rate = get_player_regen_rate

mg_rings.apply_player_transference = apply_player_transference

mg_rings.get_player_reaping_rate = get_player_reaping_rate

mg_rings.get_player_wisdom_rate = get_player_wisdom_rate

mg_rings.update_desc = update_desc
mg_rings.get_ring_help = get_ring_help
mg_rings.get_ring_help_stats = get_ring_help_stats
