-- Poison effect for player and mobs - like burning
-- poison can stack - the more poisons you have the more damage you take
-- looks like it adds to the current duration and adds to the damage for the entire duration
-- one poison for 5 seconds will do 5 damage total - 1 damage per second
-- 2 uses of a poison staff will last for about 9 seconds and do 1 + 9*2 = 18 damage (2 per second)
-- 3 uses of a poison staff will last for about 13 seconds and do 1 + 2 + 3*13 = 42 damange (3 per second)

local poison_storage = {}

-- poison_start
-- poison_duration
-- poison_damage

-- store in player meta_data
--
-- poison_object
-- poison_step - every step
-- poison_tick - once per second
--
-- check players
-- mobs will check in their on_step code
-- start with players

local function add_spawner(obj, color)
	local d = 0.2
	local pos = obj:get_pos()
	minetest.add_particlespawner({
		amount = 1,
		time = 1,
		minpos = {x=pos.x-d, y=pos.y+1, z=pos.z-d},
		maxpos = {x=pos.x+d, y=pos.y+2, z=pos.z+d},
		minvel = {x=-0.1, y=0, z=-0.1},
		maxvel = {x=0.1, y=0.1, z=0.1},
		minacc = {x=-0.1, y=0, z=-0.1},
		maxacc = {x=0.1, y=.1, z=0.1},
		minexptime = 0.5,
		maxexptime = 1,
		minsize = 0.5,
		maxsize = 1,
		collisiondetection = false,
		vertical = false,
		texture = "mcl_particles_effect.png^[colorize:"..color..":127",
	})
end


local default_data = {
  damage = 0,
  duration = 0,
  damage_timer = 0,
  total_timer = 0,
  cooldown_timer = 0
}


local function get_poison_data(target)
  if target:is_player() then
    local name = target:get_player_name()
    return poison_storage[name] or {
      damage = 0,
      duration = 0,
      damage_timer = 0,
      total_timer = 0
    }
  end

  local mob = target:get_luaentity()
  return mobkit.recall(mob, 'poison') or {
    damage = 0,
    duration = 0,
    damage_timer = 0,
    total_timer = 0
  }
end

local function set_poison_data(target, data)
  if target:is_player() then
    local name = target:get_player_name()
    poison_storage[name] = data
    return
  end

  local mob = target:get_luaentity()
  return mobkit.remember(mob, 'poison', data)
end

local function start_poison(target, damage, duration)
  local data = get_poison_data(target)

  local check_cooldown = true

  if not data.damage or data.damage == 0 then
    check_cooldown = false
    data.damage_timer = 0
    data.total_timer = 0

    -- update the hudbar
    if target:is_player() then
      hb.change_hudbar(target, "health", nil, nil, "hbhunger_icon_health_poison.png", nil, "hbhunger_bar_health_poison.png")
    end
  end

  -- 1 second cool down for applying poison effects again
  -- only check cooldown after first poison
  if check_cooldown and (data.cooldown_timer or 0) <= 1 then
    return
  end


  data.cooldown_timer = 0
  data.damage = (data.damage or 0) + damage
  data.duration = (data.duration or 0) + duration

  set_poison_data(target, data)
end

local function cure_poison(target)
  local data = get_poison_data(target)
  data.damage = 0
  data.duration = 0
  data.total_timer = 0
  if target:is_player() then
    -- reset hudbar
    hb.change_hudbar(target, "health", nil, nil, "hudbars_icon_health.png", nil, "hudbars_bar_health.png")
  end
  set_poison_data(target, data)
end

local particle_color = "#4E9331"

local function poison_on_step(target, dt)
  local data = get_poison_data(target)
  -- I'm being lazy here and calling the mcl burning code here
  -- maybe I should figure out how to abstract and merge these
  -- effects together
  local nodes = mg_burning.get_touching_nodes(target, {"group:poisons"}, data)
  -- save any changes made to the cache
  --set_poison_data(target, data)

  for _, pos in pairs(nodes) do
    local node = minetest.get_node(pos)
    local node_duration = core.get_item_group(node.name, "poisons")
    if node_duration > 0 then
      start_poison(target, 1, node_duration)
      break
    end
  end

  -- means damage is not being applied
  if(not data.damage or data.damage == 0 or data.duration == 0) then
    return
  end

  --inc timers
  data.damage_timer = data.damage_timer + dt
  data.cooldown_timer = data.cooldown_timer + dt
  data.total_timer = data.total_timer + dt

  add_spawner(target, particle_color)

  if data.damage_timer > 1 then
    data.damage_timer = 0
    -- apply damage
    if target:is_player() then
      local hp = target:get_hp()
      if hp > 0 then
        target:set_hp(hp - data.damage, {_mg_type = "poison"})
      end
    else
      local mob = target:get_luaentity()
      if (mob.hp or -1) > 0 then
        mobkit.make_sound(mob,'hurt')
        mobkit.hurt(mob, data.damage)
      end
    end

    -- check to see if the total timer is done
    if data.total_timer > data.duration then
      cure_poison(target)
    end
  end

  set_poison_data(target, data)
end

core.register_globalstep(function(dt)
	for _, player in pairs(core.get_connected_players()) do
    poison_on_step(player, dt)
  end
end)

core.register_on_joinplayer(function(player)
  local name = player:get_player_name()
	local storage = table.copy(default_data)
	local poison_data = player:get_meta():get_string("mg_poison:data")
	if poison_data ~= "" then
		storage = minetest.deserialize(poison_data) or storage
	end
	poison_storage[name] = storage
end)

local function on_leaveplayer(player)
  local name = player:get_player_name()
	local storage = poison_storage[name]
	if not storage then
		storage = {}
	end
	player:get_meta():set_string("mg_poison:data", minetest.serialize(storage))
	poison_storage[name] = nil
end

core.register_on_leaveplayer(function(player)
	on_leaveplayer(player)
end)

core.register_on_shutdown(function()
	for _,player in ipairs(minetest.get_connected_players()) do
		on_leaveplayer(player)
	end
end)

minetest.register_on_respawnplayer(function(player)
  cure_poison(player)
end)

local poison_api = {
  get_poison_data = get_poison_data,
  start_poison = start_poison,
  poison_on_step = poison_on_step,
  cure_poison = cure_poison
}

return poison_api
