local function S(s)
  return s
end

local function get_description(stack)
  local meta = stack:get_meta()
  local desc = meta:get_string("description")
  if not desc or desc == '' then
    local def = stack:get_definition()
    desc = def.description
  end
  return desc
end

local function update_desc(stack)
  if not stack then
    return
  end

  local tc = stack:get_tool_capabilities()
  local def = stack:get_definition()
  local meta = stack:get_meta()

  local uses = meta:get_int("uses")
  local is_identified = meta:get_int("identified")

  local magic_polarity = meta:get_int("magic_polarity")
  local magic_polarity_revealed = meta:get_int("magic_polarity_revealed")

  local protected = (tc.damage_groups.protected or 0)
  local enchant = (tc.damage_groups.enchant or 0)
  local unknown = ''

  -- if not identified
  -- only show enchants player has put into it
  if is_identified == 0 then
    enchant = (tc.damage_groups.player_enchant or 0)
    unknown = '?'
  end

  local magic_desc = ''
  if magic_polarity == -1 and (is_identified or magic_polarity_revealed) then
    magic_desc = 'cursed '
  end

  local sign = (enchant >= 0 and '+') or ''

  local show_protect = (protected > 0 and ' (protected)') or ''
  local description = sign..enchant..unknown..' '..magic_desc..def.description .. show_protect .. ' (' .. uses .. ' kills)'

  meta:set_string("description", description)
  return stack
end

--get_wielded_item


local wield_cache = {}

local function string_starts(str,start)
   return string.sub(str, 1, string.len(start)) == start
end

local function get_weapon_info(item, str, weakness, extra_enchant)
  local tool_capabilities = item:get_tool_capabilities()
  local meta = item:get_meta()
  local is_identified = meta:get_int("identified")
  local magic_polarity = meta:get_int("magic_polarity")
  local magic_polarity_revealed = meta:get_int("magic_polarity_revealed")
  local weapon_is_stuck = meta:get_int("weapon_is_stuck")
  local uses = meta:get_int("uses")

  extra_enchant = extra_enchant or 0

  local enchant = (tool_capabilities.damage_groups.enchant or 0) + extra_enchant
  local player_enchant = (tool_capabilities.damage_groups.player_enchant or 0) + extra_enchant
  local str_required = tool_capabilities.damage_groups.strength_required or 12
  local protected = (tool_capabilities.damage_groups.protected or 0)

  -- if you haven't id'd the weapon only count
  -- enchants player has put into it including
  -- strength requirements as well
  if is_identified == 0 then
    enchant = player_enchant
    str_required = str_required + enchant - player_enchant
  end

  local player_weakness = weakness or 0
  local player_str = str or 12
  local str_bonus = mg_strength.strengthModifier(str_required, player_str, player_weakness)
  local diff = player_str - str_required

  local dmg_min = tool_capabilities.damage_groups.fleshy or 1
  local dmg_max = tool_capabilities.damage_groups.fleshy_max or dmg_min

  local bonus = mg_strength.netEnchant(enchant, str_required, player_str, player_weakness)
  local multiplier = mg_strength.damageFraction(bonus)
  local lower_dmg = dmg_min * multiplier
  local upper_dmg = dmg_max * multiplier
  if lower_dmg < 1 then lower_dmg = 1 end
  if upper_dmg < 1 then upper_dmg = 1 end
  local avg_dmg = (lower_dmg + upper_dmg) / 2

  local accuracy = math.floor(mg_strength.damageFraction(mg_strength.netEnchant(
    enchant,
    str_required,
    player_str,
    player_weakness
  ))*100)

  return {
    enchant = enchant,
    protected = protected,
    uses = uses,
    avg_dmg = avg_dmg,
    accuracy = accuracy,
    player_str = player_str,
    str_diff = diff,
    str_bonus = str_bonus,
    is_identified = is_identified,
    magic_polarity = magic_polarity,
    magic_polarity_revealed = magic_polarity_revealed,
    weapon_is_stuck = weapon_is_stuck
  }
end

local function get_weapon_stats_desc(item, player)
  local player_str = mg_strength.get_strength_raw(player) or 12
  local player_weakness = mg_strength.get_weakness(player) or 0
  local def = item:get_definition()
  local info = get_weapon_info(item, player_str, player_weakness)
  local info_enchant = get_weapon_info(item, player_str, player_weakness, 1)

  local protected = (info.protected or 0)
  local enchant = (info.enchant or 0)
  local unknown = ''
  local unknown_desc = ''
  local protected_desc = ''

  if info.is_identified == 0 then
    unknown = '?'
    unknown_desc = 'This weapon will identify itself with ' .. (20 - info.uses) .. ' more kills\n'
  end

  if protected > 0 then
    protected_desc = 'Protected from acid'
  end

  local magic_desc = ''
  if info.magic_polarity == -1 and (info.is_identified or info.magic_polarity_revealed) then
    magic_desc = 'cursed '
  end

  local sign = (enchant >= 0 and '+') or ''

  local avg_dmg = string.format("%.1f", info.avg_dmg)
  local avg_dmg_enchant = string.format("%.1f", info_enchant.avg_dmg)
  local hit_desc = '\nYour hit will average around ' .. avg_dmg .. ' damage and '.. info.accuracy ..'% accuracy.'
  local range_reset = "  Range: " .. def.range ..
                      "  Reset Time: " .. def.tool_capabilities.full_punch_interval .. '\n'
  local enchant_hit_desc = '\nIf you enchant it, your hit will average around ' .. avg_dmg_enchant .. ' damage and '.. info_enchant.accuracy ..'% accuracy.'
  local message = ''
  if info.str_bonus < 0 then
    message = message .. 'Your inadequate strength of ' .. info.player_str .. ' gives you a penalty of ' .. info.str_bonus .. '.  ' .. (info.str_diff * -1) .. ' more strength would be ideal. '
  elseif info.str_bonus == 0 then
    message = message .. 'Your strength of ' .. info.player_str .. ' is adequate. '
  else
    message = message .. 'Your excess strength of ' .. info.player_str .. ' gives you a bonus of ' .. info.str_bonus .. '. '
  end

  local description = '<b>'..sign..enchant..unknown..' '..magic_desc..def.description..'</b> ' ..
    def.mg_help .. '\n' ..
    unknown_desc .. protected_desc .. range_reset .. message .. hit_desc .. enchant_hit_desc
  return description
end

local function get_wielded_weapon_info(player)
  local item = player:get_wielded_item()
  local player_str = mg_strength.get_strength_raw(player) or 12
  local player_weakness = mg_strength.get_weakness(player) or 0

  return get_weapon_info(item, player_str, player_weakness)
end

local function stick_weapon(stack)
  local meta = stack:get_meta()
  meta:set_int("weapon_is_stuck", 1)
  meta:set_int("magic_polarity_revealed", 1)
  return stack
end

local function check_weapon_change(player)
  local player_name = player:get_player_name()
  local item = player:get_wielded_item()
  local name = item:get_name()

  -- Already cached - return
  if wield_cache[player_name] == name then
    return
  end

  -- update cache
  wield_cache[player_name] = name

  -- not wield a weapon, then we dont care
  if not string_starts(name, "mg_tools:") then
    return
  end


  local info = get_wielded_weapon_info(player)

  -- if the weapon is cursed is wasnt already stuck
  -- make it stick
  if info.magic_polarity == -1 and info.weapon_is_stuck ~= 1 then
    local new_item = update_desc(stick_weapon(item))
    player:set_wielded_item(new_item)
    core.chat_send_player(player_name, "The weapon wields itself to your hand.")
  end

  -- if not currently wielding a cursed weapons
  -- search through player inventory to see if they
  -- previous wielded a cursed weapon
  if info.weapon_is_stuck ~= 1 then
    local inv = player:get_inventory()
    local main_list = inv:get_list("main")
    local cursed_idx = nil
    for i, search_item in ipairs(main_list) do
      if search_item then
        local meta = search_item:get_meta()
        local weapon_is_stuck = meta:get_int("weapon_is_stuck")
        if weapon_is_stuck == 1 then
          cursed_idx = i
          break
        end
      end
    end

    -- if found a cursed item - swap it out
    if cursed_idx and cursed_idx > 0 then
      local cursed_item = main_list[cursed_idx]
      player:set_wielded_item(cursed_item)
      inv:set_stack("main", cursed_idx, item)
      -- return otherwise chat message will be wrong
      return
    end
  end


  local avg_dmg = string.format("%.1f", info.avg_dmg)


  local hit_desc = 'Your hit will average around ' .. avg_dmg .. ' damage and hit with '.. info.accuracy ..'% accuracy.'

  -- TODO update HUD as well with this info
  local message = 'You wield ' .. get_description(item) .. '. '
  if info.str_bonus < 0 then
    message = message .. 'Your inadequate strength of ' .. info.player_str .. ' gives you a penalty of ' .. info.str_bonus .. '.  ' .. (info.str_diff * -1) .. ' more strength would be ideal. ' .. hit_desc
  elseif info.str_bonus == 0 then
    message = message .. 'Your strength of ' .. info.player_str .. ' is adequate. ' .. hit_desc
  else
    message = message .. 'Your excess strength of ' .. info.player_str .. ' gives you a bonus of ' .. info.str_bonus .. '. ' .. hit_desc
  end
  core.chat_send_player(player_name, message)
end

local timer = 0

core.register_globalstep(function(dt)
  timer = timer + dt

  -- Timer not done - exist
  if timer < 0.5 then
    return
  end

  timer = 0

	for _, player in pairs(core.get_connected_players()) do
    check_weapon_change(player)
  end
end)

local function register_weapon(def)
  minetest.register_tool("mg_tools:"..def.name, {
    description = def.description,
    mg_help = def.mg_help,
    inventory_image = def.image,
    tool_capabilities = {
      full_punch_interval = def.full_punch_interval,
      max_drop_level=0,
      damage_groups = def.damage_groups,
    },
    sound = def.sound or {breaks = "default_tool_breaks"},
    groups = def.groups or {sword = 1},
    range = def.range or 1.5,
    on_drop = function(item, dropper, pos)
      -- can't drop a stuck item
      local meta = item:get_meta()
      local weapon_is_stuck = meta:get_int("weapon_is_stuck")
      if weapon_is_stuck > 0 then
        return item
      end
      -- can't drop a cursed item
      return core.item_drop(item, dropper, pos)
    end
  })
end

-- Tools from voxel dungeon
--
register_weapon({
  name = "dagger",
	description = S("Dagger"),
  mg_help = "Not much, but better than your bare hands.",
	image = "voxeldungeon_tool_weapon_dagger.png",
  full_punch_interval = 1,
  damage_groups = {
    fleshy=3,
    fleshy_max=4,
    strength_required = 12,
  },
  range = 1.5,
})


register_weapon({
  name = "shortsword",
	description = S("Short Sword"),
  mg_help = "One step up from a dagger.",
	image = "voxeldungeon_tool_weapon_shortsword.png",
  full_punch_interval = 1.2,
  damage_groups = {
    fleshy=4, fleshy_max=7,
    strength_required = 13
  },
  range = 1.75,
})

register_weapon({
  name = "sword",
	description = S("Sword"),
  mg_help = "A fine weapon.",
	image = "voxeldungeon_tool_weapon_sword.png",
  full_punch_interval = 1.2,
  damage_groups = {
    fleshy=7, fleshy_max=9,
    strength_required = 14
  },
  range = 2,
})


register_weapon({
  name = "greatsword",
	description = S("Great Sword"),
  mg_help = "A bulky heavy weapon with a long reach.  It does a lot of damage, but requires a lot of strength.",
	image = "voxeldungeon_tool_weapon_greatsword.png",
  full_punch_interval = 1.5,
  damage_groups = {
    fleshy=14, fleshy_max=22,
    strength_required = 19
  },
  range = 3,
})


register_weapon({
  name = "quarterstaff",
	description = S("Quarterstaff"),
  mg_help = "A light weapon with good reach.",
	image = "voxeldungeon_tool_weapon_quarterstaff.png",
  full_punch_interval = 1.2,
  damage_groups = {
    fleshy=4, fleshy_max=7,
    strength_required = 13
  },
  range = 3,
})

register_weapon({
  name = "handaxe",
	description = S("Hand Axe"),
  mg_help = "Light quick weapon.",
	image = "voxeldungeon_tool_weapon_handaxe.png",
  full_punch_interval = 0.9,
  damage_groups = {
    fleshy=7, fleshy_max=9,
    strength_required = 15
  },
  range = 1.5,
})

register_weapon({
  name = "battleaxe",
	description = S("Battle Axe"),
	image = "voxeldungeon_tool_weapon_battleaxe.png",
  mg_help = "Heavy weapon with a quick reset.",
  groupcaps={
    choppy={times={[1]=2.20, [2]=1.00, [3]=0.60}, uses=20, maxlevel=3},
  },
  full_punch_interval = 1.5,
  damage_groups = {
    fleshy=12, fleshy_max=17,
    strength_required = 19
  },
  range = 1.75,
})


register_weapon({
  name = "mace",
	description = S("Mace"),
	image = "voxeldungeon_tool_weapon_mace.png",
  mg_help = "Heavy weapon with a long reset. Pushes the enemy back.",
  full_punch_interval = 3,
  damage_groups = {
    fleshy=16, fleshy_max=20,
    strength_required = 16,
    pushback=2
  },
	groups = {axe = 1},
  range = 1.75,
})

register_weapon({
  name = "warhammer",
	description = S("War Hammer"),
  mg_help = "Heaviest weapon with a long reset. Pushes the enemy back far.",
	image = "voxeldungeon_tool_weapon_warhammer.png",
  full_punch_interval = 3,
  damage_groups = {
    fleshy=25, fleshy_max=35,
    strength_required=20,
    pushback=5
  },
	groups = {axe = 1},
  range = 1.75,
})

register_weapon({
  name = "spear",
	description = S("Spear"),
  mg_help = "A light weapon with long reach and a quick reset.",
	image = "voxeldungeon_tool_weapon_spear.png",
  full_punch_interval = 1.2,
  damage_groups = {
    fleshy=4, fleshy_max=5,
    strength_required=13
  },
  range = 4,
})


register_weapon({
  name = "glaive",
	description = S("Glaive"),
  mg_help = "A heavy weapon with long reach and a quick reset.",
	image = "voxeldungeon_tool_weapon_glaive.png",
  full_punch_interval = 1.2,
  damage_groups = {
    fleshy=12, fleshy_max=17,
    strength_required=19
  },
  range = 5,
})

local function get_weapon_help(itemname)
    local def = core.registered_items[itemname]

    if not def then return "" end

    local help = ""

    if def.mg_help and def.mg_help ~= "" then
      local img = "<img name=".. def.inventory_image .." float=left width=64 height=64> "
      help = img .. "<b>" .. def.description .. ".</b> " .. def.mg_help
    end

    if def.tool_capabilities == nil then
      return help
    end

    local avg_dmg = (def.tool_capabilities.damage_groups.fleshy + def.tool_capabilities.damage_groups.fleshy_max)/2

    local pushback = def.tool_capabilities.damage_groups.pushback or "0"
    local pushback_text = ""
    if pushback ~= "0" then
      pushback_text = "  Pushback: " .. pushback
    end

    help = help .. "\nStrength Required: " .. def.tool_capabilities.damage_groups.strength_required ..
                   "  Avg Dmg: " .. avg_dmg ..
                   "  Range: " .. def.range ..
                   "  Reset Time: " .. def.tool_capabilities.full_punch_interval ..
                   pushback_text

    return help
end



local function degrade_weapon(stack, amount)
  local tc = stack:get_tool_capabilities()
  amount = amount or 1

  local protected = (tc.damage_groups.protected or 0)

  if protected > 0 then
    return
  end

  -- enchantment goes up
  local enchant = (tc.damage_groups.enchant or 0) - amount
  local player_enchant = (tc.damage_groups.player_enchant or 0) - amount

  -- set values in damage groups because thats the
  -- data we get on the punch callbacks
  tc.damage_groups.enchant = enchant
  tc.damage_groups.player_enchant = player_enchant

  -- update the item stack
  local meta = stack:get_meta()
  meta:set_tool_capabilities(tc)
  update_desc(stack)

  return stack
end

local function degrade_player_weapon(player, amount)
  if not player:is_player() then
    return
  end

  local item = player:get_wielded_item()
  local degraded_item = degrade_weapon(item, amount)
  if degraded_item then
    player:set_wielded_item(degraded_item)
  end
end

local function protect_weapon(stack)
  local tc = stack:get_tool_capabilities()

  tc.damage_groups.protected = 1

  -- update the item stack
  local meta = stack:get_meta()
  meta:set_tool_capabilities(tc)
  update_desc(stack)

  return stack
end

local function curse_weapon(stack)
  local meta = stack:get_meta()
  meta:set_int("magic_polarity", -1)
  return stack
end

local function enchant_weapon(stack, amount, from_player)
  local tc = stack:get_tool_capabilities()

  amount = amount or 1

  -- enchantment goes up
  local enchant = (tc.damage_groups.enchant or 0) + amount
  -- strength requirement goes down
  local strength_required = (tc.damage_groups.strength_required or 0) - amount
  if amount < 0 then
    strength_required = tc.damage_groups.strength_required or 0
  end

  -- set values in damage groups because thats the
  -- data we get on the punch callbacks
  tc.damage_groups.enchant = enchant
  tc.damage_groups.strength_required = strength_required

  if from_player then
    tc.damage_groups.player_enchant = (tc.damage_groups.player_enchant or 0) + amount
  end

  -- update the item stack
  local meta = stack:get_meta()

  -- Undo weapon curse
  if amount > 0 then
    meta:set_int("weapon_is_stuck", 0)
    meta:set_int("magic_polarity", 0)
  end

  meta:set_tool_capabilities(tc)
  update_desc(stack)

  return stack
end

-- for auto identification
-- only increment if kill a monster with the weapon
local function set_use(stack, amount)
  amount = amount or 1
  local meta = stack:get_meta()


  -- might be fun to actually track this...
  -- local is_identified = meta:get_int("identified")
  -- if is_identified > 0 then
    -- dont care if it has already been identified
    -- return nil
  -- end

  local uses = meta:get_int("uses") + amount

  meta:set_int("uses", uses)

  if uses >= 20 then
    meta:set_int("identified", 1)
  end
  update_desc(stack)

  return stack
end

local function set_player_weapon_use(player, amount)
  if not player:is_player() then
    return
  end

  local item = player:get_wielded_item()
  local name = item:get_name()

  -- only care about weapons
  if not string_starts(name, "mg_tools:") then
    return
  end

  local updated_item = set_use(item, amount)
  if updated_item then
    player:set_wielded_item(updated_item)
  end
end


mg_tools = {}
mg_tools.enchant_weapon = enchant_weapon
mg_tools.curse_weapon = curse_weapon
mg_tools.degrade_player_weapon = degrade_player_weapon
mg_tools.protect_weapon = protect_weapon
mg_tools.get_wielded_weapon_info = get_wielded_weapon_info
mg_tools.update_desc = update_desc
mg_tools.set_player_weapon_use = set_player_weapon_use
mg_tools.get_weapon_help = get_weapon_help
mg_tools.get_weapon_stats_desc = get_weapon_stats_desc
