local path=minetest.get_modpath("mg_bolts")
local charm = dofile(path..'/charms.lua')
local recharge = dofile(path..'/recharge.lua')

local get_charm_enchant = recharge.get_charm_enchant
local use_charge = recharge.use_charge
local calc_staff_duration = recharge.calc_staff_duration
local calc_staff_duration_from_def = recharge.calc_staff_duration_from_def

local function heal_object(obj, health_percent)
  if obj:is_player() then
    local hp = obj:get_hp()
    local props = obj:get_properties()
    local hp_delta = math.floor(props.hp_max * health_percent / 100)
    local new_hp = hp + hp_delta
    print('old hp '..hp..' new hp '..new_hp .. ' percent ' .. health_percent)
    obj:set_hp(new_hp)
    minetest.chat_send_all(obj:get_player_name() .. " looks healthier.")
    return
  elseif obj:get_luaentity().mg_id ~= nil then
    local mob = obj:get_luaentity()
    -- full health - don't heal
    if not mob.max_hp or mob.hp == mob.max_hp then
      return
    end

    local hp_delta = math.floor(mob.max_hp * health_percent / 100)
    mob.hp = mob.hp + hp_delta
    if mob.hp > mob.max_hp then
      mob.hp = mob.max_hp
    end

    minetest.chat_send_all(mob.name .. " looks healthier.")
  end
end


mg_bolts.register_bolt("fire",{
	description = "Fire Bolt",
	texture = "mg_bolts_fire.png",
	damage = 3,
  calc_damage = function(enchant)
    enchant = enchant or 1
    local dmg_min = math.floor(0.75 * (enchant + 2))
    local dmg_max = math.floor(2.5 * enchant + 4)
    local dmg = math.random(dmg_min, dmg_max)
    return dmg
  end,
  stop_on_node = 1,
  stop_on_object = 1,
	on_hit_sound = "mg_spell_end",
	on_hit_node = function(_self, _pos, _user, oldpos)
    -- Find a nearby air node and light it on fire
    local p = minetest.find_node_near(oldpos, 1, {"air", "group:air"})
    if p then
      minetest.set_node(p, {name = 'fire:basic_flame'})
    end
  end,
	on_timeout = function(_self, pos, _user, _oldpos)
    -- when fire node ends, set an air node near it on fire
    local pa = core.find_node_near(pos, 1, {"air", "group:air"})
    if pa then
      core.set_node(pa, {name = 'fire:basic_flame'})
    end
  end,
  -- Fire bolt sets the air it passes through on fire
  -- If it passes near flammable terrain
	on_step = function(_self, pos, _user, _oldpos)
    local pf = minetest.find_node_near(pos, 1, {"group:flammable"})
    local flame = minetest.find_node_near(pos, 1, {'fire:basic_flame'})
    -- Light on fire if near flammable terrain and not yet near fire
    if pf and not flame then
      local pa = minetest.find_node_near(pf, 1, {"air", "group:air"})
      if pa then
        minetest.set_node(pa, {name = 'fire:basic_flame'})
      end
    end
  end,
	on_hit_object = function(_self, target, _hp, _user, pos)
    -- Find a nearby air node and light it on fire
    local p = minetest.find_node_near(pos, 1, {"air", "group:air"})
    if p then
      minetest.set_node({x = p.x, y = p.y, z = p.z}, {name = 'fire:basic_flame'})
    end

    -- burn the target
    mg_oe.start_effect("burning", target, 7)
	end
})

mg_bolts.register_bolt("blinking",{
	description = "Blinking Bolt",
	texture = "mg_bolts_light.png",
	damage = 0,
	drop_chance = 0,
  stop_on_object = 0,
  stop_on_node = 1,
	on_hit_sound = "bows_arrow_hit",
	on_remove = function(_self, player, oldpos)
    local i = 1
    local p = core.find_node_near(oldpos, i, {"air", "mg_torch:air_light_8"})
    while p == nil and i < 10 do
      i = i + 1
      p = core.find_node_near(oldpos, i, {"air", "mg_torch:air_light_8"})
    end
    local prop = player:get_properties()
    local adjust = (prop.eye_height or 1.23)
    player:set_pos(vector.add(p, {x = 0, y = -1*adjust, z = 0}))
    core.sound_play("enderpearl_teleport", {max_hear_distance = 10, pos = player:get_pos()})
  end,
})

mg_bolts.register_bolt("tunneling",{
	description = "Tunneling Bolt",
	texture = "mg_bolts_light.png",
	damage = 0,
  stop_on_object = 0,
  stop_on_node = 0,
	on_hit_sound = "bows_arrow_hit",
  -- Fire bolt sets the air it passes through on fire
	on_hit_node = function(_self, pos, _user, _oldpos)
    minetest.set_node({x = pos.x, y=pos.y, z=pos.z}, {name = 'air'})
    minetest.set_node({x = pos.x, y=pos.y - 1, z=pos.z}, {name = 'air'})
  end,
  on_hit_object = function(self, target, _hp, _user, _pos)
    if target:is_player() then
      return
    end
    local mob = target:get_luaentity()
    if mob.behaviors and mob.behaviors.in_wall then
      mg_mobs.hq_die(mob)
    end
  end
})

-- TODO only apply damage once per entity
mg_bolts.register_bolt("lightning",{
	description = "Lightning Bolt",
	texture = "mg_bolts_light.png",
	damage = 1,
  -- This is the same as the firebolt as well
  calc_damage = function(enchant)
    enchant = enchant or 1
    local dmg_min = math.floor(0.75 * (enchant + 2))
    local dmg_max = math.floor(2.5 * enchant + 4)
    local dmg = math.random(dmg_min, dmg_max)
    return dmg
  end,
	drop_chance = 0,
  stop_on_object = 0,
  stop_on_node = 0,
  reflection = 1,
	on_hit_sound = "bows_arrow_hit"
})

mg_bolts.register_bolt("poison",{
	description = "Poison Bolt",
	texture = "mg_bolts_poison.png",
	damage = 1,
	drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
	on_hit_sound = "bows_arrow_hit",
	on_hit_object = function(self, target, _hp, _user, _pos)
    local enchant = self.enchant
    local duration = math.floor(5 * 1.3^ (enchant-2))
    mg_oe.start_effect("poison", target, duration, 1)
  end
})

mg_bolts.register_bolt("protection",{
  description = "Protection Bolt",
  texture = "mg_bolts_light.png",
  damage = 0,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_sound = "mg_good_spell",
  on_hit_object = function(self, target, _hp, _user, _pos)
    local enchant = self.enchant
    local shield = math.ceil(13 * 1.4^(enchant-2))
    local duration = 20
    mg_oe.start_effect("protection", target, duration,  shield)
  end
})

mg_bolts.register_bolt("negation",{
  description = "Negation Bolt",
  texture = "mg_bolts_negation.png",
  damage = 0,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_object = function(self, target, _hp, _user, _pos)
    if target:is_player() and target:get_hp() > 0 and target:get_properties().pointable then
      core.sound_play("enderpearl_teleport", {max_hear_distance = 10, pos = target:get_pos()})
      mg_oe.negate_effects(target)
    end
    if not target:is_player() and target:get_hp() > 0 then
      local mob = target:get_luaentity()
      if (mob.hp or -1) > 0 then
        mg_mobs.negate(mob)
        core.sound_play("enderpearl_teleport", {max_hear_distance = 10, pos = target:get_pos()})
      end
    end
  end
})

mg_bolts.register_bolt("domination",{
  description = "Domination Bolt",
  texture = "mg_bolts_negation.png",
  damage = 0,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_object = function(self, target, _hp, user, _pos)
    local player_name = user:get_player_name()
    if target:is_player() and target:get_hp() > 0 and target:get_properties().pointable then
      -- nothing happens
      local target_name = target:get_player_name()
      local message = "You hit " .. target_name .. " but nothing happens.  You cannot use a wand of domination on other players."
      core.chat_send_player(player_name, message)
    end
    if not target:is_player() and target:get_hp() > 0 then
      local mob = target:get_luaentity()
      if (mob.hp or -1) > 0 then
        local success = mg_mobs.dominate(mob, player_name)
        if success then
          local message = "You've dominated the monster and it has become your ally."
          core.chat_send_player(player_name, message)
          core.sound_play("enderpearl_teleport", {max_hear_distance = 10, pos = target:get_pos()})
        else
          local message = "You failed to dominate the monster."
          core.chat_send_player(player_name, message)
        end
      end
    end
  end
})

mg_bolts.register_bolt("haste",{
  description = "Haste Bolt",
  texture = "mg_bolts_light.png",
  damage = 0,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_sound = "mg_good_spell",
  on_hit_object = function(self, target, _hp, _user, _pos)
    local duration = 20
    mg_oe.start_effect("speed", target, duration, true)
  end
})

mg_bolts.register_bolt("spark",{
  description = "Spark Bolt",
  texture = "mg_bolts_light.png",
  damage = 2,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_sound = "zap15"
})

mg_bolts.register_bolt("web",{
	description = "Web Bolt",
	texture = "mg_bolts_light.png",
	damage = 0,
	drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
	on_hit_sound = "bows_arrow_hit",
	on_hit_object = function(_self, _target, _hp, _user, pos)
    mg_effects.net({pos=pos})
	end,
	on_hit_node = function(_self, pos, _user, _oldpos)
    mg_effects.net({pos=pos})
  end
})

mg_bolts.register_bolt("teleportation",{
	description = "Teleportation Bolt",
	texture = "mg_bolts_light.png",
	damage = 0,
	drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
	on_hit_sound = "enderpearl_teleport",
	on_hit_object = function(_self, target, _hp, _user, _pos)
    mg_effects.teleport(target)
	end
})

mg_bolts.register_bolt("polymorph",{
	description = "Polymorph Bolt",
	texture = "mg_bolts_light.png",
	damage = 0,
	drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
	on_hit_sound = "enderpearl_teleport",
	on_hit_object = function(_self, target, _hp, _user, pos)
    -- won't work on players...
    if target:is_player() then
      return
    end
    -- get current hp percentage of target
    local old_mob = target:get_luaentity()
    local percent = old_mob.hp / old_mob.max_hp

    -- remove the old mob
		target:remove()

    -- pick a random monster
    local m_list = mg_arch.monsters.name_list
    local new_monster_name = m_list[math.random(1, #m_list)]
    print('try to spawn: ' .. new_monster_name)
    -- spawn a new one with current velociy
    local obj = mg_mobs.spawn(pos, new_monster_name)
    local mob = obj:get_luaentity()
    -- hp should get percentage
    mob.hp = mob.max_hp * percent
	end
})

mg_bolts.register_bolt("obstruction",{
	description = "Obstruction Bolt",
	texture = "mg_bolts_light.png",
	damage = 0,
	drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
	on_hit_sound = "bows_arrow_hit",
	on_hit_node = function(self, _pos, _user, arrow_pos)
    mg_effects.obstruction({pos = arrow_pos, radius = self.enchant})
  end,
	on_hit_object = function(self, _target, _hp, _user, arrow_pos)
    mg_effects.obstruction({pos = arrow_pos, radius = self.enchant})
	end,
	on_remove = function(self, _player, oldpos)
    mg_effects.obstruction({pos = oldpos, radius = self.enchant})
  end
})

mg_bolts.register_bolt("slowness",{
  description = "Slowness Bolt",
  texture = "mg_bolts_light.png",
  damage = 0,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_sound = "bows_arrow_hit",
  on_hit_object = function(_self, target, _hp, _user, _pos)
    mg_oe.start_effect("slowness", target, 30, true)
  end
})

mg_bolts.register_bolt("paralysis",{
  description = "Paralysis Bolt",
  texture = "mg_bolts_light.png",
  damage = 0,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_sound = "bows_arrow_hit",
  on_hit_object = function(_self, target, _hp, _user, _pos)
    mg_oe.start_effect("gas_paralysis", target, 20)
  end
})

mg_bolts.register_bolt("confusion",{
  description = "Confusion Bolt",
  texture = "mg_bolts_light.png",
  damage = 0,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_sound = "bows_arrow_hit",
  on_hit_object = function(_self, target, _hp, _user, _pos)
    mg_oe.start_effect("confusion", target, 12)
  end
})

mg_bolts.register_bolt("healing",{
  description = "Healing Bolt",
  texture = "mg_bolts_fire.png",
  damage = 0,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_sound = "enderpearl_teleport",
  on_hit_object = function(_self, target, _hp, _user, _pos)
    heal_object(target, 20)
  end
})

mg_bolts.register_bolt("discord",{
  description = "Discord Bolt",
  texture = "mg_bolts_negation.png",
  damage = 0,
  drop_chance = 0,
  stop_on_object = 1,
  stop_on_node = 1,
  reflection = 0,
  on_hit_sound = "bows_arrow_hit",
  on_hit_object = function(self, target, _hp, _user, _pos)
    local enchant = self.enchant or 1
    local duration = enchant * 4
    mg_oe.start_effect("discordant", target, duration)
  end
})


minetest.register_craftitem("mg_bolts:wand_paralysis", {
  description = "Wand of Paralysis",
  inventory_image = "mg_wand_teleport.png",
  stack_max = 1,
  mg_help = "Shoots a magical bolts that will paralyze a monster.",
  _enchant = 0,
  _recharge_inc = 10,
  _base_duration = 0,
  on_use = function(itemstack, user, _pointed_thing)

    local enchant = get_charm_enchant(itemstack)
    local range = 2
    local meta = itemstack:get_meta()
    local max_charges = meta:get_int("recharge_max_charges")

    return use_charge(
      max_charges,
      0,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:paralysis', user, range, enchant)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:wand_confusion", {
  description = "Wand of Paralysis",
  inventory_image = "mg_wand_teleport.png",
  stack_max = 1,
  mg_help = "Shoots a magical bolts that will confuse a monster.",
  _enchant = 0,
  _recharge_inc = 10,
  _base_duration = 0,
  on_use = function(itemstack, user, _pointed_thing)

    local enchant = get_charm_enchant(itemstack)
    local range = 2
    local meta = itemstack:get_meta()
    local max_charges = meta:get_int("recharge_max_charges")

    return use_charge(
      max_charges,
      0,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:confusion', user, range, enchant)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:wand_teleportation", {
  description = "Wand of Teleportation",
  inventory_image = "mg_wand_teleport.png",
  stack_max = 1,
  mg_help = "Shoots a magical bolts that teleports the monster it hits to a random place in the level. Does not recharge.",
  _enchant = 0,
  _recharge_inc = 2,
  _base_duration = 0,
  on_use = function(itemstack, user, _pointed_thing)

    local enchant = get_charm_enchant(itemstack)
    local range = 2
    local meta = itemstack:get_meta()
    local max_charges = meta:get_int("recharge_max_charges")

    return use_charge(
      max_charges,
      0,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:teleportation', user, range, enchant)
      end
    )
  end
})

core.register_craftitem("mg_bolts:wand_negation", {
  description = "Wand of Negation",
  inventory_image = "mg_wand_negation.png",
  stack_max = 1,
  mg_help = "Shoots a magical bolts that will negate what it hits.  The monster it hits will lose their magical abilities.  Enchanting this wand will give it four more charges.",
  _enchant = 0,
  _recharge_inc = 4,
  _base_duration = 0,
  on_use = function(itemstack, user, _pointed_thing)

    local enchant = get_charm_enchant(itemstack)
    local range = 2
    local meta = itemstack:get_meta()
    local max_charges = meta:get_int("recharge_max_charges")

    return use_charge(
      max_charges,
      0,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:negation', user, range, enchant)
      end
    )
  end
})

core.register_craftitem("mg_bolts:wand_domination", {
  description = "Wand of Domination",
  inventory_image = "mg_wand_domination.png",
  stack_max = 1,
  mg_help = "Shoots magical bolts that will tame a monster and make it your ally.  The more hurt the mob is, the more likely it is to succeed.  Enchanting this wand will give it 2 more charges.  This wand does nothing against other players.",
  _enchant = 0,
  _recharge_inc = 2,
  _base_duration = 0,
  on_use = function(itemstack, user, _pointed_thing)

    local enchant = get_charm_enchant(itemstack)
    local range = 2
    local meta = itemstack:get_meta()
    local max_charges = meta:get_int("recharge_max_charges")

    return use_charge(
      max_charges,
      0,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:domination', user, range, enchant)
      end
    )
  end
})


minetest.register_craftitem("mg_bolts:wand_polymorph", {
  description = "Wand of Polymorph",
  inventory_image = "mg_wand_polymorph.png",
  mg_help = "Shoots a bolt magical bolt that will transform one type of monster into another type. Does not recharge.",
  stack_max = 1,
  _enchant = 0,
  _recharge_inc = 2,
  _base_duration = 0,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local range = 2
    local meta = itemstack:get_meta()
    local max_charges = meta:get_int("recharge_max_charges")

    return use_charge(
      max_charges,
      0,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:polymorph', user, range, enchant)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:wand_slowness", {
  description = "Wand of Slowness",
  inventory_image = "mg_wand_slowness.png",
  stack_max = 1,
  mg_help = "This wand will cause a creature to move at half its ordinary speed for 30 seconds.",
  _enchant = 0,
  _recharge_inc = 2,
  _base_duration = 0,
  on_use = function(itemstack, user, _pointed_thing)

    local enchant = get_charm_enchant(itemstack)
    local range = 2
    local meta = itemstack:get_meta()
    local max_charges = meta:get_int("recharge_max_charges")

    return use_charge(
      max_charges,
      0,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:slowness', user, range, enchant)
      end
    )
  end
})



--- Staves
--
minetest.register_craftitem("mg_bolts:fire_staff", {
  description = "A fire staff",
  inventory_image = "mg_staff_fire.png",
  mg_help = "Shoots a fire bolt that will catch the air around it on fire and any monster it hits.",
  stack_max = 1,
  _enchant = 2,
  _base_duration = 500,
  range = 1.5,
  mg_calc_range = function(enchant)
    return (2 + (2 * enchant))/10
  end,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = calc_staff_duration(user, itemstack)
    -- range is in seconds - we travel 10 blocks per second
    -- local range = (2 + (2 * enchant))/10
    local def = itemstack:get_definition()
    local range = def.mg_calc_range(enchant)
    local max_charges = enchant
    return recharge.use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:fire', user, range, enchant)
      end
    )
  end
})


minetest.register_craftitem("mg_bolts:blinking_staff", {
  description = "Staff of Blinking",
  inventory_image = "mg_staff_blinking.png",
  mg_help = "Shoots a magical bolt that will teleport the player to where it ends or hits a block.",
  stack_max = 1,
  _enchant = 2,
  _base_duration = 1000,
  range = 1.5,
  mg_calc_range = function(enchant)
    return (2 + (2 * enchant))/10
  end,
  on_use = function(itemstack, user, _pointed_thing)

    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = calc_staff_duration(user, itemstack)
    -- range is in seconds - we travel 10 blocks per second
    local def = itemstack:get_definition()
    local range = def.mg_calc_range(enchant)
    local max_charges = enchant

    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:blinking', user, range, enchant)
      end
    )
  end
})


minetest.register_craftitem("mg_bolts:tunneling_staff", {
  description = "A staff of tunneling",
  inventory_image = "mg_staff_tunneling.png",
  mg_help = "Shoots a bolt that will dig out the walls and floors of the dungeon.",
  stack_max = 1,
  _enchant = 1,
  _base_duration = 500,
  mg_calc_range = function(enchant)
    return enchant/10
  end,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local max_charges = enchant
    local recharge_duration = calc_staff_duration(user, itemstack)
    -- range is in seconds - we travel 10 blocks per second
    local def = itemstack:get_definition()
    local range = def.mg_calc_range(enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:tunneling', user, range, enchant)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:lightning_staff", {
  description = "A staff of lightning",
  mg_help = "Shoots a bolt of lightning that will pass and harm monsters and bounce off the walls, ceilings and floors of the dungeon.",
  inventory_image = "magic_materials_enchanted_staff.png",
  stack_max = 1,
  _enchant = 2,
  _base_duration = 500,
  mg_calc_range = function(enchant)
    return (2 + (2 * enchant))/10
  end,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local max_charges = enchant
    local recharge_duration = calc_staff_duration(user, itemstack)
    -- range is in seconds - we travel 10 blocks per second
    local def = itemstack:get_definition()
    local range = def.mg_calc_range(enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:lightning', user, range, enchant)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:poison_staff", {
  description = "A staff of poison",
  inventory_image = "mg_staff_poison.png",
  mg_help = "Shoots a magical bolt that poison any monster it hits.  The poison will take away a little HP from the monster every turn.",
  stack_max = 1,
  _enchant = 2,
  _base_duration = 500,
  mg_calc_range = function(enchant)
    return (2 + (2 * enchant))/10
  end,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local max_charges = enchant
    local recharge_duration = calc_staff_duration(user, itemstack)
    local def = itemstack:get_definition()
    local range = def.mg_calc_range(enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:poison', user, range, enchant)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:protection_staff", {
  description = "A staff of protection",
  inventory_image = "mg_staff_protection.png",
  mg_help = "Shoots a magical bolt that will add a shield of protection around any monster or player it hits.  This is useful if others are playing with you.",
  stack_max = 1,
  _enchant = 2,
  _base_duration = 500,
  mg_calc_range = function(enchant)
    return (2 + (2 * enchant))/10
  end,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local max_charges = enchant
    local recharge_duration = calc_staff_duration(user, itemstack)
    local def = itemstack:get_definition()
    local range = def.mg_calc_range(enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:protection', user, range, enchant)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:discord_staff", {
  description = "A staff of discord",
  inventory_image = "mg_staff_discord.png",
  mg_help = "The purple light from this staff will alter the perceptions of all creatures to think the target is their enemy. Strangers and allies alike will turn on an affected creature.",
  stack_max = 1,
  _enchant = 2,
  _base_duration = 500,
  mg_calc_range = function(enchant)
    return (2 + (2 * enchant))/10
  end,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local max_charges = enchant
    local recharge_duration = calc_staff_duration(user, itemstack)
    local def = itemstack:get_definition()
    local range = def.mg_calc_range(enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:discord', user, range, enchant)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:conjure_staff", {
  description = "A staff of conjuring",
  inventory_image = "mg_staff_conjure.png",
  mg_help = "A flick of this staff summons a number of phantom blades to fight on your behalf.",
  stack_max = 1,
  _enchant = 2,
  _base_duration = 250,
  mg_calc_range = function(enchant)
    return (2 + (2 * enchant))/10
  end,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local max_charges = enchant
    local recharge_duration = calc_staff_duration(user, itemstack)
    local number = math.floor(enchant * 3/2)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_mobs.player_summon(user, "spectral_blade", number)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:obstruction_staff", {
  description = "A staff of obstruction",
  inventory_image = "mg_staff_obstruction.png",
  mg_help = "Shoots a magical bolt that creates a magical obstruction of purple crystals around the location it hits.  The crystals dissipate over time. The higher it's enchanted the larger the area will be that the crystals will enclose.",
  stack_max = 1,
  _enchant = 2,
  _base_duration = 1000,
  mg_calc_range = function(enchant)
    return (2 + (2 * enchant))/10
  end,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local max_charges = enchant
    local recharge_duration = calc_staff_duration(user, itemstack)
    local def = itemstack:get_definition()
    local range = def.mg_calc_range(enchant)
    -- need to update the radius of the obstrution
    -- to depend on enchant
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_bolts.shoot('mg_bolts:obstruction', user, range, enchant)
      end
    )
  end
})

local function get_staff_help(itemname)
    local def = core.registered_items[itemname]

    if not def then return "" end

    local help = ""

    if def.mg_help and def.mg_help ~= "" then
      local img = "<img name=".. def.inventory_image .." float=left width=64 height=64> "
      help = img .. "<b>" .. def.description .. ".</b> " .. def.mg_help
    end

    if def._base_duration ~= nil then
      -- TODO need default range as well - that's a function of enchantment
      -- it's different for each staff though
      help = help .. "  Recharge: " .. def._base_duration .. "s"
    end

    return help
end

local function get_staff_help_stats(item, user)
  local def = item:get_definition()
  local enchant = get_charm_enchant(item)
  local recharge_duration = math.floor(calc_staff_duration_from_def(user, def, enchant))
  local recharge_duration_new = math.floor(calc_staff_duration_from_def(user, def, enchant + 1))
  local range = def.mg_calc_range(enchant)*10
  local range_new = def.mg_calc_range(enchant + 1)*10

  local wear = item:get_wear()
  local meta = item:get_meta()
  local now = core.get_gametime()
  local start = meta:get_int("recharge_start")
  local time_left = math.ceil(recharge_duration) - (now - start)

  local message0 = def.description .. ". " .. def.mg_help .. "\n"
  local message1 = "This staff has ".. (enchant - wear) .." of ".. enchant .." charges"
  if wear > 0 then
    message1 = message1 .. " and will gain an extra charge in " .. time_left .. " seconds"
  end
  message1 = message1 .. ".\n"
  local message2 = "The time to regain a charge is "..recharge_duration.." seconds and the range is "..range.." blocks.\n"
  local message3 = "Enchanting this staff will give it an extra charge, reduce the recharge time to "..recharge_duration_new.." seconds, increase the potency of the bolt and extend the bolt range to "..range_new.." blocks."

  return message0 .. message1 .. message2 .. message3
end

local function get_wand_help_stats(item)
  local def = item:get_definition()

  local meta = item:get_meta()
  local wear = item:get_wear() or 0
  local max_charges = meta:get_int("recharge_max_charges")
  local charges_remaining = max_charges - wear
  local recharge_inc = def._recharge_inc

  local message0 = def.description .. ". " .. def.mg_help .. "\n"
  local message1 = "This wand has ".. charges_remaining .." charges remaining.\n"
  local message2 = "Enchanting this staff will give it ".. recharge_inc .." more charge"
  if recharge_inc > 1 then
    message2 = message2 .. "s"
  end
  message2 = message2 .. "."

  return message0 .. message1 .. message2
end




minetest.register_craftitem("mg_bolts:health_charm", {
  description = "A charm of health",
  inventory_image = "mg_charm_health.png",
  mg_help = "This charm will restore a percentage of HP back to the player.  The higher its enchanted the more HP it will restore.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('health', enchant)
    local health_percent = charm.calc_charm_healing(enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        local hp = user:get_hp()
        local props = user:get_properties()
        local hp_delta = math.floor(props.hp_max * health_percent / 100)
        local new_hp = hp + hp_delta
        print('old hp '..hp..' new hp '..new_hp .. ' percent ' .. health_percent)
        user:set_hp(new_hp)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:haste_charm", {
  description = "A charm of Haste",
  inventory_image = "magic_materials_light_rune.png",
  mg_help = "This charm will temporarily increase a players speed.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('haste', enchant)
    local duration = charm.calc_charm_duration('haste', enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_oe.start_effect("speed", user, duration)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:levitation_charm", {
  description = "Charm of Levitation",
  inventory_image = "magic_materials_ice_rune.png",
  mg_help = "This charm will temporarily allow a player to float over lava or water.  If you jump while using this charm you will go up and not come down until the effect of the charm wears off.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('levitation', enchant)
    local duration = charm.calc_charm_duration('levitation', enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_oe.start_effect("levitation", user, duration)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:shattering_charm", {
  description = "A charm of shattering",
  inventory_image = "magic_materials_storm_rune.png",
  mg_help = "This charm turns the rock around the player into purple crystal that dispates over time, creating air where rock used to be.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('shattering', enchant)
    local radius = charm.calc_charm_shattering(enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        local pos = user:getpos()
        mg_effects.shattering({pos = pos, radius = radius})
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:negation_charm", {
  description = "A charm of negation",
  inventory_image = "magic_materials_void_rune.png",
  mg_help = "This charm will negate any creatures in a certain radius around the player including themselves.  Negation removes a monster's magical abilities including spell casting.  The radius is the enchant level times three plus one.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('shattering', enchant)
    local radius = charm.calc_charm_negate(enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        local pos = user:getpos()
        mg_effects.negate({pos = pos, radius = radius})
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:recharging_charm", {
  description = "A charm of recharging",
  inventory_image = "magic_materials_earth_rune.png",
  mg_help = "This charm will recharge all of a players staves.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('recharging', enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        recharge.recharge_staves(user)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:teleportation_charm", {
  description = "Charm of Teleportation",
  inventory_image = "magic_materials_teleport_rune.png",
  mg_help = "This charm will teleport the player to a random place on the current depth.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('teleportation', enchant)
    local max_charges = 1
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_effects.teleport(user)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:fire_immunity_charm", {
  description = "Charm of Fire Immunity",
  inventory_image = "magic_materials_fire_rune.png",
  mg_help = "This charm will temporarily grant the player immunity to fire and lava damage.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('fire_immunity', enchant)
    local duration = charm.calc_charm_duration('fire_immunity', enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_oe.start_effect("fire_immunity", user, duration)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:invisibility_charm", {
  description = "Charm of Invisibility",
  inventory_image = "magic_materials_invisible_rune.png",
  mg_help = "This charm will temporarily lower the players stealth to 1, making it very easy to sneak around monsters.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('invisibility', enchant)
    local duration = charm.calc_charm_duration('invisibility', enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_oe.start_effect("invisibility", user, duration)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:protection_charm", {
  description = "Charm of Protection",
  inventory_image = "mg_charm_protection.png",
  mg_help = "This charm will temporarily add a shield around the player that must be broken through before they take damage to their HP.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('protection', enchant)
    -- always lasts 20 seconds
    -- this calculates the level of the shield
    local shield = charm.calc_charm_duration('protection', enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_oe.start_effect("protection", user, 20, shield)
      end
    )
  end
})

minetest.register_craftitem("mg_bolts:guardian_charm", {
  description = "Charm of Guardian Sprit",
  inventory_image = "magic_materials_invisible_rune.png",
  mg_help = "This charm summons a guardian spirit wielding a battle axe to fight along side the player temporarily.",
  stack_max = 1,
  _enchant = 1,
  range = 1.5,
  on_use = function(itemstack, user, _pointed_thing)
    local max_charges = 1
    local enchant = get_charm_enchant(itemstack)
    local recharge_duration = charm.calc_charm_recharge('guardian', enchant)
    local duration = charm.calc_charm_duration('guardian', enchant)
    return use_charge(
      max_charges,
      recharge_duration,
      itemstack,
      user,
      function()
        mg_mobs.player_summon(user, "guardian_spirit", 1, duration)
      end
    )
  end
})

local function get_charm_help_stats(item)
  local def = item:get_definition()
  local name = item:get_name()
  local enchant = get_charm_enchant(item)

  local charm_kind0 = string.gsub(name, "mg_bolts:", "")
  local charm_kind  = string.gsub(charm_kind0, "_charm", "")


  local recharge_duration = charm.calc_charm_recharge(charm_kind, enchant)
  local recharge_duration_new = charm.calc_charm_recharge(charm_kind, enchant + 1)
  local duration = charm.calc_charm_duration(charm_kind, enchant)
  local duration_new = charm.calc_charm_duration(charm_kind, enchant + 1)
  local healing_percent = charm.calc_charm_healing(enchant)
  local healing_percent_new = charm.calc_charm_healing(enchant + 1)
  local shattering_radius = charm.calc_charm_shattering(enchant)
  local shattering_radius_new = charm.calc_charm_shattering(enchant + 1)

  local wear = item:get_wear()
  local meta = item:get_meta()
  local now = core.get_gametime()
  local start = meta:get_int("recharge_start")
  local time_left = math.ceil(recharge_duration) - (now - start)

  local message0 = def.description .. ". " .. def.mg_help .. "\n"
  local message1
  if wear > 0  then
    message1 = "This charm will gain an extra charge in " .. time_left .. " seconds.\n"
  else
    message1 = "This charm is fully charged.\n"
  end

  local message2 = "The time to fully regain recharge is "..recharge_duration.." seconds.\n"
  if charm_kind == "health" then
    message2 = message2 .. "The charm effect will heal you for "..healing_percent.." percent of your health.\n"
  elseif charm_kind == "shattering" then
    message2 = message2 .. "The charm will shatter "..shattering_radius.." blocks around you.\n"
  else
    message2 = message2 .. "The charm effect will last "..duration.." seconds.\n"
  end

  message2 = message2 .. "Enchanting this charm will reduce the recharge time to "..recharge_duration_new.." seconds "
  if charm_kind == "health" then
    message2 = message2 .. "and increase the percent it heals you to " .. healing_percent_new .." percent."
  elseif charm_kind == "shattering" then
    message2 = message2 .. "and increase charm shattering radius to "..shattering_radius_new.." blocks around you.\n"
  else
    message2 = message2 .. "and increase the duration to " .. duration_new .." seconds."
  end

  return message0 .. message1 .. message2
end


local function set_enchant_staff(itemstack, enchant)
  enchant = enchant or 2
  local meta = itemstack:get_meta()
  meta:set_int("recharge_max_charges", enchant)
  meta:set_int('enchant', enchant)
  return recharge.recharge_item(itemstack)
end

local function enchant_staff(itemstack, number)
  number = number or 1
  local meta = itemstack:get_meta()
  local enchant = get_charm_enchant(itemstack) + number
  -- max charges is always enchantment value
  meta:set_int("recharge_max_charges", enchant)
  meta:set_int('enchant', enchant)
  return recharge.recharge_item(itemstack)
end

local function set_enchant_charm(itemstack, enchant)
  enchant = enchant or 1
  local meta = itemstack:get_meta()
  meta:set_int('enchant', enchant)
  meta:set_int("recharge_max_charges", 1)
  return recharge.recharge_item(itemstack)
end

local function enchant_charm (itemstack, number)
  number = number or 1
  local meta = itemstack:get_meta()
  local enchant = get_charm_enchant(itemstack) + number
  meta:set_int('enchant', enchant)
  meta:set_int("recharge_max_charges", 1)
  return recharge.recharge_item(itemstack)
end

local function enchant_wand(itemstack)
  local meta = itemstack:get_meta()
  local def = itemstack:get_definition()
  local enchant = get_charm_enchant(itemstack) + 1
  local recharge_inc = def._recharge_inc or 1

  local max_charges = meta:get_int("recharge_max_charges")
  local wear = itemstack:get_wear() or 0
  local charges_remaining = max_charges - wear
  local total_charges = charges_remaining + recharge_inc

  -- wands only recharge by certain increments when recharged
  itemstack:set_wear(0)
  meta:set_int("recharge_max_charges", total_charges)
  -- does enchant actually mean anything here?
  meta:set_int('enchant', enchant)
  recharge.set_desc(itemstack, total_charges)
  return itemstack
end


mg_bolts.recharge_staves = recharge.recharge_staves
mg_bolts.recharge_charms = recharge.recharge_charms
mg_bolts.apply_player_reaping = recharge.apply_player_reaping

mg_bolts.enchant_charm = enchant_charm
mg_bolts.enchant_staff = enchant_staff
mg_bolts.set_enchant_charm = set_enchant_charm
mg_bolts.set_enchant_staff = set_enchant_staff
mg_bolts.enchant_wand = enchant_wand
mg_bolts.get_staff_help = get_staff_help
mg_bolts.get_staff_help_stats = get_staff_help_stats
mg_bolts.get_wand_help_stats = get_wand_help_stats
mg_bolts.get_charm_help_stats = get_charm_help_stats
