-- Taken from mcl_maps
-- TODO: improve support for larger zoom levels, benchmark raycasting, too?
-- TODO: only send texture to players that have the map
-- TODO: use ephemeral textures or base64 inline textures to eventually allow explorer maps?
-- TODO: show multiple players on the map
-- TODO: show banners on map
-- TODO: when the minimum supported Luanti version has core.get_node_raw, use it
-- Check for engine updates that allow improvements
local mgmaps = {}

local path = core.get_modpath("mg_maps")
local fov = dofile(path..'/fov.lua')

mgmaps.max_zoom = tonumber(core.settings:get("mp_maps_max_zoom")) or 3
mgmaps.map_allow_overlap = false
mgmaps.map_update_rate = 1 / (tonumber(core.settings:get("mg_maps_map_update_rate")) or 15) -- invert for the globalstep check

local modname = core.get_current_modname()
local modpath = core.get_modpath(modname)
local S = core.get_translator(modname)

local vector = vector
local table = table
local pairs = pairs
local min, max, round, floor, ceil, abs, pi = math.min, math.max, math.round, math.floor, math.ceil, math.abs, math.pi
local char = string.char
local concat = table.concat

local get_item_group = core.get_item_group
local dynamic_add_media = core.dynamic_add_media
local get_connected_players = core.get_connected_players
local get_node_light = core.get_node_light

local storage = core.get_mod_storage()
local worldpath = core.get_worldpath()
local map_textures_path = worldpath .. DIR_DELIM .. "mcl_maps" .. DIR_DELIM

-- Idea here is to keep track of player's fov map
-- by dungeon seed and level
local fov_maps = {}

core.mkdir(map_textures_path)

local function load_json_file(name)
  local file = assert(io.open(modpath .. DIR_DELIM .. name .. ".json", "r"))
  local data = core.parse_json(file:read("*all"))
  file:close()
  return data
end

local texture_colors = load_json_file("colors")

local maps_generating, maps_loading = {}, {}

-- TODO - put this in a mg_utils that can be shared
local startY = 0
local function pos_to_depth(pos)
  if pos.y > 7 then
    return 0
  end
  return math.floor((pos.y - startY - 10) / (-11) + 1)
end

local function depth_to_y_range(depth)
  if depth == 0 then
    return (8.5 - 1), (8.5 + 2)
  end

  local y = -11 * (depth - 2)  + startY - 5
  return (y-2), (y+1)
end

-- Main map generation function, called from emerge
local function do_generate_map(id, player, minp, maxp, key, callback--[[, t1]])
  --local t2 = os.clock()
  -- Generate a (usually) 128x128 linear array for the image

  -- build out a fov grid to test against
  -- I can probably run this every second
  -- and save it for the current level
  -- then update the map automatically for the
  -- player

  local pixels = {}
  local xsize, zsize = maxp.x - minp.x + 1, maxp.z - minp.z + 1
  -- Step size, for zoom levels > 1
  local xstep, zstep = 1, 1
  for z = zsize, 1, -zstep do
    local map_z = minp.z + z - 1
    local last_height
    for x = 1, xsize, xstep do
      local map_x = minp.x + x - 1
      -- Color aggregate and height information (for 3D effect)
      local cagg, height = {0, 0, 0, 0}, nil
      local solid_under_air = -1 -- anything but air, actually
      for map_y = maxp.y, minp.y, -1 do
        local node = core.get_node({x = map_x, y = map_y, z = map_z})
        local nodename = node.name
        local param2 = node.param2
        if nodename ~= "air" then
          local color = texture_colors[nodename]
          -- Use param2 if available:
          if color and type(color[1]) == "table" then
            color = color[param2 + 1] or color[1]
          end
          if color then
            if solid_under_air == 0 then
              cagg, height = {0, 0, 0, 0}, nil -- reset
              solid_under_air = 1
            end
            local alpha = cagg[4] -- 0 (transparent) to 255 (opaque)
            if alpha < 255 then
              local a = (color[4] or 255) * (255 - alpha) / 255 -- 0 to 255
              local f = a / 255 -- 0 to 1, color contribution
              -- Alpha blend the colors:
              cagg[1] = cagg[1] + f * color[1]
              cagg[2] = cagg[2] + f * color[2]
              cagg[3] = cagg[3] + f * color[3]
              alpha = cagg[4] + a -- new alpha, 0 to 255
              cagg[4] = alpha
            end

            -- Ground estimate with transparent blocks
            if alpha > 140 and not height then height = map_y end
            if alpha >= 250 and solid_under_air > 0 then
              -- Adjust color to give a 3D effect
              if last_height and height then
                local dheight = max(-48, min((height - last_height) * 8, 48))
                cagg[1] = cagg[1] + dheight
                cagg[2] = cagg[2] + dheight
                cagg[3] = cagg[3] + dheight
              end
              cagg[4] = 255 -- make fully opaque
              break
            end
          end
        elseif solid_under_air == -1 then
          solid_under_air = 0 -- first air
        end
      end

      if cagg[4] == 0 then
        cagg[4] = 255
      end

      local name = player:get_player_name()
      local seed = mg_arch.dungeon.seed
      if fov_maps[name][seed] == nil then
        fov_maps[name][seed] = {}
      end
      local fov_grid = fov_maps[name][seed][key]

      if map_z > mg_arch.DROWS + 1 or map_x > mg_arch.DCOLS + 1 or fov_grid == nil or fov_grid[x][z] == 0 then
        cagg[1] = 0
        cagg[2] = 0
        cagg[3] = 0
        cagg[4] = 0
      end
      -- Clamp colors values to 0..255 for PNG
      -- (because 3D height effect may exceed this range)
      cagg[1] = max(0, min(round(cagg[1]), 255))
      cagg[2] = max(0, min(round(cagg[2]), 255))
      cagg[3] = max(0, min(round(cagg[3]), 255))
      cagg[4] = max(0, min(round(cagg[4]), 255))
      pixels[#pixels + 1] = char(cagg[1], cagg[2], cagg[3], cagg[4])
      last_height = height
    end
  end
  -- Save as png texture
  --local t3 = os.clock()
  local filename = map_textures_path .. "mcl_maps_map_" .. id .. ".png"
  core.safe_file_write(filename, core.encode_png(xsize / xstep, zsize / zstep, concat(pixels)))
  --local t4 = os.clock()
  --core.log("action", string.format("Completed map %s after %.2fms (%.2fms emerge, %.2fms map, %.2fms png)", id, (os.clock()-t1)*1000, (t2-t1)*1000, (t3-t2)*1000, (t4-t3)*1000))
  maps_generating[id] = nil
  if callback then callback(id, filename) end
end

-- Trigger map generation
local function emerge_generate_map(id, player, minp, maxp, key, callback)
  if maps_generating[id] then return end
  maps_generating[id] = true
  --local t1 = os.clock()
  core.emerge_area(minp, maxp, function(_blockpos, _action, calls_remaining)
    if calls_remaining > 0 then return end
    -- Do a DOUBLE emerge to give mapgen the chance to place structures triggered by the initial emerge
    core.emerge_area(minp, maxp, function(_blockpos, _action, calls_remaining2)
      if calls_remaining2 > 0 then return end
      do_generate_map(id, player, minp, maxp, key, function(iid, filename)
        if callback then callback(iid, filename) end
        dynamic_add_media(filename, function()
          if not maps_loading[id] then -- avoid repeated callbacks
            maps_loading[id] = true
          end
        end)
      end)
    end)
  end)
end

---@param itemstack core.ItemStack
---@return boolean?
function mgmaps.is_map(itemstack)
  local item_def = core.registered_items[itemstack:get_name()]
  return (item_def.groups.vl_map or 0) ~= 0
end

local function calc_bounds(cx, dim, cz, zoom)
  zoom = max(zoom or 1, 1)
  -- Texture size is 128
  local size = 64 * (2 ^ zoom)
  local halfsize = size / 2

  cx, cz = (floor(cx / size) + 0.5) * size, (floor(cz / size) + 0.5) * size

  local miny, maxy = depth_to_y_range(dim)

  local minp = vector.new(cx - halfsize, miny, cz - halfsize)
  local maxp = vector.new(cx + halfsize - 1, maxy, cz + halfsize - 1)
  return minp, maxp
end

local function configure_map(itemstack, player, cx, dim, cz, zoom, key, callback)
  zoom = max(zoom or 1, 1)
  -- Texture size is 128
  local size = 64 * (2 ^ zoom)
  local halfsize = size / 2

  local meta = itemstack:get_meta()

  -- If enabled, round to halfsize grid, otherwise to size grid
  if mgmaps.map_allow_overlap then
    cx, cz = (floor(cx / halfsize) + 0.5) * halfsize, (floor(cz / halfsize) + 0.5) * halfsize
  else
    cx, cz = (floor(cx / size) + 0.5) * size, (floor(cz / size) + 0.5) * size
  end

  -- Y range to use for mapping. In nether, if we begin above bedrock, maps will be bedrock only, similar to MC
  -- Prefer smaller ranges for performance!
  local miny, maxy = depth_to_y_range(dim)

  -- File name conventions, including a unique number in case someone maps the same area twice (old and new)
  local seq = storage:get_int("next_id")
  storage:set_int("next_id", seq + 1)
  local id = concat({key, seq}, "_")
  local minp = vector.new(cx - halfsize, miny, cz - halfsize)
  local maxp = vector.new(cx + halfsize - 1, maxy, cz + halfsize - 1)

  meta:set_string("mg_maps:id", id)
  meta:set_int("mg_maps:cx", cx)
  meta:set_string("mg_maps:dim", dim)
  meta:set_int("mg_maps:cz", cz)
  meta:set_int("mg_maps:zoom", zoom)
  meta:set_string("mg_maps:minp", core.pos_to_string(minp))
  meta:set_string("mg_maps:maxp", core.pos_to_string(maxp))


  emerge_generate_map(id, player, minp, maxp, key, function(iid, filename)
    if callback then
      callback(iid, filename)
    end
    -- only set map id after map is done generating
  end)
  return itemstack
end

function mgmaps.load_map(id, callback)
  if id == "" or maps_generating[id] then return false end

  local texture = "mcl_maps_map_" .. id .. ".png"

  if maps_loading[id] then
    if callback then callback(texture) end
    return texture
  end

  -- core.dynamic_add_media() never blocks in Luanti 5.5, callback runs after load
  -- TODO: send only to the player that needs it!
  dynamic_add_media(map_textures_path .. texture, function()
    if not maps_loading[id] then -- avoid repeated callbacks
      maps_loading[id] = true
      if callback then callback(texture) end
    end
  end)
end

local function load_map_item(itemstack)
  if itemstack:get_name() == "mg_maps:filled_map" then
    local map_id = itemstack:get_meta():get_string("mg_maps:id")
    if map_id == nil then
      return "blank.png"
    end
    -- only returns texture if its done loading
    local texture = mgmaps.load_map(map_id)
    if texture then
      return texture
    else
      return 'loading'
    end
  end
  return 'nomap'
end

local function regenerate_map(itemstack, player, pos, key, callback)
  local dim = pos_to_depth(pos)
  local cx = mg_arch.DCOLS/2
  local cz = mg_arch.DROWS/2
  configure_map(itemstack, player, cx, dim, cz, 1, key, callback)
end

local function fill_map(itemstack, player)
  local name = player:get_player_name()
  local pos = player:get_pos()
  local depth = pos_to_depth(pos)
  local seed = mg_arch.dungeon.seed
  local key = name .. '_' .. seed .. '_' .. depth
  regenerate_map(itemstack, player, pos, key)
  return itemstack
end

local filled_def = {
  description = S("Map"),
  mg_help = "A map of the current depth you are in.  Use map to fill in the area that you can see at the moment.  It show the depth at which you last used it.  If you find a scroll of magic mapping, you can use that to fill out the entire level without exploring it.",
  _tt_help = S("Shows a map image."),
  _doc_items_longdesc = S("When created, the map saves the nearby area as an image that can be viewed any time by holding the map."),
  _doc_items_usagehelp = S("Hold the map in your hand. This will display a map on your screen."),
  inventory_image = "mcl_maps_map_filled.png^(mcl_maps_map_filled_markings.png^[colorize:#000000)",
  groups = {not_in_creative_inventory = 1, tool = 1, mp_map = 1},
  on_use = fill_map,
}

core.register_craftitem("mg_maps:filled_map", filled_def)


-- Avoid dropping detached hands with held maps
local old_add_item = core.add_item
function core.add_item(pos, stack)
  if not pos then
    core.log("warning", "Trying to add item with missing pos: " .. dump(stack))
    return
  end
  stack = ItemStack(stack)
  if get_item_group(stack:get_name(), "filled_map") > 0 then
    stack:set_name("mg_maps:filled_map")
  end
  return old_add_item(pos, stack)
end

-- Render handheld maps as part of HUD overlay
local maps = {}
local huds = {}

core.register_on_joinplayer(function(player)
  local map_def = {
    type = "image",
    text = "blank.png",
    position = {x = 0.85, y = 0.9},
    alignment = {x = 0, y = -1},
    offset = {x = 0, y = 0},
    scale = {x = 10, y = 10},
  }
  local marker_def = table.copy(map_def)
  marker_def.alignment = {x = 0, y = 0}
  huds[player] = {
    map = player:hud_add(map_def),
    marker = player:hud_add(marker_def),
  }

  local name = player:get_player_name()
  fov_maps[name] = {}

  local seed = mg_arch.dungeon.seed
  local key = "fov-maps-"..name.."-"..seed
  print("load fov data from " .. key)
  local raw_fov_maps = storage:get_string(key)
  fov_maps[name][seed] = core.deserialize(raw_fov_maps)
end)

core.register_on_leaveplayer(function(player)
  local name = player:get_player_name()
  local seed = mg_arch.dungeon.seed
  local key = "fov-maps-"..name.."-"..seed
  storage:set_string(key, core.serialize(fov_maps[name][seed]))
  maps[player] = nil
  huds[player] = nil
end)

local player_fov_maps = {}

local etime2 = 0
core.register_globalstep(function(dtime)
  -- this is a hack
  local seed = mg_arch.dungeon.seed
  etime2 = etime2 + dtime
  if etime2 < 1 then
    return
  end
  etime2 = 0

  local cx = mg_arch.DCOLS/2
  local cz = mg_arch.DROWS/2


  for _, player in pairs(get_connected_players()) do
    -- what radius does the player see now in the game?
    local radius = mg_light.calc_player_right_radius(player)

    local pos = player:get_pos()
    local depth = pos_to_depth(pos)
    local minp, maxp = calc_bounds(cx, depth, cz, 1)
    local fov_grid = fov.calc_fov_grid(minp, maxp, pos, radius + 4)

    local name = player:get_player_name()
    local key = name .. '_' .. seed .. '_' .. depth

    if fov_maps[name][seed] == nil then
      fov_maps[name][seed] = {}
    end

    local fov_grid_acc = fov_maps[name][seed][key]

    if fov_grid_acc == nil then
      fov_grid_acc = fov_grid
    else
      for x = 1, #fov_grid, 1 do
        for y = 1, #fov_grid[1], 1 do
          if fov_grid[x][y] == 1 then
            fov_grid_acc[x][y] = 1
          end
        end
      end
    end
    fov_maps[name][seed][key] = fov_grid_acc
  end
end)


local etime = 0
core.register_globalstep(function(dtime)
  etime = etime + dtime
  if etime < mgmaps.map_update_rate then
    return
  end
  etime = 0

  for _, player in pairs(get_connected_players()) do
    local wield = player:get_wielded_item()
    local texture = load_map_item(wield)
    if texture ~= 'nomap' then
      local hud = huds[player]

      local pos = player:get_pos() -- was: vector.round(player:get_pos())
      local light = get_node_light(vector.offset(pos, 0, 0.5, 0)) or 0

      -- Change map only when necessary
      if not maps[player] or texture ~= maps[player][1] or light ~= maps[player][4] then
        local light_overlay = "^[colorize:black:" .. 255 - (light * 17)
        if texture ~= 'loading' then
          player:hud_change(hud.map, "text", "[combine:140x140:0,0=mcl_maps_map_background.png:6,6=" .. texture .. light_overlay)
        end
        local meta = wield:get_meta()
        local minp = core.string_to_pos(meta:get_string("mg_maps:minp"))
        local maxp = core.string_to_pos(meta:get_string("mg_maps:maxp"))
        if minp == nil then
          minp = {x=0, y=-1, z=0}
        end
        if maxp == nil then
          maxp = {x=mg_arch.DCOLS, y=6, z=mg_arch.DROWS + 14}
        end
        maps[player] = {texture, minp, maxp, light}
      end

      -- Map overlay with player position
      local minp, maxp = maps[player][2], maps[player][3]

      pos.z = pos.z + 6

      -- Use dots when outside of map, indicate direction
      local marker
      if pos.x < minp.x then
        marker = abs(minp.x - pos.x) < 256 and "mcl_maps_player_dot_large.png" or "mcl_maps_player_dot.png"
        pos.x = minp.x
      elseif pos.x > maxp.x then
        marker = abs(pos.x - maxp.x) < 256 and "mcl_maps_player_dot_large.png" or "mcl_maps_player_dot.png"
        pos.x = maxp.x
      end

      -- Never override the small marker
      if pos.z < minp.z then
        marker = (abs(minp.z - pos.z) < 256 and marker ~= "mcl_maps_player_dot.png")
        and "mcl_maps_player_dot_large.png" or "mcl_maps_player_dot.png"
        pos.z = minp.z
      elseif pos.z > maxp.z then
        marker = (abs(pos.z - maxp.z) < 256 and marker ~= "mcl_maps_player_dot.png")
        and "mcl_maps_player_dot_large.png" or "mcl_maps_player_dot.png"
        pos.z = maxp.z
      end

      -- Default to yaw-based player arrow
      if not marker then
        local yaw = (floor(player:get_look_horizontal() * 180 / pi / 45 + 0.5) % 8) * 45
        if yaw == 0 or yaw == 90 or yaw == 180 or yaw == 270 then
          marker = "mcl_maps_player_arrow.png^[transformR" .. yaw
        else
          marker = "mcl_maps_player_arrow_diagonal.png^[transformR" .. (yaw - 45)
        end
      end

      -- Note the alignment and scale used above
      local scale = 10
      local w = 128
      local f = scale * w / (maxp.x - minp.x + 1)
      local alignx = scale * w / 2
      local alignz = scale * w
      player:hud_change(hud.marker, "offset", {x = (pos.x - minp.x) * f - alignx, y = (maxp.z - pos.z) * f - alignz})
      player:hud_change(hud.marker, "text", marker)

    elseif maps[player] then -- disable map
      local hud = huds[player]
      player:hud_change(hud.map, "text", "blank.png")
      player:hud_change(hud.marker, "text", "blank.png")
      maps[player] = nil
    end
  end
end)

local function reveal_current_depth(player)
  local cx = mg_arch.DCOLS/2
  local cz = mg_arch.DROWS/2
  local pos = player:get_pos()
  local seed = mg_arch.dungeon.seed
  local depth = pos_to_depth(pos)
  local minp, maxp = calc_bounds(cx, depth, cz, 1)

  local name = player:get_player_name()
  local key = name .. '_' .. seed .. '_' .. depth

  if fov_maps[name][seed] == nil then
    fov_maps[name][seed] = {}
  end

  local fov_grid_acc = fov_maps[name][seed][key]

  local xsize, zsize = maxp.x - minp.x + 1, maxp.z - minp.z + 1
  for x = 1, xsize, 1 do
    for y = 1, zsize, 1 do
      fov_grid_acc[x][y] = 1
    end
  end
  fov_maps[name][seed][key] = fov_grid_acc
end

mg_scrolls.register_enhancement_scroll("magic_scroll", {
  description = "Scroll of Magic Mapping",
  image = "voxeldungeon_item_scroll_laguz.png",
  mg_help = "Read this scroll over your map to reveal the all the rooms and hallways in the current depth.",

  instruction = "Place your map below to reveal the level",
  button = "Reveal",
  error_message_no_scroll = "Identification failed.  You need a scroll of magic mapping in your inventory.",
  error_message_no_item = "Identification failed. You need to put your map in the slot to reveal the current depth.",
  error_message_no_type = "Identification failed. You need to put your map in the slot to reveal the current depth.",
  apply_scroll = function(name, stack, player)
    if name == "mg_maps:filled_map" then
      reveal_current_depth(player)
      return fill_map(stack, player)
    end
    return nil
  end,
}, "mg_maps")

mg_maps = mgmaps
