local mod_path = core.get_modpath("mg_mobs")
local ai_tools = dofile(mod_path.."/ai_tools.lua")
local ai_bolts = dofile(mod_path.."/ai_bolts.lua")

-- really need to have a utils mod where stuff like this can be re-used
local function clamp(x, low, high)
  return math.min(high, math.max(x, low))
end

local function rand_percent(percent)
  return math.random(0,99) < clamp(percent, 0, 100)
end

-- TODO check head and feet - not just middle
-- maybe just check head??
local function walkable_node_between(mob_pos, target_pos)
  -- only nodes - no object or liques
  local cast = core.raycast(mob_pos, target_pos, false, false)
  local thing = cast:next()
  while thing do
    if thing.type == "node" then
      local node = core.get_node(thing.intersection_point)
      local def = minetest.registered_nodes[node.name]
      -- A walkable node is a solid node
      if def and def.walkable then
        return true
      end
    end
    thing = cast:next()
  end
  return false
end

local function queue_message (self, publisher, message, data)
  local messages = self.mg_messages or {}
  local message_obj = {
    publisher = publisher, message = message, data = data
  }
  table.insert(messages, message_obj)
  self.mg_messages = messages
end

local startY = 0
local function send_message_to_horde (publisher, message, data)
  -- get mob position
  local pos = mobkit.get_stand_pos(publisher)

  -- convert that to a level
  local depth = math.floor((pos.y - startY - 10) / (-11) + 1)

  -- get level boundaries
  local p1 = { x = 1, y = -11 * (depth - 1)  + startY - 1, z = 1}
  local p2 = { x = mg_arch.DCOLS, y = p1.y + 11, z = mg_arch.DROWS}

  -- get all objects in that level
  -- filter to only the ones in the horde
  -- give them the event
  local objs = core.get_objects_in_area(p1, p2)
  for _, obj in ipairs(objs) do
    local mob = obj:get_luaentity()
    if mob and mob.leader_id == publisher.leader_id and
      mob.mg_id ~= publisher.mg_id
    then
      queue_message (mob, publisher, message, data)
    end
  end
end

local function check_target(self, target)
  if not target then
    return nil
  end

  -- gets top of collision box
  local middle_pos = mobkit.get_middle_pos(self)
  local target_middle_pos = mobkit.get_middle_pos(target)
  if not target_middle_pos then
    return nil
  end

  local stealth_range = mg_stealth.get_stealth(target)

  if mobkit.is_alive(target) and
    vector.distance(middle_pos, target_middle_pos) < stealth_range and
    walkable_node_between(middle_pos, target_middle_pos) == false then
    -- 49 percent chance to notice player
    if rand_percent(49) then
      return target
    end
  end
  return nil
end

local function check_is_target_enemy (self, target)
  -- nobody is an enemy when confused
  if mg_oe.has_effect("confusion", self.object) then
    return false
  end


  local target_mob = target:get_luaentity()

  -- any target that is discordant is fair game
  if target_mob and target_mob.mg_id and
    mg_oe.has_effect("discordant", target)
  then
    return true
  end

  -- if mobs is discordant than all other mobs or players are enemies
  -- but not entities like bolts, arrows or gas clouds
  if mg_oe.has_effect("discordant", self.object) then
    return mobkit.is_alive(target) and (
              target:is_player() or
              (target_mob and target_mob.mg_id and target_mob.mg_id ~= self.mg_id))
  end

  -- if an ally do not do not target players of mobs
  if self.allied_to and
    (target:is_player() or
    (target_mob and target_mob.allied_to))
  then
    return false
  end

  -- if self is an ally and target is not a player or ally
  -- already check above to not attack allies an players
  if self.allied_to and
    (target_mob and target_mob.mg_id)
  then
    return true
  end

  -- if not an ally, then attack players or allied mobs
  if not self.allied_to and
    (target:is_player() or
    (target_mob and target_mob.allied_to))
  then
    return true
  end

  return false
end

local function get_hunting_target(self)
  ai_tools.shuffle_list_in_place(self.nearby_objects)
	for _, obj in ipairs(self.nearby_objects) do
    local is_enemy = check_is_target_enemy(self, obj)
    if is_enemy then
      local noticed_target = check_target(self, obj)
      if noticed_target then
        return noticed_target
      end
    end
	end

end

local function drop_stolen_items(self)
  local mob_id = self.mg_id
  if not mob_id then
    print('no mob id for drop stolen items')
    return
  end

  local pos = mobkit.get_stand_pos(self)

  for _, player in pairs(core.get_connected_players()) do
    local inv = player:get_inventory()
    local mob_list = inv:get_list("mob_"..mob_id)
    if not mob_list then
      return
    end
    for _, item in ipairs(mob_list) do
      if item and item:get_count() > 0 then
        core.add_item(pos, item)
      end
    end
  end
end


local function hq_die(self)
  local timer = 5
  local start = true

  self.object:set_properties({
    collide_with_objects = false,
    pointable = false
  })

  self.hp = 0
  self.dead = true

  if self.mg_id and self.mg_id == self.leader_id then
    send_message_to_horde(self, "leader_dead", {})
  end

  if self.on_death then
    self.on_death(self)
  end

  local abilities = self.abilities or {}

  if abilities.hit_steal_flee then
    drop_stolen_items(self)
  end

  local func = function(self)
    if start then
      mobkit.lq_fallover(self)
      self.logic = function(self) end	-- brain dead as well
      start=false
    end
    timer = timer-self.dtime
    if timer < 0 then self.object:remove() end
  end
  mobkit.queue_high(self,func,100)
end

local function node_dps_dmg(self)
  local pos = self.object:get_pos()
  local box = self.object:get_properties().collisionbox
  local pos1 = {x = pos.x + box[1], y = pos.y + box[2], z = pos.z + box[3]}
  local pos2 = {x = pos.x + box[4], y = pos.y + box[5], z = pos.z + box[6]}
  local nodes_overlap = mobkit.get_nodes_in_area(pos1, pos2)
  local total_damage = 0

  for node_def, _ in pairs(nodes_overlap) do
    local dps = node_def.damage_per_second
    local behaviors = self.behaviors or {}
    local immune_to_fire = behaviors.immune_to_fire and behaviors.immune_to_fire > 0
    if dps and not immune_to_fire then
      total_damage = math.max(total_damage, dps)
    end
  end

  if total_damage ~= 0 then
    mobkit.hurt(self, total_damage)
  end
end



local function check_fiery (self)
  if self.behaviors and self.behaviors.fiery and self.behaviors.fiery > 0 then
    local pos = mobkit.get_stand_pos(self)
    local pf = minetest.find_node_near(pos, 1, {"group:flammable"})
    local flame = minetest.find_node_near(pos, 1, {'fire:basic_flame'})
    -- Light on fire if near flammable terrain and not yet near fire
    if pf and not flame then
      local pa = minetest.find_node_near(pf, 1, {"air"})
      if pa then
        minetest.set_node(pa, {name = 'fire:basic_flame'})
      end
    end
  end
end

local function regen(self)
  if not self.regen or self.regen == 0 or self.hp == self.max_hp then
    return
  end

  if mg_oe.has_effect("poison", self.object) then
    return
  end

  if not self.regen_timer then
    self.regen_timer = 0
  end

  self.regen_timer = self.regen_timer + self.dtime

  if self.regen_timer > self.regen then
    self.regen_timer = 0
    self.hp = self.hp + 1
    if self.hp > self.max_hp then
      self.hp = self.max_hp
    end
  end
end




-- start with these different high level states
-- monster will spawn either wandering or sleeping
--
-- sleeping
-- wandering
-- hunting
-- fleeing

--local function debug_entity(self, name, message)
  --if self.name == name then
    --print("debug for " .. name .. ": " .. message)
  --end
--end

local function predator_brain(self)
  if mobkit.timer(self, 1) then 			-- decision making needn't happen every engine step
    check_fiery(self)

    local prty = mobkit.get_queue_priority(self)

    -- if confused, then wander around without regard to anything else
    if prty < 20 and mg_oe.has_effect("confusion", self.object) then
      mobkit.clear_queue_high(self)
      mg_mobs.hq_confused(self, 20)
      return
    end

    -- get out of deep liquid
    if prty < 15 and mobkit.is_in_deep(self) then
      mobkit.clear_queue_high(self)
      mobkit.hq_liquid_recovery(self, 5)
    end

    -- hunt if inside player stealth range and has line of sight to them
    if prty < 9 then
      local target = get_hunting_target(self)
      if target then	-- and can see them
        mobkit.make_sound(self,'warn')
        send_message_to_horde(self, "attack", {target = target})
        mg_mobs.hq_hunt(self, 10, target)
      end
    end

    if prty < 7 and self.allied_to then
      -- get to 2 blocks away from player
      local player = core.get_player_by_name(self.allied_to)
      if player then
        mg_mobs.hq_follow_land(self, 7, player)
      end
    end

    -- Monsters only go through liquid if hunting the player
    -- Or allies are following the player
    if prty < 5 and self.isinliquid then
      mobkit.hq_liquid_recovery(self, 5)
    end

    local messages = self.mg_messages or {}
    local messages_to_delete = {}
    for i, message in ipairs(messages) do
      if message.message == "attack" then
        -- only start hunting if not doing something more important
        if prty < 4 then
          mg_mobs.hq_hunt(self, 10, message.data.target)
        end
        table.insert(messages_to_delete, i)
      end
    end
    for _, i in ipairs(messages_to_delete) do
      table.remove(self.mg_messages, i)
    end

    -- fool around
    if mobkit.is_queue_empty_high(self) then
      -- I want roam to be wandering, once they reach their destination they will
      -- go to idle, then ideal will set a new destination - I'll need path finding to get
      -- that working though...
      mobkit.hq_roam(self,0)
    end
  end
end


local function fly_brain(self)
  if mobkit.timer(self,1) then
    check_fiery(self)
    local prty = mobkit.get_queue_priority(self)

    if prty < 30 and mg_oe.has_effect("confusion", self.object) then
      mobkit.clear_queue_high(self)
      mg_mobs.hq_fly_confused(self, 30, 1)
      return
    end

    if prty < 20 then
      local target = get_hunting_target(self)
      if target then
        send_message_to_horde(self, "attack", {target = target})
        mobkit.make_sound(self,'warn')
        mg_mobs.hq_fly_attack(self,target, 20, self.speed)
      end
    end

    local messages = self.mg_messages or {}
    for _, message in ipairs(messages) do
      if message.message == "attack" then
        -- only start hunting if not doing something more important
        if prty < 4 then
          mg_mobs.hq_fly_attack(self, message.data.target, 20, self.speed)
        end
      end
    end
    self.mg_messages = {}

    if self.allied_to and mobkit.is_queue_empty_high(self) then
      -- get to 2 blocks away from player
      local player = core.get_player_by_name(self.allied_to)
      mg_mobs.hq_follow(self, player, 3)
    end

    if mobkit.is_queue_empty_high(self) then
      mg_mobs.hq_fly_roam(self, 10, 1, 'fly')
    end
  end
end

local function aqua_get_target_attack(self)
  local target = get_hunting_target(self)

  -- target also has to be in the water
  if mobkit.is_in_water(target) then
    return target
  end
  return nil
end

-- Liquid attack
local function liquid_brain(self)
  if mobkit.timer(self,1) then
    -- could be mobs that live in lava...?
    check_fiery(self)
    local prty = mobkit.get_queue_priority(self)
    if prty < 20 then
      local target = aqua_get_target_attack(self)
      if target then
        send_message_to_horde(self, "attack", {target = target})
        mobkit.clear_queue_high(self)
        mobkit.make_sound(self,'warn')
        mg_mobs.hq_water_attack(self, target, 20, self.speed, true)
      end
    end

    local messages = self.mg_messages or {}
    for _, message in ipairs(messages) do
      if message.message == "attack" then
        -- only start hunting if not doing something more important
        if prty < 4 then
          mg_mobs.hq_water_attack(self, message.data.target, 20, self.speed, true)
        end
      end
    end
    self.mg_messages = {}

    if mobkit.is_queue_empty_high(self) then mg_mobs.hq_aqua_roam(self,10, self.speed/2 ,'def') end
  end
end

local function static_brain(self)
  if mobkit.timer(self,1) then
    check_fiery(self)

    local prty = mobkit.get_queue_priority(self)
    if prty < 20 then
      local target = get_hunting_target(self)
      if target then
        send_message_to_horde(self, "attack", {target = target})
        mobkit.make_sound(self,'warn')
        mg_mobs.hq_static_attack(self, 20, target)
      end
    end

    local messages = self.mg_messages or {}
    for _, message in ipairs(messages) do
      if message.message == "attack" then
        -- only start hunting if not doing something more important
        if prty < 4 then
          mg_mobs.hq_static_attack(self, 20, message.data.target)
        end
      end
    end
    self.mg_messages = {}

    -- not sure what this will do
    if mobkit.is_queue_empty_high(self) then
      mobkit.hq_roam(self,0)
    end
  end
end

local function mg_brain(self)
  -- if in node that causes damage per second, apply that
  -- damage once per second
  if mobkit.timer(self,1) then node_dps_dmg(self) end

  -- run effect timers on mob
  mg_oe.on_step(self.object, self.dtime)

  -- regeneration
  regen(self)

  -- vitals should be checked every step
  mobkit.vitals(self)

  if self.hp <= 0 and not self.dead then
    mobkit.clear_queue_high(self)
    hq_die(self)
    return
  end


  if self.timeout then
    self.timeout = self.timeout - self.dtime
    if self.timeout < 0 then
      mobkit.clear_queue_high(self)
      hq_die(self)
      return
    end
  end


  -- TODO abstract this to something like process_message(self, 'message type', cb)
  self.mg_messages = self.mg_messages or {}
  local messages_to_delete = {}
  for i, message in ipairs(self.mg_messages) do
    if message.message == "leader_dead" then
      if self.behaviors and self.behaviors.dies_on_leader_death then
        hq_die(self)
      end
      table.insert(messages_to_delete, i)
    end
  end

  for _, i in ipairs(messages_to_delete) do
    table.remove(self.mg_messages, i)
  end


  -- punch timer
  -- some monsters have a cool down before they can punch
  -- the player again
  self.punch_timer = self.punch_timer or 0
  self.punch_timer = self.punch_timer + self.dtime

  -- increment ability timer here
  -- punch and ability timer should really be an attack timer
  self.ability_timer = self.ability_timer or 0
  self.ability_timer = self.ability_timer + self.dtime

  -- flit timer for some mobs
  self.flit_timer = self.flit_timer or 0
  self.flit_timer = self.flit_timer + self.dtime


  if self.behaviors and self.behaviors.flies then
    fly_brain(self)
  elseif self.behaviors and self.behaviors.restricted_to_liquid then
    liquid_brain(self)
  elseif self.max_speed <= 0 then
    static_brain(self)
  else
    predator_brain(self)
  end
end

local function mob_attack(self, target)
  if self.behaviors and self.behaviors.flies then
    mg_mobs.hq_fly_attack(self,target, 20, self.speed)
  elseif self.behaviors and self.behaviors.restricted_to_liquid then
    mg_mobs.hq_water_attack(self, target, 20, self.speed, true)
  elseif self.max_speed <= 0 then
    mg_mobs.hq_static_attack(self, 20, target)
  else
    mg_mobs.hq_hunt(self, 10, target)
  end
end

local function handle_on_punch(self, puncher, time_from_last_punch, tool_capabilities, dir)
  if mobkit.is_alive(self) then

    local player_str = mg_strength.get_strength_raw(puncher) or 12
    local player_weakness = mg_strength.get_weakness(puncher) or 0
    local dmg_min = tool_capabilities.damage_groups.fleshy or 0.5
    local dmg_max = tool_capabilities.damage_groups.fleshy_max or dmg_min
    -- for things like arrows or something else i forgot to set
    local str_required = tool_capabilities.damage_groups.strength_required or 12
    local enchant = tool_capabilities.damage_groups.enchant or 0

    local dmg = mg_strength.calcDamage(
      enchant,
      str_required,
      dmg_min,
      dmg_max,
      player_str,
      player_weakness
    )

    local accuracy = mg_strength.damageFraction(mg_strength.netEnchant(
      enchant,
      str_required,
      player_str,
      player_weakness
    ))

    local defense = mg_armor.calc_defense_fraction(self.defense or 0)
    print('mob defense: ' .. defense*100)

    local time_frac = time_from_last_punch / tool_capabilities.full_punch_interval

    if time_frac > 1 then
      time_frac = 1
    end

    print('player accuracy: ' .. accuracy*100)
    print('time fraction: ' .. time_frac*100)
    local hitProb = math.floor(accuracy * 100 * time_frac * defense)
    print('hit prob: ' .. hitProb)
    local hit = mg_arch.rand_percent(hitProb)

    if mg_oe.has_effect("gas_paralysis", self.object) then
      mg_oe.clear_effect("gas_paralysis", self.object)
      hit = true
      if tool_capabilities.damage_groups.sneak_attack_bonus then
        dmg = dmg * 5
      else
        dmg = dmg * 3
      end
      print('mob ' .. self.name .. ' is paralyzed ' .. ' give an attack bonus, damage is now ' .. dmg)
    end

    -- you missed, no damage is done
    if not hit then
      print('player missed')
      local pos = puncher:get_pos()
      minetest.sound_play("miss", {
        pos = pos,
        gain = 1.0,
        max_hear_distance = 5,
      })
      -- play a miss sound here
      return true
    end


    print('player hit mob - apply hit to mob ' .. self.name)

    local str_diff = mg_strength.player_strength_diff(puncher, str_required)
    -- Only apply knockback if player is strong enough
    if (tool_capabilities.damage_groups.pushback or 0) > 0 and str_diff >= 0 and time_frac >= 1 then
      local hvel = vector.multiply(vector.normalize({x=dir.x,y=0,z=dir.z}), 3)
      self.object:set_velocity({x=hvel.x,y=1,z=hvel.z})
    end

    -- Mob is hit, now attack player if puncher is an enemy
    if check_is_target_enemy(self, puncher) then
      -- don't clear queue - fleeing mobs still need to flee
      mob_attack(self, puncher)
      send_message_to_horde(self, "attack", {target = puncher})
    end

    dmg = mg_oe.apply_protection(self.object, dmg or 1)

    if dmg == 0 then
      print('no damage because protected')
      local pos = puncher:get_pos()
      minetest.sound_play("protection", {
        pos = pos,
        gain = 1.0,
        max_hear_distance = 5,
      })
      return true
    end

    if self.behaviors and self.behaviors.defend_degrade_weapon then
      mg_tools.degrade_player_weapon(puncher, 1)
    end

    ai_bolts.apply_hitter_abilities(self.object, puncher, dmg)

    -- no damage because immune to convetional weapon
    if self.behaviors and self.behaviors.immune_to_weapons then
      local pos = puncher:get_pos()
      minetest.sound_play("protection", {
        pos = pos,
        gain = 1.0,
        max_hear_distance = 5,
      })
      return true
    end

    mobkit.make_sound(self,'hurt')
    mobkit.hurt(self, dmg or 1)

    -- puncher killed mob
    -- increment kill count for weapon
    -- TODO - dont apply to purely magical creatures
    if self.hp <= 0 then
      mg_tools.set_player_weapon_use(puncher, 1)
    end

    mg_rings.apply_player_transference(puncher, dmg)
    mg_bolts.apply_player_reaping(puncher, dmg)

    if self.behaviors and self.behaviors.flees_near_death then
      local percent = (self.hp / self.max_hp) * 100
      if percent < 24 then
        mobkit.hq_runfrom(self, 15, puncher)
      end
    end

    if self.abilities and self.abilities.clone_self_on_defend and self.hp > 1 then
      -- TODO move to it's own function somewhere
      local new_hp = self.hp / 2
      self.hp = new_hp
      local pos = self.object:get_pos()
      local near = core.find_node_near(pos, 4, {"air"})
      local copy = minetest.add_entity(near, self.name)
      local new_mob = copy:get_luaentity()
      new_mob.mob_id = mg_mobs.uuid()
      new_mob.leader_id = self.leader_id
      new_mob.hp = new_hp
    end


  end

  -- return true means dont apply default luanti damage algo
  return true
end


local function on_death(self)
  local pos=self.object:get_pos()

  if self.carried_monster then
    local mobname = self.carried_monster.name:match(":(.+)")
    local obj = mg_mobs.spawn(pos, mobname, self.carried_monster.mg_id)
    local mob = obj:get_luaentity()
    mob.hp = self.carried_monster.hp
  end

  local abilities = self.abilities or {}
  if abilities.df_on_death == 'caustic_gas' then
    mg_effects.add_gas("mg_effects:caustic_gas", pos)
  elseif abilities.df_on_death == 'descent' then
    local startY = 0
    local depth = math.floor((pos.y - startY - 10) / (-11) + 1)
    local y = -11 * (depth - 2)  + startY - 5
    pos.y = y
    mg_effects.descent({pos=pos, radius=6})
  elseif abilities.df_on_death == 'explosion' then
    mg_effects.explode({pos = pos, radius = 6})
  end
end

mg_mobs.mg_brain = mg_brain;
mg_mobs.handle_on_punch = handle_on_punch;
mg_mobs.hq_die = hq_die;
mg_mobs.on_death = on_death;
