local registered_effects = {}

-- def.default_data
-- def.icon
-- def.on_start
-- def.on_clear
-- def.on_step
local function update_desc(stack)
  local name = stack:get_name()
  if string.find(name, "mg_tools") then
    return mg_tools.update_desc(stack)
  elseif string.find(name, "mg_armor") then
    mg_armor.update_desc(stack)
  elseif string.find(name, "mg_rings") then
    mg_rings.update_desc(stack)
  end
end

local function register_effect(name, def)
  -- save in global effect table
  registered_effects[name] = def

  -- storage for all players
  local storage = {}

  local namespace = 'mg_obj_effects:'..name
  local default_data = def.default_data
  if default_data then
    default_data.duration = 0
    default_data.total_timer = 0
  else
    default_data = {
      duration = 0,
      total_timer = 0
    }
  end

  def.get_data = function(target)
    if target:is_player() then
      local pname = target:get_player_name()
      if not storage[pname] then
        storage[pname] = table.copy(default_data)
      end
      return storage[pname]
    end

    local mob = target:get_luaentity()
    if not mobkit.recall(mob, namespace) then
      mobkit.remember(mob, namespace, table.copy(default_data))
    end
    return mobkit.recall(mob, namespace)
  end

  def.set_data = function(target, data)
    if target:is_player() then
      local pname = target:get_player_name()
      storage[pname] = data
      return
    end

    local mob = target:get_luaentity()
    return mobkit.remember(mob, namespace, data)
  end

  def.start = function(target, duration, ...)
    if def.on_prestart then
      local should_apply = def.on_prestart(def, target, duration, ...)
      if not should_apply then
        return
      end
    end
    local data = def.get_data(target)
    -- acculate the effect instead of restarting it
    data.duration = (data.duration or 0) + duration
    def.set_data(target, data)
    -- on start can override these assignments
    if def.on_start then
      def.on_start(def, target, duration, ...)
    end
  end

  def.clear = function(target, ...)
    if def.on_preclear then
      def.on_preclear(def, target, ...)
    end
    local new_data = table.copy(default_data)
    def.set_data(target, new_data)
    if def.on_clear then
      def.on_clear(def, target, ...)
    end
  end

  def.step = function(target, dt, ...)
    if def.on_prestep then
      def.on_prestep(def, target, dt, ...)
    end
    local data = def.get_data(target)
    -- duration of 0 means no effect
    if not data.duration or data.duration == 0 then
      return
    end
    data.total_timer = (data.total_timer or 0) + dt
    def.set_data(target, data)

    if data.total_timer > data.duration then
      def.clear(target)
    end
    if def.on_step then
      def.on_step(def, target, dt, ...)
    end
  end

  def.on_player_join = function(player)
    local playername = player:get_player_name()
    local default = table.copy(default_data)
    local saved_data = player:get_meta():get_string(namespace)
    local data = default
    if saved_data ~= "" then
      data = core.deserialize(saved_data) or default
    end
    storage[playername] = data
    -- restart the effect
    if def.on_join then
      def.on_join(def, player, data.duration)
    end
    if def.on_start and data.duration > 0 then
      def.on_start(def, player, data.duration)
    end
  end

  def.on_player_leave = function(player)
    local playername = player:get_player_name()
    local data = storage[playername]
    if not data then
      data = table.copy(default_data)
    end
    player:get_meta():set_string(namespace, core.serialize(data))
    storage[playername] = nil
  end

end

local icon_ids = {}

local function effects_init_icons(player)
	local name = player:get_player_name()
	icon_ids[name] = {}

  local effect_icon_ct = 1
  for _, effect_def in pairs(registered_effects) do
    if effect_def.icon then
      effect_icon_ct = effect_icon_ct + 1
    end
  end

  -- allocate spaces for effects in HUD
  -- only that have an icon
	for e=1, effect_icon_ct do
		local x = -52 * e - 2
		local id = {}
		id.img = player:hud_add({
			type = "image",
			text = "blank.png",
			position = { x = 1, y = 0 },
			offset = { x = x, y = 3 },
			scale = { x = 0.375, y = 0.375 },
			alignment = { x = 1, y = 1 },
			z_index = 100,
		})
		id.label = player:hud_add({
			type = "text",
			text = "",
			position = { x = 1, y = 0 },
			offset = { x = x+22, y = 50 },
			scale = { x = 50, y = 15 },
			alignment = { x = 0, y = 1 },
			z_index = 100,
			style = 1,
			number = 0xFFFFFF,
		})
		id.timestamp = player:hud_add({
			type = "text",
			text = "",
			position = { x = 1, y = 0 },
			offset = { x = x+22, y = 65 },
			scale = { x = 50, y = 15 },
			alignment = { x = 0, y = 1 },
			z_index = 100,
			style = 1,
			number = 0xFFFFFF,
		})
		table.insert(icon_ids[name], id)
	end
end

local function effects_set_icons(player)
	local name = player:get_player_name()
	if not icon_ids[name] then
		return
	end
	local active_effects = {}
  for namespace, effect_def in pairs(registered_effects) do
    local data = effect_def.get_data(player)
    if effect_def.icon and data.duration and data.total_timer > 0 then
      active_effects[namespace] = data
    end
  end

  local effect_icon_ct = 1
  for _, effect_def in pairs(registered_effects) do
    if effect_def.icon then
      effect_icon_ct = effect_icon_ct + 1
    end
  end

  local hud_ids = icon_ids[name]
	local i = 1
	for effect_name, def in pairs(registered_effects) do
		local icon = hud_ids[i].img
		local label = hud_ids[i].label
		local timestamp = hud_ids[i].timestamp
		local vals = active_effects[effect_name]
		if vals then
			player:hud_change(icon, "text", def.icon .. "^[resize:128x128")
      player:hud_change(label, "text", "")
      local dur = math.floor(vals.duration - vals.total_timer + 0.5)
      player:hud_change(timestamp, "text", math.floor(dur/60)..string.format(":%02d",math.floor(dur % 60)))
			i = i + 1
		end
	end
	while i < effect_icon_ct do
		player:hud_change(hud_ids[i].img, "text", "blank.png")
		player:hud_change(hud_ids[i].label, "text", "")
		player:hud_change(hud_ids[i].timestamp, "text", "")
		i = i + 1
	end
end

local function get_effect_data(name, target)
  return registered_effects[name].get_data(target)
end

local function has_effect(name, target)
  local data = registered_effects[name].get_data(target)
  return data and data.duration and data.duration > 0
end

local function set_effect_data(name, target, data)
  return registered_effects[name].set_data(target, data)
end

local function start_effect(name, target, duration, ...)
  local mob = target:get_luaentity()
  if target:get_properties().pointable and
     ( target:is_player() and
       target:get_hp() > 0
     ) or
     ( mob and mob.mg_id ~= nil and mob.hp and mob.hp > 0) then
    return registered_effects[name].start(target, duration, ...)
  end
end

local function clear_effect(name, target, ...)
  return registered_effects[name].clear(target, ...)
end

local function negate_effects(target)
  for _, effect_def in pairs(registered_effects) do
    if not effect_def.cannot_negate then
      effect_def.clear(target)
    end
  end
end

local function on_step(target, dt)
  for _, effect_def in pairs(registered_effects) do
    effect_def.step(target, dt)
  end
end

local hud_timer = 0
local heartbeat = 0
core.register_globalstep(function(dt)
  if dt > 0.5 then
    core.log("warning", "Global step too long " .. dt .. " > 0.5s")
  end

  heartbeat = heartbeat + dt
  if heartbeat > 10 then
    core.log("action", "10s heartbeat")
    heartbeat = 0
  end
  -- run on step for effects
	for _, player in pairs(core.get_connected_players()) do
    on_step(player, dt)
  end

  -- update hud for players
  hud_timer = hud_timer + dt
  if hud_timer > 0.2 then
    hud_timer = 0
    for _, player in pairs(core.get_connected_players()) do
      effects_set_icons(player)
    end
  end
end)

core.register_on_joinplayer(function(player)
    effects_init_icons(player)
    for _, effect_def in pairs(registered_effects) do
      effect_def.on_player_join(player)
    end
end)

core.register_on_leaveplayer(function(player)
  for _, effect_def in pairs(registered_effects) do
    effect_def.on_player_leave(player)
  end
end)

core.register_on_shutdown(function()
	for _,player in ipairs(core.get_connected_players()) do
    for _, effect_def in pairs(registered_effects) do
      effect_def.on_player_leave(player)
    end
	end
end)

local function clear_all_effects(player)
  for _, effect_def in pairs(registered_effects) do
    effect_def.clear(player)
  end
end

core.register_on_respawnplayer(function(player)
  clear_all_effects(player)
end)

-- helpers
local function get_collisionbox(obj, smaller, storage)
	local cache = storage.collisionbox_cache
	if cache then
		local box = cache[smaller and 2 or 1]
		return box[1], box[2]
	else
		local box = obj:get_properties().collisionbox
		local minp, maxp = vector.new(box[1], box[2], box[3]), vector.new(box[4], box[5], box[6])
		local s_vec = vector.new(0.1, 0.1, 0.1)
		local s_minp = vector.add(minp, s_vec)
		local s_maxp = vector.subtract(maxp, s_vec)
		storage.collisionbox_cache = {{minp, maxp}, {s_minp, s_maxp}}
		return minp, maxp
	end
end

local function get_touching_nodes(obj, nodenames, storage)
	local pos = obj:get_pos()
	local minp, maxp = get_collisionbox(obj, true, storage)
	local nodes = core.find_nodes_in_area(
    vector.add(pos, minp),
    vector.add(pos, maxp),
  nodenames)
	return nodes
end

mg_oe.get_collisionbox = get_collisionbox
mg_oe.get_touching_nodes = get_touching_nodes
mg_oe.register_effect = register_effect
mg_oe.has_effect = has_effect
mg_oe.get_effect_data = get_effect_data
mg_oe.set_effect_data = set_effect_data
mg_oe.start_effect = start_effect
mg_oe.clear_effect = clear_effect
mg_oe.clear_all_effects = clear_all_effects
mg_oe.negate_effects = negate_effects
mg_oe.on_step = on_step
