local registered_gases = {}

local function uuid()
  local template ='xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'
  return string.gsub(template, '[xy]', function (c)
    local v = (c == 'x') and math.random(0, 0xf) or math.random(8, 0xb)
    return string.format('%x', v)
  end)
end

local function add_spawner(obj, amount, texture)
	local d = 0.90
	local pos = obj:get_pos()
	core.add_particlespawner({
		amount = amount,
		time = 2,
		minpos = {x=pos.x-d, y=pos.y-d, z=pos.z-d},
		maxpos = {x=pos.x+d, y=pos.y+d, z=pos.z+d},
		minvel = {x=-0.1, y=0, z=-0.1},
		maxvel = {x=0.1, y=0.1, z=0.1},
		minacc = {x=-0.1, y=0, z=-0.1},
		maxacc = {x=0.1, y=.1, z=0.1},
		minexptime = 0.5,
		maxexptime = 2,
		minsize = 2,
		maxsize = 8,
		collisiondetection = false,
		vertical = false,
		texture = texture or "toxicgas.png"
	})
end

local function clamp(x, low, high)
  return math.min(high, math.max(x, low))
end

local function rand_percent(percent)
  return math.random(0,99) < clamp(percent, 0, 100)
end

local function rand_range(x,y)
  return math.random(x,y)
end

local function register_gas(name, def)
	if name == nil or name == "" then
		return false
	end

  def.particle_texture = def.particle_texture or "toxicgas.png"
	def.name = "mg_effects:" .. name .. "_gas"
	def.initial_volume = def.initial_volume or 50
	def.decay_rate = def.decay_rate or 0
  registered_gases[def.name] = def
end

local function add_new_gas(gas_name, pos, volume)
  local obj = core.add_entity({
    x = pos.x,
    y = pos.y,
    z = pos.z,
  }, "mg_effects:gas")
  obj:set_velocity({x = 0, y = 0, z = 0})
  obj:set_acceleration({x = 0, y = 0, z = 0})
  local gas = obj:get_luaentity()
  local rgas = registered_gases[gas_name]
  if not rgas then
    print("cannot find gas type")
    obj:remove()
    return nil
  end
  gas.gas_id = uuid()
  gas.decay_rate = rgas.decay_rate or 0
  gas.initial_volume = rgas.initial_volume
  gas.volume = volume or rgas.initial_volume
  gas.effect_name = rgas.effect_name
  gas.particle_texture = rgas.particle_texture or "toxicgas.png"
  gas.effect_duration = rgas.effect_duration or 1
  gas.nextVolume = gas.volume
  gas.gasType = rgas.name
  gas.igniter = rgas.igniter
  gas.flammable = rgas.flammable
  gas.is_new = true
  return obj
end

local function add_gas(gas_name, pos, volume)
  -- check to see if threw gas into a spot where gas is already present
  -- if so then switch the existing gas to the new gas type
  -- if not create a new gas
  for check_obj in core.objects_inside_radius(pos, 1) do
    local gas = check_obj:get_luaentity()
    if gas and gas.name == "mg_effects:gas" then
      local rgas = registered_gases[gas_name]
      if not rgas then
        print("cannot find gas type")
        return nil
      end
      print('found gas in area thrown - switch from ' .. gas.gasType .. ' to ' .. gas_name)
      gas.decay_rate = rgas.decay_rate or 0
      gas.initial_volume = rgas.initial_volume
      gas.volume = volume or rgas.initial_volume or gas.volume
      gas.effect_name = rgas.effect_name
      gas.particle_texture = rgas.particle_texture or "toxicgas.png"
      gas.effect_duration = rgas.effect_duration or 1
      gas.igniter = rgas.igniter
      gas.flammable = rgas.flammable
      gas.nextVolume = gas.volume
      gas.gasType = rgas.name
      return check_obj
    end
  end
  return add_new_gas(gas_name, pos, volume)
end

local function swap_gas(self, new_gas_type, volume)
  local rgas = registered_gases[new_gas_type]
  if not rgas then
    print("cannot find gas type to swap to")
    return nil
  end

  self.decay_rate = rgas.decay_rate or 0
  self.initial_volume = rgas.initial_volume
  self.effect_name = rgas.effect_name
  self.effect_duration = rgas.effect_duration or 1
  self.particle_texture = rgas.particle_texture or "toxicgas.png"
  self.gasType = rgas.name
  self.igniter = rgas.igniter
  self.flammable = rgas.flammable
  self.volume = volume or self.volume
  self.nextVolume = volume or self.nextVolume
  -- keep volume and next volume the same
end

local dirs = {
  {x= 1,y= 0,z= 0},
  {x= 0,y= 1,z= 0},
  {x= 0,y= 0,z= 1},
  {x=-1,y= 0,z= 0},
  {x= 0,y=-1,z= 0},
  {x= 0,y= 0,z=-1},
}

-- Had to do this so it would turn up in the raycast
-- otherwise it wouldn't and nodes would get added on top of each other
local gas_queue = {}
local function add_gas_queue(gas_name, pos, volume)
  local in_pos = false
  for _, item in ipairs(gas_queue) do
    if item.pos.x == pos.x and
       item.pos.y == pos.y and
       item.pos.z == pos.z
    then
      in_pos = true
      break
    end
  end
  if not in_pos then
    table.insert(gas_queue, {name = gas_name, pos = pos, volume = volume})
  end
end


local timer = 0.5
core.register_globalstep(function(dtime)
  timer = timer - dtime
  if timer < 0 then
    timer = 0.5
    -- processes queue
    for _, item in ipairs(gas_queue) do
      add_new_gas(item.name, item.pos, item.volume)
    end
    gas_queue = {}
  end
end)

local function check_neighbor(obj, pos, dir)
  -- cast ray from pos to direction
  -- if no other gas entity or walkable node in ray
  -- then place another gas entity
  -- if there is a gas node or created one,
  -- return the gas type and volume
  -- new gas node volume will be 0
  local start_pos = vector.add(pos, dir)
  local target_pos = vector.add(pos, vector.multiply(dir, 2))

  -- check for neighbor first
  -- Maybe this will work?
  for check_obj in core.objects_inside_radius(target_pos, 1) do
    local gas = check_obj:get_luaentity()
    if gas and gas.name == "mg_effects:gas" then
      return {
        gas_id = gas.gas_id,
        volume = gas.volume or 0,
        gasType = gas.gasType or "",
        pos = target_pos,
        is_new = false
      }
    end
  end

  -- if hit a node then bug out
  local cast = core.raycast(start_pos, target_pos, true, false)
  local thing = cast:next()
  while thing do
    if thing.type == "node" then
      return nil
    end
    thing = cast:next()
  end


  local gas = obj:get_luaentity()
  return {
    gas_id = '0',
    volume = 0,
    gasType = gas.gasType,
    pos = target_pos,
    is_new = true
  }
end

local function apply_gas_effect(self)
  local pos = self.object:get_pos()
  -- should use bounding box instead...
  -- core.objects_in_area(min_pos, max_pos)
	-- local minp, maxp = mg_oe.get_collisionbox(obj, true, storage)
  for target in core.objects_inside_radius(pos, 1) do
    mg_oe.start_effect(self.effect_name, target, self.effect_duration, pos)
  end
end

local function apply_igniter(self)
  local nodes = mg_oe.get_touching_nodes(
    self.object,
    {"air", "group:air"},
    self
  )
  for _, node in ipairs(nodes) do
    core.set_node(node, {name = "fire:basic_flame"})
  end
end

local function apply_flammable(self)
  -- fire nodes and lava will set gases on fire
  local nodes = mg_oe.get_touching_nodes(
    self.object,
    {"group:set_on_fire"},
    self
  )
  if #nodes > 0 then
    print('should turn into fire gas!')
    swap_gas(self, self.flammable, self.nextVolume*2)
  end
end

core.register_entity("mg_effects:gas",{
	initial_properties = {
    pointable = false,
    visual = "sprite",
		visual_size = {x=1.99, y=1.99, z=1.99},
		collisionbox = {-1, -1, -1, 1, 1, 1},
    use_texture_alpha = true,
		physical = false,
    static_save = true,
    textures = { "blank.png" },
	},

	timer = 0,

	on_activate = function(self, _staticdata)
		if not self then
			self.object:remove()
			return
		end
	end,

	on_step = function(self, dtime)
    if not self.gasType then
      self.object:remove()
      return
    end
		self.timer = self.timer - dtime
    -- set volume to new volume from last update
    if self.timer < 0 then
      self.volume = (self.nextVolume or 0)
      self.timer = 0.5

      local totalVolume = self.volume
      local highestNeighborVolume = self.volume
      local gasType = self.gasType
      local numSpaces = 1
      local pos = self.object:get_pos()
      local new_gases = {}
      for _, dir in ipairs(dirs) do
        local result = check_neighbor(self.object, pos, dir)
        if result then
          totalVolume = totalVolume + result.volume
          numSpaces = numSpaces + 1
          if result.volume > highestNeighborVolume then
            highestNeighborVolume = result.volume
            gasType = result.gasType
          end

          if result.is_new then
            table.insert(new_gases, result)
          end
        end
      end

      self.nextVolume = math.floor(totalVolume / numSpaces)

      -- stochastic rounding
      if rand_range(0, numSpaces - 1) < (totalVolume % numSpaces) then
        self.nextVolume = self.nextVolume + 1
      end

      -- switch gas type
      -- if a neighbor of a different gas type has a higher
      -- volume, then switch gas type
      -- however fire cannot switch
      if not self.igniter and gasType ~= self.gasType and self.nextVolume > 3 then
        swap_gas(self, gasType)
      end

      -- old gases 0 and under die
      -- only allow new gases to live if their volume is greater than 3
      if self.nextVolume < 1 or (self.is_new and self.nextVolume < 3) then
        self.object:remove()
      else
        self.is_new = false

        -- apply gas effect
        if self.effect_name then
          apply_gas_effect(self)
        end

        -- can't be an igniter and
        -- flammable at the same time!!
        if self.igniter then
          apply_igniter(self)
        elseif self.flammable then
          apply_flammable(self)
        end

        local amount = 4 + math.min(30, math.floor(16 * self.nextVolume / 40 ))
        add_spawner(self.object, amount, self.particle_texture)
        -- disspate quickly
        if self.decay_rate > 0 and rand_percent(self.decay_rate) then
          self.nextVolume = self.nextVolume - 1
        end
        -- queue up new gases to create
        for _, newGas in ipairs(new_gases) do
          add_gas_queue(newGas.gasType, newGas.pos, 0)
        end
      end
    end
	end

})

register_gas("flame", {
  particle_texture = "mg_effects_flame.png",
  initial_volume = 1000,
  effect_name = "burning",
  effect_duration = 8,
  decay_rate = 60,
  igniter = true
})


register_gas("caustic", {
  particle_texture = "mg_effects_toxicgas.png",
  initial_volume = 1000,
  effect_name = "gas_damage",
  effect_duration = 0.75,
  flammable = "mg_effects:flame_gas"
})

register_gas("paralytic", {
  particle_texture = "mg_effects_paralyticgas.png",
  initial_volume = 1000,
  effect_name = "gas_paralysis",
  effect_duration = 20,
  flammable = "mg_effects:flame_gas"
})

register_gas("confusion", {
  particle_texture = "mg_effects_confusiongas.png",
  initial_volume = 1000,
  effect_name = "confusion",
  effect_duration = 25,
  flammable = "mg_effects:flame_gas"
})


mg_effects.add_gas = add_gas
