-- Defined effects that can happen to the map
-- when reading a scroll, throwing a potion, or using a charm

local crystal = "mg_effects:crystal"
minetest.register_node(crystal ,{
	description = "Tinted Glass",
	tiles = {"mcl_amethyst_tinted_glass.png"},
	drawtype = "glasslike",
	use_texture_alpha = "blend",
	sunlight_propagates = false,
	groups = {handy = 1, building_block = 1, deco_block = 1},
	is_ground_content = false,
	light_source = 4
})

-- Use an ABM to get rid of the crystal
-- only if its exposed to air
core.register_abm({
  nodenames = {crystal},
  neighbors = {"air"},
  interval = 2,
  chance = 10,
  action = function(pos)
    core.set_node(pos, {name = "air"})
  end
})

-- TODO use MT groups for this
local to_crystal_list = {
  "mg_mapgen:sand",
  "mg_mapgen:stone",
  "mg_torch:light",
  "mg_mapgen:blackstone",
  "mg_mapgen:chasm_edge",
  "mg_mapgen:stone_block",
  "mg_mapgen:cobble",
  "mg_mapgen:mossycobble",
  crystal
}

local ignore_list = {
  "mg_ladder:ladder_wood",
  "mg_mapgen:water_source",
  "mg_mapgen:magma_source",
  'mg_chest:chest',
  'mg_mapgen:goldblock',
  'mg_mapgen:gold_pile_1',
  'mg_arch:crystal',
  'mg_game:crystal',
}

local function some(tbl, f)
  for _,v in ipairs(tbl) do
    if f(v) then
      return true
    end
  end
  return false
end


local function shattering(params)
  local radius = params.radius or 5
  local pos = params.pos or {x = 0, y = 0, z = 0}

  print('shatter radius '..radius)
  print('at x: '..params.pos.x..', y: '..params.pos.y..', z: '..(params.pos.z or 0))

  -- Turns walls, doors into glowing green crystals
  -- Could use that purplse stuff from librevox
  -- kills turrets

  for i = 1, radius*2 do
    for j = 1, radius*2 do
      local x = i + pos.x - radius
      local z = j + pos.z - radius
      local baseY = pos.y

      local ti = i - radius - 1
      local tj = j - radius - 1
      if (ti*ti + tj*tj < radius * radius) then

        local list = {
          { x = x, y = baseY + 0, z = z },
          { x = x, y = baseY + 1, z = z },
          { x = x, y = baseY + 2, z = z },
          { x = x, y = baseY + 3, z = z },
          { x = x, y = baseY + 4, z = z },
        }

        for li = 1, #list do
          local node = core.get_node(list[li])
          local to_crystal = some(to_crystal_list, function(test_name)
            return test_name == node.name
          end)
          local ignore = some(ignore_list, function(test_name)
            return test_name == node.name
          end)
          if to_crystal then
            core.set_node(list[li], {name=crystal})
          elseif not ignore then
            -- Stuff like grass and stones
            core.set_node(list[li], {name='air'})
          end
        end
      end

    end
  end

  for _, v in pairs(core.get_objects_inside_radius(pos, radius)) do
    if not v:is_player() and v:get_hp() > 0 then
      local mob = v:get_luaentity()
      if mob.behaviors and mob.behaviors.in_wall then
        mg_mobs.hq_die(mob)
      end
    end
  end
end

local function fireball(params)
  local radius = params.radius or 5
  local pos = params.pos or {x = 0, y = 0, z = 0}

  for i = 1, radius*2 do
    for j = 1, radius*2 do
      local x = i + pos.x - radius
      local z = j + pos.z - radius
      local baseY = pos.y

      local ti = i - radius
      local tj = j - radius
      if (ti*ti + tj*tj < radius * radius) then

        local list = {
          { x = x, y = baseY + 3, z = z },
          { x = x, y = baseY + 2, z = z },
          { x = x, y = baseY + 1, z = z },
          { x = x, y = baseY + 0, z = z },
          { x = x, y = baseY - 1, z = z },
          { x = x, y = baseY - 2, z = z },
        }

        for li = 1, #list do
          local node = core.get_node(list[li])
          local nodedef = core.registered_nodes[node.name]
          if
            (node.name == 'air' or nodedef.drawtype == "airlike")
          then
            core.set_node(list[li], {name='fire:basic_flame'})
          end
        end
      end

    end
  end
end

local function has_line_of_sight(source_pos, target)
  local target_pos = mobkit.get_middle_pos(target)
  local cast = core.raycast(source_pos, target_pos, false, false)
  local thing = cast:next()
  while thing do
    if thing.type == "node" then
      local node = core.get_node(thing.intersection_point)
      local def = core.registered_nodes[node.name]
      -- A walkable node is a solid node
      if def and def.walkable then
        return false
      end
    end
    thing = cast:next()
  end
  return true
end

local function is_player_or_mob(target)
  local mob = target:get_luaentity()
  return target:get_properties().pointable and
    ((target:is_player() and target:get_hp() > 0) or
    (mob and mob.mg_id and mob.hp > 0))
end

local function apply_discord(params)
  local radius = params.radius or 10
  local pos = params.pos or {x = 0, y = 0, z = 0}

  for _, v in pairs(core.get_objects_inside_radius(pos, radius)) do
    if is_player_or_mob(v) and has_line_of_sight(pos, v) then
      mg_oe.start_effect("discordant", v, 30)
    end
  end
end

local function negate(params)
  local radius = params.radius or 10
  local pos = params.pos or {x = 0, y = 0, z = 0}

  for _, v in pairs(core.get_objects_inside_radius(pos, radius)) do
    -- should do a ray check to see if mob is visible from position
    -- only checking nodes
    local los = has_line_of_sight(pos, v)
    if v:is_player() and
      v:get_hp() > 0 and
      v:get_properties().pointable and
      los
    then
      mg_oe.negate_effects(v)
    end
    if not v:is_player() and
       v:get_hp() > 0 and
       los
    then
      local mob = v:get_luaentity()
      if (mob.hp or -1) > 0 then
        mg_mobs.negate(mob)
      end
    end
  end
end

local function explode(params)
  -- add fireball
  fireball(params)
  -- find players and mobs near in radius
  -- damage them for half their health
  -- knockback
  local radius = params.radius or 5
  local pos = params.pos or {x = 0, y = 0, z = 0}

  minetest.sound_play("grenades_explode", {
    pos = pos,
    gain = 1.0,
    max_hear_distance = 32,
  })

  for _, v in pairs(core.get_objects_inside_radius(pos, radius)) do
    -- should probably do a los check...
    local op = v:get_pos()
    local pushback = vector.multiply(
      vector.normalize({x=op.x - pos.x,y=op.y - pos.y,z=op.z - pos.z}),
      radius*3)

    if v:is_player() and v:get_hp() > 0 and v:get_properties().pointable then
      local hp = v:get_hp()
      if hp > 0 then
        v:set_hp(math.floor(hp/2 + 0.5) , {_mcl_reason = {type = "explosion"}})
        v:add_velocity(pushback)
      end
    end
    if not v:is_player() and v:get_hp() > 0 then
      local mob = v:get_luaentity()
      if (mob.hp or -1) > 0 then
        mobkit.make_sound(mob,'hurt')
        mobkit.hurt(mob, math.floor(mob.hp/2 + 0.5))
        mobkit.hq_runfrom_pos(mob, 10, pos)
        v:add_velocity(pushback)
      end
    end
  end
end

local function descent(params)
  local radius = params.radius or 6
  local upper_radius = params.upper_radius or radius
  local pos = params.pos or {x = 0, y = 0, z = 0}
  local time_offset = 5

  for i = 1, upper_radius*2 do
    for j = 1, upper_radius*2 do
      local x = i + pos.x - upper_radius
      local z = j + pos.z - upper_radius
      local baseY = pos.y

      local ti = i - upper_radius
      local tj = j - upper_radius
      if (ti*ti + tj*tj < upper_radius * upper_radius) then

        local list = {
          { x = x, y = baseY + 3, z = z },
          { x = x, y = baseY + 2, z = z },
          { x = x, y = baseY + 1, z = z },
          { x = x, y = baseY + 0, z = z },
          { x = x, y = baseY - 1, z = z },
          { x = x, y = baseY - 2, z = z },
          { x = x, y = baseY - 3, z = z },
        }

        for li = 1, #list do
          local node = core.get_node(list[li])
          local oldNodeName = node.name
          local oldParam2 = node.param2
          -- I'll set a timeout and restore starting from the outside
          local time = (radius*radius - (ti*ti + tj*tj)) / radius + time_offset
          --local time = 5
          -- in the old nodes
          if node.name ~= 'air' then
            -- check if its a chest and drop its contents
            -- old node will then be air
            if string.find(node.name, "mg_chest") then
              local meta = minetest.get_meta(list[li])
              local inv = meta:get_inventory()
              local main_list = inv:get_list("main")

              for idx, item in ipairs(main_list) do
                core.item_drop(item, nil, list[li])
                inv:set_stack("main", idx, nil)
              end
              oldNodeName = "air"
            end
            core.set_node(list[li], {name='air'})
            core.after(time, function()
              core.set_node(list[li], {name=oldNodeName, param2 = oldParam2})
            end)
          end
        end
      end

    end
  end

  for i = 1, radius*2 do
    for j = 1, radius*2 do
      local x = i + pos.x - radius
      local z = j + pos.z - radius
      local baseY = pos.y

      local ti = i - radius
      local tj = j - radius
      if (ti*ti + tj*tj < radius * radius) then

        local list = {
          { x = x, y = baseY - 4, z = z },
          { x = x, y = baseY - 5, z = z },
          { x = x, y = baseY - 6, z = z },
          { x = x, y = baseY - 7, z = z },
          { x = x, y = baseY - 8, z = z },
          { x = x, y = baseY - 9, z = z },
        }

        for li = 1, #list do
          local node = core.get_node(list[li])
          local oldNodeName = node.name
          -- I'll set a timeout and restore starting from the outside
          local time = (radius*radius - (ti*ti + tj*tj)) / radius + time_offset
          --local time = 5
          -- in the old nodes
          core.set_node(list[li], {name='air'})
          core.after(time, function()
            core.set_node(list[li], {name=oldNodeName})
          end)
        end
      end

    end
  end
end

local function lichen(params)
  local pos = params.pos or {x = 0, y = 0, z = 0}

  local node = core.get_node(pos)
  local y = pos.y
  while node.name == 'air' or core.get_item_group(node.name, 'replaced_by_lichen') > 0 do
    y = y - 1
    node = core.get_node({x = pos.x, y = y, z = pos.z})
  end
  -- lichen only grows on rocks
  if core.get_item_group(node.name, 'cracky') > 0 then
    local new_pos = {x = pos.x, y = y + 1, z = pos.z}
    core.set_node(new_pos, {name='mg_mapgen:deadly_lichen'})
    core.get_node_timer(new_pos):start(math.random(5, 10))
  end
end

local function obstruction(params)
  local radius = params.radius or 2
  local pos = params.pos or {x = 0, y = 0, z = 0}

  for i = 1, radius*2 do
    for j = 1, radius*2 do
      local x = i + pos.x - radius
      local z = j + pos.z - radius
      local baseY = pos.y

      local ti = i - radius
      local tj = j - radius
      if (ti*ti + tj*tj < radius * radius) then

        local list = {
          { x = x, y = baseY + 3, z = z },
          { x = x, y = baseY + 2, z = z },
          { x = x, y = baseY + 1, z = z },
          { x = x, y = baseY + 0, z = z },
          { x = x, y = baseY - 1, z = z },
          { x = x, y = baseY - 2, z = z },
          { x = x, y = baseY - 3, z = z },
          { x = x, y = baseY - 4, z = z },
        }

        for li = 1, #list do
          local node = core.get_node(list[li])
          local oldNodeName = node.name
          -- should use an lbm / abm instead like fire
          if(oldNodeName == 'air' or core.get_item_group(oldNodeName, 'replaced_by_lichen') > 0) then
            core.set_node(list[li], {name=crystal})
          end
        end
      end
    end
  end
end

local passableArcCount = mg_arch.passableArcCount
local tt = mg_arch.tileType
local startY = 0

local function teleport(obj)
  -- get the level the player is on
  local pos = obj:get_pos()
  local depth = math.floor((pos.y - startY - 10) / (-11) + 1)

  if depth < 1 then
    return
  end

  -- would like to figure this out without pmap
  local pmap = mg_arch.load_pmap(depth)
  local height = -11 * (depth - 1) + startY + 6
  -- get the dungeon from the pmap array
  -- get y pos floor

  local x, y, loc
  repeat
    loc = mg_arch.randomMatchingLocation(pmap, tt.FLOOR, tt.NOTHING, -1)
    x = loc.x
    y = loc.y
  until passableArcCount(pmap, loc.x, loc.y) <= 1 -- not in a hallway

  obj:set_pos({x = x, y = height, z = y})
end


-- The worst
local mod_path = minetest.get_modpath("mg_effects")
local df = dofile(mod_path.."/spawn_df.lua")


mg_effects = {}
mg_effects.shattering = shattering
mg_effects.descent = descent
mg_effects.fireball = fireball
mg_effects.explode = explode
mg_effects.lichen = lichen
mg_effects.obstruction = obstruction
mg_effects.teleport = teleport
mg_effects.negate = negate
mg_effects.apply_discord = apply_discord
mg_effects.net = df.net
mg_effects.flood = df.flood
mg_effects.timed_flood = df.timed_flood

dofile(mod_path.."/gas.lua")
dofile(mod_path.."/cobwebs.lua")
dofile(mod_path.."/suffocation.lua")

