
local item_physics = {}

item_physics.settings = {
  item_scale = nil,
  rotation = true,
  initial_rotation = true
}

if (item_physics.settings.item_scale ~= nil) and item_physics.settings.item_scale <= 0.05 then
  item_physics.settings.item_scale = nil
end

local flat_node_types = {
  signlike = true,
  raillike = true,
  torchlike = true,
  plantlike = true,
}

minetest.register_on_mods_loaded(function()
  local builtin_item = minetest.registered_entities["__builtin:item"]

  -- rot = vector.new(0,0,1):rotate(rot)
  local new_item_ent = {
    _get_is_node = function(self, itemname)
      local idef = minetest.registered_items[itemname]
      if not idef then return false end
      if idef.type ~= "node" then return false end
      if flat_node_types[idef.drawtype] ~= nil then return false end
      if idef.wield_image ~= "" then return false end
      return true
    end,
    _item_stick = function(self)
      self._is_stuck = true
      if (not self._is_node) then
        local rot = self.object:get_rotation()
        rot = vector.new(math.pi*0.5, rot.y, 0)
        self.object:set_rotation(rot)
      end
    end,
    _item_unstick = function(self)
      self._is_stuck = false
      local rot = self.object:get_rotation()
      rot = vector.new(0, rot.y, 0)
      self.object:set_rotation(rot)
    end,
    enable_physics = function(self)
      local ret = builtin_item.enable_physics(self)
      self:_item_unstick()
      return ret
    end,
    disable_physics = function(self)
      local ret = builtin_item.disable_physics(self)
      self:_item_stick()
      return ret
    end,
    set_item = function(self, item, ...)
      local ret = builtin_item.set_item(self, item, ...)
      local stack = ItemStack(item or self.itemstring)
      local itemname = stack:get_name()

      self._is_node = self:_get_is_node(itemname)

      local props = self.object:get_properties()
      if not props.collisionbox then
        props.collisionbox = {-0.3, -0.3, -0.3, 0.3, 0.3, 0.3}
      end
      -- allow custom size
      if item_physics.settings.item_scale then
        local s = item_physics.settings.item_scale
        local c = s - 0.1
        props.collisionbox = {-c,-c,-c, c, c, c}
        props.visual_size = vector.new(s,s,s)
      else
        -- use the other sizes as a reference to make it compat with more things
        props.collisionbox[2] = props.collisionbox[1] * 0.6
      end
      -- items lie flat
      if (not self._is_node) then
        props.collisionbox[2] = -0.03
      end
      -- remove `automatic_rotate` because it's broken
      self.object:set_properties({
        automatic_rotate = 0,
        collisionbox = props.collisionbox,
        visual_size = props.visual_size,
      })
      if item_physics.settings.initial_rotation then
        self.object:set_rotation(vector.new(0,math.random()*math.pi*2, 0))
      end
      if self._is_stuck then
        self:_item_stick()
      else
        self:_item_unstick()
      end
      return ret
    end,
    try_merge_with = function(self, own_stack, object, entity, ...)
      return builtin_item.try_merge_with(self, own_stack, object, entity, ...)
    end,
    _do_idle_rotation = function(self, amount)
      local rot = self.object:get_rotation()
      if not rot then return end
      rot = vector.new(0, rot.y - amount * self._spin_dir, 0)
      self.object:set_rotation(rot)
    end,
    on_step = function(self, dtime, moveresult, ...)
      -- if not self.itemstring then return end
      local pos = self.object:get_pos()
      -- prevent unloaded nil errors just in case
      if not pos then return end
      local vel = self.object:get_velocity()

      -- buoyancy
      local height = 0.6
      local buoyancy = 0.5
      local surface = nil
      local spos = mobkit.get_stand_pos(self)
      spos.y = spos.y+0.02

      -- get surface height
      local snodepos = mobkit.get_node_pos(spos)

      local lcounter = 0
      local surfnode = mobkit.nodeatpos(spos)
      while surfnode and surfnode.drawtype == 'liquid' do
        lcounter = lcounter + 1
        surface = snodepos.y + 0.5
        if surface > spos.y + height then break end
        snodepos.y = snodepos.y + 1
        surfnode = mobkit.nodeatpos(snodepos)
      end
      if surface then
        local submergence = math.min(surface - spos.y, height)/ height
        local buoyacc = -9.81 * ( buoyancy - submergence)
        self.object:set_acceleration({
          x=-vel.x,
          y=buoyacc-vel.y*math.abs(vel.y)*0.4,
          z=-vel.z
        })
      end

      -- if it's not moving, stick
      -- items lay flat while sliding, nodes don't
      if not self._is_node then vel.x = 0; vel.z = 0; end
      local should_stick = (math.abs(vel.x) + math.abs(vel.y) + math.abs(vel.z) < 0.1)
      if should_stick and not self._is_stuck and not surface then
        self:_item_stick()
      elseif self._is_stuck and not should_stick and not surface then
        self:_item_unstick()
      end

      if not self._spin_dir then
        self._spin_dir = math.random(0,1)*2-1
      end
      if (not self._is_stuck) and item_physics.settings.rotation then
        local spin_speed = math.min(0.3, dtime * vel:length())
        self._do_idle_rotation(self, spin_speed)
      end

      local dnodepos = mobkit.get_node_pos(spos)
      local dnode1 = mobkit.nodeatpos(dnodepos)
      dnodepos.y = dnodepos.y - 1
      local dnode2 = mobkit.nodeatpos(dnodepos)
      local in_deep = dnode1 and dnode2 and
        (dnode1.drawtype=='liquid' or dnode1.drawtype=='flowingliquid')  and
        (dnode2.drawtype=='liquid' or dnode2.drawtype=='flowingliquid')

      -- should "wander" around in the water
      if in_deep then
        self._water_timer = self._water_timer or 0
        self._water_timer = self._water_timer + dtime
        if self._water_timer > 1 then
          self._water_timer = 0
          local random_yaw = math.random() * 2 * math.pi
          local dir = core.yaw_to_dir(random_yaw)
          self.object:add_velocity(vector.add(
            vector.multiply(vel, -1),
            vector.multiply(dir, 2))
          )
        end
      -- not in deep water anymore but timer is still running
      elseif self._water_timer and self._water_timer > 0 then
        self._water_timer = self._water_timer + dtime
        -- stop the object from moving
        if self._water_timer > 1 then
          self._water_timer = 0
          self.object:add_velocity(vector.multiply(vel, -1))
        end
      end

      builtin_item.on_step(self, dtime, moveresult, ...)
    end,
  }

  setmetatable(new_item_ent, { __index = builtin_item })
  minetest.register_entity(":__builtin:item", new_item_ent)
end)

local function player_drop(player)
  local inv = player:get_inventory()
  local wi = player:get_wield_index()

  local main_list = inv:get_list("main")
  local pick_list = {}

  for i, item in ipairs(main_list) do
    if item and i ~= wi and item:get_count() > 0 then
      table.insert(pick_list, i)
    end
  end

  if #pick_list == 0 then
    return
  end

  local rand_pick_idx = math.random(1, #pick_list)
  local rand_list_id = pick_list[rand_pick_idx]
  local item = main_list[rand_list_id]
  local meta = item:get_meta()
  local def = item:get_definition()

  local desc = meta:get_string("description")
  if desc == '' then
    desc = def.description
  end

  local player_name = player:get_player_name()
  core.chat_send_player(player_name, "You lost your " .. desc .. " in the deep water!")

  core.item_drop(item, player, player:get_pos())
  inv:set_stack("main", rand_list_id, nil)
end

local drop_timer = 0
core.register_globalstep(function(dt)
  -- update hud for players
  drop_timer = drop_timer + dt
  if drop_timer > 0.75 then
    drop_timer = 0
    for _, player in pairs(core.get_connected_players()) do
      if mobkit.is_in_deep(player) and not mobkit.is_in_deep_lava(player) then
        player_drop(player)
      end
    end
  end
end)

