local mod_path = core.get_modpath("mg_mobs")
local ai_tools = dofile(mod_path.."/ai_tools.lua")
local spawn = dofile(mod_path.."/spawn.lua")

local function check_is_mob(obj)
  if obj:is_player() then
    return false
  end
  return obj:get_luaentity().mg_id ~= nil
end

local function get_name(obj)
  if obj:is_player() then
    return obj:get_player_name()
  end
  return obj:get_luaentity().name
end

local function has_line_of_sight(self, target)
  local mob_pos = mobkit.get_middle_pos(self)
  local target_pos = mobkit.get_middle_pos(target)
  local cast = core.raycast(mob_pos, target_pos, true, false)
  local thing = cast:next()
  while thing do
    if thing.type == "node" then
      local node = core.get_node(thing.intersection_point)
      local def = core.registered_nodes[node.name]
      -- A walkable node is a solid node
      if def and def.walkable then
        return false
      end
    end

    if thing.type == "object" then
      -- if it's a player or mob and not target or self, then it should
      -- break LOS
      local is_player = thing.ref:is_player()
      local is_mob = check_is_mob(thing.ref)
      local can_block = is_player or is_mob
      if can_block and thing.ref ~= self.object and thing.ref ~= target then
        return false
      end
    end
    thing = cast:next()
  end
  return true
end

-- https://www.gamedeveloper.com/programming/shooting-a-moving-target
local function aim_ahead(pos, vel, speed)
  local a = vector.dot(vel, vel) - speed*speed
  local b = 2*vector.dot(vel, pos)
  local c = vector.dot(pos, pos)

  local desc = b*b - 4*a*c;

  -- If the discriminant is negative, then there is no solution
  if desc > 0 then
    return 2*c/(math.sqrt(desc) - b)
  else
    return -1
  end
end

local function aim_point(tpos, tvel, spos, speed)
  local dt = aim_ahead(vector.subtract(tpos, spos), tvel, speed)
  if dt == -1 then
    return nil
  end
  local point = vector.add(tpos, vector.multiply(tvel, dt))
  return point
end


local function obj_has_health_max(obj)
  if obj:is_player() then
    local hp = obj:get_hp()
    local props = obj:get_properties()
    return hp == props.hp_max
  elseif obj:get_luaentity().mg_id ~= nil then
    local mob = obj:get_luaentity()
    -- full health - don't heal
    return not mob.max_hp or mob.hp == mob.max_hp
  end
  return true
end

local function blink_toward(self, target, target_pos, dist)
  local can_see = has_line_of_sight(self, target)
  if not can_see then
    return false
  end
  -- follow the line form mob to target
  mg_bolts.mob_shoot("mg_bolts:blinking", self.object, target_pos, (dist - 2)/10, 0)
  return true
end

local function blink_away(self, target_pos)
  -- to a radial search from target pos for block that is > 5 blocks and < 10 blocks away from target pos
  -- and isn't behind a wall and is in an non-walkable node
  local locations = ai_tools.find_loc(target_pos, 5, 10, true, 1)
  if #locations > 0 then
    local middle_pos = mobkit.get_middle_pos(self)
    local dist = vector.distance(middle_pos, locations[1])
    mg_bolts.mob_shoot("mg_bolts:blinking", self.object, locations[1], dist/10, 0)
  end
  return false
end


local function blink(self,  target)
  local middle_pos = mobkit.get_middle_pos(self)
  local target_pos = mobkit.get_stand_pos(target)
  local target_middle_pos = mobkit.get_middle_pos(target)
  local vel = target:get_velocity()
  local future_target_pos = aim_point(target_middle_pos, vel, middle_pos, 10)

  -- this happens sometimes - not sure why
  if future_target_pos == nil then
    return false
  end

  local dist = vector.distance(middle_pos, future_target_pos)
  local hp_percent = self.hp / self.max_hp * 100

  if not self.behaviors then
    self.behaviors = {}
  end

  if hp_percent <= 30 then
    self.behaviors.maintains_distance = 1
  elseif hp_percent > 30 then
    self.behaviors.maintains_distance = nil
  end

  -- if hp is > 30% and target is > 3 blocks away blink toward pos 2 block away
  if hp_percent > 30 and dist > 3 then
    -- All better, bring it!
    self.behaviors.maintains_distance = nil
    return blink_toward(self, target, future_target_pos, dist)
  -- if hp is <= 25% and target is < 4 blocks away blink away 5 blocks
  elseif hp_percent <= 30 and dist <= 4 then
    -- and also blink away when too close
    return blink_away(self, target_pos)
  end
  return false

end


-- protection is a bolt you shoot on allies to protect them
local function shoot_bolt_friend(self, bolt, check_effect)
  local nearby_group_members = {}

  -- looks for mobs nearby in the same group
	for _,obj in ipairs(self.nearby_objects) do
    local mob = obj:get_luaentity()
    if mob and
       -- in the same group - have the same leader
       mob.leader_id == self.leader_id and
       -- mob from mg_mobs
       mob.mg_id and
       mobkit.is_alive(mob) and
       -- not self
       mob.mg_id ~= self.mg_id
    then
      -- Healing is a special case
      if bolt == "healing" and not obj_has_health_max(mob.object) then
        table.insert(nearby_group_members, mob)
      elseif check_effect ~= nil then
        -- if bolt is healing, then should check if target is fully healed
        local p = mg_oe.get_effect_data(check_effect, obj)
        -- not already under effect
        if not p.duration or p.duration == 0 then
          table.insert(nearby_group_members, mob)
        end
      end
    end
  end

  if #nearby_group_members == 0 then
    return
  end

  ai_tools.shuffle_list_in_place(nearby_group_members)

  local middle_pos = mobkit.get_middle_pos(self)
  local target = nil

	for _, mob in ipairs(nearby_group_members) do
    local can_see = has_line_of_sight(self, mob.object)
    if can_see then
      target = mob
      break
    end
  end

  if not target then
    return
  end


  local target_middle_pos = mobkit.get_middle_pos(target)
  -- TODO handle when friend is a player as well
  local vel = target.object:get_velocity()
  local future_target_pos = aim_point(target_middle_pos, vel, middle_pos, 10)

  if future_target_pos then
    mg_bolts.mob_shoot("mg_bolts:"..bolt, self.object, future_target_pos, 1, 0)
    return true
  end
end

-- webs shoot at players
local function shoot_bolt_enemy(self, bolt, duration, enchant)
  local nearby_players = {}
	for _,obj in ipairs(self.nearby_objects) do
    if obj:is_player() and mobkit.is_alive(obj) then
      table.insert(nearby_players, obj)
    end
  end

  if #nearby_players == 0 then
    return
  end

  ai_tools.shuffle_list_in_place(nearby_players)

  local middle_pos = mobkit.get_middle_pos(self)
  local target = nil

	for _, player in ipairs(nearby_players) do
    local can_see = has_line_of_sight(self, player)
    if can_see then
      target = player
      break
    end
  end

  if not target then
    return
  end

  local target_middle_pos = mobkit.get_middle_pos(target)
  local vel = target:get_velocity()
  local future_target_pos = aim_point(target_middle_pos, vel, middle_pos, 10)

  if future_target_pos then
    mg_bolts.mob_shoot("mg_bolts:"..bolt, self.object, future_target_pos, duration, enchant)
    return true
  end
end

local function shoot_arrow(self, arrow_kind)
  arrow_kind =  arrow_kind or "arrow"
  local nearby_players = {}
	for _,obj in ipairs(self.nearby_objects) do
    if obj:is_player() and mobkit.is_alive(obj) then
      table.insert(nearby_players, obj)
    end
  end

  if #nearby_players == 0 then
    return
  end

  ai_tools.shuffle_list_in_place(nearby_players)

  local middle_pos = mobkit.get_middle_pos(self)
  local target = nil

	for _, player in ipairs(nearby_players) do
    local can_see = has_line_of_sight(self, player)
    if can_see then
      target = player
      break
    end
  end

  if not target then
    return
  end

  local target_middle_pos = mobkit.get_middle_pos(target)
  local vel = target:get_velocity()
  local future_target_pos = aim_point(target_middle_pos, vel, middle_pos, 20)
  future_target_pos.y = future_target_pos.y + 1

  if future_target_pos then
    bows.mob_shoot("bows:"..arrow_kind, self.object, future_target_pos)
    return true
  end
end

local function try_shoot(self, bolt, target)
  if bolt == 'blink' then
    return blink(self, target)
  -- Friendly bolts
  elseif bolt == 'protection' then
    return shoot_bolt_friend(self, "protection", "protection")
  elseif bolt == 'haste' then
    return shoot_bolt_friend(self, "haste", "speed")
  elseif bolt == 'healing' then
    return shoot_bolt_friend(self, "healing")

  -- Enemy Bolts
  elseif bolt == 'spiderweb' then
    return shoot_bolt_enemy(self, "web", 4, 1)
  elseif bolt == 'slow' then
    return shoot_bolt_enemy(self, "slowness", 4, 1)
  elseif bolt == 'spark' then
    return shoot_bolt_enemy(self, "lightning", 4, 2)
  elseif bolt == 'fire_bolt' then
    return shoot_bolt_enemy(self, "fire", 4, 2)
  elseif bolt == 'negation' then
    return shoot_bolt_enemy(self, "negation", 5, 2)

  -- Enemy Arrows
  elseif bolt == 'arrow' then
    return shoot_arrow(self, "arrow")
  elseif bolt == 'steel_arrow' then
    return shoot_arrow(self, "steel_arrow")
  elseif bolt == 'fire_arrow' then
    return shoot_arrow(self, "fire_arrow")
  end
  return false
end


local function shoot_bolts(self, target)
  if not self.bolts or #self.bolts == 0 then
    return false
  end

  ai_tools.shuffle_list_in_place(self.bolts)

  -- if mob can shoot multiple bolts
  -- can't can shoot one, then try another next
  -- bolt
	for _, bolt in ipairs(self.bolts) do
    local did_shoot = try_shoot(self, bolt, target)
    if did_shoot then
      return true
    end
  end

  return false
end


local function use_abilities(self, target)
  self.ability_timer = self.ability_timer or 0

  local duration = 2

  if self.behaviors and self.behaviors.cast_spells_slowly == 1 then
    duration = 4
  end

  -- every 4 seconds now that I got this working right
  -- if speed up should increase duration
  if self.ability_timer < duration then
    return
  end
  self.ability_timer = 0

  local abilities = {
    "bolts",
    "summon"
  }

  ai_tools.shuffle_list_in_place(abilities)

  -- for mobs that can cast summon and bolts
  -- randomize what they try to do first
	for _, ability in ipairs(abilities) do
    if ability == "bolts" then
      if self.bolts and #self.bolts > 0 then
        local did_shoot = shoot_bolts(self, target)
        if did_shoot then
          mobkit.make_sound(self,'warn')
          -- return so dont summon and shoot on same turn
          return true
        end
      end
    elseif ability == "summon" then
      -- if monster didn't shoot and they can summon, let them try that now
      if self.abilities and self.abilities.cast_summon then
        local did_summon = spawn.summon_horde(self, 50, target)
        if did_summon then
          return true
        end
      end
    end
  end
  return false
end

local function steal(self, player)
  if not player:is_player() then
    return
  end

  local mob_id = self.mg_id
  if not mob_id then
    mob_id = mg_mobs.uuid()
    self.mg_id = mob_id
  end

  -- Use a new list on the player for this
  -- detached inventories don't save between game restarts
  -- so you could potentially lose a good item if the server crashed
  local inv = player:get_inventory()
  inv:set_size("mob_"..mob_id, 8)

  local wi = player:get_wield_index()

  local main_list = inv:get_list("main")
  local pick_list = {}

  for i, item in ipairs(main_list) do
    if item and i ~= wi and item:get_count() > 0 then
      table.insert(pick_list, i)
    end
  end

  if #pick_list == 0 then
    return
  end

  local rand_pick_idx = math.random(1, #pick_list)
  local rand_list_id = pick_list[rand_pick_idx]
  local item = main_list[rand_list_id]
  local meta = item:get_meta()
  local def = item:get_definition()

  local desc = meta:get_string("description")
  if desc == '' then
    desc = def.description
  end

  local player_name = player:get_player_name()
  core.chat_send_player(player_name, "The monkey stole your " .. desc)

  inv:set_stack("main", rand_list_id, nil)
  inv:add_item("mob_"..mob_id, item)
end


local function apply_hitter_abilities(player, hitter, damage)
  local mob = hitter:get_luaentity()

  if not mob then
    print('hitter not a mob')
    return
  end

  if not mob.abilities then
    print('hitter has no abilities')
    return
  end

  local abilities = mob.abilities

  if abilities.poisons then
    -- Poison does stack correctly
    mg_oe.start_effect("poison", player, 5, 1)
  end

  if abilities.causes_weakness then
    mg_oe.start_effect("weakness", player, 300)
  end

  if abilities.kamikaze then
    mg_mobs.hq_die(mob)
  end

  if abilities.hit_degrade_armor then
    core.sound_play("mg_acid", {max_hear_distance = 10, pos = player:get_pos()})
    mg_armor.degrade_player_armor(player, 1)
  end

  if abilities.hit_steal_flee then
    steal(mob, player)
    mobkit.hq_runfrom(mob, 15, player)
  end


  if abilities.hit_burn then
    mg_oe.start_effect("burning", player, 7)
  end

  if abilities.transference then
    local transfer_hp = math.floor(damage * 0.9)
    mob.hp = mob.hp + transfer_hp
    if mob.hp > mob.max_hp then
      mob.hp = mob.max_hp
    end
  end

  if abilities.hit_hallucinate then
    mg_oe.start_effect("darkness", player, 120, 40)
  end

  if abilities.seizes then
    -- need to figure this one out
    -- basically set speed to 0
    mg_oe.start_effect("gas_paralysis", player, 5)
  end

end



return {
  shoot_bolts = shoot_bolts,
  use_abilities = use_abilities,
  apply_hitter_abilities = apply_hitter_abilities
}
