local function clamp(x, low, high)
  return math.min(high, math.max(x, low))
end

local function rand_percent(percent)
  return math.random(0,99) < clamp(percent, 0, 100)
end

local function uuid()
  local template ='xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'
  return string.gsub(template, '[xy]', function (c)
    local v = (c == 'x') and math.random(0, 0xf) or math.random(8, 0xb)
    return string.format('%x', v)
  end)
end

local function shuffle_list_in_place(list)
  for i=1, #list do
    local r = math.random(i, #list)
    if i~= r then
      local buf = list[r]
      list[r] = list[i]
      list[i] = buf
    end
  end
end

local bolts_cannot_negate = {
  "arrow",
  "steel_arrow"
}

local behaviors_cannot_negate = {
  "maintains_distance",
  "flees_near_death",
  "restricted_to_liquid",
  "submerges",
  "never_sleeps",
  "attacks_penetrate",
  "avoid_corridors",
  "carry_item_25"
}

local function negate(mob)
  -- All statuses gone
  mg_oe.negate_effects(mob.object)

  -- only physical bolts can stay
  local bolts = {}
  for _, bolt in ipairs(mob.bolts or {}) do
    for _, good_bolt in ipairs(bolts_cannot_negate) do
      if bolt == good_bolt then
        table.insert(bolts, bolt)
      end
    end
  end

  local behaviors = {}
  for behavior, value in pairs(mob.behaviors or {}) do
    -- bye bye
    if behavior == "dies_if_negated" then
      mg_mobs.hq_die(mob)
    end

    for _, good_behavior in ipairs(behaviors_cannot_negate) do
      if behavior == good_behavior then
        behaviors[behavior] = value
      end
    end
  end

  mob.abilities = {}
  mob.bolts = bolts
  mob.behaviors = behaviors
end

-- Dominate a mob an ally it to a player
local function dominate(mob, player_name, always_succeed)
  local health_percent = math.ceil(mob.hp / mob.max_hp * 100)
  local success_chance = 100

  if not always_succeed and health_percent >= 20 then
    success_chance = 100 - health_percent
  end

  if success_chance > 0 and rand_percent(success_chance) then
    mobkit.clear_queue_high(mob)
    mob.allied_to = player_name
    return true
  end
end

local function can_punch(mob_pos, target_pos, target)
  -- only nodes - no object or liques
  local cast = core.raycast(mob_pos, target_pos, true, false)
  local thing = cast:next()
  while thing do
    if thing.type == "node" then
      local node = core.get_node(thing.intersection_point)
      local def = minetest.registered_nodes[node.name]
      -- A walkable node is a solid node
      if def and def.walkable then
        return false
      end
    end
    if thing.type == "object" and thing.ref == target then
      return true
    end
    thing = cast:next()
  end
  return false
end

local function get_punch_target(self, mob_pos, target_pos)
  -- only nodes - no object or liques
  local cast = core.raycast(mob_pos, target_pos, true, false)
  local thing = cast:next()
  while thing do
    local target = thing.ref
    if thing.type == "node" then
      local node = core.get_node(thing.intersection_point)
      local def = minetest.registered_nodes[node.name]
      -- A walkable node is a solid node
      if def and def.walkable then
        return nil
      end
    end
    if thing.type == "object" then
      local mob = target:get_luaentity()
      if target:get_properties().pointable and
         target ~= self.object and
         ( target:is_player() and
           target:get_hp() > 0
         ) or
         ( mob and
           mob.mg_id ~= nil and
           mob.mg_id ~= self.mg_id and
           mob.hp > 0
         )
       then
        return target
       end
    end
    thing = cast:next()
  end
  return nil
end

-- A walkable node is a solid node that can be walked on
local function is_node_walkable(pos)
  local node = core.get_node(pos)
  local def = core.registered_nodes[node.name]
  return def and def.walkable
end

local function has_los(pos, target_pos)
  local cast = core.raycast(pos, target_pos, false, false)
  local thing = cast:next()
  while thing do
    if thing.type == "node" then
      local node = core.get_node(thing.intersection_point)
      local def = core.registered_nodes[node.name]
      if def and def.walkable then
        return false
      end
    end
    thing = cast:next()
  end
  return true
end

local function find_in_loop(cx, cz, cy, r, needs_ground, number_needed)
  local validate_spots = {}
  local z = cz - r
  while z <= cz + r do
    local x = cx - r
    while x <= cx + r do
      -- check if spot is valid
      local pos_test = {x = x, y=cy, z=z}
      local node = is_node_walkable(pos_test)
      local groundNode = is_node_walkable({x = x, y=cy - 1, z=z})
      local underGroundNode = is_node_walkable({x = x, y=cy - 2, z=z})
      -- if doesn't need ground and node is not walkable, it's valid - return it
      if ((not needs_ground and not node) or
         -- if does need ground, node is not walkable and ground node or underground node is walkable - it's valid
         (needs_ground and not node and (
            groundNode or underGroundNode))) and
          -- also check los to pos as well - no walkable nodes in the way
          -- do this last as it can be expensive
          has_los({x = cx, y=cy, z=cz}, pos_test)
        then
        table.insert(validate_spots, pos_test)
      end

      if #validate_spots >= number_needed then
        return validate_spots
      end

      if z == cz - r or
         z == cz + r
      then
        x = x + 1
      elseif x == cx - r then
        x = cx + r
      else
        print('missed a condition - iterating anyways to avoid infinte loop')
        x = x + 1
      end
    end
    -- iterate
    z = z + 1
  end
  return validate_spots
end

local function find_loc(pos, start_radius, end_radius, needs_ground, number_needed)
  local cx = math.floor(pos.x + 0.5)
  local cz = math.floor(pos.z + 0.5)
  local cy = math.floor(pos.y + 0.5)

  -- todo iterate from start radius to end radius until
  -- number of valid locations is met
  local all_locations = {}
  for r=start_radius, end_radius do
    print('get location in radius ' .. r)
    local sub_locations = find_in_loop(
      cx, cz, cy, r, needs_ground,
      number_needed - #all_locations
    )
    for _,l in ipairs(sub_locations) do
       table.insert(all_locations, l)
    end
    if #all_locations >= number_needed then
      return all_locations
    end
  end

  -- means didnt fill up all the needed spots
  return all_locations
end


return {
  uuid = uuid,
  shuffle_list_in_place = shuffle_list_in_place,
  negate = negate,
  can_punch = can_punch,
  get_punch_target = get_punch_target,
  find_loc = find_loc,
  dominate = dominate
}
