local S = core.get_translator(core.get_current_modname())

function metalworking.make_unrepairable(item_name)
    local item_def = core.registered_items[item_name]
    if item_def then
        local groups = table.copy(item_def.groups)
        groups.not_repaired_by_anvil = 1
        core.override_item(item_name, { groups = groups })
    end
end

function metalworking.register_hammer(name, def)
    local groups = def.groups or {}
    groups.blacksmith_hammer = 1
    if not metalworking.config.hammer_repairable then
        groups.not_repaired_by_anvil = 1
    end
    core.register_tool(name, {
        description = def.description or 'Blacksmithing Hammer',
        _doc_items_longdesc = def._doc_items_longdesc or
            S("A tool for repairing other tools at a blacksmith's metalworking."),
        _doc_items_usagehelp = def._doc_items_usagehelp or S(
            "Use this hammer to strike blows upon an anvil bearing a damaged tool and you can repair it. "
            .. "Also use it to create tool heads and blades on an anvil. "
            .. "It can also be used for smashing stone, but it is not well suited to this task."),
        image = def.image or "anvil_tool_steelhammer.png",
        inventory_image = def.image or "anvil_tool_steelhammer.png",
        tool_capabilities = {
            full_punch_interval = def.tool_capabilities.full_punch_interval or 0.8,
            max_drop_level = def.tool_capabilities.max_drop_level or 1,
            groupcaps = def.tool_capabilities.groupcaps or {
                -- about equal to a stone pick (it's not intended as a tool)
                cracky = { times = { [2] = 2.00, [3] = 1.20 }, uses = 30, maxlevel = 1 },
            },
            damage_groups = def.tool_capabilities.damage_groups or { fleshy = 6 },
        },
        groups = groups,
    })
    metalworking.registered_hammers[name] = def
end

local tmp = {}

core.register_entity("metalworking:item", {
    initial_properties = {
        hp_max = 1,
        visual = "wielditem",
        visual_size = { x = .33, y = .33 },
        collisionbox = { 0, 0, 0, 0, 0, 0 },
        physical = false,
        textures = { "air" },
    },
    on_activate = function(self, staticdata)
        if tmp.nodename ~= nil and tmp.texture ~= nil then
            self.nodename = tmp.nodename
            tmp.nodename = nil
            self.texture = tmp.texture
            tmp.texture = nil
        else
            if staticdata ~= nil and staticdata ~= "" then
                local data = staticdata:split(";")
                if data and data[1] and data[2] then
                    self.nodename = data[1]
                    self.texture = data[2]
                end
            end
        end
        if self.texture ~= nil then
            self.object:set_properties({ textures = { self.texture } })
        end
    end,
    get_staticdata = function(self)
        if self.nodename ~= nil and self.texture ~= nil then
            return self.nodename .. ";" .. self.texture
        end
        return ""
    end,
    on_blast = function()
        return false, false, {}
    end,
})

local function remove_item(pos, node)
    local npos = vector.new(pos.x, pos.y + metalworking.config.item_displacement, pos.z)
    local objs = core.get_objects_inside_radius(npos, .5)
    if objs then
        for _, obj in ipairs(objs) do
            if obj and obj:get_luaentity() and obj:get_luaentity().name == "metalworking:item" then
                obj:remove()
            end
        end
    end
end

local function update_item(pos, node)
    local meta = core.get_meta(pos)
    local inv = meta:get_inventory()
    if not inv:is_empty("input") then
        pos.y = pos.y + metalworking.config.item_displacement
        tmp.nodename = node.name
        tmp.texture = inv:get_stack("input", 1):get_name()
        local e = core.add_entity(pos, "metalworking:item")
        local yaw = math.pi * 2 - node.param2 * math.pi / 2
        e:set_rotation({ x = -1.5708, y = yaw, z = 0 }) -- x is pitch, 1.5708 is 90 degrees.
    end
end


local function has_access(pos, player)
    local name = player:get_player_name()
    local meta = core.get_meta(pos)
    local owner = meta:get_string("owner")
    local shared = meta:get_int("shared") == 1
    if shared or name == owner then
        return true
    else
        return false
    end
end

local hud_info_by_puncher_name = {}

core.register_globalstep(function()
    local now = os.time()

    for puncher_name, hud_info in pairs(hud_info_by_puncher_name) do
        local hud2, hud3, hud_expire_time = table.unpack(hud_info)
        if now > hud_expire_time then
            local puncher = core.get_player_by_name(puncher_name)
            if puncher then
                local hud2_def = puncher:hud_get(hud2)
                if hud2_def and hud2_def.name == "anvil_background" then
                    puncher:hud_remove(hud2)
                end

                local hud3_def = puncher:hud_get(hud3)
                if hud3_def and hud3_def.name == "anvil_foreground" then
                    puncher:hud_remove(hud3)
                end
            end

            hud_info_by_puncher_name[puncher_name] = nil
        end
    end
end)

local function anvil_rightclick(pos, node, clicker, itemstack)
    if not clicker or not itemstack then
        return
    end
    local meta = core.get_meta(pos)
    local name = clicker:get_player_name()
    local owner = meta:get_string("owner")
    local shared = meta:get_int("shared") == 1

    if name ~= owner and not shared then
        return itemstack
    end

    if itemstack:get_count() == 0 then
        local inv = meta:get_inventory()
        if not inv:is_empty("input") then
            local return_stack = inv:get_stack("input", 1)
            inv:set_stack("input", 1, nil)
            local wield_index = clicker:get_wield_index()
            clicker:get_inventory():set_stack("main", wield_index, return_stack)
            if shared then
                meta:set_string("infotext", S("Shared anvil"))
            else
                meta:set_string("infotext", S("@1's anvil", owner))
            end
            remove_item(pos, node)
            return return_stack
        end
    end

    local this_def = core.registered_nodes[node.name]
    if this_def.allow_metadata_inventory_put(pos, "input", 1, itemstack:peek_item(), clicker) > 0 then
        local s = itemstack:take_item()
        local inv = meta:get_inventory()
        inv:add_item("input", s)
        local meta_description = s:get_meta():get_string("description")
        if "" ~= meta_description then
            if shared then
                meta:set_string("infotext", S("Shared anvil"))
            else
                meta:set_string("infotext", S("@1's anvil", owner) .. "\n" .. meta_description)
            end
        end
        meta:set_int("informed", 0)
        update_item(pos, node)
    end
    return itemstack
end

local function anvil_rotate(pos, node, user, mode, new_param2)
    if core.get_modpath("screwdriver") ~= nil and mode == screwdriver.ROTATE_FACE then
        return
    end

    if not core.is_player(user) then
        return
    end

    local player_name = user:get_player_name()
    local wield_list = user:get_wield_list()
    local wield_index = user:get_wield_index()
    local wielded_original = user:get_inventory():get_stack(wield_list, wield_index)

    core.after(0, function()
        local player = core.get_player_by_name(player_name)
        if not player then
            return
        end

        local inv = player:get_inventory()
        local wielded = inv:get_stack(wield_list, wield_index)

        if wielded:get_name() ~= wielded_original:get_name() then
            return
        end

        local node_now = core.get_node(pos)
        if node_now.name ~= 'metalworking:steel_anvil' then
            return
        end

        local tool_after_rightclicking = anvil_rightclick(pos, node_now, player, wielded)
        inv:set_stack(wield_list, wield_index, tool_after_rightclicking)
    end)

    return false
end

function metalworking.register_anvil(name, def)
    local groups = def.groups or {}
    groups.anvil = 1
    core.register_node(name, {
        drawtype = "nodebox",
        description = def.description or S("Anvil"),
        _doc_items_longdesc = def._doc_items_longdesc or
            S("A tool for repairing other tools in conjunction with a blacksmith's hammer."),
        _doc_items_usagehelp = def._doc_items_usagehelp or
            S("Right-click on this anvil with a damaged tool to place the damaged tool upon it. " ..
                "You can then repair the damaged tool by striking it with a blacksmith's hammer. " ..
                "Repeated blows may be necessary to fully repair a badly worn tool. " ..
                "To retrieve the tool either punch or right-click the anvil with an empty hand."),
        tiles = def.tiles or { "default_stone.png" },
        paramtype = "light",
        paramtype2 = "facedir",
        groups = groups,
        sounds = def.sounds or xcompat.sounds.node_sound_metal_defaults(),
        -- the nodebox model comes from realtest
        node_box = {
            type = "fixed",
            fixed = {
                { -0.5,  -0.5, -0.3,  0.5,  -0.4, 0.3 },
                { -0.35, -0.4, -0.25, 0.35, -0.3, 0.25 },
                { -0.3,  -0.3, -0.15, 0.3,  -0.1, 0.15 },
                { -0.35, -0.1, -0.2,  0.35, 0.1,  0.2 },
            },
        },
        selection_box = {
            type = "fixed",
            fixed = {
                { -0.5,  -0.5, -0.3,  0.5,  -0.4, 0.3 },
                { -0.35, -0.4, -0.25, 0.35, -0.3, 0.25 },
                { -0.3,  -0.3, -0.15, 0.3,  -0.1, 0.15 },
                { -0.35, -0.1, -0.2,  0.35, 0.1,  0.2 },
            }
        },

        on_construct = function(pos)
            if def.on_construct then
                def.on_construct(pos)
            end
            local meta = core.get_meta(pos)
            local inv = meta:get_inventory()
            inv:set_size("input", 1)
        end,

        after_place_node = function(pos, placer, itemstack)
            if def.after_place_node then
                def.after_place_node(pos, placer, itemstack)
            end
            local meta = core.get_meta(pos)
            local stackmeta = itemstack:get_meta()
            if stackmeta:get_int("shared") == 1 then
                meta:set_int("shared", 1)
                meta:set_string("infotext", S("Shared anvil"))
            else
                meta:set_string("owner", placer:get_player_name() or "")
                meta:set_string("infotext", S("@1's anvil", placer:get_player_name()))
            end
        end,

        preserve_metadata = function(pos, oldnode, oldmeta, drops)
            if def.preserve_metadata then
                def.preserve_metadata(pos, oldnode, oldmeta, drops)
            end
            if drops[1] and tonumber(oldmeta.shared) == 1 then
                drops[1] = ItemStack(shared_anvil_item)
            end
            return drops
        end,

        can_dig = function(pos, player)
            if def.can_dig then
                def.can_dig(pos, player)
            end
            local meta = core.get_meta(pos)
            local inv = meta:get_inventory()

            if not inv:is_empty("input") then
                return false
            end
            return true
        end,

        allow_metadata_inventory_put = function(pos, listname, index, stack, player)
            if def.allow_metadata_inventory_put then
                def.allow_metadata_inventory_put(pos, listname, index, stack, player)
            end
            local meta = core.get_meta(pos)
            if listname ~= "input" then
                return 0
            end

            if not has_access(pos, player) then
                return 0
            end

            local player_name = player:get_player_name()
            if stack:get_wear() == 0 then
                core.chat_send_player(player_name, S("This anvil is for damaged tools only."))
                return 0
            end

            local stack_name = stack:get_name()
            if core.get_item_group(stack_name, "not_repaired_by_anvil") ~= 0 then
                local item_def = core.registered_items[stack_name]
                core.chat_send_player(player_name, S("@1 cannot be repaired with an anvil.", item_def.description))
                return 0
            end

            if meta:get_inventory():room_for_item("input", stack) then
                return stack:get_count()
            end
            return 0
        end,

        allow_metadata_inventory_take = function(pos, listname, index, stack, player)
            if def.allow_metadata_inventory_take then
                def.allow_metadata_inventory_take(pos, listname, index, stack, player)
            end
            if not has_access(pos, player) then
                return 0
            end

            if listname ~= "input" then
                return 0
            end
            return stack:get_count()
        end,

        on_rotate = anvil_rotate,
        on_rightclick = anvil_rightclick,

        on_punch = function(pos, node, puncher)
            if not pos or not node or not puncher then
                return
            end
            if def.on_punch then
                def.on_punch(pos, node, puncher)
            end

            local wielded = puncher:get_wielded_item()
            local meta = core.get_meta(pos)
            local inv = meta:get_inventory()
            local owner = meta:get_string("owner")
            local shared = meta:get_int("shared") == 1
            local puncher_name = puncher:get_player_name()
            if not shared and owner ~= puncher_name then
                return
            end

            if wielded:get_count() == 0 then
                if not inv:is_empty("input") then
                    local return_stack = inv:get_stack("input", 1)
                    inv:set_stack("input", 1, nil)
                    local wield_index = puncher:get_wield_index()
                    puncher:get_inventory():set_stack("main", wield_index, return_stack)
                    if shared then
                        meta:set_string("infotext", S("Shared anvil"))
                    else
                        meta:set_string("infotext", S("@1's anvil", owner))
                    end
                    remove_item(pos, node)
                end
            end
            local hammers = {}
            local n = 0
            for k,v in pairs(metalworking.registered_hammers) do
              n=n+1
              hammers[n]=k
            end

            -- only punching with the hammer is supposed to work
            if hammers[wielded:get_name()] == nil then
                return
            end
            local input = inv:get_stack("input", 1)

            -- only tools can be repaired
            if not input or input:is_empty() then
                return
            end

            -- 65535 is max damage
            local damage_state = 40 - math.floor(input:get_wear() / 1638)

            local tool_name = input:get_name()

            if input:get_wear() > 0 then
                local hud2, hud3, hud3_def

                if hud_info_by_puncher_name[puncher_name] then
                    hud2, hud3 = table.unpack(hud_info_by_puncher_name[puncher_name])
                    hud3_def = puncher:hud_get(hud3)
                end

                if hud3_def and hud3_def.name == "anvil_foreground" then
                    puncher:hud_change(hud3, "number", damage_state)
                else
                    hud2 = puncher:hud_add({
                        name = "anvil_background",
                        [core.features.hud_def_type_field and "type" or "hud_elem_type"] = "statbar",
                        text = "default_cloud.png^[colorize:#ff0000:256",
                        number = 40,
                        direction = 0, -- left to right
                        position = { x = 0.5, y = 0.65 },
                        alignment = { x = 0, y = 0 },
                        offset = { x = -320, y = 0 },
                        size = { x = 32, y = 32 },
                    })
                    hud3 = puncher:hud_add({
                        name = "anvil_foreground",
                        [core.features.hud_def_type_field and "type" or "hud_elem_type"] = "statbar",
                        text = "default_cloud.png^[colorize:#00ff00:256",
                        number = damage_state,
                        direction = 0, -- left to right
                        position = { x = 0.5, y = 0.65 },
                        alignment = { x = 0, y = 0 },
                        offset = { x = -320, y = 0 },
                        size = { x = 32, y = 32 },
                    })
                end

                hud_info_by_puncher_name[puncher_name] = { hud2, hud3, os.time() + metalworking.config.hud_timeout }
            end

            -- tell the player when the job is done
            if input:get_wear() == 0 then
                -- but only once
                if 0 < meta:get_int("informed") then
                    return
                end
                meta:set_int("informed", 1)
                local tool_desc
                local meta_description = input:get_meta():get_string("description")
                if "" ~= meta_description then
                    tool_desc = meta_description
                elseif core.registered_items[tool_name] and core.registered_items[tool_name].description then
                    tool_desc = core.registered_items[tool_name].description
                else
                    tool_desc = tool_name
                end
                core.chat_send_player(puncher_name, S("Your @1 has been repaired successfully.", tool_desc))
                return
            else
                pos.y = pos.y + metalworking.config.item_displacement
                core.sound_play({ name = "anvil_clang" }, { pos = pos })
                core.add_particlespawner({
                    amount = 10,
                    time = 0.1,
                    minpos = pos,
                    maxpos = pos,
                    minvel = { x = 2, y = 3, z = 2 },
                    maxvel = { x = -2, y = 1, z = -2 },
                    minacc = { x = 0, y = -10, z = 0 },
                    maxacc = { x = 0, y = -10, z = 0 },
                    minexptime = 0.5,
                    maxexptime = 1,
                    minsize = 1,
                    maxsize = 1,
                    collisiondetection = true,
                    vertical = false,
                    texture = "anvil_spark.png",
                })
            end

            -- do the actual repair
            input:add_wear(-5000) -- equals to what technic toolshop does in 5 seconds
            inv:set_stack("input", 1, input)

            -- damage the hammer slightly
            wielded:add_wear(100)
            puncher:set_wielded_item(wielded)
        end,
        is_ground_content = false,

        on_blast = function(pos, intensity)
            if def.on_blast then
                def.on_blast(pos, intensity)
            end
            local drops = {}
            local meta = core.get_meta(pos)
            if meta:get_int("shared") == 1 then
                drops[1] = ItemStack(shared_anvil_item)
            else
                drops[1] = ItemStack(name)
            end
            local inv = meta:get_inventory()
            local input = inv:get_stack("input", 1)
            if not input:is_empty() then
                drops[2] = input:to_string()
            end
            remove_item(pos)
            core.remove_node(pos)

            return drops
        end,
    })
    local shared_anvil_item = ItemStack({
        name = name,
        meta = {
            shared = 1,
            description = def.description and ('Shared ' .. def.description) or S("Shared Anvil")
        }
    })
    core.register_craft({
        output = shared_anvil_item:to_string(),
        type = "shapeless",
        recipe = { name, "default:paper" },
    })
    metalworking.registered_anvils[name] = def
end

-- automatically restore entities lost due to /clearobjects or similar
core.register_lbm({
    name = "metalworking:anvil_item_restoration",
    nodenames = { 'group:anvil' },
    run_at_every_load = true,
    action = function(pos, node, active_object_count, active_object_count_wider)
        local test_pos = { x = pos.x, y = pos.y + metalworking.config.item_displacement, z = pos.z }
        if #core.get_objects_inside_radius(test_pos, 0.5) > 0 then
            return
        end
        update_item(pos, node)
    end
})

-- Transfer the hammer from the old hammer storage slot to the main slot, or else drop it in world
core.register_lbm({
    name = "metalworking:hammer_ejection",
    nodenames = { 'group:anvil' },
    run_at_every_load = false,
    action = function(pos, node)
        local meta = core.get_meta(pos)
        local inv = meta:get_inventory()
        if not inv:is_empty("hammer") then
            local hammer = inv:get_stack("hammer", 1)
            inv:set_stack("hammer", 1, nil)
            inv:set_size("hammer", 0)
            if inv:is_empty("input") then
                inv:set_stack("input", 1, hammer) -- the abm will ensure there's an entity showing the hammer is here
            else
                core.add_item({ x = pos.x, y = pos.y + 1, z = pos.z }, hammer)
            end
        end
    end
})

local fallback_texture = "knapping_stone.png"
local initial_properties = {
	visual = "cube",
    -- mesh = "small_ingot.obj",
	collisionbox = { -0.5 / 8, -0.5 / 8, -0.5 / 8, 0.5 / 8, 0.5 / 8, 0.5 / 8 },
	textures = { fallback_texture, fallback_texture, fallback_texture, fallback_texture, fallback_texture, fallback_texture },
	visual_size = { x = 1 / 8, y = 1 / 8, z = 1 / 8 },
	physical = true,
	static_save = false,
    collide_with_objects = true,
}

core.register_entity('metalworking:stone_piece_go', {
	initial_properties = initial_properties,
	on_punch = function(self, puncher)
		if metalworking.registered_hammers[puncher:get_wielded_item():get_name()] ~= nil and self.complete ~= true then
            -- local props = self:get_properties()
            local textures = table.copy(self.initial_properties.textures)
            textures[0] = textures[1] .. '^[multiply:' .. self.material
            self.textures = textures
            self.complete = true
            self.get_removed()
            -- self.object:remove()
            core.sound_play("thunk", { gain = 0.3, to_player = puncher:get_player_name() }, true)
        end
	end,
})

-- core.register_entity('metalworking:stone_piece', {
-- 	initial_properties = initial_properties,
-- })

local function get_empty_slot(t)
	local i = 1
	while t[i] do
		i = i + 1
	end
	return i
end

local function clear_craft(id, cp_table)
	-- first check if this crafting process even still exists
	if not metalworking.crafting_processes[id] then return end
	-- if cp_table and metalworking.crafting_processes[id] ~= cp_table then return end

	if cp_table and metalworking.crafting_processes[id] == cp_table then
		core.chat_send_player(cp_table.player,
			S("The metalworking process you started at: @1 crumbled away due to time.",
			core.pos_to_string(cp_table.pos)))
	end

	local pos = metalworking.crafting_processes[id].pos
	local bits = core.get_objects_inside_radius(pos, 1)
	for _, v in pairs(bits) do
		if not v:is_player() and v:get_luaentity().id == id then
			v:remove()
		end
	end
	metalworking.crafting_processes[id] = nil
end

local function finish_craft(id)
	local crafting_process = metalworking.crafting_processes[id]
	local pos = crafting_process.pos
	core.add_item(pos, crafting_process.recipe.output)
	for _, func in pairs(metalworking.registered_callbacks[metalworking.crafting_processes[id].recipe.type]) do
		func(crafting_process.recipe, pos, crafting_process.player)
	end
	clear_craft(id)
end

local function place_metalworking_plane(id)
	local crafting_process = metalworking.crafting_processes[id]
	local recipe = crafting_process.recipe
	local node_pos = crafting_process.pos

	local start_pos = vector.add(node_pos, -1 / 16)

	-- instead of matching the in-world pattern to the recipe, we just count up
	-- how many diggable pieces there are and finnish once they are gone
	local to_remove = 0

	for x = 1, #recipe.recipe do
		for z = 1, #recipe.recipe[#recipe.recipe] do -- - 0.89
			local pos = { x = (start_pos.x + (1 / 8) * (x - 1)), y = start_pos.y - 0.83, z = (start_pos.z + (1 / 8) * (z - 1)) - 0.05 }

			-- local texture = recipe.texture and recipe.texture .. "^[sheet:8x8:" .. x - 1 .. "," .. 8 - z or
			-- 	"knapping_stone.png"
            local texture = 'default_steel_block.png' .. "^[sheet:8x8:" .. x - 1 .. "," .. 8 - z
            local texture_top = 'default_steel_block.png' --recipe.texture and recipe.texture .. "^[sheet:8x8:" .. x - 1 .. "," .. 8 - z or
            -- "knapping_stone.png^" ..
            -- core.registered_items[recipe.input].inventory_image --.. '^[transform3)'

			if recipe.recipe[x][z] == 1 then
				to_remove = to_remove + 1
				local objref = core.add_entity(pos, 'metalworking:stone_piece_go')
				objref:set_properties({ textures = { texture, texture, texture, texture, texture, texture } })

				local luaent = objref:get_luaentity()
				-- used during cleanup so recipes in adjacent nodes don't get messed up
				luaent.id = id
                luaent.material = recipe.input
                luaent.get_removed = function()
					to_remove = to_remove - 1
					if to_remove == 0 then
						finish_craft(id)
					end
				end
			-- else
			-- 	to_remove = to_remove + 1
			-- 	local objref = core.add_entity(pos, 'metalworking:stone_piece_go')
			-- 	texture = texture .. "^[multiply:#b0b0b0" -- darken texture to distinguish the pattern
            --     texture_top = texture_top .. "^[multiply:#b0b0b0"
			-- 	objref:set_properties({ textures = { texture_top, texture, texture, texture, texture, texture } })

			-- 	local luaent = objref:get_luaentity()
			-- 	luaent.id = id
			-- 	luaent.material = recipe.input
			-- 	luaent.get_removed = function()
			-- 		to_remove = to_remove - 1
			-- 		if to_remove == 0 then
			-- 			finish_craft(id)
			-- 		end
			-- 	end
			end
		end
	end
	-- so the map doesn't get cluttered, this isn't supposed to be a permanent decoration after all
	core.after(metalworking.config.timeout, clear_craft, id, crafting_process)
end

local function filter_inplace(arr, func)
    -- local new_index = 1
    -- local size_orig = #arr
    local new_arr = table.copy(arr)
    for old_index, v in ipairs(arr) do
        if func(v, old_index) then
            new_arr[old_index] = v
            -- new_index = new_index + 1
        end
    end
    -- for i = new_index, size_orig do new_arr[i] = nil end
    return new_arr
end


local function get_metalworking_formspec(type, itemname, pos)
    local spos = core.pos_to_string(pos)
    local recipes = filter_inplace(metalworking.registered_recipes[itemname], function(v, i) return v.type == type end)
    local formspec = {}
    formspec[#formspec + 1] = "size["
    formspec[#formspec + 1] = #recipes
    formspec[#formspec + 1] = ",1]"

    local x = 0
    for _, v in pairs(recipes) do
        formspec[#formspec + 1] = "item_image_button["
        formspec[#formspec + 1] = x
        formspec[#formspec + 1] = ",0.125;1,1;"
        formspec[#formspec + 1] = v.output
        formspec[#formspec + 1] = ";"
        -- is there a a way to avoid passing the position through the formspec?
        formspec[#formspec + 1] = itemname .. "|" .. v.output .. "|" .. spos
        formspec[#formspec + 1] = ";]"
        x = x + 1
    end

    return table.concat(formspec)
end

local function recipe_item_count(recipe)
    local count = 0
    for x = 1, #recipe.recipe do
		for z = 1, #recipe.recipe[#recipe.recipe] do
            if recipe.recipe[x][z] == 1 then
                count = count + 1
            end
        end
    end
    return count
end

local function handle_input(player, formname, fields)
    if formname ~= "metalworking:choose_recipe" then return end
    if fields.quit then return end

    local player_name = player:get_player_name()

    for k, v in pairs(fields) do
        if v then
            local a = k:find("|")
            local b = k:find("|", a + 1)
            local itemname, output, pos = k:sub(1, a - 1), k:sub(a + 1, b - 1), core.string_to_pos(k:sub(b + 1, -1))
            local recipe
            for _, v in pairs(metalworking.registered_recipes[itemname]) do
                if v.output == output then recipe = v end
            end

            core.close_formspec(player_name, formname)

            local id = get_empty_slot(metalworking.crafting_processes)

            if id > metalworking.config.max_crafts then
                core.chat_send_player(player_name,
                    S("There are too many metalworking processes going on at once."))
                core.chat_send_player(player_name, S("Try again in a few minutes."))
                return true
            end

            metalworking.crafting_processes[id] = {
                pos = pos,
                recipe = recipe,
                player = player_name,
            }

            place_metalworking_plane(id)

            -- remove input item if not creative
            if not core.settings:get_bool("creative_mode") and
                not core.check_player_privs(player_name, { creative = true }) then
                local player_inv = player:get_inventory()
                local main = player_inv:get_list('main')
                local cost = recipe_item_count(recipe)
                local count = 0
                local stacks = {}
                for i, itemstack in ipairs(main) do
                    if itemstack:get_name() == recipe.input then
                        table.insert(stacks, {stack = itemstack, i = i})
                        count = count + itemstack:get_count()
                    end
                end
                if count < cost then return false end
                for _, stack in ipairs(stacks) do
                    local scount = stack.stack:get_count()
                    local remove = scount
                    if scount > cost then
                        remove = cost
                    end
                    stack.stack:take_item(remove)
                    cost = cost - remove
                    player_inv:set_stack('main', stack.i, stack.stack)
                end
                -- local itemstack = player_inv:get_stack('main', player_inv:get_list('main'))
                -- itemstack:take_item(recipe_item_count(recipe))
                -- player:set_wielded_item(itemstack)
            end
        end
    end
    return true
end

core.register_on_player_receive_fields(handle_input)

local function can_forge_weld(itemstack, placer, pointed_thing)
    local placer_inv = placer:get_inventory()
    if not placer:get_player_control().sneak then return false end
    if metalworking.registered_anvils[core.get_node(pointed_thing.under).name] ~= nil then
        if pointed_thing.under.y ~= pointed_thing.above.y - 1 then return false end
        if core.get_node(pointed_thing.above).name ~= "air" then return false end
        local has_hammer = false
        for item, _ in pairs(metalworking.registered_hammers) do
           if placer_inv:contains_item('main', item) then
                has_hammer = true
           end
        end
        if has_hammer then
            -- if placer_inv:contains_item('') then
                return true
            -- end
        else 
            core.chat_send_player(placer:get_player_name(), S('Need a blacksmith hammer to forge weld.'))
        end
    end
    return false
end

local function add_forge_welding(itemname)
	assert(core.registered_items[itemname],
		"Trying to register a forge welding recipe using the nonexistent item: '" .. itemname .. "'")
	local og_on_place = core.registered_items[itemname].on_place
	core.override_item(itemname, {
		description = core.registered_items[itemname].description ..
			core.colorize("#ababab", "\n" .. S("Use on the top of an anvil while sneaking to start forge welding.")),
		on_place = function(itemstack, placer, pointed_thing)
			if can_forge_weld(itemstack, placer, pointed_thing) then
				core.show_formspec(placer:get_player_name(), "metalworking:choose_recipe",
					get_metalworking_formspec('forge_weld', itemname, pointed_thing.above))
			else
				return og_on_place(itemstack, placer, pointed_thing)
			end
		end,
	})
end

function metalworking.register_on_forge_weld_craft(func)
    assert(type(func) == "function", "'register_on_forge_weld_craft' expects a function, got: '" .. type(func) .. "'")
    table.insert(metalworking.registered_callbacks.forge_weld, func)
end

function metalworking.register_craft(recipe)
    -- def expects type, input, output, recipe
    if not metalworking.registered_recipes[recipe.input] then
		metalworking.registered_recipes[recipe.input] = {}
        if recipe.type == 'forge_weld' then
    		add_forge_welding(recipe.input)
        end
	end
	table.insert(metalworking.registered_recipes[recipe.input], recipe)
end