thermometer = {
    celsius = core.settings:get_bool('thermometer.celsius', true),
    temp_nodes = {
        ["snowy"] = { add = -1, rad = 2 },
        ["cools_lava"] = { add = -1, rad = 2 },
        ["puts_out_fire"] = { add = -2, rad = 3 },
        ["water"] = { add = -10, rad = 0 },
        ["torch"] = { add = 5, rad = 5 },
        ["igniter"] = { add = 15, rad = 5 },
        ["lava"] = { add = 22, rad = 20 },
        ['lava_heated'] = { add = 17, rad = 5 },
        ["default:furnace_active"] = { add = 10, rad = 10 },
        ["magma:magma"] = { add = 28, rad = 20 },
        ["magma:obsidian_vent"] = { add = 28, rad = 20 },
        ["magma:basalt_vent"] = { add = 28, rad = 20 },
        ["magma:pumice_vent"] = { add = 28, rad = 20 },
        ["bucket:bucket_lava"] = { add = 22, rad = 20 },
        ["water_bucket"] = { add = -5, rad = 0 },
    },
    board = 'group:sign',
    storage_nodes = {
        'default:chest', 'default:chest_locked',
        'default:chest_open', 'default:chest_locked_open',
    }
}

core.log('info', 'Thermometer is set to use ' .. (thermometer.celsius and 'Celsius' or 'Fahrenheit'))

function thermometer.c2f(c)
    local temp = (c * 9 / 5) + 32
    return math.floor(temp * 10) * 0.1
end

if core.global_exists('tempsurvive') then
    tempsurvive.nodes = thermometer.temp_nodes
    core.clear_recipe({ output = 'tempsurvive:thermometer' })
    core.register_alias_force('tempsurvive:thermometer', 'thermometer:spirit_thermometer')
    tempsurvive.mod = 'tempsurvive'
else
    dofile(core.get_modpath('thermometer') .. '/functions.lua')
    core.register_alias('tempsurvive:thermometer', 'thermometer:spirit_thermometer')
    tempsurvive.mod = 'thermometer'
end
if core.global_exists('default') then -- MTG-compatible
    thermometer.spirit = 'vessels:glass_bottle'
    thermometer.cork = 'vessels:glass_fragments'
    thermometer.red_dye = 'dye:red'
    thermometer.red_dye_texture = 'dye_red.png'
    thermometer.board_texture = 'default_sign_wood.png'
    thermometer.board_tile = 'default_sign_wall_wood.png'

    local signs = { 'default:sign_wall_wood', 'default:sign_wall_steel' }
    for _, sign in ipairs(signs) do
        local def_group = table.copy(core.registered_items[sign].groups)
        def_group.sign = 1
        core.override_item(sign, {
            groups = def_group
        })
    end
end

if core.global_exists('mcl_vars') then -- Mineclone-compatible
    thermometer.spirit = 'mcl_potions:glass_bottle'
    thermometer.cork = 'mcl_nether:quartz'
    thermometer.board_texture = 'mcl_trees_planks_oak.png'
    thermometer.board_tile = 'mcl_trees_planks_oak.png'
end
if core.global_exists('mcl_dyes') then
    thermometer.red_dye = 'mcl_dyes:red'
    thermometer.red_dye_texture = 'dye_red.png'
end

if core.global_exists('Everness') then
    thermometer.cork = 'everness:quartz_crystal'
end

if core.get_modpath('quartz') then
    -- quartz is preferred over everness by bm_quartz_compat
    thermometer.cork = 'quartz:quartz_crystal'
end

if core.global_exists('basic_materials') then
    if core.global_exists('bottles') then
        bottles.register_filled_bottle({
            target = 'basic_materials:paraffin',
            sound = 'default_water_footstep'
        })
        thermometer.spirit = 'bottles:bottle_of_paraffin'
    else
        thermometer.spirit = 'basic_materials:paraffin'
    end
end

if core.global_exists('farming') and farming.mod == 'redo' then
    -- set last because this is preferred
    thermometer.spirit = 'farming:bottle_ethanol'
end

local thermometer_texture = '^([combine:16x16:0,0=' ..
    thermometer.red_dye_texture .. ':0,-2=vessels_glass_bottle_mask.png^[makealpha:0,254,0)'

local texture = thermometer.board_texture .. '^[transform1' .. thermometer_texture

local tile = thermometer.board_tile .. '^[transform1' .. thermometer_texture

function thermometer.getTemp(pos)
    local temp = tempsurvive.get_bio_temperature(pos)
    temp = tempsurvive.get_artificial_temperature(pos, temp)
    temp = math.floor(temp * 10) * 0.1
    local tempStr = thermometer.celsius and (temp .. '°C') or (thermometer.c2f(temp) .. '°F')
    return temp, tempStr
end

function thermometer.updateInvTemp(pos, inventory, listname)
    if listname == nil then
        listname = 'main'
    end
    local temp = thermometer.getTemp(pos)
    if inventory:is_empty('main') == false then
        local list = inventory:get_list(listname)
        local thermometers = {}
        for _, stack in ipairs(list) do
            if stack:get_name() == 'thermometer:spirit_thermometer' then
                table.insert(thermometers, { itemstack = stack, i = _ })
            else
                local name = stack:get_name()
                local add = 0
                if core.get_item_group(name, "tempsurvive_temp_by_meta") > 0 then
                    add = add + stack:get_meta():get_int("temp")
                else
                    add = core.get_item_group(name, "tempsurvive_add")
                end
                temp = temp + (add * stack:get_count())
            end
        end
        local tempStr = thermometer.celsius and (temp .. '°C') or (thermometer.c2f(temp) .. '°F')
        for _, t in ipairs(thermometers) do
            if t ~= nil and t.itemstack ~= nil then
                local meta = t.itemstack:get_meta()
                if meta ~= nil then
                    meta:set_string('temperature', temp)
                    meta:set_string('infotext', tempStr)
                    meta:set_string('description', t.itemstack:get_definition().description .. '\n' .. tempStr)
                    inventory:set_stack(listname, t.i, t.itemstack)
                    core.log('info',
                        'Set thermometer description to ' .. meta:get_string('description') .. '. Should be ' .. tempStr)
                end
            end
        end
    end
    return temp
end

core.register_node('thermometer:spirit_thermometer', {
    description = 'Alcohol Thermometer',
    drawtype = "nodebox",
    tiles = { tile },
    inventory_image = texture,
    wield_image = texture,
    paramtype = "light",
    paramtype2 = "wallmounted",
    sunlight_propagates = true,
    liquids_pointable = true,
    is_ground_content = false,
    walkable = false,
    use_texture_alpha = "opaque",
    node_box = {
        type        = "wallmounted",
        wall_top    = { -0.3125, 0.4375, -0.4375, 0.3125, 0.5, 0.4375 },
        wall_bottom = { -0.3125, -0.5, -0.4375, 0.3125, -0.4375, 0.4375 },
        wall_side   = { -0.5, -0.4375, -0.3125, -0.4375, 0.4375, 0.3125 },
    },
    sounds = xcompat.sounds.node_sound_wood_defaults(),
    groups = { choppy = 2, attached_node = 1, flammable = 2, oddly_breakable_by_hand = 3 },
    on_use = function(itemstack, user, pointed_thing)
        local pos = pointed_thing.above or user:get_pos()
        local temp, tempStr = thermometer.getTemp(pos)
        minetest.chat_send_player(user:get_player_name(), tempStr)
    end,
    on_construct = function(pos)
        local meta = minetest.get_meta(pos)
        local temp, tempStr = thermometer.getTemp(pos)
        meta:set_string('temperature', temp)
        meta:set_string("infotext", tempStr)
        minetest.get_node_timer(pos):start(2)
    end,
    on_timer = function(pos, elapsed)
        local meta = minetest.get_meta(pos)
        local temp, tempStr = thermometer.getTemp(pos)
        meta:set_string('temperature', temp)
        meta:set_string("infotext", tempStr)
        return true
    end,
})

core.register_craft({
    output = 'thermometer:spirit_thermometer',
    type = "shaped",
    recipe = {
        { '',                thermometer.cork,    '' },
        { '',                thermometer.red_dye, '' },
        { thermometer.board, thermometer.spirit,  '' },
    }
})

core.register_on_player_inventory_action(function(player, action, inventory, inventory_info)
    local temp
    if action == 'move' then
        temp = thermometer.updateInvTemp(player:get_pos(), inventory, inventory_info.from_list)
        temp = thermometer.updateInvTemp(player:get_pos(), inventory, inventory_info.to_list)
    else
        temp = thermometer.updateInvTemp(player:get_pos(), inventory, inventory_info.listname)
    end
    player:get_meta():set_int('inv_temp', temp)
end)

core.register_on_mods_loaded(function()
    for _, chest in ipairs(thermometer.storage_nodes) do
        local def = table.copy(core.registered_nodes[chest])
        if core.registered_items[chest] then
            core.override_item(chest, {
                on_rightclick = function(pos, node, clicker)
                    core.log('info', 'Attempting to update temp in on_rightclick')
                    local meta = core.get_meta(pos)
                    local temp = thermometer.updateInvTemp(pos, meta:get_inventory())
                    core.log('action', temp)
                    meta:set_int('inv_temp', temp)
                    local on_rightclick = def.on_rightclick or function (_pos, _node, _clicker) end
                    return on_rightclick(pos, node, clicker)
                end,
                on_metadata_inventory_move = function(pos, from_list, from_index, to_list, to_index, count, player)
                    core.log('info', 'Attempting to update temp in on_metadata_inventory_move')
                    local meta = core.get_meta(pos)
                    local temp = thermometer.updateInvTemp(pos, meta:get_inventory())
                    meta:set_int('inv_temp', temp)
                    local on_move = def.on_metadata_inventory_move or function (_pos, _from_list, _from_index, _to_list, _to_index, _count, _player) end
                    return on_move(pos, from_list, from_index, to_list, to_index, count, player)
                end,
                on_metadata_inventory_take = function(pos, listname, index, stack, player)
                    core.log('info', 'Attempting to update temp in on_metadata_inventory_take')
                    local meta = core.get_meta(pos)
                    local temp = thermometer.updateInvTemp(pos, meta:get_inventory())
                    meta:set_int('inv_temp', temp)
                    local on_take = def.on_metadata_inventory_take or function (_pos, _listname, _index, _stack, _player) end
                    return on_take(pos, listname, index, stack, player)
                end,
                on_metadata_inventory_put = function(pos, listname, index, stack, player)
                    core.log('info', 'Attempting to update temp in on_metadata_inventory_put')
                    local meta = core.get_meta(pos)
                    local temp = thermometer.updateInvTemp(pos, meta:get_inventory())
                    meta:set_int('inv_temp', temp)
                    local on_put = def.on_metadata_inventory_put or function (_pos, _listname, _index, _stack, _player) end
                    return on_put(pos, listname, index, stack, player)
                end,
            })
        end
    end
end)
