-- Minetest mod: bucket (Fork of Minetest mod)
-- See README.md for licensing and other information.

-- Load support for MT game translation.
local S = minetest.get_translator("bucket")

-- bucket item definitions
bucket = {}

bucket.material = "default:steel_ingot"
bucket.empty = "bucket:bucket_empty"
bucket.water = "bucket:bucket_water_uni"
bucket.lava = "bucket:bucket_lava_uni"
bucket.default_material = "steel"
bucket.liquids = {}

-- Bucket aliases - default bucket ist steel
minetest.register_alias("bucket", bucket.empty.."_"..bucket.default_material)
minetest.register_alias("bucket_water", bucket.water.."_"..bucket.default_material)
minetest.register_alias("bucket_lava", bucket.lava.."_"..bucket.default_material)
-- for compatibility witk old bucket mod
minetest.register_alias("bucket:bucket_empty", bucket.empty.."_"..bucket.default_material)
minetest.register_alias("bucket:bucket_water", bucket.water.."_"..bucket.default_material)
minetest.register_alias("bucket:bucket_lava", bucket.lava.."_"..bucket.default_material)

-- define bucket liquids and Liquid_name
dofile(minetest.get_modpath("bucket") .. "/bucket_liquids.lua")

--craft diferent types of bucket
local craft_ingreds = {
	wood = {"group:wood", "default_wood.png"},
	stone = {"group:stone", "default_stone.png"},
	steel = {"default:steel_ingot", "default_steel_block.png"},
	bronze = {"default:bronze_ingot", "default_bronze_block.png"},
	mese = {"default:mese_crystal", "default_mese_block.png"},
	diamond = {"default:diamond", "default_diamond_block.png"},
	gold = {"default:gold_ingot", "default_gold_block.png"}
}

-- according to source (liquid type) assign bucket with "bucket" liquid
-- with this you can override the mod setting
bucket.giving_back={
		-- return bucket for standard water and lava
		-- [Liquid_name.lava.source]=bucket.lava,
		-- [Liquid_name.water.source]=bucket.water,
		-- return bucket for river water
		-- [Liquid_name.water.river_source ]=bucket.water,
		-- return bucket for bucket water and lava
		-- [Liquid_name.lava.bucket_source]=bucket.lava,
		-- [Liquid_name.water.bucket_source]=bucket.water,
 		}

-- if true, the flowing water can be taken in the bucket
-- load settingtypes
local get_flowing_liquid = minetest.settings:get_bool("bucket_get_flowing", true)

-- craft empty bucket
minetest.register_craft({
	output = bucket.empty.." 1",
	recipe = {
		{bucket.material, "", bucket.material},
		{"", bucket.material, ""},
	}
})


for name, mat in pairs(craft_ingreds) do
	minetest.register_craft({
		output = "bucket:bucket_empty".."_"..name,
		recipe = {
			{mat[1], "", mat[1]},
			{"", mat[1], ""},
		}
	})
end

-- for compatibility
minetest.register_alias("bucket:bucket_empty", "bucket:bucket_empty_steel")


local function check_protection(pos, name, text)
	if minetest.is_protected(pos, name) then
		minetest.log("action", (name ~= "" and name or "A mod")
			.. " tried to " .. text
			.. " at protected position "
			.. minetest.pos_to_string(pos)
			.. " with a bucket")
		minetest.record_protection_violation(pos, name)
		return true
	end
	return false
end

-- Register a new liquid
--    source = name of the source node
--    flowing = name of the flowing node
--    itemname = name of the new bucket item (or nil if liquid is not takeable)
--    inventory_image = texture of the new bucket item (ignored if itemname == nil)
--    name = text description of the bucket item
--    groups = (optional) groups of the bucket item, for example {water_bucket = 1}
--    force_renew = (optional) bool. Force the liquid source to renew if it has a
--                  source neighbour, even if defined as 'liquid_renewable = false'.
--                  Needed to avoid creating holes in sloping rivers.
-- This function can be called from any mod (that depends on bucket).

local function register_full_bucket(source, flowing, itemname, inventory_image, name,
		groups, force_renew, giving_back)
	minetest.register_craftitem(itemname, {
		description = name,
		inventory_image = inventory_image,
		stack_max = 1,
		liquids_pointable = true,
		groups = groups,

		on_place = function(itemstack, user, pointed_thing)
			-- Must be pointing to node
			if pointed_thing.type ~= "node" then
				return
			end

			local node = minetest.get_node_or_nil(pointed_thing.under)
			local ndef = node and minetest.registered_nodes[node.name]

			-- Call on_rightclick if the pointed node defines it
			if ndef and ndef.on_rightclick and
					not (user and user:is_player() and
					user:get_player_control().sneak) then
				return ndef.on_rightclick(
					pointed_thing.under,
					node, user,
					itemstack)
			end

			local lpos

			-- Check if pointing to a buildable node
			if ndef and ndef.buildable_to then
				-- buildable; replace the node
				lpos = pointed_thing.under
			else
				-- not buildable to; place the liquid above
				-- check if the node above can be replaced

				lpos = pointed_thing.above
				node = minetest.get_node_or_nil(lpos)
				local above_ndef = node and minetest.registered_nodes[node.name]

				if not above_ndef or not above_ndef.buildable_to then
					-- do not remove the bucket with the liquid
					return itemstack
				end
			end

			if check_protection(lpos, user
					and user:get_player_name()
					or "", "place "..source) then
				return
			end

			minetest.set_node(lpos, {name = source})
			return ItemStack(giving_back)
		end
	})
end

function bucket.register_liquid(source, flowing, itemname, inventory_image, name,
		groups, force_renew)
	bucket.liquids[source] = {
		source = source,
		flowing = flowing,
		itemname = itemname,
		force_renew = force_renew,
	}
	bucket.liquids[flowing] = bucket.liquids[source]

	if itemname ~= nil then
		if type(inventory_image) == "string" then
			register_full_bucket(source, flowing, itemname, inventory_image, name,
			groups, force_renew)
		else
			bucket.liquids[source].itemname = {}
			for mat_name, mat in pairs(inventory_image["materials"]) do
				local Mat_name = mat_name:gsub("^%l", string.upper)
				local bucket_name = S(Mat_name.." "..name)
				local bucket_itemname = itemname.."_"..mat_name
				bucket.liquids[source].itemname[mat_name]=bucket_itemname
				local bucket_inv_image = mat[2].."^[resize:16x16".."^[colorize:#fff:30^bucket_uni.png^"..inventory_image["fill"].."^[noalpha^[makealpha:0,255,0"

				register_full_bucket(source, flowing, bucket_itemname, bucket_inv_image, bucket_name,
				groups, force_renew, bucket.empty.."_"..mat_name)
			end
		end
	end
end

-- Register Empty buckets
for mat_name, mat in pairs(craft_ingreds) do
	-- Capitalise first character
	local Name = mat_name:gsub("^%l", string.upper)
	-- register all empty buckets
	minetest.register_craftitem(bucket.empty.."_"..mat_name, {
		description = S(Name.." ".."Empty Bucket"),
		--inventory_image = "bucket.png",
		-- inventory_image created from bucket_uni.png and defined material texture mat[2] (craft_ingreds)
		inventory_image = mat[2].."^[resize:16x16".."^[colorize:#fff:30^bucket_uni.png^[noalpha^[makealpha:0,255,0",
		groups = {tool = 1},
		liquids_pointable = true,
		on_use = function(itemstack, user, pointed_thing)
			if pointed_thing.type == "object" then
				pointed_thing.ref:punch(user, 1.0, { full_punch_interval=1.0 }, nil)
				return user:get_wielded_item()
			elseif pointed_thing.type ~= "node" then
				-- do nothing if it's neither object nor node
				return
			end
			-- Check if pointing to a liquid source
			local node = minetest.get_node(pointed_thing.under)
			local liquiddef = bucket.liquids[node.name]
			local item_count = user:get_wielded_item():get_count()

			-- node.name == liquiddef.flowing return water even from flowing liquid
				-- get_flowing_liquid - if this is enabled
			if liquiddef ~= nil
			and liquiddef.itemname ~= nil
			and (node.name == liquiddef.source
				or (node.name == liquiddef.flowing and get_flowing_liquid)) then
				if check_protection(pointed_thing.under,
									user:get_player_name(),
									"take ".. node.name) then
									return
				end

				-- find source for the flowing liquid
				if node.name == liquiddef.flowing then
					-- source_dist max. distance from flowing node, if Nil, the max. = 8 is used
					local source_dist = tonumber(minetest.registered_nodes[liquiddef.flowing].liquid_range) or 8
					-- Find source node position
					local source_pos = minetest.find_node_near(pointed_thing.under, source_dist, liquiddef.source) -- position of the source node
					-- If found, then replace with flowing node
					if source_pos ~= nil then
						minetest.set_node(source_pos, {name = liquiddef.flowing})
					else
						-- if source not found, then do nothing
						return
					end
				end


				-- default set to return filled bucket
				local giving_back = bucket.giving_back[liquiddef.source]

				-- giving_back from original bucket
				if giving_back == nil then
					if type(liquiddef.itemname) == "string" then
						giving_back = liquiddef.itemname
					else
						giving_back = liquiddef.itemname[mat_name]
					end
				end

				-- check if holding more than 1 empty bucket
				if item_count > 1 then

					-- if space in inventory add filled bucked, otherwise drop as item
					local inv = user:get_inventory()
					if inv:room_for_item("main", {name=giving_back}) then
						inv:add_item("main", giving_back)
					else
						local pos = user:get_pos()
						pos.y = math.floor(pos.y + 0.5)
						minetest.add_item(pos, giving_back)
					end

					-- set to return empty buckets minus 1
					giving_back = bucket.empty.." "..tostring(item_count-1)

				end

				-- force_renew requires a source neighbour
				local source_neighbor = false
				if liquiddef.force_renew then
					source_neighbor =
						minetest.find_node_near(pointed_thing.under, 1, liquiddef.source)
				end
				if not (source_neighbor and liquiddef.force_renew) then
					minetest.add_node(pointed_thing.under, {name = "air"})
				end

				return ItemStack(giving_back)
			else
				-- non-liquid nodes will have their on_punch triggered
				local node_def = minetest.registered_nodes[node.name]
				if node_def then
					node_def.on_punch(pointed_thing.under, node, user, pointed_thing)
				end
				return user:get_wielded_item()
			end
		end,
	})
end

-- original bucket - to take the original liquit sources
-- uses new bucket uni
bucket.register_liquid(
	Liquid_name.water.source,
	Liquid_name.water.flowing,
	bucket.water,
	{empty = "bucket_uni.png", materials = craft_ingreds, fill = "bucket_uni_water.png"},
	"Water Bucket",
	{tool = 1, water_bucket = 1},
	false
)

-- River water source is 'liquid_renewable = false' to avoid horizontal spread
-- of water sources in sloping rivers that can cause water to overflow
-- riverbanks and cause floods.
-- River water source is instead made renewable by the 'force renew' option
-- used here.

bucket.register_liquid(
	Liquid_name.water.river_source ,
	Liquid_name.water.river_flowing,
	bucket.water,
	{empty = "bucket_uni.png", materials = craft_ingreds, fill = "bucket_uni_water.png"},
	"Water Bucket",
	{tool = 1, water_bucket = 1},
	false
)

bucket.register_liquid(
	Liquid_name.lava.source,
	Liquid_name.lava.flowing,
	bucket.lava,
	{empty = "bucket_uni.png", materials = craft_ingreds, fill = "bucket_uni_lava.png"},
	"Lava Bucket",
	{tool = 1},
	false
)

-- Unified bucket
-- every bucket will be change to this

bucket.register_liquid(
	Liquid_name.water.bucket_source,
	Liquid_name.water.bucket_flowing,
	bucket.water,
	{empty = "bucket_uni.png", materials = craft_ingreds, fill = "bucket_uni_water.png"},
	"Water Bucket",
	{tool = 1, water_bucket = 1},
	false
)

bucket.register_liquid(
	Liquid_name.lava.bucket_source,
	Liquid_name.lava.bucket_flowing,
	bucket.lava,
	{empty = "bucket_uni.png", materials = craft_ingreds, fill = "bucket_uni_lava.png"},
	"Lava Bucket",
	{tool = 1},
	false
)

----

minetest.register_craft({
	type = "fuel",
	recipe = bucket.lava,
	burntime = 60,
	replacements = {{bucket.lava, bucket.empty}},
})

-- Register buckets as dungeon loot
if minetest.global_exists("dungeon_loot") then
	dungeon_loot.register({
		{name = bucket.empty, chance = 0.55},
		-- water in deserts/ice or above ground, lava otherwise
		{name = bucket.water, chance = 0.45,
			types = {"sandstone", "desert", "ice"}},
		{name = bucket.water, chance = 0.45, y = {0, 32768},
			types = {"normal"}},
		{name = bucket.lava, chance = 0.45, y = {-32768, -1},
			types = {"normal"}},
	})
end
