--[[
Copyright 2023 ekl

This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this
file, You can obtain one at https://mozilla.org/MPL/2.0/.
]]
local modName = minetest.get_current_modname()
local modDir = minetest.get_modpath(modName)

_G.cid = {
	backends = {},
	tunes = {},
	---@type {[string]: {name: string?, gain: number?, pitch: number?}}
	instruments = {
		piano = { name = "cid_piano", gain = 1 / 2, pitch = 60 },
		piano2 = { name = "cid_piano_alt", gain = 1 / 4, pitch = 60 },
		guitar = { name = "cid_guitar", gain = 1 / 2, pitch = 60 },
		eguitar = { name = "cid_eguitar", gain = 1 / 8, pitch = 60 },
		recorder = { name = "cid_recorder", gain = 1 / 16, pitch = 67 },
		bagpipe = { name = "cid_bagpipe", gain = 1 / 16, pitch = 65 },
		saxophone = { name = "cid_saxophone", gain = 1 / 4, pitch = 60 },
		clarinet = { name = "cid_clarinet", gain = 1 / 4, pitch = 58 },
		ebass = { name = "cid_ebass", gain = 1 / 2, pitch = 48 },
	},
}
cid.backends.midi = dofile(modDir .. "/backends/midi.lua")
cid.backends.table = dofile(modDir .. "/backends/table.lua")

local midiPlayers = {}
minetest.register_globalstep(function(dtime)
	for playername, midiPlayer in pairs(midiPlayers) do
		objRef = minetest.get_player_by_name(playername)
		if (not objRef) or midiPlayer:step(dtime, objRef) then
			midiPlayers[playername] = nil
		end
	end
end)


local function stopMidi(name)
	local midiPlayer = midiPlayers[name]
	if midiPlayer then
		midiPlayer:destroy()
		midiPlayers[name] = nil
	end
end

minetest.register_chatcommand("tunestop", {
	description = "Stop playing tune",
	params = "<tune name> [instrument name] [speed]",
	func = stopMidi
})

minetest.register_chatcommand("tuneplay", {
	description = "Play a tune",
	params = "<tune id> [instrument name]",
	func = function(name, param)
		local params = param:split(" ")
		local tuneName = params[1]
		local tune = cid.tunes[tuneName]
		if not tune then
			return false, "Unknown tune requested"
		end
		local instrument = cid.instruments.piano
		if params[2] then
			local userInstrument = cid.instruments[params[2]]
			if userInstrument then
				instrument = userInstrument
			else
				return false, "Unknown instrument requested"
			end
		end
		stopMidi(name)
		minetest.chat_send_player(name, ("Now playing: %s"):format(tune.title or tuneName))
		midiPlayers[name] = cid.backends[tune.backend or "midi"].play(tune, instrument)
		return true
	end
})

minetest.register_chatcommand("tunes", {
	description = "List available tunes",
	params = "",
	func = function(name, param)
		local tunes = {}
		for tuneID, tune in pairs(cid.tunes) do
			--Check if the tune has a name
			if tune.title then
				tunes[#tunes + 1] = ("(%s) - %s"):format(tuneID, tune.title)
			else
				tunes[#tunes + 1] = ("(%s)"):format(tuneID)
			end
		end
		table.sort(tunes)
		minetest.chat_send_player(name, [[\
Note: License information can be found with /tuneinfo <tune id>
The following tunes are available:
(tune id) - Description]])
		minetest.chat_send_player(name, table.concat(tunes, "\n"))
		return true
	end
})

minetest.register_chatcommand("tuneinfo", {
	description = "View license information, if provided",
	params = "<tune id>",
	func = function(name, param)
		local tune = cid.tunes[param]
		if not tune then
			return false, "Not found"
		end
		if not tune.license then
			return false, "The specified tune's license is unknown"
		end
		minetest.chat_send_player(name, ("Displaying license information for: %s\n%s"):format(param, tune.license))
		return true
	end
})

minetest.register_chatcommand("instruments", {
	description = "List available instruments",
	params = "",
	func = function(name, param)
		local instruments = {}
		for instrument, _ in pairs(cid.instruments) do
			instruments[#instruments + 1] = instrument
		end
		table.sort(instruments)
		minetest.chat_send_player(name, table.concat(instruments, "\n"))
		return true
	end
})
