--[[
This program is free software: you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 
]]

---@class Leg
---@field upper ObjectRef?
---@field lower ObjectRef?
---@field target Vector
---@field interpolation {from:Vector,completion:number}?
local Leg = {
	stepPressure = 0, ---Amount of pressure to take a step
	onGround = false,
	upper_ent = "giad:leg_upper",
	lower_ent = "giad:leg_lower",
}
Leg.__index = Leg

---Create a new leg with given position
---@param target Vector
---@return Leg
function Leg.new(target)
	return setmetatable({
		target = target,
	}, Leg)
end

function Leg:destroy()
	if self.upper then
		self.upper:remove()
	end
	if self.lower then
		self.lower:remove()
	end
end

---Update the leg
---@param dtime number
---@param attachment Vector
function Leg:update(dtime, attachment)
	self.onGround = self.interpolation == nil
	self:reposition(dtime, attachment)
end

function Leg:step(target)
	self.interpolation = { from = self.target, completion = 0 }
	self.target = target
end

---Update positioning of leg
---@param dtime number
---@param attachment Vector
function Leg:reposition(dtime, attachment)
	local target = self.target
	local interpolation = self.interpolation
	if interpolation then
		local completion = interpolation.completion + dtime * 4
		if completion < 1 then
			interpolation.completion = completion
			target = target * completion + interpolation.from * (1 - completion)
			target.y = target.y + (completion * (1 - completion) * 8)
		else
			self.interpolation = nil
		end
	end
	local upper = self.upper
	local lower = self.lower
	if not (upper and upper:get_pos()) then
		upper = minetest.add_entity(attachment, self.upper_ent)
		self.upper = upper
	end
	if not (lower and lower:get_pos()) then
		lower = minetest.add_entity(attachment, self.lower_ent)
		self.lower = lower
	end


	local kneeOffset = target - attachment
	---@type Vector
	local knee = kneeOffset:offset(0, 3, 0)
	for _ = 1, 3 do
		---@type Vector
		knee = (((knee:normalize() * 3) - kneeOffset):normalize() * 4) + kneeOffset
	end

	upper:set_pos(attachment)
	upper:set_rotation(knee:dir_to_rotation())
	lower:set_pos(target)
	lower:set_rotation((knee - kneeOffset):dir_to_rotation())
end

---@type entity_definition
local Meta = {
	_legSpec = {}, -- Gets filled with a default legSpecs
	_getRider = function(self)
		local child = self.object:get_children()[1]
		if child and child:is_player() then
			return child
		end
	end,
	on_step = function(self, dtime, moveresult)
		local spec = self._legSpec
		local object = self.object
		local rotation = object:get_rotation()
		local pos = self.object:get_pos()
		local velocity = object:get_velocity()

		---@type ObjectRef?
		local rider = self:_getRider()
		local acceleration = velocity * -1
		local wantJump = false
		if rider then
			local controls = rider:get_player_control()
			local x = 0
			local z = 0
			if controls.left then
				x = x - 1
			end
			if controls.right then
				x = x + 1
			end
			if controls.up then
				z = z + 1
			end
			if controls.down then
				z = z - 1
			end
			if controls.jump then
				--Disabled until ground detection is improved
				--wantJump = true
			end
			local lookYaw = rider:get_look_horizontal()
			acceleration = acceleration + (vector.new(x * 5, 0, z * 5):rotate(vector.new(0, lookYaw, 0)))
			local rotationalVelocity = (lookYaw - rotation.y + math.pi) % (math.pi * 2) - math.pi
			rotation = rotation:offset(0, rotationalVelocity * dtime, 0)
			object:set_rotation(rotation)
		end

		if not self.legs then
			---@type Leg[]
			self.legs = {}
			local upper = "giad:leg_upper"
			local lower = "giad:leg_lower"
			for i, data in ipairs(spec) do
				upper = data[3] or upper
				lower = data[4] or lower
				self.legs[i] = Leg.new(pos)
				self.legs[i].upper_ent = upper
				self.legs[i].lower_ent = lower
			end
		end

		---@type Leg?
		local priorityLeg
		local priority = 1
		---@type Vector
		local priorityLegTarget
		local inAir = #self.legs
		local verticalForce = -9.8
		for i, leg in ipairs(self.legs) do
			local attachment = pos + spec[i][1]:rotate(rotation)
			leg:update(dtime, attachment)
			if leg.onGround then
				inAir = inAir - 1

				local idlePos = attachment + spec[i][2]:rotate(rotation) + velocity * 0.75
				local pred = idlePos
				local hit = Raycast(pred:offset(0, 4, 0), pred, false, false):next()
				if hit then
					pred = hit.intersection_point
					verticalForce = verticalForce + idlePos:distance(pred)
				end
				local stepPriority = pred:distance(leg.target)
				if stepPriority > priority then
					priorityLeg       = leg
					priorityLegTarget = pred
					priority          = stepPriority
				end
			end
		end

		if inAir < 1 and wantJump then
			object:set_velocity(velocity:offset(0, 50 - velocity.y, 0))
		end

		if inAir < 3 and priorityLeg then
			priorityLeg:step(priorityLegTarget)
		end

		object:set_acceleration(acceleration:offset(0, verticalForce, 0))
	end,
	on_rightclick = function(self, clicker)
		local rider = self:_getRider()
		if rider then
			if rider == clicker then
				rider:set_detach()
			end
		else
			clicker:set_attach(self.object)
		end
	end,
	on_death = function(self)
		for _, leg in ipairs(self.legs) do
			leg:destroy()
		end
	end,
	on_deactivate = function(self)
		for _, leg in ipairs(self.legs) do
			leg:destroy()
		end
	end
}
Meta.__index = Meta

for i = 1, 4 do
	local angle = vector.new(0, (i + 2) * math.pi / 6, 0)
	Meta._legSpec[i] = { vector.new(1, 0, 1):rotate(angle), vector.new(4, -3, 5):rotate(angle) }
end
for i = 5, 8 do
	local angle = vector.new(0, (i + 6) * math.pi / 7, 0)
	Meta._legSpec[i] = { vector.new(1, 0, 1):rotate(angle), vector.new(4, -3, 5):rotate(angle) }
end

-- Exported stuff
_G.giad = {
	meta = Meta, -- Use this as a metatable for your entity definition, or copy functions from it, but DO NOT MODIFY! You can override legData in your child definition
}

-- Testing entities
minetest.register_entity("giad:leg_upper", {
	initial_properties = {
		visual = "mesh",
		mesh = "giad_leg_upper.obj",
	},
	textures = { "default_wood.png" },
	static_save = false,
})

minetest.register_entity("giad:leg_lower", {
	initial_properties = {
		visual = "mesh",
		mesh = "giad_leg_lower.obj",
	},
	textures = { "default_wood.png" },
	static_save = false,
})
minetest.register_entity("giad:testbed", setmetatable({
	initial_properties = {
		visual = "mesh",
		mesh = "boats_boat.obj",
	},
	textures = { "default_wood.png" },
}, giad.meta))
