--[[
Copyright 2023 ekl

This program is free software: you can redistribute it and/or modify it under the terms of the GNU Lesser General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License along with this program. If not, see <https://www.gnu.org/licenses/>. 
]]

local v = vector.new
local upVector = vector.new(0, 1, 0)

---@class Leg
---@field upperObj ObjectRef?
---@field lowerObj ObjectRef?
---@field target Vector
---@field interpolation {from:Vector,completion:number}?
local Leg = {
	stepping = false,
	onGround = true,
	upper = "giad:leg_upper",
	lower = "giad:leg_lower",
	upperLength = 3,
	lowerLength = 4,
	offset = vector.new(0, 0, 0),
	restPos = vector.new(2, -3, 0),
	force = 1, ---Additional y force, 1 is double
}
Leg.__index = Leg

---Create a new leg with given position
---@param target Vector
---@return Leg
function Leg.new(target)
	return setmetatable({
		target = target,
	}, Leg)
end

function Leg:destroy()
	if self.upperObj then
		self.upperObj:remove()
	end
	if self.lowerObj then
		self.lowerObj:remove()
	end
end

---Update the leg. Note: Attachment will likely be eliminated from future usage
---@param dtime number
---@param parent ObjectRef
---@param attachment Vector
function Leg:update(dtime, parent, attachment)
	if not self.target then
		self.target = attachment
	end
	local up = upVector:rotate(parent:get_rotation())
	self:reposition(dtime, attachment, up)
	self.stepping = self.interpolation ~= nil
end

---Initiate a step
---@param target Vector
function Leg:step(target)
	self.interpolation = { from = self.target, completion = 0 }
	self.stepping = true
	self.target = target
end

---Update positioning of leg
---@param dtime number
---@param attachment Vector
---@param up Vector upwards direction
function Leg:reposition(dtime, attachment, up)
	local target = self.target
	local interpolation = self.interpolation
	if interpolation then
		local completion = interpolation.completion + dtime * 4
		if completion < 1 then
			interpolation.completion = completion
			target = target * completion + interpolation.from * (1 - completion)
			target.y = target.y + (completion * (1 - completion) * 8)
		else
			self.interpolation = nil
			self.onGround = Raycast(self.target, self.target + (up * -.1), false, false):next() ~= nil
		end
	end
	local upper = self.upperObj
	local lower = self.lowerObj
	if not (upper and upper:get_pos()) then
		upper = minetest.add_entity(attachment, self.upper)
		if not upper then
			return
		end
		self.upperObj = upper
	end
	if not (lower and lower:get_pos()) then
		lower = minetest.add_entity(attachment, self.lower)
		if not lower then
			return
		end
		self.lowerObj = lower
	end


	local kneeOffset = target - attachment
	local upperLength = self.upperLength
	local lowerLength = self.lowerLength
	local knee = kneeOffset + up
	for _ = 1, 3 do
		knee = (((knee:normalize() * upperLength) - kneeOffset):normalize() * lowerLength) + kneeOffset
	end

	upper:set_pos(attachment)
	upper:set_rotation(knee:dir_to_rotation())
	lower:set_pos(target)
	lower:set_rotation((knee - kneeOffset):dir_to_rotation())
end

---@class GiadMeta: entity_definition, Luaentity
---@field _legs Leg[] Should be created by _createLegs
local Meta = {
	_createLegs = function(self)
		self._legs = {
			setmetatable({ offset = v(1, -3, 0) }, Leg)
		}
	end,
	_maxStepping = 1,
	_canJump = false,

	_destroyLegs = function(self)
		for _, leg in ipairs(self._legs) do
			leg:destroy()
		end
	end,

	---Returns the entity riding, if it exists
	---@param self GiadMeta
	---@return ObjectRef?
	_getRider = function(self)
		local child = self.object:get_children()[1]
		if child and child:is_player() then
			return child
		end
	end,

	---Get control information
	---@param self GiadMeta
	---@return GiadControl?
	_getControl = function(self)
		local rider = self:_getRider() ---@type ObjectRef?
		if not rider then
			return
		end
		local control = {} ---@type GiadControl
		local controls = rider:get_player_control()
		local x = 0
		local z = 0
		if controls.left then
			x = -3
		end
		if controls.right then
			x = x + 3
		end
		if controls.up then
			z = 3
		end
		if controls.down then
			z = z - 3
		end
		if controls.jump and self._canJump then
			control.jump = true
		end
		local lookYaw = rider:get_look_horizontal()
		local rotation = self.object:get_rotation()
		control.acceleration = vector.new(x, 0, z):rotate(vector.new(0, lookYaw, 0))
		control.yaw_velocity = (lookYaw - rotation.y + math.pi) % (math.pi * 2) - math.pi
		return control
	end,

	---Handles activation
	---@param self GiadMeta
	on_activate = function(self)
		self:_createLegs()
	end,

	---Main driver code that handles movement
	---@param self GiadMeta
	---@param dtime number
	---@param moveresult Vector
	on_step = function(self, dtime, moveresult)
		local object = self.object
		local rotation = object:get_rotation()
		local pos = self.object:get_pos()
		local velocity = object:get_velocity()
		local control = self:_getControl() or {} ---@type GiadControl

		local legControlForce = control.acceleration or v(0, 0, 0)
		local wantJump = control.jump or false
		if control.yaw_velocity then
			rotation.y = rotation.y + control.yaw_velocity * dtime
		end

		local yawForce = 0

		local priorityLeg ---@type Leg?
		local priority = 1
		local priorityLegTarget ---@type Vector

		local steppingCount = #self._legs
		local inAir = #self._legs
		local acceleration = vector.new(0, -9.81, 0)
		local localUp = vector.new(0, 1, 0):rotate(rotation) --- "up" vector with respect to the hull
		for _, leg in ipairs(self._legs) do
			local attachment = pos + leg.offset:rotate(rotation)
			leg:update(dtime, object, attachment)
			if not leg.stepping then
				steppingCount = steppingCount - 1
				local idlePos = attachment + leg.restPos:rotate(rotation) + velocity * 0.5
				if leg.onGround then
					inAir = inAir - 1
					local compression = leg.target - idlePos
					acceleration = acceleration + compression + legControlForce
					acceleration.y = acceleration.y + compression.y * leg.force --Increase force
					--yawForce = yawForce + compression.y * math.sign(spec[i][1].x, 0)
				end
				local pred = idlePos
				local hit = Raycast(pred + (localUp * 4), pred, false, false):next()

				local stepTarget = (hit and hit.intersection_point) or pred
				local stepPriority = stepTarget:distance(leg.target)
				if stepPriority > priority then
					priorityLeg       = leg
					priorityLegTarget = stepTarget
					priority          = stepPriority
				end
			end
		end

		if inAir <= self._maxStepping and wantJump then
			--This is a bit janky, but essentially this just unsticks all of the legs
			for _, leg in ipairs(self._legs) do
				leg.interpolation = nil
				leg.onGround = false
			end
			object:set_velocity(velocity:offset(0, 20 - velocity.y, 0))
		elseif steppingCount < self._maxStepping and priorityLeg then
			priorityLeg:step(priorityLegTarget)
		end

		object:set_acceleration(acceleration)
		rotation.z = rotation.z - yawForce * dtime / 10
		object:set_rotation(rotation)
	end,
	on_rightclick = function(self, clicker)
		local rider = self:_getRider()
		if rider then
			if rider == clicker then
				rider:set_detach()
			end
		else
			clicker:set_attach(self.object)
		end
	end,
	on_death = function(self)
		self:_destroyLegs()
	end,
	on_deactivate = function(self)
		self:_destroyLegs()
	end
}
Meta.__index = Meta


-- Exported stuff
_G.giad = {
	meta = Meta, -- Use this as a metatable for your entity definition, or copy functions from it, but DO NOT MODIFY! You can override legSpec in your child definition
	leg = Leg -- A class for creating legs
}

local modName = minetest.get_current_modname()
local modPath = minetest.get_modpath(modName)
dofile(modPath .. "/test_entities.lua")
