-------------------------------------------------------------------------------
--Default Data Structures
-------------------------------------------------------------------------------

simple_meters = {}
local player_context = {}
local registered_meters = {}

-------------------------------------------------------------------------------
--Settingtypes
-------------------------------------------------------------------------------

local mod_name = core.get_current_modname()

local function get_setting(name, def)

    if type(def) == "boolean" then
        local inp = core.settings:get_bool(mod_name .. "_" .. name, def)
        return inp
    elseif type(def) == "string" or type(def) == "table" then
        local inp = core.settings:get(mod_name .. "_" .. name) or def
        return inp
    else
        local inp = core.settings:get(mod_name .. "_" .. name) or def
        return tonumber(inp)
    end

end

local update_interval = get_setting("update_interval", 0.25)
local bottom_padding = get_setting("hud_bottom_padding", 10)
local hud_padding = get_setting("hud_padding", 0)
local hud_scale = get_setting("hud_scale", 5)

-------------------------------------------------------------------------------
--Internal variables
-------------------------------------------------------------------------------

local default_hud_def = {
    hud_elem_type = "image",
    position  = {x = 0.5, y = 1},
    offset    = {x = 0, y = 0},
    text      = "",
    scale     = {x = hud_scale, y = hud_scale},
    alignment = {x = 1, y = -1},
}

local default_hud_elem_size = 16
local center_offset = 275

-------------------------------------------------------------------------------
--Utilities
-------------------------------------------------------------------------------

local function msg(level, input)
    core.log(level, "[" .. mod_name .. "] " .. input)
end

-------------------------------------------------------------------------------
--Internal functions
-------------------------------------------------------------------------------

local function update_single_meter(p_ref, stack_name, id)

    local img = registered_meters[stack_name].on_update(p_ref)

    if not img then return end

    p_ref:hud_change(id, "text", img)

end

local function update_meters()

    for _,v in pairs(player_context) do

        for k,m in pairs(v.enabled_meters) do

            update_single_meter(v.ref, k, m)

        end

    end
end

local function recalculate_meter_positions(name)

    local sort_table = {}

    for k,v in pairs(player_context[name].enabled_meters) do
        local weight = registered_meters[k].weight
        table.insert(sort_table, { stack = k, id = v, weight = weight })
    end
    
    table.sort(sort_table, function(a, b)
        return a.weight < b.weight
    end)

    for k,v in ipairs(sort_table) do

        local ind = k - 1

        local dir = 1
        local shift = math.floor((ind - (ind % 2))/2)

        if ind % 2 == 0 then dir = -1 end

        local base_offset = center_offset * dir
        local offset = (shift * default_hud_elem_size * hud_scale * dir) +
        (shift * hud_padding * dir)

        player_context[name].ref:hud_change(
            v.id,
            "offset",
            { x = base_offset + offset, y = -bottom_padding })

        player_context[name].ref:hud_change(
            v.id,
            "alignment",
            { x = dir, y = -1 })

    end
end

local function sync_ids(player, name, stack_name)

    if player:get_inventory():contains_item("main", { name = stack_name }) then

        if not player_context[name].enabled_meters[stack_name] then
            local id = player:hud_add(default_hud_def)
            player_context[name].enabled_meters[stack_name] = id
            update_single_meter(player, stack_name, id)
        end
        
    else

        if player_context[name].enabled_meters[stack_name] then
            player:hud_remove(player_context[name].enabled_meters[stack_name])
            player_context[name].enabled_meters[stack_name] = nil
        end

    end

end

local function repopulate_player_context(name)

    for k,v in pairs(registered_meters) do
        sync_ids(player_context[name].ref, name, k)
    end
    
    recalculate_meter_positions(name)
end

core.register_on_player_inventory_action(function(
    player,
    action,
    inventory,
    inventory_info)

    if not player then return end

    local stack_name = ""

    if action == "put" or action == "take" then

        stack_name = inventory_info.stack:get_name()

    elseif action == "move" then
    
        local inv_name = inventory_info.to_list
        local inv_ind = inventory_info.to_index
        
        stack_name = inventory:get_stack(inv_name, inv_ind):get_name()

    else

        return

    end

    local meter_def = registered_meters[stack_name]
    
    if not meter_def then return end

    local name = player:get_player_name()

    sync_ids(player, name, stack_name)

    recalculate_meter_positions(name)
end)

-------------------------------------------------------------------------------
--API
-------------------------------------------------------------------------------

simple_meters.register_meter = function(def)

    if not def.item or not def.weight or not def.on_update then
        msg("error", "Missing meter definition fields!")
        return
    end

    local ret = {
        weight = def.weight,
        on_update = def.on_update,
    }

    registered_meters[def.item] = ret
end

-------------------------------------------------------------------------------
--Registrations
-------------------------------------------------------------------------------

local timer = 0
core.register_globalstep(function(dtime)

    timer = timer + dtime
    if timer < update_interval then return end
    update_meters()
    timer = 0

end)

core.register_on_joinplayer(function(player)
    local name = player:get_player_name()
    player_context[name] = {
        ref = player,
        enabled_meters = {},
    }
    repopulate_player_context(name)
end)

core.register_on_leaveplayer(function(player)
    local name = player:get_player_name()
    player_context[name] = nil
end)
