-------------------------------------------------------------------------------
--Internal Data structures
-------------------------------------------------------------------------------

simple_sprint = {}
local player_context = {}

-------------------------------------------------------------------------------
--Settingtype initialization
-------------------------------------------------------------------------------

local mod_name = core.get_current_modname()

local function get_setting(name, def)

    if type(def) == "boolean" then
        local inp = core.settings:get_bool(mod_name .. "_" .. name, def)
        return inp
    elseif type(def) == "string" or type(def) == "table" then
        local inp = core.settings:get(mod_name .. "_" .. name) or def
        return inp
    else
        local inp = core.settings:get(mod_name .. "_" .. name) or def
        return tonumber(inp)
    end

end

local max_stamina = get_setting("max_stamina", 20)
local speed_multiplier = get_setting("speed_multiplier", 2)
local jump_multiplier = get_setting("jump_multiplier", 1.25)
local step_interval = get_setting("step_interval", 0.15)
local drain_hunger = get_setting("drain_hunger", true)
local drain_hunger_amount = get_setting("hunger_amount", 0.05)
local stamina_drain = get_setting("stamina_drain", 0.3)
local stamina_regen = get_setting("stamina_regen", 0.4)
local stamina_threshold = get_setting("stamina_threshold", 8)
local spawn_particles = get_setting("spawn_particles", true)
local respawn_stamina = get_setting("respawn_stamina", max_stamina/4)
local require_ground = get_setting("require_ground", false)

-------------------------------------------------------------------------------
--Internal Variables
-------------------------------------------------------------------------------

local hudbars_enabled = false

-------------------------------------------------------------------------------
--Utilities
-------------------------------------------------------------------------------

local function msg(level, input)
    core.log(level, "[" .. mod_name .. "] " .. input)
end

-------------------------------------------------------------------------------
--API Functions
-------------------------------------------------------------------------------

simple_sprint.set_stamina = function(name, amount, add)
    if type(name) ~= "string" or type(amount) ~= "number" then
        msg("error", "set_stamina: Wrong input data! Expected string \
        and number, got " .. type(name) .. " and " .. type(amount) .. "!")
        return false
    end

    if not player_context[name] then
        msg("error", "set_stamina: Can't find player " .. name .. "!")
        return false
    end

    local stamina = 0
    if not add and amount < 0 then
        msg("error", "set_stamina: value can't be lower than 0 when \
        setting stamina!")
        return false
    end

    if add then
        stamina = player_context[name].stamina + amount
    else
        stamina = amount
    end

    if stamina > max_stamina then
        stamina = max_stamina
    elseif stamina < 0 then
        stamina = 0
    end

    player_context[name].stamina = stamina
    return player_context[name].stamina
end

simple_sprint.get_stamina = function(name)
    if not player_context[name] then
        msg("error", "get_stamina: Can't find player " .. name .. "!")
        return false
    end

    return player_context[name].stamina
end

-------------------------------------------------------------------------------
--Integrations
-------------------------------------------------------------------------------

if core.get_modpath("hudbars") then
    hb.register_hudbar(
        "stamina",
        0xFFFFFF,
        "Stamina",
        {
            bar = mod_name .. "_bar.png",
            icon = mod_name .. "_icon.png",
            bgicon = mod_name .. "_bg.png"
        },
        0,
        max_stamina,
        false)
    hudbars_enabled = true
end

if not core.get_modpath("hbhunger") and drain_hunger then
    msg("error","hbhunger is not enabled/installed! Hunger drain is disabled.")
    drain_hunger = false
end

-------------------------------------------------------------------------------
--Registrations
-------------------------------------------------------------------------------

core.register_on_joinplayer(function(player)
    local name = player:get_player_name()
    player_context[name] = {
        ref = player,
        stamina = respawn_stamina,
        previous_stamina = 0,
        sprinting = false,
        grounded = false
    }
    if hudbars_enabled then
        hb.init_hudbar(
            player,
            "stamina",
            player_context[name].stamina,
            max_stamina)
    end
end)

core.register_on_leaveplayer(function(player)
    local name = player:get_player_name()
    player_context[name] = nil
end)

core.register_on_respawnplayer(function(player)
    local name = player:get_player_name()
    player_context[name].stamina = respawn_stamina
end)

-------------------------------------------------------------------------------
--Main Globalstep function
-------------------------------------------------------------------------------

local sprint_timer = 0

core.register_globalstep(function(dtime)

    sprint_timer = sprint_timer + dtime
    if sprint_timer < step_interval then return end

    for playername,playerstats in pairs(player_context) do

        local pos = playerstats.ref:get_pos()
        local keys = playerstats.ref:get_player_control()
        local node = core.get_node_or_nil({
            x = pos.x,
            y = pos.y - 0.5,
            z = pos.z})

        if node and core.registered_nodes[node.name] and
        (core.registered_nodes[node.name].walkable or
        core.registered_nodes[node.name].liquidtype ~= "none") then
            playerstats.grounded = true
        else
            playerstats.grounded = false
        end

        if keys.aux1 and keys.up and not keys.left and not keys.right and
        not keys.down and not keys.sneak then
            if ((require_ground and playerstats.grounded) or not require_ground)
            and ((not playerstats.sprinting and
            playerstats.stamina > stamina_threshold) or
            (playerstats.sprinting and playerstats.stamina > 0)) and
            playerstats.ref:get_hp() > 0 then
                if not playerstats.sprinting then
                    playerstats.sprinting = true
                    player_monoids.speed:add_change(
                        playerstats.ref,
                        speed_multiplier,
                        mod_name .. "_sprinting")
                    player_monoids.jump:add_change(
                        playerstats.ref,
                        jump_multiplier,
                        mod_name .. "_jumping")
                end
            else
                playerstats.sprinting = false
                player_monoids.speed:del_change(
                    playerstats.ref,
                 mod_name .. "_sprinting")
                player_monoids.jump:del_change(
                    playerstats.ref,
                    mod_name .. "_jumping")
            end
        else
            if playerstats.sprinting then
                playerstats.sprinting = false
                player_monoids.speed:del_change(
                    playerstats.ref,
                    mod_name .. "_sprinting")
                player_monoids.jump:del_change(
                    playerstats.ref,
                    mod_name .. "_jumping")
            end
        end

        if playerstats.sprinting and playerstats.stamina > 0 then

            playerstats.stamina = playerstats.stamina - stamina_drain
            if playerstats.stamina < 0 then playerstats.stamina = 0 end

            if drain_hunger then
                local hunger = hbhunger.hunger[playername]
                hunger = hunger - drain_hunger_amount
                if hunger < 0 then hunger = 0 end
                hbhunger.hunger[playername] = hunger
                hbhunger.set_hunger_raw(playerstats.ref)
            end

            if spawn_particles then

                local texture = mod_name .. "_fallback.png"
                local glow = 0
                local acceleration = {x = 0, y = -9.8, z = 0}

                if playerstats.grounded and
                core.registered_nodes[node.name] then

                    if core.registered_nodes[node.name].tiles and
                    type(core.registered_nodes[node.name].tiles[1])
                    == "string" then
                        texture = core.registered_nodes[node.name].tiles[1]
                    end

                    if core.registered_nodes[node.name].liquidtype
                    ~= "none" then
                        acceleration = {x = 0, y = 0, z = 0}
                    end

                    glow =
                    core.registered_nodes[node.name].light_source or 0
                end

                core.add_particlespawner({
                    amount = math.random(4, 8),
                    time = 0.05,
                    minpos = {x=-0.35, y=-0.4, z=-0.35},
                    maxpos = {x=0.35, y=-0.4, z=0.35},
                    minvel = {x=-0.25, y=1, z=-0.25},
                    maxvel = {x=0.25, y=3, z=0.25},
                    minacc = acceleration,
                    maxacc = acceleration,
                    minexptime = 1.5,
                    maxexptime = 2.5,
                    minsize = 0.3,
                    maxsize = 1.25,
                    collisiondetection = true,
                    collision_removal = false,
                    attached = playerstats.ref,
                    texture = texture,
                    glow = glow
                })
            end

        elseif playerstats.stamina < max_stamina and
        playerstats.ref:get_hp() > 0 then
            playerstats.stamina = playerstats.stamina + stamina_regen
            if playerstats.stamina > max_stamina then
                playerstats.stamina = max_stamina
            end
        end

        if playerstats.stamina ~= playerstats.previous_stamina then
            playerstats.previous_stamina = playerstats.stamina
            if hudbars_enabled then
                hb.change_hudbar(
                    playerstats.ref,
                    "stamina",
                    playerstats.stamina)
            end
        end

    end
    sprint_timer = 0
end)
