-------------------------------------------------------------------------------
--Default data structures
-------------------------------------------------------------------------------

simple_temperature = {}

local player_context = {}
local temperature_nodes = {}
local override_zones = {}
local liquid_nodes = {}

-------------------------------------------------------------------------------
--Settingtype initialization
-------------------------------------------------------------------------------

local mod_name = core.get_current_modname()

local function get_setting(name, def)

    if type(def) == "boolean" then
        local inp = core.settings:get_bool(mod_name .. "_" .. name, def)
        return inp
    elseif type(def) == "string" or type(def) == "table" then
        local inp = core.settings:get(mod_name .. "_" .. name) or def
        return inp
    else
        local inp = core.settings:get(mod_name .. "_" .. name) or def
        return tonumber(inp)
    end

end

local interval = get_setting("interval", 3)
local low_temp_threshold = get_setting("low_damage", 5)
local high_temp_threshold = get_setting("high_damage", 95)
local biome_temp_y_min = get_setting("biome_temp_y_min", -256)
local biome_temp_y_max = get_setting("biome_temp_y_max", 256)
local height_temp_y_min = get_setting("height_temp_y_min", -16384)
local height_temp_y_max = get_setting("height_temp_y_max", 1024)
local temp_at_max_height = get_setting("temp_at_y_max", -30)
local temp_at_min_height = get_setting("temp_at_y_min", 60)
local shelter_height = get_setting("shelter_height", 12)
local change_multiplier = get_setting("change_multiplier", 0.75)
local damage_per_step = get_setting("damage", 1)
local node_search_dist = get_setting("node_search_dist", 2)
local temp_min = get_setting("min_temp", 0)
local temp_max = get_setting("max_temp", 100)
local default_target_temp = get_setting("default_target_temp", 50)
local default_heat_mult = get_setting("default_heat_mult", 0.5)
local night_temperature_drop = get_setting("night_temperature_drop", -20)
local show_dmg_notification = get_setting("show_dmg_notification", true)
local enable_3d_armor = get_setting("enable_3d_armor", true)
local armor_multiplier = get_setting("armor_multiplier", 1.25)
local armor_protection_cap = get_setting("armor_protection_cap", 95)
local enable_radiant = get_setting("enable_radiant", true)
local enable_overrides = get_setting("enable_overrides", true)
local enable_liquid = get_setting("enable_liquid", true)

-------------------------------------------------------------------------------
--Utilities
-------------------------------------------------------------------------------

local function msg(level, input)
    core.log(level, "[" .. mod_name .. "] " .. input)
end

local function user_message(name, message, color)

    local message = core.colorize(color,
    "<" .. mod_name .. "> " .. message)

    core.chat_send_player(name, message)
end

local function fastpow(a, b)
    local state = a
    for i = 1, b - 1 do
        state = state * a
    end
    return state
end

local function fastabs(a)
    if a >= 0 then
        return a
    else
        return -a
    end
end

local function clamp(min, max, input)
    if input > max then
        return max
    elseif input < min then
        return min
    else
        return input
    end
end

local function distance(pos1, pos2)
    return math.sqrt(
    fastpow(pos1.x - pos2.x, 2) +
    fastpow(pos1.y - pos2.y, 2) +
    fastpow(pos1.z - pos2.z, 2)
    )
end

local function lerp(a, b, factor)
    return a + (b - a) * factor
end

local function inverse_lerp(a, b, val)
    return (val - a) / (b - a)
end

-------------------------------------------------------------------------------
--Internal variables
-------------------------------------------------------------------------------

local hudbars_enabled = false
local armor_enabled = false

local biome_midpoint = lerp(biome_temp_y_min, biome_temp_y_max, 0.5)
local biome_halflength = (biome_temp_y_max - biome_temp_y_min) / 2

local height_midpoint = lerp(height_temp_y_min, height_temp_y_max, 0.5)
local height_halflength = (height_temp_y_max - height_temp_y_min) / 2

local cold_color = "#1c7eff"
local hot_color = "#ff551c"
local freeze_msg = "You're freezing!"
local overheat_msg = "You're overheating!"

-------------------------------------------------------------------------------
--Internal functions
-------------------------------------------------------------------------------

local function notify_player_dmg(name, temp)

    if not show_dmg_notification then return end

    if temp > high_temp_threshold then
        user_message(name, overheat_msg, hot_color)
    elseif temp < low_temp_threshold then
        user_message(name, freeze_msg, cold_color)
    end

end

local function is_sheltered(pos)

    for i = 1,shelter_height do
        local node = core.get_node_or_nil({
            x = pos.x,
            y = pos.y + i,
            z = pos.z
        })
        if node and node.name ~= "air" then
            return true
        end
    end

    return false

end

local function get_heat_multiplier(temp, hum)
    local t = clamp(0, 100, temp)
    local h = clamp(0, 100, hum)
    return 1 - math.sqrt(((((t/temp_max)-0.5)*((h/temp_max)-0.5))+0.25)*2)
end

local function get_time_temperature()

    local time = core.get_timeofday()

    local time_factor = math.sin(time * math.pi)
    return lerp(night_temperature_drop, 0, time_factor)

end

local function get_height_temperature(y)

    local biome_factor =
    1 - math.min(fastabs(biome_midpoint - y)/biome_halflength, 1)

    local height_factor =
    clamp(0, 1, inverse_lerp(height_temp_y_max, height_temp_y_min, y))

    local temp = lerp(
        temp_at_max_height,
        temp_at_min_height,
        height_factor) * (1 - biome_factor)

    local time_temp = get_time_temperature() * biome_factor

    return temp + time_temp
end

local function get_node_liquid_temperature(pos)

    for k,v in pairs(liquid_nodes) do
        if core.find_node_near(pos, 0, k, true) then
            return v
        end
    end

    return nil
end

local function get_node_env_temperature(pos)

    local total_t = 0
    local amount = 0

    local pos1 = {
        x = pos.x + node_search_dist,
        y = pos.y + node_search_dist,
        z = pos.z + node_search_dist
   }

    local pos2 = {
        x = pos.x - node_search_dist,
        y = pos.y - node_search_dist,
        z = pos.z - node_search_dist
   }

    for k,v in pairs(temperature_nodes) do

        local found_nodes = core.find_nodes_in_area(pos1, pos2, k, false)

        for _,node_pos in pairs(found_nodes) do
            amount = amount + 1
            total_t = total_t +
            ((v / math.max(distance(pos, node_pos), 1)) * (1 / amount))
        end

    end

    if amount == 0 then
        return 0
    else
        return total_t
    end
end

local function recalculate_player_armor(player_ref)

    local total_armor = 0
    local worn_armor = armor:get_weared_armor_elements(player_ref)

    if not worn_armor then return end

    for _,v in pairs(worn_armor) do

        local def = core.registered_items[v]

        if def and def.groups.armor_heal then
            total_armor = total_armor + def.groups.armor_heal
        end

    end

    total_armor = clamp(0, armor_protection_cap, total_armor)

    local name = player_ref:get_player_name()

    if player_context[name] then
        local mult = clamp(0, 1, 1 - ((total_armor / 100) * armor_multiplier))
        player_context[name].worn_armor_mult = mult
    end
end

-------------------------------------------------------------------------------
--Mod integrations
-------------------------------------------------------------------------------

if core.get_modpath("hudbars") then
    hb.register_hudbar(
        "temperature",
        0xFFFFFF,
        "Temperature",
        {
            bar = mod_name .. "_bar.png",
            icon = mod_name .. "_icon.png",
            bgicon = mod_name .. "_bg.png"
        },
        temp_min,
        temp_max,
        false)
    hudbars_enabled = true
end

if core.get_modpath("3d_armor") and enable_3d_armor then
    armor_enabled = true
    armor:register_on_update(recalculate_player_armor)
end

-------------------------------------------------------------------------------
--API
-------------------------------------------------------------------------------

simple_temperature.register_liquid_node = function(name, add_temp)

    if not enable_liquid then return end

    if not liquid_nodes[name] then
        liquid_nodes[name] = add_temp
    else
        msg("error", "Node " .. name .. " is already registered!")
    end
end

simple_temperature.register_temperature_node = function(name, add_temp)

    if not enable_radiant then return end

    if not temperature_nodes[name] then
        temperature_nodes[name] = add_temp
    else
        msg("error", "Node " .. name .. " is already registered!")
    end
end

simple_temperature.register_override_zone = function(target_temp,
    heat_multiplier, y_min, y_max)

    if not enable_overrides then return end

    if target_temp > temp_max or target_temp < temp_min or heat_multiplier > 1
    or heat_multiplier < 0 then
        msg("error", "heat values for override zone are out of bounds!")
        return
    end

    if y_min >= y_max then
        msg("error", "y_min can't be higher than y_max for override zone!")
        return
    end

    for _,v in pairs(override_zones) do
        if not (y_min > v.y_max or y_max < v.y_min) then
            msg("error", "override zone limits intersect with another!")
            return
        end
    end

    table.insert(override_zones, {
        target_temp = target_temp,
        heat_multiplier = heat_multiplier,
        y_min = y_min,
        y_max =y_max,
    })
end

simple_temperature.get_stats_at = function(pos)

    local stats = core.get_biome_data(pos)

    local target_heat = default_target_temp
    local heat_multiplier = default_heat_mult

    local overridden = false
    
    for _,v in pairs(override_zones) do
        if pos.y > v.y_min and pos.y < v.y_max then
            overridden = true
            target_heat = v.target_temp
            heat_multiplier = v.heat_multiplier
        end
    end

    if not overridden and pos.y > biome_temp_y_min and pos.y < biome_temp_y_max
    and not is_sheltered(pos) then
        target_heat = stats.heat
        heat_multiplier = get_heat_multiplier(stats.heat, stats.humidity)
    end

    local feet_pos = { x = pos.x, y = pos.y - 0.5, z = pos.z }

    local liquid_node_temp = get_node_liquid_temperature(feet_pos)
    if liquid_node_temp then
        target_heat = target_heat + liquid_node_temp
        heat_multiplier = 1
    end

    if not overridden then
        target_heat = target_heat + get_height_temperature(pos.y)
    end

    target_heat =
        clamp(temp_min, temp_max, target_heat + get_node_env_temperature(pos))

    return {heat = target_heat, heat_multiplier = heat_multiplier}
end

simple_temperature.get_player_stats = function(name)
    if player_context[name] then
        return player_context[name]
    else
        return nil
    end
end

-------------------------------------------------------------------------------
--Registrations
-------------------------------------------------------------------------------

core.register_on_joinplayer(function(player)

    local name = player:get_player_name()
    local stats = simple_temperature.get_stats_at(player:get_pos())

    player_context[name] = {
        ref = player,
        temperature = stats.heat,
        ext_stats = stats,
        worn_armor_mult = 1,
    }

    if armor_enabled then
        recalculate_player_armor(player)
    end

    if hudbars_enabled then
        hb.init_hudbar(player,
        "temperature",
        player_context[name].temperature,
        temp_max - temp_min)
    end
end)

core.register_on_leaveplayer(function(player)
    local name = player:get_player_name()
    player_context[name] = nil
end)

-------------------------------------------------------------------------------
--Globalstep
-------------------------------------------------------------------------------

local function update_players()

    for k,v in pairs(player_context) do

        local stats = simple_temperature.get_stats_at(v.ref:get_pos())

        v.ext_stats = stats

        v.temperature = v.temperature + ((stats.heat - v.temperature) *
        stats.heat_multiplier * change_multiplier * v.worn_armor_mult)

        if v.temperature > high_temp_threshold or
        v.temperature < low_temp_threshold then
            if v.ref:get_hp() > 0 then
                v.ref:set_hp(math.max(0, v.ref:get_hp() - damage_per_step))
            end
            notify_player_dmg(v.ref:get_player_name(), v.temperature)
        end

        if hudbars_enabled then
            hb.change_hudbar(v.ref, "temperature", v.temperature)
        end

    end
end

local timer = 0

core.register_globalstep(function(dtime)
    timer = timer + dtime
    if timer < interval then return end
       update_players()
    timer = 0
end)

-------------------------------------------------------------------------------
--Node registrations
-------------------------------------------------------------------------------

dofile(core.get_modpath(mod_name) .. "/registrations.lua")
