-------------------------------------------------------------------------------
--Default data structures
-------------------------------------------------------------------------------

simple_treasures = {}

local loot_table = {}

-------------------------------------------------------------------------------
--Settingtype initialization
-------------------------------------------------------------------------------

local mod_name = core.get_current_modname()

local function get_setting(name, def)

    if type(def) == "boolean" then
        local inp = core.settings:get_bool(mod_name .. "_" .. name, def)
        return inp
    elseif type(def) == "string" or type(def) == "table" then
        local inp = core.settings:get(mod_name .. "_" .. name) or def
        return inp
    else
        local inp = core.settings:get(mod_name .. "_" .. name) or def
        return tonumber(inp)
    end

end

local disable_spawn = get_setting("disable_global_spawn", false)
local disable_lbm = get_setting("disable_marker_lbm", false)
local global_multiplier = get_setting("global_multiplier", 1.0)
local slot_multiplier = get_setting("slot_multiplier", 1.0)

-------------------------------------------------------------------------------
--Utilities
-------------------------------------------------------------------------------

local function msg(level, input)
    core.log(level, "[" .. mod_name .. "] " .. input)
end


local function fastpow(a, b)
    local state = a
    for i = 1, b - 1 do
        state = state * a
    end
    return state
end

-------------------------------------------------------------------------------
--Internal functions
-------------------------------------------------------------------------------

local function chest_on_construct(pos, description, slots)

    local formspec = {}

    formspec[#formspec + 1] = "size[8,9]"
    formspec[#formspec + 1] = default.gui_bg
    formspec[#formspec + 1] = default.gui_bg_img
    formspec[#formspec + 1] = default.gui_slots
    formspec[#formspec + 1] = "list[current_name;main;0,0;8,5;]"
    formspec[#formspec + 1] = "list[current_player;main;0,5;8,4;]"
    formspec[#formspec + 1] = "listring[]"

    local meta = core.get_meta(pos)

    meta:set_string("formspec", table.concat(formspec))
    meta:set_string("infotext", description)

    local inv = meta:get_inventory()
    inv:set_size("main", slots)

end

local function test_if_empty(pos, player)

    local meta = core.get_meta(pos)
    local inv = meta:get_inventory()

    return inv:is_empty("main")

end

local function marker_lbm(pos, node, chest_name, slots, slot_spawn_chance)

    core.set_node(pos, {
        name = chest_name,
        param2 = core.get_node(pos).param2
    })

    if not loot_table[chest_name] then return end
    
    local world_seed = core.get_mapgen_setting("seed")

    local rng = PcgRandom(pos.x * pos.y * pos.z + world_seed)

    local inv = core.get_inventory({ type = "node", pos = pos })
        
    for i = 1, slots do
            
        if rng:next(0, 100) <= slot_spawn_chance * slot_multiplier then

            local chest_lt_size = #loot_table[chest_name]

            local item_def = loot_table[chest_name][rng:next(1, chest_lt_size)]
            local stack = ItemStack(item_def[1])

            if core.registered_tools[item_def[1]] then
                stack:set_wear(rng:next(1, 65535))
            else
                stack:set_count(rng:next(1, item_def[2]))
            end

            inv:set_stack("main", i, stack)

        end

    end
end

-------------------------------------------------------------------------------
--API
-------------------------------------------------------------------------------

simple_treasures.add_to_loot_table = function(
    chest_name,
    item_string,
    max_stack)

    if not chest_name or not item_string or not max_stack then
        msg("error", "Missing loot parameters!")
        return
    end

    if not loot_table[chest_name] then
        loot_table[chest_name] = {}
    end
    
    table.insert(loot_table[chest_name], {
        item_string,
        max_stack
    })

end

simple_treasures.merge_loot_tables = function(chest_name, add_table)

    if not loot_table[chest_name] then
        loot_table[chest_name] = {}
    end

    for _,v in pairs(add_table) do
        table.insert(loot_table[chest_name], v)
    end
    
end

simple_treasures.register_loot_table = function(chest_name, def)

    loot_table[chest_name] = def

end

simple_treasures.get_loot_table = function(chest_name)

    return loot_table[chest_name]

end

simple_treasures.register_lootchest = function(def)

    if not def.name or not def.description then
        msg("error", "Missing fields in chest definition!")
        return
    end

    local sounds = def.sounds or default.node_sound_wood_defaults()
    local groups = def.groups or { choppy = 2, oddly_breakable_by_hand = 2 }
    local tiles = def.tiles or {
        "default_chest_top.png",
        "default_chest_top.png",
        "default_chest_side.png",
        "default_chest_side.png",
        "default_chest_front.png",
    }

    local rarity = def.spawn_in_rarity or 512
    local fill_ratio = def.spawn_on_rarity or 512
    local y_max = def.y_max or 31000
    local y_min = def.y_min or -31000
    local slot_spawn_chance = def.slot_spawn_chance or 0.25
    local slots = def.slots or 32

    local node_box = def.node_box or {
        type = "fixed",
        fixed = {-0.5, -0.5, -0.5, 0.5, 0.5, 0.5},
    }

    local marker_drawtype = "airlike"
    local marker_groups = { dig_immediate = 2, not_in_creative_inventory = 1 }

    if disable_lbm then
        marker_drawtype = nil
        marker_groups = { dig_immediate = 2 }
    end

    core.register_node(def.name .. "_marker", {
        drawtype = marker_drawtype,
        description = def.description .. " spawn marker",
        tiles = {
            mod_name .. "_marker_top.png",
            mod_name .. "_marker_side.png"
        },
        groups = marker_groups,
        paramtype2 = "facedir",
    })

    core.register_node(def.name, {
        description = def.description,
        drawtype = def.drawtype,
        tiles = tiles,
        node_box = node_box,
        selection_box = node_box,
        groups = groups,
        sounds = sounds,
        paramtype = "light",
        on_construct = function(pos)
            chest_on_construct(pos, def.description, slots)
        end, 
        can_dig = test_if_empty,
    })

    if not disable_spawn and def.spawn_in and global_multiplier > 0 then
        core.register_ore({
            ore_type = "scatter",
            ore = def.name .. "_marker",
            wherein = def.spawn_in,
            clust_scarcity = fastpow(rarity, 3) / global_multiplier,
            clust_num_ores = 1,
            clust_size = 25,
            y_min = y_min,
            y_max = y_max,
        })
    end

    if not disable_spawn and def.spawn_on and global_multiplier > 0 then
        core.register_decoration({
            deco_type = "simple",
            place_on = def.spawn_on,
            sidelen = 16,
            fill_ratio = (1 / fill_ratio) * global_multiplier,
            y_min = y_min,
            y_max = y_max,
            flags = "force_placement, all_floors",
            decoration = def.name .. "_marker",
        })
    end

    if not disable_lbm then
        core.register_lbm({
            label = "Replace " .. def.description,
            name = def.name .. "_marker_replace",
            nodenames = def.name .. "_marker",
            run_at_every_load = true,
            action = function(pos, node)
                marker_lbm(pos, node, def.name, slots, slot_spawn_chance)
            end
            })
    end
end
