local acid_puddle = {
   initial_properties = {
      hp_max = 999,
      physical = false,
      visual_size = {x = 1, y = 0.2},
      textures = {
         "fbrawl_acid_puddle_entity.png", "fbrawl_acid_puddle_entity.png",
         "fbrawl_transparent.png", "fbrawl_transparent.png",
         "fbrawl_transparent.png", "fbrawl_transparent.png"
      },
      initial_sprite_basepos = {x = 0, y = 0},
      pointable = false,
      visual = "cube",
      static_save = false,
   },
   damage = 4.5,
   time_accumulator = 0,
   has_to_shrink = false,
   pl_name = "",
   id = -1,
   pool_id = nil,
   is_queued_for_update = false
}

function acid_puddle:on_activate(staticdata, dtime_s)
   local data = core.deserialize(staticdata)
   local pl_name = data[1]
   local obj = self.object
   local player = core.get_player_by_name(pl_name or "")

   if player then
      self.id = math.random(1, 100000)
      self.pl_name = pl_name
      self.max_size = data[2]
      self.duration = data[3]
      self.fast_grow = data[4] or false
      self.pool_id = data[5] -- 5th param is pool id (string) or nil
      self.spread_speed = self.fast_grow and self.max_size * 3 or self.max_size * 2
      core.after(self.duration, function() self.has_to_shrink = true end)
   else
      obj:remove()
   end
end



function acid_puddle:on_step(dtime)
   local obj = self.object
   local props = obj:get_properties()
   local player = core.get_player_by_name(self.pl_name)

   self.time_accumulator = self.time_accumulator + dtime

   if not player then
      obj:remove()
      return
   end

   if self.time_accumulator >= 0.25 then
      fbrawl.damage_players_near(player, obj:get_pos(), props.visual_size.x / 2.5, 0, nil, function(pl_name)
         pl_name:unlock_skill("fbrawl:hit_by_acid_puddle")
         pl_name:get_skill("fbrawl:hit_by_acid_puddle"):stack(self.pl_name, self.id)
      end)

      if vector.distance(obj:get_pos(), player:get_pos()) < props.visual_size.x / 2.5 then
         self.pl_name:unlock_skill("fbrawl:hit_by_acid_puddle")
         self.pl_name:get_skill("fbrawl:hit_by_acid_puddle"):stack(self.pl_name, self.id)
      end

      self.time_accumulator = self.time_accumulator - 0.25
   end

   -- Only queue for update if not already queued and needs to grow/shrink
   local needs_update = false
   if not self.has_to_shrink then
      if props.visual_size.x < self.max_size then
         needs_update = true
      end
   else
      if props.visual_size.x > 0 then
         needs_update = true
      end
   end

   if needs_update and not self.is_queued_for_update then
      self.is_queued_for_update = true
      if self.pool_id then
         table.insert(fbrawl.puddle_update_queue, {obj = obj, dtime = dtime})
      else
         table.insert(fbrawl.puddle_immediate_update_queue, {obj = obj, dtime = dtime})
      end
   end

   if props.visual_size.x == 0 then
      obj:remove()
      return
   end
end



function acid_puddle:puddle_size_update(dtime)
   local obj = self.object
   if not obj then return end
   local props = obj:get_properties()
   local increase_size_per_step = self.spread_speed * dtime
   local changed = false

   -- grow
   if not self.has_to_shrink then
      local new_x = math.min(props.visual_size.x + increase_size_per_step, self.max_size)
      if new_x ~= props.visual_size.x then
         props.visual_size.x = new_x
         props.visual_size.z = new_x
         props.visual_size.y = 0.2
         changed = true
      end
      -- shrink
   else
      local decrease_size_per_step = self.fast_grow and increase_size_per_step or increase_size_per_step / 3
      local new_x = math.max(props.visual_size.x - decrease_size_per_step, 0)
      if new_x ~= props.visual_size.x then
         props.visual_size.x = new_x
         props.visual_size.z = new_x
         props.visual_size.y = 0.2
         changed = true
      end
   end

   if changed then
      obj:set_properties(props)
   end

   self.is_queued_for_update = false

   if props.visual_size.x == 0 then
      obj:remove()
   end
end

core.register_entity("fantasy_brawl:acid_puddle", acid_puddle)





---
--- PUDDLES SCALING SYSTEM
---

fbrawl.puddle_update_queue = {}           -- Queue for puddles with pool_id {{obj=..., dtime=...}}
fbrawl.puddle_immediate_update_queue = {} -- Queue for puddles without pool_id (immediate updates)

local PUDDLE_UPDATES_PER_STEP = 15

core.register_globalstep(function(dtime)
   local processed_count = 0
   local rate_limited_queue = fbrawl.puddle_update_queue
   local immediate_queue = fbrawl.puddle_immediate_update_queue

   -- Process rate-limited queue (puddles with pool_id)
   while processed_count < PUDDLE_UPDATES_PER_STEP and #rate_limited_queue > 0 do
      local entry = table.remove(rate_limited_queue, 1)
      local obj = entry and entry.obj
      local luaentity = obj and obj:get_luaentity()

      if luaentity and luaentity.puddle_size_update then
         luaentity:puddle_size_update(entry.dtime)
      end

      processed_count = processed_count + 1
   end

   -- Process immediate queue (all puddles without pool_id)
   for i = #immediate_queue, 1, -1 do
      local entry = table.remove(immediate_queue, i)
      local obj = entry and entry.obj
      local luaentity = obj and obj:get_luaentity()
      if luaentity and luaentity.puddle_size_update then
         luaentity:puddle_size_update(entry.dtime)
      end
   end
end)





--
-- SKILL SECTION
--

skills.register_skill("fbrawl:hit_by_acid_puddle", {
   loop_params = {
      duration = 0.3,
      cast_rate = 0.29
   },
   sounds = {
      bgm = {name = "fbrawl_puddle", to_player = true},
   },
   physics = {
      operation = "multiply",
      speed = 0.5
   },

   heal_multiplicator = 0.50,
   damage_per_tick = 0.4,
   puddles = {},
   current_hitter = "",
   max_puddles_overlapping = 3,

   stack = function(self, hitter_name, puddle_id)
      -- calculate heal multiplier
      self.puddles[puddle_id] = true
      local puddles_count = math.min(table.count(self.puddles), self.max_puddles_overlapping)
      self.heal_moltiplicator = puddles_count * 0.50

      -- modifier counter hud text
      local counter_hud = self._hud and self._hud["counter"] or false
      if counter_hud then
         self.player:hud_change(counter_hud, "text", "x" .. puddles_count)
      end

      -- activate skill
      if not self.is_active then
         if self.pl_name == hitter_name then
            self.physics = {
               operation = "multiply",
               speed = 1.3
            }
            self.hud = {
               {
                  name = "heal",
                  [fbrawl.hud_type] = "image",
                  text = "fbrawl_healing_hud.png",
                  scale = {x = 2.4, y = 2.4},
                  position = {x = 0.5, y = 0.73 - fbrawl.get_display_scale_offset(self.pl_name)},
               },
               {
                  name = "counter",
                  [fbrawl.hud_type] = "text",
                  text = "x1",
                  number = 0xFFFFFF,
                  style = 1,
                  position = {x = 0.5, y = 0.73 - fbrawl.get_display_scale_offset(self.pl_name)},
                  offset = {x = 20, y = 0}
               }
            }
         end
         self.current_hitter = hitter_name
         self.pl_name:start_skill("fbrawl:hit_by_acid_puddle", hitter_name)

      -- prolong skill if on a puddle of the same caster
      elseif self.current_hitter == hitter_name then
         if self._stop_job then self._stop_job:cancel() end
         self._stop_job = core.after(self.loop_params.duration, function()
            self:stop()
         end)
      end
   end,

   on_start = function(self, hitter_name)
      self.heal_moltiplicator = 0.50
      self.puddles = {}
   end,

   cast = function(self, hitter_name)
      local hitter = core.get_player_by_name(hitter_name)
      if not hitter then return false end
      fbrawl.hit_player(hitter, self.player, self.damage_per_tick, nil, false)
      if self.pl_name == hitter_name then
         self.player:set_hp(self.player:get_hp() + self.damage_per_tick * self.heal_moltiplicator * 20)
      end
   end,

   on_stop = function(self)
      self.physics = {
         operation = "multiply",
         speed = 0.7
      }
      self.hud = nil
   end
})
