--- 3D matrix data structure with 1-based indexing.
-- Supports sparse arrays when default_value is nil/falsy.

--#region types
---@class Matrix3D
---@field data any[] Flat array of values indexed by (z-1)*size.x*size.y + (y-1)*size.x + x
---@field size vector Dimensions {x, y, z}

--#endregion

---@class matrix3d
matrix3d = {}


---@param self Matrix3D
---@param x integer
---@param y integer
---@param z integer
---@param value any
function matrix3d.set(self, x, y, z, value)
	if x < 1 or x > self.size.x or
		y < 1 or y > self.size.y or
		z < 1 or z > self.size.z then
		error("Out of bounds access: " ..
			x .. "," .. y .. "," .. z .. " vs size " .. core.pos_to_string(self.size))
	end
	local idx = (z - 1) * self.size.x * self.size.y +
		(y - 1) * self.size.x + x
	self.data[idx] = value
end



---@param self Matrix3D
---@param x integer
---@param y integer
---@param z integer
---@return any
function matrix3d.get(self, x, y, z)
	if x < 1 or x > self.size.x or
		y < 1 or y > self.size.y or
		z < 1 or z > self.size.z then
		error("Out of bounds access: " ..
			x .. "," .. y .. "," .. z .. " vs size " .. core.pos_to_string(self.size))
	end
	local idx = (z - 1) * self.size.x * self.size.y +
		(y - 1) * self.size.x + x
	return self.data[idx]
end



---Create a new 3D matrix with given dimensions.
---@param x integer
---@param y integer
---@param z integer
---@param default_value any? Pre-fill value (nil allows sparse arrays)
---@return Matrix3D
function matrix3d.new(x, y, z, default_value)
	local data = {}
	if default_value then
		local total = x * y * z
		for i = 1, total do
			data[i] = default_value
		end
	end

	return {
		data = data,
		size = vector.new(x, y, z)
	}
end



local NEIGHBOR_DIRECTIONS = {
	{x = -1, y = 0,  z = 0}, {x = 1, y = 0, z = 0},
	{x = 0,  y = -1, z = 0}, {x = 0, y = 1, z = 0},
	{x = 0, y = 0, z = -1}, {x = 0, y = 0, z = 1}
}


---Get values of 6-connected neighbors at (x,y,z).
---@param matrix Matrix3D
---@param x integer
---@param y integer
---@param z integer
---@return any[]
function matrix3d.get_neighbors(matrix, x, y, z)
	local neighbors = {}
	for _, dir in ipairs(NEIGHBOR_DIRECTIONS) do
		local nx, ny, nz = x + dir.x, y + dir.y, z + dir.z
		if nx >= 1 and nx <= matrix.size.x and
			ny >= 1 and ny <= matrix.size.y and
			nz >= 1 and nz <= matrix.size.z then
			table.insert(neighbors, matrix3d.get(matrix, nx, ny, nz))
		end
	end
	return neighbors
end



---Iterate over all cells in the matrix.
---@param matrix Matrix3D
---@param f fun(x: integer, y: integer, z: integer)
function matrix3d.iterate(matrix, f)
	for i = 1, matrix.size.x do
		for j = 1, matrix.size.y do
			for k = 1, matrix.size.z do
				f(i, j, k)
			end
		end
	end
end
