--- Octree structural hashing utilities.
-- Used by octcache for deduplication.

map_octree.chunk_hash = {}
local chunk_hash = map_octree.chunk_hash

local ID = octchunk.ID
local CHILDREN = octchunk.CHILDREN
local PARAM2 = octchunk.PARAM2

-- Weak-key table: GC automatically releases when nodes are no longer referenced.
local hash_memoization = setmetatable({}, {__mode = "k"})



-- Precomputed replacement patterns for run-length encoding
local same_hash_replace = {}
for i = 1, 100 do
	same_hash_replace[i] = "r" .. i
end
same_hash_replace[1] = ""



---Generate a structural hash for an octree node.
---Uses run-length encoding for repeated children.
---param1 is intentionally excluded: it varies heavily and would prevent effective dedup,
---and in typical Luanti usage it defaults to 0.
---@param octnode OctNode
---@return string
local function generate_hash(octnode)
	if hash_memoization[octnode] then
		return hash_memoization[octnode]
	end

	local children = octnode[CHILDREN]

	if not children then
		local p2 = octnode[PARAM2] or 0
		local hash
		if p2 == 0 then
			hash = tostring(octnode[ID])
		else
			hash = tostring(octnode[ID]) .. ":" .. tostring(p2)
		end
		hash_memoization[octnode] = hash
		return hash
	end

	local components = {}
	do
		local p2 = octnode[PARAM2] or 0
		if p2 == 0 then
			components[1] = tostring(octnode[ID])
		else
			components[1] = tostring(octnode[ID]) .. ":" .. tostring(p2)
		end
	end
	local last_child_hash = nil
	local repeat_count = 0

	for i = 1, 8 do
		local child = children[i]
		local child_hash

		if child then
			if child[CHILDREN] then
				child_hash = generate_hash(child)
			else
				local p2 = child[PARAM2] or 0
				if p2 == 0 then
					child_hash = tostring(child[ID])
				else
					child_hash = tostring(child[ID]) .. ":" .. tostring(p2)
				end
			end
		else
			child_hash = ""
		end

		if child_hash == last_child_hash then
			repeat_count = repeat_count + 1
		else
			if last_child_hash then
				components[#components + 1] = same_hash_replace[repeat_count] .. last_child_hash
			end
			last_child_hash = child_hash
			repeat_count = 1
		end
	end
	if last_child_hash then
		components[#components + 1] = same_hash_replace[repeat_count] .. last_child_hash
	end

	local hash = table.concat(components, "|")
	hash_memoization[octnode] = hash
	return hash
end


---Generate a structural hash for an octree node.
---Wrapper uses the local recursive implementation for speed.
---@param octnode OctNode
---@return string
function chunk_hash.generate(octnode)
	return generate_hash(octnode)
end



---Return memoized hash count (debugging only).
---@return integer
function chunk_hash.get_memoized_count()
	local memo_count = 0
	for _ in pairs(hash_memoization) do
		memo_count = memo_count + 1
	end
	return memo_count
end



---Clear memoized hashes (used when discarding a map).
function chunk_hash.clear_memoization()
	for k in pairs(hash_memoization) do
		hash_memoization[k] = nil
	end
end
