local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

---@diagnostic disable: undefined-field

-- Async tests cannot block - results are logged via core.after callbacks
local async_results = {}


local function pos(x, y, z)
    return {x = x, y = y, z = z}
end


local function lcg(seed)
    local state = seed
    return function()
        state = (1103515245 * state + 12345) % 2147483648
        return state
    end
end


local function rand_int(rng, min, max)
    return min + (rng() % (max - min + 1))
end


local function fill_region(area, data, param2_data, pmin, pmax, cid, p2)
    for x = pmin.x, pmax.x do
        for y = pmin.y, pmax.y do
            for z = pmin.z, pmax.z do
                local idx = area:index(x, y, z)
                data[idx] = cid
                if param2_data then
                    param2_data[idx] = p2
                end
            end
        end
    end
end


local function apply_expected(expected, expected_p2, area, pos1, pos2, cid, p2val)
    for x = pos1.x, pos2.x do
        for y = pos1.y, pos2.y do
            for z = pos1.z, pos2.z do
                local idx = area:index(x, y, z)
                expected[idx] = cid
                if expected_p2 then
                    expected_p2[idx] = p2val
                end
            end
        end
    end
end


local function run_voxel_op(op, done)
    core.load_area(op.p1, op.p2)
    local manip = core.get_voxel_manip()
    local e1, e2 = manip:read_from_map(op.p1, op.p2)
    local area = VoxelArea(e1, e2)
    local data = {}
    local param2_data = {}
    manip:get_data(data)
    manip:get_param2_data(param2_data)

    fill_region(area, data, param2_data, op.p1, op.p2, op.cid, op.param2)
    manip:set_data(data)
    manip:set_param2_data(param2_data)
    manip:write_to_map(false)
    manip:close()
    if done then done() end
end


local function run_octree_op(op, done)
    local m = map_octree.new_octree_manip()
    m:read_from_map(op.p1, op.p2)
    for x = op.p1.x, op.p2.x do
        for y = op.p1.y, op.p2.y do
            for z = op.p1.z, op.p2.z do
                m:set_node_at(x, y, z, op.name, op.param2)
            end
        end
    end
    assert(m:write_to_map(), "write_to_map failed")
    if done then done() end
end


local function verify_region(pmin, pmax, expected, expected_p2, fail)
    core.load_area(pmin, pmax)
    local manip = core.get_voxel_manip()
    local e1, e2 = manip:read_from_map(pmin, pmax)
    local area = VoxelArea(e1, e2)
    local data = {}
    local param2_data = {}
    manip:get_data(data)
    manip:get_param2_data(param2_data)
    manip:close()

    for x = pmin.x, pmax.x do
        for y = pmin.y, pmax.y do
            for z = pmin.z, pmax.z do
                local idx = area:index(x, y, z)
                if data[idx] ~= expected[idx] then
                    local got_name = core.get_name_from_content_id(data[idx])
                    local exp_name = core.get_name_from_content_id(expected[idx])
                    fail(string.format("content mismatch at %d,%d,%d: got=%s expected=%s", x, y, z, got_name, exp_name))
                    return
                end
                if expected_p2 and param2_data then
                    local got_p2 = param2_data[idx] or 0
                    local exp_p2 = expected_p2[idx] or 0
                    if got_p2 ~= exp_p2 then
                        fail(string.format("param2 mismatch at %d,%d,%d: got=%d expected=%d", x, y, z, got_p2, exp_p2))
                        return
                    end
                end
            end
        end
    end
end


map_octree.tests.register("concurrent ops: overlapping mixed manips", function(ctx)
    assert(ctx and ctx.player, "ctx.player required")
    if map_octree.tests.async_pending() > 0 then
        core.log("warning", "[test] skipping concurrent overlap: async pending=" .. tostring(map_octree.tests.async_pending()))
        return
    end

    local base = vector.add(octchunk.snap_to_center(testutil.get_test_region(ctx.player)), pos(0, 0, 8192))
    local size = octchunk.SIZE
    local half = size / 2
    local pmin = vector.subtract(base, pos(half, half, half))
    local pmax = vector.add(base, pos(half - 1, half - 1, half - 1))

    local nodes = {
        "default:stone",
        "default:dirt",
        "default:sand",
        "default:cobble",
    }
    local cids = {}
    for i = 1, #nodes do
        cids[i] = core.get_content_id(nodes[i])
        assert(type(cids[i]) == "number", "missing node: " .. nodes[i])
    end
    local cid_air = core.get_content_id("air")
    assert(type(cid_air) == "number", "missing node: air")

    local test_id = "concurrent_overlap_mixed_" .. os.time()
    async_results[test_id] = {status = "pending", start = core.get_us_time()}
    map_octree.tests.async_begin(test_id)

    local r1 = {p1 = vector.add(pmin, pos(0, 0, 0)), p2 = vector.add(pmin, pos(7, 7, 7))}
    local r2 = {p1 = vector.add(pmin, pos(4, 4, 0)), p2 = vector.add(pmin, pos(11, 11, 7))}
    local r3 = {p1 = vector.add(pmin, pos(2, 2, 2)), p2 = vector.add(pmin, pos(9, 9, 9))}
    local r4 = {p1 = vector.add(pmin, pos(6, 0, 4)), p2 = vector.add(pmin, pos(13, 7, 11))}

    local ops = {
        {kind = "voxel", name = nodes[1], cid = cids[1], param2 = 11, delay = 0.05, p1 = r1.p1, p2 = r1.p2},
        {kind = "oct",   name = nodes[2], cid = cids[2], param2 = 22, delay = 0.10, p1 = r2.p1, p2 = r2.p2},
        {kind = "voxel", name = nodes[3], cid = cids[3], param2 = 33, delay = 0.15, p1 = r3.p1, p2 = r3.p2},
        {kind = "oct",   name = nodes[4], cid = cids[4], param2 = 44, delay = 0.20, p1 = r4.p1, p2 = r4.p2},
    }

    core.log("action", string.format("[test] %s: bounds=%s..%s", test_id, core.pos_to_string(pmin), core.pos_to_string(pmax)))

    testutil.with_voxel_region_async(pmin, pmax, function(manip, area, data, param2_data, restore)
        fill_region(area, data, param2_data, pmin, pmax, cid_air, 0)
        manip:set_data(data)
        manip:set_param2_data(param2_data)
        manip:write_to_map(false)
        manip:close()

        local expected = {}
        local expected_p2 = nil
        for i = 1, #data do
            expected[i] = cid_air
        end
        if param2_data then
            expected_p2 = {}
            for i = 1, #data do
                expected_p2[i] = 0
            end
        end

        local pending = #ops
        local function fail(msg)
            local res = async_results[test_id]
            res.status = "FAIL"
            res.error = tostring(msg)
            core.log("error", "[test] " .. test_id .. ": " .. res.error)
            if map_octree.tests.async_end then
                map_octree.tests.async_end(test_id)
            end
            restore()
        end
        local function pass()
            local res = async_results[test_id]
            res.status = "PASS"
            local elapsed_ms = (core.get_us_time() - res.start) / 1000
            core.log("action", string.format("[test] %s: PASS - %.1fms", test_id, elapsed_ms))
            if map_octree.tests.async_end then
                map_octree.tests.async_end(test_id)
            end
            restore()
        end

        local function on_done(op)
            apply_expected(expected, expected_p2, area, op.p1, op.p2, op.cid, op.param2)
            pending = pending - 1
            if pending == 0 then
                verify_region(pmin, pmax, expected, expected_p2, fail)
                if async_results[test_id].status ~= "FAIL" then
                    pass()
                end
            end
        end

        for _, op in ipairs(ops) do
            core.after(op.delay, function()
                if op.kind == "voxel" then
                    run_voxel_op(op, function() on_done(op) end)
                else
                    run_octree_op(op, function() on_done(op) end)
                end
            end)
        end
    end)

    core.log("action", "[test] " .. test_id .. ": async operations started (result will be logged)")
end)


map_octree.tests.register("concurrent ops: fuzzy overlaps", function(ctx)
    assert(ctx and ctx.player, "ctx.player required")
    if map_octree.tests.async_pending and map_octree.tests.async_pending() > 0 then
        core.log("warning", "[test] skipping concurrent fuzzy: async pending=" .. tostring(map_octree.tests.async_pending()))
        return
    end

    local base = vector.add(octchunk.snap_to_center(testutil.get_test_region(ctx.player)), pos(0, 0, 12288))
    local size = octchunk.SIZE
    local half = size / 2
    local pmin = vector.subtract(base, pos(half, half, half))
    local pmax = vector.add(base, pos(half - 1, half - 1, half - 1))

    local nodes = {
        "default:stone",
        "default:dirt",
        "default:sand",
        "default:cobble",
        "default:brick",
    }
    local cids = {}
    for i = 1, #nodes do
        cids[i] = core.get_content_id(nodes[i])
        assert(type(cids[i]) == "number", "missing node: " .. nodes[i])
    end
    local cid_air = core.get_content_id("air")
    assert(type(cid_air) == "number", "missing node: air")

    local test_id = "concurrent_fuzzy_" .. os.time()
    async_results[test_id] = {status = "pending", start = core.get_us_time()}
    map_octree.tests.async_begin(test_id)

    local seed_start = 1337
    local seed_step = 97
    local seed_count = 5
    local delay_max_ms = 500
    local op_count = 64

    testutil.with_voxel_region_async(pmin, pmax, function(_, area, data, param2_data, restore)
        local finished = false
        local function fail(msg)
            if finished then
                return
            end
            finished = true
            local res = async_results[test_id]
            res.status = "FAIL"
            res.error = tostring(msg)
            core.log("error", "[test] " .. test_id .. ": " .. res.error)
            if map_octree.tests.async_end then
                map_octree.tests.async_end(test_id)
            end
            restore()
        end
        local function pass()
            if finished then
                return
            end
            finished = true
            local res = async_results[test_id]
            res.status = "PASS"
            local elapsed_ms = (core.get_us_time() - res.start) / 1000
            core.log("action", string.format("[test] %s: PASS - %.1fms", test_id, elapsed_ms))
            if map_octree.tests.async_end then
                map_octree.tests.async_end(test_id)
            end
            restore()
        end

        local function reset_region()
            core.load_area(pmin, pmax)
            local manip = core.get_voxel_manip()
            local e1, e2 = manip:read_from_map(pmin, pmax)
            local area2 = VoxelArea(e1, e2)
            local data2 = {}
            local param2_data2 = {}
            manip:get_data(data2)
            manip:get_param2_data(param2_data2)
            fill_region(area2, data2, param2_data2, pmin, pmax, cid_air, 0)
            manip:set_data(data2)
            manip:set_param2_data(param2_data2)
            manip:write_to_map(false)
            manip:close()
        end

        local function run_seed(seed_idx)
            if seed_idx > seed_count then
                pass()
                return
            end

            reset_region()

            local seed = seed_start + (seed_idx - 1) * seed_step
            local rng = lcg(seed)
            local ops = {}
            for i = 1, op_count do
                local w = rand_int(rng, 2, 6)
                local h = rand_int(rng, 2, 6)
                local d = rand_int(rng, 2, 6)
                local ox = rand_int(rng, 0, size - w)
                local oy = rand_int(rng, 0, size - h)
                local oz = rand_int(rng, 0, size - d)
                local p1 = vector.add(pmin, pos(ox, oy, oz))
                local p2 = vector.add(pmin, pos(ox + w - 1, oy + h - 1, oz + d - 1))
                local node_idx = rand_int(rng, 1, #nodes)
                local op_kind = (rand_int(rng, 1, 2) == 1) and "voxel" or "oct"
                local delay = rand_int(rng, 0, delay_max_ms) / 1000
                ops[#ops + 1] = {
                    kind = op_kind,
                    name = nodes[node_idx],
                    cid = cids[node_idx],
                    param2 = (i * 7) % 256,
                    delay = delay,
                    p1 = p1,
                    p2 = p2,
                }
            end

            core.log("action", string.format("[test] %s: seed=%d (%d/%d) bounds=%s..%s ops=%d delay_max_ms=%d",
                test_id, seed, seed_idx, seed_count, core.pos_to_string(pmin), core.pos_to_string(pmax), op_count, delay_max_ms))

            local expected = {}
            local expected_p2 = nil
            for i = 1, #data do
                expected[i] = cid_air
            end
            if param2_data then
                expected_p2 = {}
                for i = 1, #data do
                    expected_p2[i] = 0
                end
            end

            local pending = #ops
            local function on_done(op)
                apply_expected(expected, expected_p2, area, op.p1, op.p2, op.cid, op.param2)
                pending = pending - 1
                if pending == 0 then
                    verify_region(pmin, pmax, expected, expected_p2, fail)
                    if async_results[test_id].status ~= "FAIL" then
                        core.after(0, function() run_seed(seed_idx + 1) end)
                    end
                end
            end

            for _, op in ipairs(ops) do
                core.after(op.delay, function()
                    if op.kind == "voxel" then
                        run_voxel_op(op, function() on_done(op) end)
                    else
                        run_octree_op(op, function() on_done(op) end)
                    end
                end)
            end
        end

        run_seed(1)
    end)

    core.log("action", "[test] " .. test_id .. ": async operations started (result will be logged)")
end)
