--- Restore tracking tests.

local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

local function pick_different_node(map, pos)
	local name = map:get_node_at(pos.x, pos.y, pos.z)
	if name == "air" then
		return "mapgen_stone"
	end
	return "air"
end


map_octree.tests.register("restore_tracking:enable_disable", function(ctx)
	local minp = vector.new(0, 0, 0)
	local maxp = vector.new(31, 31, 31)
	testutil.with_voxel_region(minp, maxp, function()
		local map = octmap.new(minp, maxp, {store_chunk_blobs = true, cache_mb = 8})
		local src_min = map:get_emerged_area()
		map_octree.place(map, src_min)

		map:enable_tracking()
		assert(map._tracker_id, "map should have _tracker_id after enable_tracking")

		local status = map:get_tracking_status()
		assert(status, "get_tracking_status should return table")
		assert(status.pending == 0, "no pending initially")
		assert(status.dirty == 0, "no dirty initially")

		map:disable_tracking()

		status = map:get_tracking_status()
		assert(status == nil, "status should be nil after disable")
	end)
end)


map_octree.tests.register("restore_tracking:dirty_detection", function(ctx)
	local minp = vector.new(0, 0, 0)
	local maxp = vector.new(31, 31, 31)
	testutil.with_voxel_region(minp, maxp, function()
		local map = octmap.new(minp, maxp, {store_chunk_blobs = true, cache_mb = 8})
		local src_min = map:get_emerged_area()
		map_octree.place(map, src_min)
		map:enable_tracking()

		map:_enqueue_chunk_for_verification(1, 1, 1)
		map:_verify_chunk_now(1, 1, 1)

		local status = map:get_tracking_status()
		assert(status and status.dirty == 0, "chunk should not be dirty before modification")

		core.set_node(minp, {name = pick_different_node(map, minp)})

		map:_enqueue_chunk_for_verification(1, 1, 1)
		map:_verify_chunk_now(1, 1, 1)

		status = map:get_tracking_status()
		assert(status and status.dirty == 1, "chunk should be dirty after modification")

		map:disable_tracking()
	end)
end)


map_octree.tests.register("restore_tracking:restore_dirty_sync", function(ctx)
	local minp = vector.new(0, 0, 0)
	local maxp = vector.new(31, 31, 31)
	testutil.with_voxel_region(minp, maxp, function()
		local map = octmap.new(minp, maxp, {store_chunk_blobs = true, cache_mb = 8})
		local src_min = map:get_emerged_area()
		map_octree.place(map, src_min)
		map:enable_tracking()

		local test_pos = vector.new(minp.x + 5, minp.y + 5, minp.z + 5)
		core.set_node(test_pos, {name = pick_different_node(map, test_pos)})

		map:_verify_chunk_now(1, 1, 1)

		local status = map:get_tracking_status()
		assert(status and status.dirty >= 1, "should have dirty chunks")

		local ok, err = map:restore()
		assert(ok, "restore should succeed: " .. tostring(err))

		status = map:get_tracking_status()
		assert(status and status.dirty == 0, "no dirty after restore")

		local node = core.get_node(test_pos)
		assert(node.name == "air", "node should be restored to air, got: " .. node.name)

		map:disable_tracking()
	end)
end)


map_octree.tests.register("restore_tracking:schedule_restore_async", function(ctx)
	if map_octree.tests.async_should_skip and map_octree.tests.async_should_skip("restore_tracking_async") then
		return
	end
	local minp = vector.new(0, 0, 0)
	local maxp = vector.new(31, 31, 31)
	testutil.with_voxel_region(minp, maxp, function()
		local map = octmap.new(minp, maxp, {store_chunk_blobs = true, cache_mb = 8})
		local src_min = map:get_emerged_area()
		map_octree.place(map, src_min)
		map:enable_tracking()

		local end_async = nil
		if map_octree.tests.async_start then
			end_async = map_octree.tests.async_start("restore_tracking_async_" .. os.time())
		end

		local test_pos = vector.new(minp.x + 3, minp.y + 3, minp.z + 3)
		core.set_node(test_pos, {name = pick_different_node(map, test_pos)})

		map:_verify_chunk_now(1, 1, 1)

		local status = map:get_tracking_status()
		assert(status and status.dirty >= 1, "should have dirty chunks before restore")

		map:schedule_restore(function(ok, err)
			if not ok then
				-- This test intentionally cancels quickly via disable_tracking() below.
				-- Treat cancellation as expected to avoid noisy ERROR logs.
				err = tostring(err or "")
				if err:find("cancel") ~= nil or err:find("disabled") ~= nil then
					if end_async then end_async() end
					return
				end
				core.log("warning", "[test] schedule_restore_async failed: " .. err)
			end
			if end_async then end_async() end
		end)

		map:disable_tracking()
	end)
end)


map_octree.tests.register("restore_tracking:multiple_trackers", function(ctx)
	local minp = vector.new(0, 0, 0)
	local maxp = vector.new(63, 31, 31)
	testutil.with_voxel_region(minp, maxp, function()
		local map1_maxp = vector.new(31, 31, 31)
		local map1 = octmap.new(minp, map1_maxp, {store_chunk_blobs = true, cache_mb = 8})
		local map1_src_min = map1:get_emerged_area()
		map_octree.place(map1, map1_src_min)

		local map2_minp = vector.new(32, 0, 0)
		local map2 = octmap.new(map2_minp, maxp, {store_chunk_blobs = true, cache_mb = 8})
		local map2_src_min = map2:get_emerged_area()
		map_octree.place(map2, map2_src_min)

		map1:enable_tracking()
		map2:enable_tracking()

		assert(map1._tracker_id ~= map2._tracker_id, "tracker ids should be unique")

		local status1 = map1:get_tracking_status()
		local status2 = map2:get_tracking_status()
		assert(status1, "status1 should exist")
		assert(status2, "status2 should exist")

		map1:disable_tracking()

		status1 = map1:get_tracking_status()
		status2 = map2:get_tracking_status()
		assert(status1 == nil, "status1 should be nil after disable")
		assert(status2, "status2 should still exist")

		map2:disable_tracking()
	end)
end)


map_octree.tests.register("restore_tracking:generic_coords", function(ctx)
	local minp = vector.new(100, 200, 50)
	local maxp = vector.new(131, 231, 81)
	testutil.with_voxel_region(minp, maxp, function()
		local map = octmap.new(minp, maxp, {store_chunk_blobs = true, cache_mb = 8})
		local src_min = map:get_emerged_area()
		map_octree.place(map, src_min)
		map:enable_tracking()

		local test_pos = vector.new(minp.x + 5, minp.y + 5, minp.z + 5)
		core.set_node(test_pos, {name = pick_different_node(map, test_pos)})

		map:_verify_chunk_now(1, 1, 1)

		local status = map:get_tracking_status()
		assert(status and status.dirty == 1, "chunk should be dirty after modification (generic coords)")

		local ok, err = map:restore()
		assert(ok, "restore should succeed with generic coords: " .. tostring(err))

		status = map:get_tracking_status()
		assert(status and status.dirty == 0, "no dirty after restore (generic coords)")

		local node = core.get_node(test_pos)
		assert(node.name == "air", "node should be restored to air, got: " .. node.name)

		map:disable_tracking()
	end)
end)


map_octree.tests.register("restore_tracking:pending_queue_hole_regression", function(ctx)
	local minp = vector.new(0, 0, 0)
	local maxp = vector.new(63, 31, 31)
	testutil.with_voxel_region(minp, maxp, function()
		local map = octmap.new(minp, maxp, {store_chunk_blobs = true, cache_mb = 8})
		local src_min = map:get_emerged_area()
		map_octree.place(map, src_min)
		map:enable_tracking()

		-- Enqueue multiple chunks so that popping creates holes in the internal list.
		local coords = {
			{1, 1, 1},
			{2, 1, 1},
			{3, 1, 1},
			{4, 1, 1},
		}

		for i = 1, #coords do
			local c = coords[i]
			map:_enqueue_chunk_for_verification(c[1], c[2], c[3])
		end

		assert(map:_pending_count_for_tests() == #coords, "pending should match enqueued count")

		-- Pop all keys; should get exactly N non-nil keys.
		local popped = 0
		while true do
			local key = map:_pop_pending_key_for_tests()
			if not key then
				break
			end
			popped = popped + 1
			-- Ensure pending count stays consistent while draining.
			local expected_left = #coords - popped
			assert(map:_pending_count_for_tests() == expected_left, "pending count mismatch while draining")
		end

		assert(popped == #coords, "should pop all enqueued keys")
		assert(map:_pending_count_for_tests() == 0, "pending should be empty after draining")
		assert(map:_pop_pending_key_for_tests() == nil, "pop on empty queue should return nil")

		-- Ensure a popped key is truly removed from the pending set (can be re-enqueued).
		map:_enqueue_chunk_for_verification(coords[1][1], coords[1][2], coords[1][3])
		assert(map:_pending_count_for_tests() == 1, "should be able to re-enqueue after pop")
		assert(map:_pop_pending_key_for_tests() ~= nil, "re-enqueued key should be poppable")
		assert(map:_pending_count_for_tests() == 0, "pending should be empty after re-pop")

		map:disable_tracking()
	end)
end)


map_octree.tests.register("restore_tracking:flush_tracking_clears_state", function(ctx)
	local minp = vector.new(0, 0, 0)
	local maxp = vector.new(31, 31, 31)
	testutil.with_voxel_region(minp, maxp, function()
		local map = octmap.new(minp, maxp, {store_chunk_blobs = true, cache_mb = 8})
		local src_min = map:get_emerged_area()
		map_octree.place(map, src_min)
		map:enable_tracking()

		-- Make one dirty tree.
		local test_pos = vector.new(minp.x + 1, minp.y + 1, minp.z + 1)
		core.set_node(test_pos, {name = pick_different_node(map, test_pos)})
		map:_verify_chunk_now(1, 1, 1)

		local status = map:get_tracking_status()
		assert(status and status.dirty == 1, "expected dirty before flush")

		-- Also enqueue something to pending.
		map:_enqueue_chunk_for_verification(1, 1, 1)
		assert(map:_pending_count_for_tests() == 1, "expected pending before flush")

		local info = map:flush_tracking()
		assert(info, "flush_tracking should return info")
		assert(info.pending_cleared >= 1, "flush should clear pending")
		assert(info.dirty_cleared >= 1, "flush should clear dirty")

		status = map:get_tracking_status()
		assert(status and status.pending == 0, "pending should be 0 after flush")
		assert(status and status.dirty == 0, "dirty should be 0 after flush")

		map:disable_tracking()
	end)
end)


map_octree.tests.register("restore_tracking:restore_state_and_cancel", function(ctx)
	local minp = vector.new(0, 0, 0)
	local maxp = vector.new(31, 31, 31)
	testutil.with_voxel_region(minp, maxp, function()
		local map = octmap.new(minp, maxp, {store_chunk_blobs = true, cache_mb = 8})
		map_octree.place(map, minp)
		map:enable_tracking()

		-- Force schedule_restore into the "waiting" phase by adding pending work.
		map:_enqueue_chunk_for_verification(1, 1, 1)
		assert(map:_pending_count_for_tests() == 1, "pending should be 1 before schedule_restore")

		local cb_called = false
		local cb_ok = nil
		local cb_err = nil

		map:schedule_restore(function(ok, err)
			cb_called = true
			cb_ok = ok
			cb_err = err
		end)

		local st = map:get_restore_state()
		assert(st and st.active == true, "restore should be active after schedule_restore")
		assert(st.phase == "waiting", "restore phase should be waiting")
		assert(map:is_restoring() == true, "is_restoring should be true during inflight restore")

		local info = map:flush_tracking()
		assert(info and info.restore_cancelled == true, "flush should cancel inflight restore")
		assert(cb_called == true, "schedule_restore callback should be called on cancel")
		assert(cb_ok == false, "schedule_restore callback should receive ok=false on cancel")
		assert(type(cb_err) == "string" and cb_err:find("cancel") ~= nil, "cancel err should mention cancel")

		st = map:get_restore_state()
		assert(st and st.active == false, "restore should no longer be active after cancel")
		assert(map:is_restoring() == false, "is_restoring should be false after cancel")

		-- Cancel also happens when disabling tracking.
		cb_called = false
		cb_ok = nil
		cb_err = nil
		map:_enqueue_chunk_for_verification(1, 1, 1)
		map:schedule_restore(function(ok, err)
			cb_called = true
			cb_ok = ok
			cb_err = err
		end)

		map:disable_tracking({flush = true})
		assert(cb_called == true, "callback should be called when disabling tracking during inflight restore")
		assert(cb_ok == false, "callback ok should be false when tracking disabled")
		assert(type(cb_err) == "string", "callback err should be a string")
		assert(map:get_tracking_status() == nil, "tracking status should be nil after disable")
	end)
end)