local testutil = dofile(core.get_modpath("map_octree") .. "/src/tests/util.lua")

local async_results = {}

map_octree.tests.register("save_to_file_async restores server_unload_unused_data_timeout after concurrent jobs", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	if map_octree.tests.async_pending and map_octree.tests.async_pending() > 0 then
		core.log("warning", "[test] skipping async restore: async pending=" .. tostring(map_octree.tests.async_pending()))
		return
	end

	local oc = octchunk
	local pos0 = testutil.get_test_region(ctx.player)
	local center = vector.add(oc.snap_to_center(pos0), {x = 0, y = 0, z = 8960})
	local half = oc.SIZE / 2
	local pmin = vector.subtract(center, {x = half, y = half, z = half})
	local pmax = vector.new(
		pmin.x + oc.SIZE * 2 - 1,
		pmin.y + oc.SIZE - 1,
		pmin.z + oc.SIZE - 1
	)

	local original = core.settings:get("server_unload_unused_data_timeout") or "29"
	local reduce_setting = core.settings:get("map_octree_reduce_unload_timeout")
	core.settings:set("map_octree_reduce_unload_timeout", "true")
	local test_id = "save_to_file_async_restore_" .. os.time()
	async_results[test_id] = {status = "pending", start = core.get_us_time()}
	if map_octree.tests.async_begin then
		map_octree.tests.async_begin(test_id)
	end
	local done = false
	local function end_async()
		if done then
			return
		end
		done = true
		if map_octree.tests.async_end then
			map_octree.tests.async_end(test_id)
		end
	end

	-- Ensure the area exists
	core.load_area(pmin, pmax)

	local remaining = 2
	local function on_one_done(ok, result)
		remaining = remaining - 1
		if not ok then
			core.log("error", "[test] " .. test_id .. ": save failed: " .. tostring(result))
		end
		if remaining > 0 then
			return
		end

		core.after(0, function()
			local now = core.settings:get("server_unload_unused_data_timeout") or "29"
			if tostring(now) ~= tostring(original) then
				async_results[test_id].status = "FAIL"
				core.log("error", string.format(
					"[test] %s: FAIL (expected unload_timeout=%s, got %s)",
					test_id, tostring(original), tostring(now)
				))
				end_async()
				if reduce_setting then
					core.settings:set("map_octree_reduce_unload_timeout", reduce_setting)
				else
					core.settings:remove("map_octree_reduce_unload_timeout")
				end
				return
			end

			async_results[test_id].status = "PASS"
			local elapsed_ms = (core.get_us_time() - async_results[test_id].start) / 1000
			core.log("action", string.format("[test] %s: PASS - %.1fms", test_id, elapsed_ms))
			end_async()
			if reduce_setting then
				core.settings:set("map_octree_reduce_unload_timeout", reduce_setting)
			else
				core.settings:remove("map_octree_reduce_unload_timeout")
			end
		end)
	end

	map_octree.save_to_file_async(pmin, pmax, {
		file_name = test_id .. "_a",
		subdir = "tests",
		async_inflight = 1,
		max_voxelmanip_volume = (oc.SIZE + 1) ^ 3,
	}, on_one_done)

	map_octree.save_to_file_async(pmin, pmax, {
		file_name = test_id .. "_b",
		subdir = "tests",
		async_inflight = 1,
		max_voxelmanip_volume = (oc.SIZE + 1) ^ 3,
	}, on_one_done)

	local during = core.settings:get("server_unload_unused_data_timeout") or "29"
	core.log("action", string.format("[test] %s: started (during=%s, original=%s)", test_id, tostring(during), tostring(original)))
end)

map_octree.tests.register("save_to_file_async does not overwrite stored unload timeout", function(ctx)
	assert(ctx and ctx.player, "ctx.player required")
	if map_octree.tests.async_pending and map_octree.tests.async_pending() > 0 then
		core.log("warning", "[test] skipping async storage guard: async pending=" .. tostring(map_octree.tests.async_pending()))
		return
	end

	local oc = octchunk
	local pos0 = testutil.get_test_region(ctx.player)
	local center = vector.add(oc.snap_to_center(pos0), {x = 0, y = 0, z = 9024})
	local half = oc.SIZE / 2
	local pmin = vector.subtract(center, {x = half, y = half, z = half})
	local pmax = vector.new(
		pmin.x + oc.SIZE * 2 - 1,
		pmin.y + oc.SIZE - 1,
		pmin.z + oc.SIZE - 1
	)

	local original = core.settings:get("server_unload_unused_data_timeout") or "29"
	local reduce_setting = core.settings:get("map_octree_reduce_unload_timeout")
	local original_storage = map_octree.storage_get_last_timeout()
	local stored_marker = "123"
	core.settings:set("map_octree_reduce_unload_timeout", "true")
	core.settings:set("server_unload_unused_data_timeout", "1")
	map_octree.storage_set_last_timeout(stored_marker)
	local test_id = "save_to_file_async_storage_guard_" .. os.time()
	if map_octree.tests.async_begin then
		map_octree.tests.async_begin(test_id)
	end
	local done = false
	local function end_async()
		if done then
			return
		end
		done = true
		if map_octree.tests.async_end then
			map_octree.tests.async_end(test_id)
		end
	end

	local function cleanup()
		if reduce_setting then
			core.settings:set("map_octree_reduce_unload_timeout", reduce_setting)
		else
			core.settings:remove("map_octree_reduce_unload_timeout")
		end
		core.settings:set("server_unload_unused_data_timeout", original)
		if original_storage ~= "" then
			map_octree.storage_set_last_timeout(original_storage)
		else
			map_octree.storage_clear_last_timeout()
		end
		core.settings:write()
	end

	core.load_area(pmin, pmax)
	map_octree.save_to_file_async(pmin, pmax, {
		file_name = test_id .. "_a",
		subdir = "tests",
		async_inflight = 1,
		max_voxelmanip_volume = (oc.SIZE + 1) ^ 3,
	}, function(ok, result)
		core.after(0, function()
			if not ok then
				core.log("error", "[test] " .. test_id .. ": save failed: " .. tostring(result))
				end_async()
				cleanup()
				return
			end
			local stored = map_octree.storage_get_last_timeout()
			if stored ~= stored_marker then
				core.log("error", string.format(
					"[test] %s: FAIL (expected storage=%s, got %s)",
					test_id, stored_marker, tostring(stored)
				))
				end_async()
				cleanup()
				return
			end
			core.log("action", string.format("[test] %s: PASS", test_id))
			end_async()
			cleanup()
		end)
	end)

	local during = map_octree.storage_get_last_timeout()
	core.log("action", string.format("[test] %s: started (stored=%s, original=%s)", test_id, tostring(during), tostring(original)))
end)

map_octree.tests.register("restore_unload_timeout_from_storage restores and clears storage", function(ctx)
	assert(ctx, "ctx required")
	local original = core.settings:get("server_unload_unused_data_timeout") or "29"
	local original_storage = map_octree.storage_get_last_timeout()
	local test_id = "restore_unload_timeout_from_storage_" .. os.time()

	core.settings:set("server_unload_unused_data_timeout", "1")
	map_octree.storage_set_last_timeout("27")
	map_octree.restore_unload_timeout_from_storage()

	local now = core.settings:get("server_unload_unused_data_timeout") or "29"
	local stored = map_octree.storage_get_last_timeout()
	local failed = false
	if tostring(now) ~= "27" then
		failed = true
		core.log("error", string.format(
			"[test] %s: FAIL (expected unload_timeout=27, got %s)",
			test_id, tostring(now)
		))
	end
	if stored ~= "" then
		failed = true
		core.log("error", string.format(
			"[test] %s: FAIL (expected storage cleared, got %s)",
			test_id, tostring(stored)
		))
	end
	if not failed then
		core.log("action", string.format("[test] %s: PASS", test_id))
	end

	core.settings:set("server_unload_unused_data_timeout", original)
	if original_storage ~= "" then
		map_octree.storage_set_last_timeout(original_storage)
	else
		map_octree.storage_clear_last_timeout()
	end
	core.settings:write()
end)